import logging, time
from arcnagios.utils import nth

busy_timeout = 10
try:
    import sqlite3
    from sqlite3 import OperationalError
    def dconnect(fp):
        return sqlite3.connect(fp, busy_timeout)
    def dexec(c, stmt):
        return c.execute(stmt)
    def dcommit(c):
        c.commit()
    def dclose(c):
        c.close()
except ImportError:
    import sqlite, random
    from sqlite import DatabaseError as OperationalError
    def dconnect(fp):
        return sqlite.connect(fp, timeout = busy_timeout)
    def dexec(c, stmt):
        # execute seems to be non-blocking even with timeout set.
        for i in range(0, 100*busy_timeout):
            try:
                return c.db.execute(stmt).row_list
            except OperationalError, xc:
                if not 'database is locked' in str(xc):
                    raise xc
                time.sleep(0.01 + random.uniform(-0.002, 0))
    def dcommit(c):
        c.commit()
    def dclose(c):
        c.close()

_default_log = logging.getLogger(__name__)

_create_sql = """\
CREATE TABLE %s (
    n_attempts integer NOT NULL,
    t_sched integer NOT NULL,
    task_type varchar(16) NOT NULL,
    arg text NOT NULL
)"""

def format_time(t):
    return time.strftime('%Y-%m-%d %H:%M', time.localtime(t))

class TaskType(object):
    def __init__(self, h, min_delay = 3600, max_attempts = 12):
	self.handler = h
	self.min_delay = min_delay
	self.max_attempts = max_attempts

class Rescheduler(object):
    def __init__(self, db_path, table_name, log = _default_log):
	self._db = dconnect(db_path)
	self._table = table_name
	self._handlers = {}
	self._log = log
	try:
	    dexec(self._db, _create_sql % self._table)
	except OperationalError:
	    pass

    def close(self):
	self._db.close()

    def _update(self, stmt):
	r = dexec(self._db, stmt)
	dcommit(self._db)
	return r

    def _query(self, stmt):
	return dexec(self._db, stmt)

    def register(self, task_type, h, min_delay = 3600, max_attempts = 12):
	self._handlers[task_type] = TaskType(h, min_delay, max_attempts)

    def schedule(self, task_type, arg, n_attempts = 0):
	t_sched = time.time() + self._handlers[task_type].min_delay
	self._update('INSERT INTO %s (n_attempts, t_sched, task_type, arg) '
		     'VALUES (%d, %d, "%s", "%s")'
		     % (self._table, n_attempts, t_sched, task_type, arg))

    def call(self, task_type, arg):
	if self._handlers[task_type].handler(arg, 0):
	    return True
	else:
	    self.schedule(task_type, arg, n_attempts = 1)
	    return False

    def run(self):
	t_now = time.time()
	r = self._query('SELECT ROWID, n_attempts, t_sched, task_type, arg '
			'FROM %s WHERE t_sched <= %d'
			% (self._table, t_now))
	for id, n_attempts, t_sched, task_type_name, arg in r:
	    if not task_type_name in self._handlers:
		self._log.warning('No task type %s.' % task_type_name)
		continue
	    task_type = self._handlers[task_type_name]
	    try:
		ok = task_type.handler(arg, n_attempts)
	    except Exception, xc:
		self._log.error('Task %s(%r) raised exception: %s'
				% (task_type_name, arg, xc))
		ok = False
	    if ok or n_attempts == task_type.max_attempts:
		self._log.info('Unscheduling %s(%r)' % (task_type_name, arg))
		self._update('DELETE FROM %s WHERE ROWID = %d'
			      % (self._table, id))
	    else:
		t_sched = t_now + (task_type.min_delay << n_attempts)
		n_attempts += 1
		self._log.info('Scheduling %s attempt at %s to %s(%r)'
			       % (nth(n_attempts), format_time(t_sched),
				  task_type_name, arg))
		self._update('UPDATE %s SET n_attempts = %d, t_sched = %d '
			     'WHERE ROWID = %d'
			     % (self._table, n_attempts, t_sched, id))

    def run_and_close(self):
	self.run()
	self.close()
