/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( '@stdlib/math/base/assert/is-nan' );
var pow = require( '@stdlib/math/base/special/pow' );
var SQRT2 = require( '@stdlib/constants/float64/sqrt-two' );


// MAIN //

/**
* Returns the skewness of a triangular distribution.
*
* @param {number} a - minimum support
* @param {number} b - maximum support
* @param {number} c - mode
* @returns {number} skewness
*
* @example
* var v = skewness( 0.0, 1.0, 0.5 );
* // returns 0.0
*
* @example
* var v = skewness( 4.0, 12.0, 9.0 );
* // returns ~-0.236
*
* @example
* var v = skewness( -4.0, 4.0, -1.0 );
* // returns ~0.236
*
* @example
* var v = skewness( 1.0, -0.1, 0.5 );
* // returns NaN
*
* @example
* var v = skewness( 0.0, 1.0, 2.0 );
* // returns NaN
*
* @example
* var v = skewness( NaN, 4.0, 2.0 );
* // returns NaN
*
* @example
* var v = skewness( 0.0, NaN, 2.0 );
* // returns NaN
*
* @example
* var v = skewness( 0.0, 4.0, NaN );
* // returns NaN
*/
function skewness( a, b, c ) {
	var out;
	if (
		isnan( a ) ||
		isnan( b ) ||
		isnan( c ) ||
		!( a <= c && c <= b )
	) {
		return NaN;
	}
	out = SQRT2 * ( a + b - (2*c) ) * ( (2*a) - b - c ) * ( a - (2*b) + c );
	out /= 5.0 * pow( (a*a) + (b*b) + (c*c) - (a*b) - (a*c) - (b*c), 1.5 );
	return out;
}


// EXPORTS //

module.exports = skewness;
