/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright(c) 2016-2019 Intel Corporation
 */

#ifndef TEST_CRYPTODEV_AES_TEST_VECTORS_H_
#define TEST_CRYPTODEV_AES_TEST_VECTORS_H_

/* test vectors */
static const uint8_t plaintext_aes128ctr[] = {
	0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
	0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A,
	0xAE, 0x2D, 0x8A, 0x57, 0x1E, 0x03, 0xAC, 0x9C,
	0x9E, 0xB7, 0x6F, 0xAC, 0x45, 0xAF, 0x8E, 0x51,
	0x30, 0xC8, 0x1C, 0x46, 0xA3, 0x5C, 0xE4, 0x11,
	0xE5, 0xFB, 0xC1, 0x19, 0x1A, 0x0A, 0x52, 0xEF,
	0xF6, 0x9F, 0x24, 0x45, 0xDF, 0x4F, 0x9B, 0x17,
	0xAD, 0x2B, 0x41, 0x7B, 0xE6, 0x6C, 0x37, 0x10
};

static const uint8_t ciphertext64_aes128ctr[] = {
	0x87, 0x4D, 0x61, 0x91, 0xB6, 0x20, 0xE3, 0x26,
	0x1B, 0xEF, 0x68, 0x64, 0x99, 0x0D, 0xB6, 0xCE,
	0x98, 0x06, 0xF6, 0x6B, 0x79, 0x70, 0xFD, 0xFF,
	0x86, 0x17, 0x18, 0x7B, 0xB9, 0xFF, 0xFD, 0xFF,
	0x5A, 0xE4, 0xDF, 0x3E, 0xDB, 0xD5, 0xD3, 0x5E,
	0x5B, 0x4F, 0x09, 0x02, 0x0D, 0xB0, 0x3E, 0xAB,
	0x1E, 0x03, 0x1D, 0xDA, 0x2F, 0xBE, 0x03, 0xD1,
	0x79, 0x21, 0x70, 0xA0, 0xF3, 0x00, 0x9C, 0xEE
};

static const uint8_t ciphertext64_aes128ctr_IV_12bytes[] = {
	0x28, 0x80, 0x28, 0xC7, 0x15, 0x99, 0xC5, 0xA8,
	0xDD, 0x53, 0xC2, 0x67, 0x1B, 0x86, 0xB8, 0x13,
	0xAB, 0x25, 0x39, 0x7A, 0xD2, 0x1F, 0x8B, 0x4B,
	0x94, 0x89, 0x2B, 0x65, 0xCF, 0x89, 0x1E, 0xDD,
	0xD4, 0x7C, 0xFD, 0x8D, 0x0E, 0xCD, 0x23, 0xA4,
	0xEB, 0x8C, 0x05, 0x58, 0x45, 0x4A, 0x63, 0x44,
	0x11, 0x42, 0x07, 0x17, 0xB4, 0xD2, 0xCC, 0x75,
	0xB7, 0x23, 0x99, 0xA9, 0xC5, 0x89, 0x7F, 0x66
};

static const uint8_t plaintext_aes_docsis_bpi_cfb[] = {
	0x00, 0x01, 0x02, 0x88, 0xEE, 0x59, 0x7E
};

static const uint8_t ciphertext_aes_docsis_bpi_cfb[] = {
	0xFC, 0x68, 0xA3, 0x55, 0x60, 0x37, 0xDC
};

static const uint8_t plaintext_aes_docsis_bpi_cbc_cfb[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x91,
	0xD2, 0xD1, 0x9F
};

static const uint8_t ciphertext_aes_docsis_bpi_cbc_cfb[] = {
	0x9D, 0xD1, 0x67, 0x4B, 0xBA, 0x61, 0x10, 0x1B,
	0x56, 0x75, 0x64, 0x74, 0x36, 0x4F, 0x10, 0x1D,
	0x44, 0xD4, 0x73
};

static const uint8_t plaintext_aes192ctr[] = {
	0x01, 0x0F, 0x10, 0x1F, 0x20, 0x1C, 0x0E, 0xB8,
	0xFB, 0x5C, 0xCD, 0xCC, 0x1F, 0xF9, 0xAF, 0x0B,
	0x95, 0x03, 0x74, 0x99, 0x49, 0xE7, 0x62, 0x55,
	0xDA, 0xEA, 0x13, 0x20, 0x1D, 0xC6, 0xCC, 0xCC,
	0xD1, 0x70, 0x75, 0x47, 0x02, 0x2F, 0xFB, 0x86,
	0xBB, 0x6B, 0x23, 0xD2, 0xC9, 0x74, 0xD7, 0x7B,
	0x08, 0x03, 0x3B, 0x79, 0x39, 0xBB, 0x91, 0x29,
	0xDA, 0x14, 0x39, 0x8D, 0xFF, 0x81, 0x50, 0x96,
};

static const uint8_t ciphertext64_aes192ctr[] = {
	0x4A, 0x6C, 0xC8, 0xCC, 0x96, 0x2A, 0x13, 0x84,
	0x1C, 0x36, 0x88, 0xE9, 0xE5, 0x94, 0x70, 0xB2,
	0x14, 0x5B, 0x13, 0x80, 0xEA, 0xD8, 0x8D, 0x37,
	0xFD, 0x70, 0xA8, 0x83, 0xE8, 0x2B, 0x88, 0x1E,
	0xBA, 0x94, 0x3F, 0xF6, 0xB3, 0x1F, 0xDE, 0x34,
	0xF3, 0x5B, 0x80, 0xE9, 0xAB, 0xF5, 0x1C, 0x29,
	0xB6, 0xD9, 0x76, 0x2B, 0x06, 0xC6, 0x74, 0xF1,
	0x59, 0x5E, 0x9E, 0xA5, 0x7B, 0x2D, 0xD7, 0xF0
};

static const uint8_t ciphertext64_aes192ctr_IV_12bytes[] = {
	0x67, 0x65, 0xa9, 0xee, 0xfd, 0x31, 0x62, 0xfc,
	0xad, 0xfd, 0xc7, 0x25, 0xb7, 0x25, 0x16, 0xbe,
	0x25, 0xce, 0xc0, 0x1d, 0xda, 0xa9, 0xd3, 0xda,
	0x1b, 0x7d, 0x68, 0x6a, 0x6f, 0x06, 0xea, 0x47,
	0xa0, 0xe0, 0x15, 0xf4, 0xbd, 0x1b, 0x70, 0x34,
	0xd4, 0x6d, 0x1c, 0x84, 0x17, 0x91, 0x46, 0x0c,
	0xe8, 0xbc, 0x7a, 0xfb, 0x9f, 0x2a, 0x8f, 0xb4,
	0xd4, 0xf3, 0x6e, 0x5b, 0x75, 0xa0, 0xce, 0x32
};

static const uint8_t plaintext_aes256ctr[] = {
	0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
	0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A,
	0xAE, 0x2D, 0x8A, 0x57, 0x1E, 0x03, 0xAC, 0x9C,
	0x9E, 0xB7, 0x6F, 0xAC, 0x45, 0xAF, 0x8E, 0x51,
	0x30, 0xC8, 0x1C, 0x46, 0xA3, 0x5C, 0xE4, 0x11,
	0xE5, 0xFB, 0xC1, 0x19, 0x1A, 0x0A, 0x52, 0xEF,
	0xF6, 0x9F, 0x24, 0x45, 0xDF, 0x4F, 0x9B, 0x17,
	0xAD, 0x2B, 0x41, 0x7B, 0xE6, 0x6C, 0x37, 0x10
};

static const uint8_t ciphertext64_aes256ctr[] = {
	0x60, 0x1E, 0xC3, 0x13, 0x77, 0x57, 0x89, 0xA5,
	0xB7, 0xA7, 0xF5, 0x04, 0xBB, 0xF3, 0xD2, 0x28,
	0xF4, 0x43, 0xE3, 0xCA, 0x4D, 0x62, 0xB5, 0x9A,
	0xCA, 0x84, 0xE9, 0x90, 0xCA, 0xCA, 0xF5, 0xC5,
	0x2B, 0x09, 0x30, 0xDA, 0xA2, 0x3D, 0xE9, 0x4C,
	0xE8, 0x70, 0x17, 0xBA, 0x2D, 0x84, 0x98, 0x8D,
	0xDF, 0xC9, 0xC5, 0x8D, 0xB6, 0x7A, 0xAD, 0xA6,
	0x13, 0xC2, 0xDD, 0x08, 0x45, 0x79, 0x41, 0xA6
};

static const uint8_t ciphertext64_aes256ctr_IV_12bytes[] = {
	0x7B, 0x7A, 0x7D, 0x83, 0x85, 0xF8, 0x81, 0xF3,
	0x32, 0x33, 0xD9, 0xFB, 0x04, 0x73, 0xD4, 0x2F,
	0x70, 0xDE, 0x90, 0x3E, 0xD0, 0xA9, 0x93, 0x8A,
	0x91, 0xF3, 0xB5, 0x29, 0x4D, 0x2A, 0x74, 0xD0,
	0xDC, 0x4E, 0x5C, 0x9B, 0x97, 0x24, 0xD8, 0x02,
	0xFE, 0xAB, 0x38, 0xE8, 0x73, 0x51, 0x29, 0x7E,
	0xF1, 0xF9, 0x40, 0x78, 0xB1, 0x04, 0x7A, 0x78,
	0x61, 0x07, 0x47, 0xE6, 0x8C, 0x0F, 0xA8, 0x76
};

static const uint8_t plaintext_aes_common[] = {
	"What a lousy earth! He wondered how many people "
	"were destitute that same night even in his own "
	"prosperous country, how many homes were "
	"shanties, how many husbands were drunk and "
	"wives socked, and how many children were "
	"bullied, abused, or abandoned. How many "
	"families hungered for food they could not "
	"afford to buy? How many hearts were broken? How "
	"many suicides would take place that same night, "
	"how many people would go insane? How many "
	"cockroaches and landlords would triumph? How "
	"many winners were losers, successes failures, "
	"and rich men poor men? How many wise guys were "
	"stupid? How many happy endings were unhappy "
	"endings? How many honest men were liars, brave "
	"men cowards, loyal men traitors, how many "
	"sainted men were corrupt, how many people in "
	"positions of trust had sold their souls to "
	"bodyguards, how many had never had souls? How "
	"many straight-and-narrow paths were crooked "
	"paths? How many best families were worst "
	"families and how many good people were bad "
	"people? When you added them all up and then "
	"subtracted, you might be left with only the "
	"children, and perhaps with Albert Einstein and "
	"an old violinist or sculptor somewhere."
};

static const uint8_t ciphertext512_aes128cbc[] = {
	0x8B, 0x4D, 0xDA, 0x1B, 0xCF, 0x04, 0xA0, 0x31,
	0xB4, 0xBF, 0xBD, 0x68, 0x43, 0x20, 0x7E, 0x76,
	0xB1, 0x96, 0x8B, 0xA2, 0x7C, 0xA2, 0x83, 0x9E,
	0x39, 0x5A, 0x2F, 0x7E, 0x92, 0xB4, 0x48, 0x1A,
	0x3F, 0x6B, 0x5D, 0xDF, 0x52, 0x85, 0x5F, 0x8E,
	0x42, 0x3C, 0xFB, 0xE9, 0x1A, 0x24, 0xD6, 0x08,
	0xDD, 0xFD, 0x16, 0xFB, 0xE9, 0x55, 0xEF, 0xF0,
	0xA0, 0x8D, 0x13, 0xAB, 0x81, 0xC6, 0x90, 0x01,
	0xB5, 0x18, 0x84, 0xB3, 0xF6, 0xE6, 0x11, 0x57,
	0xD6, 0x71, 0xC6, 0x3C, 0x3F, 0x2F, 0x33, 0xEE,
	0x24, 0x42, 0x6E, 0xAC, 0x0B, 0xCA, 0xEC, 0xF9,
	0x84, 0xF8, 0x22, 0xAA, 0x60, 0xF0, 0x32, 0xA9,
	0x75, 0x75, 0x3B, 0xCB, 0x70, 0x21, 0x0A, 0x8D,
	0x0F, 0xE0, 0xC4, 0x78, 0x2B, 0xF8, 0x97, 0xE3,
	0xE4, 0x26, 0x4B, 0x29, 0xDA, 0x88, 0xCD, 0x46,
	0xEC, 0xAA, 0xF9, 0x7F, 0xF1, 0x15, 0xEA, 0xC3,
	0x87, 0xE6, 0x31, 0xF2, 0xCF, 0xDE, 0x4D, 0x80,
	0x70, 0x91, 0x7E, 0x0C, 0xF7, 0x26, 0x3A, 0x92,
	0x4F, 0x18, 0x83, 0xC0, 0x8F, 0x59, 0x01, 0xA5,
	0x88, 0xD1, 0xDB, 0x26, 0x71, 0x27, 0x16, 0xF5,
	0xEE, 0x10, 0x82, 0xAC, 0x68, 0x26, 0x9B, 0xE2,
	0x6D, 0xD8, 0x9A, 0x80, 0xDF, 0x04, 0x31, 0xD5,
	0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
	0x58, 0x34, 0x85, 0x61, 0x1C, 0x42, 0x10, 0x76,
	0x73, 0x02, 0x42, 0xC9, 0x23, 0x18, 0x8E, 0xB4,
	0x6F, 0xB4, 0xA3, 0x54, 0x6E, 0x88, 0x3B, 0x62,
	0x7C, 0x02, 0x8D, 0x4C, 0x9F, 0xC8, 0x45, 0xF4,
	0xC9, 0xDE, 0x4F, 0xEB, 0x22, 0x83, 0x1B, 0xE4,
	0x49, 0x37, 0xE4, 0xAD, 0xE7, 0xCD, 0x21, 0x54,
	0xBC, 0x1C, 0xC2, 0x04, 0x97, 0xB4, 0x10, 0x61,
	0xF0, 0xE4, 0xEF, 0x27, 0x63, 0x3A, 0xDA, 0x91,
	0x41, 0x25, 0x62, 0x1C, 0x5C, 0xB6, 0x38, 0x4A,
	0x88, 0x71, 0x59, 0x5A, 0x8D, 0xA0, 0x09, 0xAF,
	0x72, 0x94, 0xD7, 0x79, 0x5C, 0x60, 0x7C, 0x8F,
	0x4C, 0xF5, 0xD9, 0xA1, 0x39, 0x6D, 0x81, 0x28,
	0xEF, 0x13, 0x28, 0xDF, 0xF5, 0x3E, 0xF7, 0x8E,
	0x09, 0x9C, 0x78, 0x18, 0x79, 0xB8, 0x68, 0xD7,
	0xA8, 0x29, 0x62, 0xAD, 0xDE, 0xE1, 0x61, 0x76,
	0x1B, 0x05, 0x16, 0xCD, 0xBF, 0x02, 0x8E, 0xA6,
	0x43, 0x6E, 0x92, 0x55, 0x4F, 0x60, 0x9C, 0x03,
	0xB8, 0x4F, 0xA3, 0x02, 0xAC, 0xA8, 0xA7, 0x0C,
	0x1E, 0xB5, 0x6B, 0xF8, 0xC8, 0x4D, 0xDE, 0xD2,
	0xB0, 0x29, 0x6E, 0x40, 0xE6, 0xD6, 0xC9, 0xE6,
	0xB9, 0x0F, 0xB6, 0x63, 0xF5, 0xAA, 0x2B, 0x96,
	0xA7, 0x16, 0xAC, 0x4E, 0x0A, 0x33, 0x1C, 0xA6,
	0xE6, 0xBD, 0x8A, 0xCF, 0x40, 0xA9, 0xB2, 0xFA,
	0x63, 0x27, 0xFD, 0x9B, 0xD9, 0xFC, 0xD5, 0x87,
	0x8D, 0x4C, 0xB6, 0xA4, 0xCB, 0xE7, 0x74, 0x55,
	0xF4, 0xFB, 0x41, 0x25, 0xB5, 0x4B, 0x0A, 0x1B,
	0xB1, 0xD6, 0xB7, 0xD9, 0x47, 0x2A, 0xC3, 0x98,
	0x6A, 0xC4, 0x03, 0x73, 0x1F, 0x93, 0x6E, 0x53,
	0x19, 0x25, 0x64, 0x15, 0x83, 0xF9, 0x73, 0x2A,
	0x74, 0xB4, 0x93, 0x69, 0xC4, 0x72, 0xFC, 0x26,
	0xA2, 0x9F, 0x43, 0x45, 0xDD, 0xB9, 0xEF, 0x36,
	0xC8, 0x3A, 0xCD, 0x99, 0x9B, 0x54, 0x1A, 0x36,
	0xC1, 0x59, 0xF8, 0x98, 0xA8, 0xCC, 0x28, 0x0D,
	0x73, 0x4C, 0xEE, 0x98, 0xCB, 0x7C, 0x58, 0x7E,
	0x20, 0x75, 0x1E, 0xB7, 0xC9, 0xF8, 0xF2, 0x0E,
	0x63, 0x9E, 0x05, 0x78, 0x1A, 0xB6, 0xA8, 0x7A,
	0xF9, 0x98, 0x6A, 0xA6, 0x46, 0x84, 0x2E, 0xF6,
	0x4B, 0xDC, 0x9B, 0x8F, 0x9B, 0x8F, 0xEE, 0xB4,
	0xAA, 0x3F, 0xEE, 0xC0, 0x37, 0x27, 0x76, 0xC7,
	0x95, 0xBB, 0x26, 0x74, 0x69, 0x12, 0x7F, 0xF1,
	0xBB, 0xFF, 0xAE, 0xB5, 0x99, 0x6E, 0xCB, 0x0C
};

/* NULL cipher NULL auth 8-byte multiple test vector */
static const struct blockcipher_test_data null_test_data_chain_x8_multiple = {
	.crypto_algo = RTE_CRYPTO_CIPHER_NULL,
	.cipher_key = {	/* arbitrary data - shouldn't be used */
		.data = {
			0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
			0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
		},
		.len = 16
	},
	.iv = {			/* arbitrary data - shouldn't be used */
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_NULL,
	.auth_key = {		/* arbitrary data - shouldn't be used */
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00
		},
		.len = 20,
		.truncated_len = 12
	}
};

/* NULL cipher NULL auth 4-byte multiple test vector */
static const struct blockcipher_test_data null_test_data_chain_x4_multiple = {
	.crypto_algo = RTE_CRYPTO_CIPHER_NULL,
	.cipher_key = {	/* arbitrary data - shouldn't be used */
		.data = {
			0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
			0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
		},
		.len = 16
	},
	.iv = {			/* arbitrary data - shouldn't be used */
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes128ctr,
		.len = 20
	},
	.ciphertext = {
		.data = plaintext_aes128ctr,
		.len = 20
	},
	.auth_algo = RTE_CRYPTO_AUTH_NULL,
	.auth_key = {		/* arbitrary data - shouldn't be used */
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00
		},
		.len = 20,
		.truncated_len = 12
	}
};

/* NULL cipher NULL auth 1-byte multiple test vector */
static const struct blockcipher_test_data null_test_data_chain_x1_multiple = {
	.crypto_algo = RTE_CRYPTO_CIPHER_NULL,
	.cipher_key = {	/* arbitrary data - shouldn't be used */
		.data = {
			0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
			0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
		},
		.len = 16
	},
	.iv = {			/* arbitrary data - shouldn't be used */
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes128ctr,
		.len = 21
	},
	.ciphertext = {
		.data = plaintext_aes128ctr,
		.len = 21
	},
	.auth_algo = RTE_CRYPTO_AUTH_NULL,
	.auth_key = {		/* arbitrary data - shouldn't be used */
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00
		},
		.len = 20,
		.truncated_len = 12
	}
};

static const uint8_t ciphertext512_aes128cbc_aad[] = {
	0x57, 0x68, 0x61, 0x74, 0x20, 0x61, 0x20, 0x6C,
	0x6F, 0x75, 0x73, 0x79, 0x6D, 0x70, 0xB4, 0xAD,
	0x09, 0x7C, 0xD7, 0x52, 0xD6, 0xF2, 0xBF, 0xD1,
	0x9D, 0x79, 0xC6, 0xB6, 0x8F, 0x94, 0xEB, 0xD8,
	0xBA, 0x5E, 0x01, 0x49, 0x7D, 0xB3, 0xC5, 0xFE,
	0x18, 0xF4, 0xE3, 0x60, 0x8C, 0x84, 0x68, 0x13,
	0x33, 0x06, 0x85, 0x60, 0xD3, 0xE7, 0x8A, 0xB5,
	0x23, 0xA2, 0xDE, 0x52, 0x5C, 0xB6, 0x26, 0x37,
	0xBB, 0x23, 0x8A, 0x38, 0x07, 0x85, 0xB6, 0x2E,
	0xC3, 0x69, 0x57, 0x79, 0x6B, 0xE4, 0xD7, 0x86,
	0x23, 0x72, 0x4C, 0x65, 0x49, 0x08, 0x1E, 0xF3,
	0xCC, 0x71, 0x4C, 0x45, 0x97, 0x03, 0xBC, 0xA0,
	0x9D, 0xF0, 0x4F, 0x5D, 0xEC, 0x40, 0x6C, 0xC6,
	0x52, 0xC0, 0x9D, 0x1C, 0xDC, 0x8B, 0xC2, 0xFA,
	0x35, 0xA7, 0x3A, 0x00, 0x04, 0x1C, 0xA6, 0x91,
	0x5D, 0xEB, 0x07, 0xA1, 0xB9, 0x3E, 0xD1, 0xB6,
	0xCA, 0x96, 0xEC, 0x71, 0xF7, 0x7D, 0xB6, 0x09,
	0x3D, 0x19, 0x6E, 0x75, 0x03, 0xC3, 0x1A, 0x4E,
	0x5B, 0x4D, 0xEA, 0xD9, 0x92, 0x96, 0x01, 0xFB,
	0xA3, 0xC2, 0x6D, 0xC4, 0x17, 0x6B, 0xB4, 0x3B,
	0x1E, 0x87, 0x54, 0x26, 0x95, 0x63, 0x07, 0x73,
	0xB6, 0xBA, 0x52, 0xD7, 0xA7, 0xD0, 0x9C, 0x75,
	0x8A, 0xCF, 0xC4, 0x3C, 0x4A, 0x55, 0x0E, 0x53,
	0xEC, 0xE0, 0x31, 0x51, 0xB7, 0xB7, 0xD2, 0xB4,
	0xF3, 0x2B, 0x70, 0x6D, 0x15, 0x9E, 0x57, 0x30,
	0x72, 0xE5, 0xA4, 0x71, 0x5F, 0xA4, 0xE8, 0x7C,
	0x46, 0x58, 0x36, 0x71, 0x91, 0x55, 0xAA, 0x99,
	0x3B, 0x3F, 0xF6, 0xA2, 0x9D, 0x27, 0xBF, 0xC2,
	0x62, 0x2C, 0x85, 0xB7, 0x51, 0xDD, 0xFD, 0x7B,
	0x8B, 0xB5, 0xDD, 0x2A, 0x73, 0xF8, 0x93, 0x9A,
	0x3F, 0xAD, 0x1D, 0xF0, 0x46, 0xD1, 0x76, 0x83,
	0x71, 0x4E, 0xD3, 0x0D, 0x64, 0x8C, 0xC3, 0xE6,
	0x03, 0xED, 0xE8, 0x53, 0x23, 0x1A, 0xC7, 0x86,
	0xEB, 0x87, 0xD6, 0x78, 0xF9, 0xFB, 0x9C, 0x1D,
	0xE7, 0x4E, 0xC0, 0x70, 0x27, 0x7A, 0x43, 0xE2,
	0x5D, 0xA4, 0x10, 0x40, 0xBE, 0x61, 0x0D, 0x2B,
	0x25, 0x08, 0x75, 0x91, 0xB5, 0x5A, 0x26, 0xC8,
	0x32, 0xA7, 0xC6, 0x88, 0xBF, 0x75, 0x94, 0xCC,
	0x58, 0xA4, 0xFE, 0x2F, 0xF7, 0x5C, 0xD2, 0x36,
	0x66, 0x55, 0xF0, 0xEA, 0xF5, 0x64, 0x43, 0xE7,
	0x6D, 0xE0, 0xED, 0xA1, 0x10, 0x0A, 0x84, 0x07,
	0x11, 0x88, 0xFA, 0xA1, 0xD3, 0xA0, 0x00, 0x5D,
	0xEB, 0xB5, 0x62, 0x01, 0x72, 0xC1, 0x9B, 0x39,
	0x0B, 0xD3, 0xAF, 0x04, 0x19, 0x42, 0xEC, 0xFF,
	0x4B, 0xB3, 0x5E, 0x87, 0x27, 0xE4, 0x26, 0x57,
	0x76, 0xCD, 0x36, 0x31, 0x5B, 0x94, 0x74, 0xFF,
	0x33, 0x91, 0xAA, 0xD1, 0x45, 0x34, 0xC2, 0x11,
	0xF0, 0x35, 0x44, 0xC9, 0xD5, 0xA2, 0x5A, 0xC2,
	0xE9, 0x9E, 0xCA, 0xE2, 0x6F, 0xD2, 0x40, 0xB4,
	0x93, 0x42, 0x78, 0x20, 0x92, 0x88, 0xC7, 0x16,
	0xCF, 0x15, 0x54, 0x7B, 0xE1, 0x46, 0x38, 0x69,
	0xB8, 0xE4, 0xF1, 0x81, 0xF0, 0x08, 0x6F, 0x92,
	0x6D, 0x1A, 0xD9, 0x93, 0xFA, 0xD7, 0x35, 0xFE,
	0x7F, 0x59, 0x43, 0x1D, 0x3A, 0x3B, 0xFC, 0xD0,
	0x14, 0x95, 0x1E, 0xB2, 0x04, 0x08, 0x4F, 0xC6,
	0xEA, 0xE8, 0x22, 0xF3, 0xD7, 0x66, 0x93, 0xAA,
	0xFD, 0xA0, 0xFE, 0x03, 0x96, 0x54, 0x78, 0x35,
	0x18, 0xED, 0xB7, 0x2F, 0x40, 0xE3, 0x8E, 0x22,
	0xC6, 0xDA, 0xB0, 0x8E, 0xA0, 0xA1, 0x62, 0x03,
	0x63, 0x34, 0x11, 0xF5, 0x9E, 0xAA, 0x6B, 0xC4,
	0x14, 0x75, 0x4C, 0xF4, 0xD8, 0xD9, 0xF1, 0x76,
	0xE3, 0xD3, 0x55, 0xCE, 0x22, 0x7D, 0x4A, 0xB7,
	0xBB, 0x7F, 0x4F, 0x09, 0x88, 0x70, 0x6E, 0x09,
	0x84, 0x6B, 0x24, 0x19, 0x2C, 0x20, 0x73, 0x75
};

/* AES128-CTR-SHA1 test vector */
static const struct blockcipher_test_data aes_test_data_1 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CTR,
	.cipher_key = {
		.data = {
			0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
			0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes128ctr,
		.len = 64
	},
	.ciphertext = {
		.data = ciphertext64_aes128ctr,
		.len = 64
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x9B, 0x6F, 0x0C, 0x43, 0xF5, 0xC1, 0x3E, 0xB0,
			0xB1, 0x70, 0xB8, 0x2B, 0x33, 0x09, 0xD2, 0xB2,
			0x56, 0x20, 0xFB, 0xFE
		},
		.len = 20,
		.truncated_len = 12
	}
};

/** AES-192-CTR XCBC test vector */
static const struct blockcipher_test_data aes_test_data_2 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CTR,
	.cipher_key = {
		.data = {
			0xCB, 0xC5, 0xED, 0x5B, 0xE7, 0x7C, 0xBD, 0x8C,
			0x50, 0xD9, 0x30, 0xF2, 0xB5, 0x6A, 0x0E, 0x5F,
			0xAA, 0xAE, 0xAD, 0xA2, 0x1F, 0x49, 0x52, 0xD4
		},
		.len = 24
	},
	.iv = {
		.data = {
			0x3F, 0x69, 0xA8, 0xCD, 0xE8, 0xF0, 0xEF, 0x40,
			0xB8, 0x7A, 0x4B, 0xED, 0x2B, 0xAF, 0xBF, 0x57
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes192ctr,
		.len = 64
	},
	.ciphertext = {
		.data = ciphertext64_aes192ctr,
		.len = 64
	},
	.auth_algo = RTE_CRYPTO_AUTH_AES_XCBC_MAC,
	.auth_key = {
		.data = {
			0x87, 0x61, 0x54, 0x53, 0xC4, 0x6D, 0xDD, 0x51,
			0xE1, 0x9F, 0x86, 0x64, 0x39, 0x0A, 0xE6, 0x59
		},
		.len = 16
	},
	.digest = {
		.data = {
			0xCA, 0x33, 0xB3, 0x3B, 0x16, 0x94, 0xAA, 0x55,
			0x36, 0x6B, 0x45, 0x46
		},
		.len = 12,
		.truncated_len = 12
	}
};

/** AES-256-CTR SHA1 test vector */
static const struct blockcipher_test_data aes_test_data_3 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CTR,
	.cipher_key = {
		.data = {
			0x60, 0x3D, 0xEB, 0x10, 0x15, 0xCA, 0x71, 0xBE,
			0x2B, 0x73, 0xAE, 0xF0, 0x85, 0x7D, 0x77, 0x81,
			0x1F, 0x35, 0x2C, 0x07, 0x3B, 0x61, 0x08, 0xD7,
			0x2D, 0x98, 0x10, 0xA3, 0x09, 0x14, 0xDF, 0xF4
		},
		.len = 32
	},
	.iv = {
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes256ctr,
		.len = 64
	},
	.ciphertext = {
		.data = ciphertext64_aes256ctr,
		.len = 64
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x3B, 0x1A, 0x9D, 0x82, 0x35, 0xD5, 0xDD, 0x64,
			0xCC, 0x1B, 0xA9, 0xC0, 0xEB, 0xE9, 0x42, 0x16,
			0xE7, 0x87, 0xA3, 0xEF
		},
		.len = 20,
		.truncated_len = 12
	}
};

/* AES128-CTR-SHA1 test vector (12-byte IV) */
static const struct blockcipher_test_data aes_test_data_1_IV_12_bytes = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CTR,
	.cipher_key = {
		.data = {
			0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
			0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB
		},
		.len = 12
	},
	.plaintext = {
		.data = plaintext_aes128ctr,
		.len = 64
	},
	.ciphertext = {
		.data = ciphertext64_aes128ctr_IV_12bytes,
		.len = 64
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x5C, 0x34, 0x6B, 0xE4, 0x9A, 0x7F, 0x4A, 0xC3,
			0x82, 0xBE, 0xA0, 0x12, 0xD1, 0xF0, 0x15, 0xFA,
			0xCF, 0xC8, 0x7F, 0x60
		},
		.len = 20,
		.truncated_len = 12
	}
};

/** AES-192-CTR XCBC test vector (12-byte IV) */
static const struct blockcipher_test_data aes_test_data_2_IV_12_bytes = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CTR,
	.cipher_key = {
		.data = {
			0xCB, 0xC5, 0xED, 0x5B, 0xE7, 0x7C, 0xBD, 0x8C,
			0x50, 0xD9, 0x30, 0xF2, 0xB5, 0x6A, 0x0E, 0x5F,
			0xAA, 0xAE, 0xAD, 0xA2, 0x1F, 0x49, 0x52, 0xD4
		},
		.len = 24
	},
	.iv = {
		.data = {
			0x3F, 0x69, 0xA8, 0xCD, 0xE8, 0xF0, 0xEF, 0x40,
			0xB8, 0x7A, 0x4B, 0xED
		},
		.len = 12
	},
	.plaintext = {
		.data = plaintext_aes192ctr,
		.len = 64
	},
	.ciphertext = {
		.data = ciphertext64_aes192ctr_IV_12bytes,
		.len = 64
	},
	.auth_algo = RTE_CRYPTO_AUTH_AES_XCBC_MAC,
	.auth_key = {
		.data = {
			0x87, 0x61, 0x54, 0x53, 0xC4, 0x6D, 0xDD, 0x51,
			0xE1, 0x9F, 0x86, 0x64, 0x39, 0x0A, 0xE6, 0x59
		},
		.len = 16
	},
	.digest = {
		.data = {
			0x0C, 0xA1, 0xA5, 0xAF, 0x3E, 0x41, 0xD2, 0xF4,
			0x4C, 0x4C, 0xAB, 0x13
		},
		.len = 12,
		.truncated_len = 12
	}
};

/** AES-256-CTR SHA1 test vector (12-byte IV) */
static const struct blockcipher_test_data aes_test_data_3_IV_12_bytes = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CTR,
	.cipher_key = {
		.data = {
			0x60, 0x3D, 0xEB, 0x10, 0x15, 0xCA, 0x71, 0xBE,
			0x2B, 0x73, 0xAE, 0xF0, 0x85, 0x7D, 0x77, 0x81,
			0x1F, 0x35, 0x2C, 0x07, 0x3B, 0x61, 0x08, 0xD7,
			0x2D, 0x98, 0x10, 0xA3, 0x09, 0x14, 0xDF, 0xF4
		},
		.len = 32
	},
	.iv = {
		.data = {
			0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
			0xF8, 0xF9, 0xFA, 0xFB
		},
		.len = 12
	},
	.plaintext = {
		.data = plaintext_aes256ctr,
		.len = 64
	},
	.ciphertext = {
		.data = ciphertext64_aes256ctr_IV_12bytes,
		.len = 64
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x57, 0x9A, 0x52, 0x6E, 0x31, 0x17, 0x57, 0x49,
			0xE7, 0xA1, 0x88, 0x6C, 0x2E, 0x36, 0x67, 0x63,
			0x3F, 0x2D, 0xA3, 0xEF
		},
		.len = 20,
		.truncated_len = 12
	}
};
/** AES-128-CBC SHA1 test vector */
static const struct blockcipher_test_data aes_test_data_4 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x9A, 0x4F, 0x88, 0x1B, 0xB6, 0x8F, 0xD8, 0x60,
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0x18, 0x8C, 0x1D, 0x32
		},
		.len = 20,
		.truncated_len = 12
	}
};

/** AES-128-CBC SHA256 test vector */
static const struct blockcipher_test_data aes_test_data_5 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA256_HMAC,
	.auth_key = {
		.data = {
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
			0x58, 0x34, 0x85, 0x61, 0x1C, 0x42, 0x10, 0x76,
			0x9A, 0x4F, 0x88, 0x1B, 0xB6, 0x8F, 0xD8, 0x60
		},
		.len = 32
	},
	.digest = {
		.data = {
			0xC8, 0x57, 0x57, 0x31, 0x03, 0xE0, 0x03, 0x55,
			0x07, 0xC8, 0x9E, 0x7F, 0x48, 0x9A, 0x61, 0x9A,
			0x68, 0xEE, 0x03, 0x0E, 0x71, 0x75, 0xC7, 0xF4,
			0x2E, 0x45, 0x26, 0x32, 0x7C, 0x12, 0x15, 0x15
		},
		.len = 32,
		.truncated_len = 16
	}
};

/** AES-128-CBC SHA512 test vector */
static const struct blockcipher_test_data aes_test_data_6 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA512_HMAC,
	.auth_key = {
		.data = {
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
			0x58, 0x34, 0x85, 0x65, 0x1C, 0x42, 0x50, 0x76,
			0x9A, 0xAF, 0x88, 0x1B, 0xB6, 0x8F, 0xF8, 0x60,
			0xA2, 0x5A, 0x7F, 0x3F, 0xF4, 0x72, 0x70, 0xF1,
			0xF5, 0x35, 0x4C, 0x3B, 0xDD, 0x90, 0x65, 0xB0,
			0x47, 0x3A, 0x75, 0x61, 0x5C, 0xA2, 0x10, 0x76,
			0x9A, 0xAF, 0x77, 0x5B, 0xB6, 0x7F, 0xF7, 0x60
		},
		.len = 64
	},
	.digest = {
		.data = {
			0x5D, 0x54, 0x66, 0xC1, 0x6E, 0xBC, 0x04, 0xB8,
			0x46, 0xB8, 0x08, 0x6E, 0xE0, 0xF0, 0x43, 0x48,
			0x37, 0x96, 0x9C, 0xC6, 0x9C, 0xC2, 0x1E, 0xE8,
			0xF2, 0x0C, 0x0B, 0xEF, 0x86, 0xA2, 0xE3, 0x70,
			0x95, 0xC8, 0xB3, 0x06, 0x47, 0xA9, 0x90, 0xE8,
			0xA0, 0xC6, 0x72, 0x69, 0x05, 0xC0, 0x0D, 0x0E,
			0x21, 0x96, 0x65, 0x93, 0x74, 0x43, 0x2A, 0x1D,
			0x2E, 0xBF, 0xC2, 0xC2, 0xEE, 0xCC, 0x2F, 0x0A
		},
		.len = 64,
		.truncated_len = 32
	}
};

/** AES-128-CBC XCBC test vector */
static const struct blockcipher_test_data aes_test_data_7 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_AES_XCBC_MAC,
	.auth_key = {
		.data = {
			0x87, 0x61, 0x54, 0x53, 0xC4, 0x6D, 0xDD, 0x51,
			0xE1, 0x9F, 0x86, 0x64, 0x39, 0x0A, 0xE6, 0x59
		},
		.len = 16
	},
	.digest = {
		.data = {
			0xE0, 0xAC, 0x9A, 0xC4, 0x22, 0x64, 0x35, 0x89,
			0x77, 0x1D, 0x8B, 0x75
		},
		.len = 12,
		.truncated_len = 12
	}
};

/** AES-128-CBC SHA224 test vector */
static const struct blockcipher_test_data aes_test_data_8 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA224_HMAC,
	.auth_key = {
		.data = {
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
			0x58, 0x34, 0x85, 0x65, 0x1C, 0x42, 0x50, 0x76,
			0x9A, 0xAF, 0x88, 0x1B, 0xB6, 0x8F, 0xF8, 0x60,
			0xA2, 0x5A, 0x7F, 0x3F, 0xF4, 0x72, 0x70, 0xF1,
			0xF5, 0x35, 0x4C, 0x3B, 0xDD, 0x90, 0x65, 0xB0,
			0x47, 0x3A, 0x75, 0x61, 0x5C, 0xA2, 0x10, 0x76,
			0x9A, 0xAF, 0x77, 0x5B, 0xB6, 0x7F, 0xF7, 0x60
		},
		.len = 64
	},
	.digest = {
		.data = {
			0xA3, 0xCA, 0xC7, 0x1D, 0xA8, 0x61, 0x30, 0x98,
			0x3B, 0x8F, 0x01, 0x19, 0xAE, 0x8D, 0xBD, 0x34,
			0x40, 0x63, 0xA8, 0x2F, 0xDF, 0x85, 0x2B, 0x7F,
			0x63, 0x7C, 0xDD, 0xB7
		},
		.len = 28,
		.truncated_len = 14
	}
};

/** AES-128-CBC SHA384 test vector */
static const struct blockcipher_test_data aes_test_data_9 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA384_HMAC,
	.auth_key = {
		.data = {
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
			0x58, 0x34, 0x85, 0x65, 0x1C, 0x42, 0x50, 0x76,
			0x9A, 0xAF, 0x88, 0x1B, 0xB6, 0x8F, 0xF8, 0x60,
			0xA2, 0x5A, 0x7F, 0x3F, 0xF4, 0x72, 0x70, 0xF1,
			0xF5, 0x35, 0x4C, 0x3B, 0xDD, 0x90, 0x65, 0xB0,
			0x47, 0x3A, 0x75, 0x61, 0x5C, 0xA2, 0x10, 0x76,
			0x9A, 0xAF, 0x77, 0x5B, 0xB6, 0x7F, 0xF7, 0x60,
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
			0x58, 0x34, 0x85, 0x65, 0x1C, 0x42, 0x50, 0x76,
			0x9A, 0xAF, 0x88, 0x1B, 0xB6, 0x8F, 0xF8, 0x60,
			0xA2, 0x5A, 0x7F, 0x3F, 0xF4, 0x72, 0x70, 0xF1,
			0xF5, 0x35, 0x4C, 0x3B, 0xDD, 0x90, 0x65, 0xB0,
			0x47, 0x3A, 0x75, 0x61, 0x5C, 0xA2, 0x10, 0x76,
			0x9A, 0xAF, 0x77, 0x5B, 0xB6, 0x7F, 0xF7, 0x60
		},
		.len = 128
	},
	.digest = {
		.data = {
			0x23, 0x60, 0xC8, 0xB1, 0x2D, 0x6C, 0x1E, 0x72,
			0x25, 0xAB, 0xF9, 0xC3, 0x9A, 0xA9, 0x4F, 0x8C,
			0x56, 0x38, 0x65, 0x0E, 0x74, 0xD5, 0x45, 0x9D,
			0xA3, 0xFD, 0x7E, 0x6D, 0x9E, 0x74, 0x88, 0x9D,
			0xA7, 0x12, 0x9D, 0xD8, 0x81, 0x3C, 0x86, 0x2F,
			0x4D, 0xF9, 0x6F, 0x0A, 0xB0, 0xC9, 0xEB, 0x0B
		},
		.len = 48,
		.truncated_len = 24
	}
};

static const uint8_t ciphertext512_aes192cbc[] = {
	0x45, 0xEE, 0x9A, 0xEA, 0x3C, 0x03, 0xFC, 0x4C,
	0x84, 0x36, 0xB0, 0xDA, 0xB0, 0xDC, 0xF3, 0x5B,
	0x75, 0xA7, 0xBE, 0x0E, 0xC0, 0x8D, 0x6C, 0xF8,
	0xC1, 0x0F, 0xD0, 0x35, 0x1D, 0x82, 0xAE, 0x7C,
	0x57, 0xC5, 0x7A, 0x55, 0x87, 0x1B, 0xD4, 0x03,
	0x0A, 0x64, 0xC9, 0xE0, 0xF4, 0xC7, 0x6F, 0x57,
	0x52, 0xC6, 0x73, 0xBA, 0x84, 0x0B, 0x5B, 0x89,
	0x21, 0xD2, 0x9B, 0x88, 0x68, 0xF5, 0xA9, 0x7F,
	0x3F, 0x49, 0xEB, 0xF4, 0xD4, 0x52, 0xD2, 0x64,
	0x80, 0xB2, 0x53, 0xDA, 0x19, 0xF6, 0x10, 0x24,
	0x23, 0x26, 0x7A, 0x7C, 0x07, 0x57, 0x4B, 0x0E,
	0x58, 0x49, 0x61, 0xD1, 0xDC, 0x9A, 0x32, 0x6B,
	0x0F, 0x43, 0x9E, 0x4D, 0xB4, 0x07, 0x4E, 0xB3,
	0x51, 0x74, 0xDE, 0x29, 0xBC, 0x98, 0xF9, 0xDF,
	0x78, 0x9A, 0x18, 0x9C, 0xD6, 0x7A, 0x55, 0x7C,
	0xE6, 0x1D, 0x5C, 0x1A, 0x99, 0xD2, 0xC3, 0x7B,
	0x9F, 0x96, 0x74, 0x2D, 0xE0, 0xEF, 0xD1, 0xE3,
	0x08, 0x9F, 0xAF, 0xE6, 0xED, 0xCA, 0xE1, 0xEA,
	0x23, 0x6F, 0x7C, 0x81, 0xA8, 0xC0, 0x5B, 0x8B,
	0x53, 0x90, 0x51, 0x2D, 0x0F, 0xF6, 0x7D, 0xA7,
	0x1C, 0xBD, 0x83, 0x84, 0x54, 0xA4, 0x15, 0xFB,
	0x3E, 0x25, 0xA7, 0x3A, 0x0A, 0x73, 0xD9, 0x88,
	0x6F, 0x80, 0x78, 0x95, 0x7F, 0x60, 0xAA, 0x86,
	0x8A, 0xFC, 0xDF, 0xC1, 0xCB, 0xDE, 0xBB, 0x25,
	0x52, 0x20, 0xC6, 0x79, 0xD4, 0x0F, 0x25, 0xE7,
	0xDB, 0xB2, 0x17, 0xA4, 0x6F, 0x3C, 0x6F, 0x91,
	0xF6, 0x44, 0x1E, 0xB6, 0x85, 0xBC, 0x7A, 0x14,
	0x10, 0x72, 0xBD, 0x16, 0x63, 0x39, 0x9E, 0x7B,
	0x84, 0x5B, 0x17, 0x61, 0xB1, 0x5D, 0x82, 0x0B,
	0x6D, 0x37, 0xD7, 0x79, 0xB8, 0x24, 0x91, 0x30,
	0x82, 0x91, 0x02, 0xB1, 0x18, 0x4B, 0xE0, 0xF4,
	0x13, 0x1B, 0xB2, 0x4C, 0xDA, 0xB8, 0x99, 0x96,
	0x83, 0x2F, 0xBE, 0x53, 0x8D, 0xDE, 0xFA, 0xAD,
	0xF6, 0x5C, 0xDB, 0xE5, 0x66, 0x26, 0x8F, 0x13,
	0x2B, 0x76, 0x47, 0x73, 0xDE, 0x1A, 0x74, 0xA6,
	0x30, 0xAF, 0x42, 0xA0, 0xE5, 0xD2, 0x8F, 0xC2,
	0xED, 0x3E, 0x9E, 0x29, 0x54, 0x3C, 0xDE, 0x9F,
	0x5D, 0x30, 0x2B, 0x63, 0xFB, 0xE3, 0xB1, 0x07,
	0xEE, 0x74, 0x4A, 0xAF, 0xB1, 0x20, 0x8D, 0xEC,
	0xE6, 0x78, 0x16, 0x8D, 0xA4, 0x6E, 0x34, 0x7D,
	0x47, 0xFB, 0x0B, 0xC1, 0x32, 0xD7, 0x0D, 0x6C,
	0x6F, 0x93, 0x9C, 0x5E, 0xEF, 0x1F, 0x9C, 0x45,
	0x80, 0x6B, 0x74, 0xA6, 0x81, 0xF2, 0xF6, 0xFA,
	0xAA, 0x9D, 0x4F, 0xCA, 0xB5, 0x90, 0x59, 0xB0,
	0x3B, 0xF2, 0xF0, 0x75, 0xFD, 0x8A, 0xD8, 0x97,
	0x65, 0x88, 0x56, 0x4C, 0x44, 0xDF, 0x73, 0xF7,
	0x56, 0x9C, 0x48, 0x7E, 0xB0, 0x1F, 0x1D, 0x7D,
	0x6A, 0x11, 0xF5, 0xC2, 0xF4, 0x17, 0xEF, 0x58,
	0xD8, 0x2A, 0xAF, 0x56, 0x2F, 0xCF, 0xEC, 0xA4,
	0x58, 0x8B, 0x60, 0xCE, 0xD4, 0x0F, 0x9C, 0x21,
	0xEC, 0x3E, 0x74, 0x7B, 0x81, 0x3D, 0x69, 0xC6,
	0x5E, 0x12, 0x83, 0xE9, 0xEF, 0x81, 0x58, 0x36,
	0x6A, 0x60, 0x0F, 0x54, 0x28, 0x11, 0xF9, 0x64,
	0x36, 0xAD, 0x79, 0xF5, 0x1C, 0x74, 0xD0, 0xC3,
	0x7B, 0x61, 0xE1, 0x92, 0xB0, 0x13, 0x91, 0x87,
	0x32, 0x1F, 0xF2, 0x5A, 0xDA, 0x25, 0x69, 0xEB,
	0xD7, 0x32, 0x7F, 0xF5, 0x23, 0x21, 0x54, 0x47,
	0x7B, 0x1B, 0x33, 0xB0, 0x3D, 0xF6, 0xE2, 0x7E,
	0x3E, 0xA2, 0x9E, 0xCA, 0x48, 0x0B, 0x4A, 0x29,
	0x81, 0xD4, 0x4E, 0xD5, 0x69, 0xFB, 0xCD, 0x37,
	0x8A, 0xC1, 0x5B, 0x50, 0xFF, 0xB5, 0x7D, 0x43,
	0x0F, 0xAE, 0xA6, 0xC2, 0xE5, 0x8F, 0x45, 0xB2,
	0x85, 0x99, 0x02, 0xA2, 0x9B, 0xBE, 0x90, 0x43,
	0x4F, 0x2F, 0x50, 0xE2, 0x77, 0x62, 0xD9, 0xCC
};

/** AES-192-CBC test vector */
static const struct blockcipher_test_data aes_test_data_10 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A,
			0xD4, 0xC3, 0xA3, 0xAA, 0x33, 0x62, 0x61, 0xE0
		},
		.len = 24
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes192cbc,
		.len = 512
	}
};

static const uint8_t ciphertext512_aes256cbc[] = {
	0xF3, 0xDD, 0xF0, 0x0B, 0xFF, 0xA2, 0x6A, 0x04,
	0xBE, 0xDA, 0x52, 0xA6, 0xFE, 0x6B, 0xA6, 0xA7,
	0x48, 0x1D, 0x7D, 0x98, 0x65, 0xDB, 0xEF, 0x06,
	0x26, 0xB5, 0x8E, 0xEB, 0x05, 0x0E, 0x77, 0x98,
	0x17, 0x8E, 0xD0, 0xD4, 0x7B, 0x92, 0x8F, 0x5C,
	0xD0, 0x74, 0x5C, 0xA8, 0x4B, 0x54, 0xB6, 0x2F,
	0x83, 0x72, 0x2C, 0xFF, 0x72, 0xE9, 0xE4, 0x15,
	0x4C, 0x32, 0xAF, 0xC8, 0xC9, 0x89, 0x3C, 0x6E,
	0x31, 0xD5, 0xC0, 0x16, 0xC0, 0x31, 0x7D, 0x11,
	0xAB, 0xCB, 0xDE, 0xD2, 0xD6, 0xAA, 0x76, 0x5E,
	0xBA, 0xF6, 0xE2, 0x92, 0xCB, 0x86, 0x07, 0xFA,
	0xD4, 0x9E, 0x83, 0xED, 0xFD, 0xB8, 0x70, 0x54,
	0x6B, 0xBE, 0xEC, 0x72, 0xDD, 0x28, 0x5E, 0x95,
	0x78, 0xA5, 0x28, 0x43, 0x3D, 0x6D, 0xB1, 0xD9,
	0x69, 0x1F, 0xC9, 0x66, 0x0E, 0x32, 0x44, 0x08,
	0xD2, 0xAE, 0x2C, 0x43, 0xF2, 0xD0, 0x7D, 0x26,
	0x70, 0xE5, 0xA1, 0xCA, 0x37, 0xE9, 0x7D, 0xC7,
	0xA3, 0xFA, 0x81, 0x91, 0x64, 0xAA, 0x64, 0x91,
	0x9A, 0x95, 0x2D, 0xC9, 0xF9, 0xCE, 0xFE, 0x9F,
	0xC4, 0xD8, 0x81, 0xBE, 0x57, 0x84, 0xC5, 0x02,
	0xDB, 0x30, 0xC1, 0xD9, 0x0E, 0xA0, 0xA6, 0x00,
	0xD6, 0xF3, 0x52, 0x7E, 0x0D, 0x23, 0x6B, 0x2B,
	0x34, 0x99, 0x1F, 0x70, 0x27, 0x6D, 0x58, 0x84,
	0x93, 0x77, 0xB8, 0x3E, 0xF1, 0x71, 0x58, 0x42,
	0x8B, 0x2B, 0xC8, 0x6D, 0x05, 0x84, 0xFF, 0x4E,
	0x85, 0xEF, 0x4A, 0x9D, 0x91, 0x6A, 0xD5, 0xE1,
	0xAF, 0x01, 0xEB, 0x83, 0x8F, 0x23, 0x7C, 0x7F,
	0x12, 0x91, 0x05, 0xF0, 0x4E, 0xD9, 0x17, 0x62,
	0x75, 0xBB, 0xAC, 0x97, 0xEE, 0x3B, 0x4E, 0xC7,
	0xE5, 0x92, 0xF8, 0x9D, 0x4C, 0xF9, 0xEE, 0x55,
	0x18, 0xBB, 0xCC, 0xB4, 0xF2, 0x59, 0xB9, 0xFC,
	0x7A, 0x0F, 0x98, 0xD4, 0x8B, 0xFE, 0xF7, 0x83,
	0x46, 0xE2, 0x83, 0x33, 0x3E, 0x95, 0x8D, 0x17,
	0x1E, 0x85, 0xF8, 0x8C, 0x51, 0xB0, 0x6C, 0xB5,
	0x5E, 0x95, 0xBA, 0x4B, 0x69, 0x1B, 0x48, 0x69,
	0x0B, 0x8F, 0xA5, 0x18, 0x13, 0xB9, 0x77, 0xD1,
	0x80, 0x32, 0x32, 0x6D, 0x53, 0xA1, 0x95, 0x40,
	0x96, 0x8A, 0xCC, 0xA3, 0x69, 0xF8, 0x9F, 0xB5,
	0x8E, 0xD2, 0x68, 0x07, 0x4F, 0xA7, 0xEC, 0xF8,
	0x20, 0x21, 0x58, 0xF8, 0xD8, 0x9E, 0x5F, 0x40,
	0xBA, 0xB9, 0x76, 0x57, 0x3B, 0x17, 0xAD, 0xEE,
	0xCB, 0xDF, 0x07, 0xC1, 0xDF, 0x66, 0xA8, 0x0D,
	0xC2, 0xCE, 0x8F, 0x79, 0xC3, 0x32, 0xE0, 0x8C,
	0xFE, 0x5A, 0xF3, 0x55, 0x27, 0x73, 0x6F, 0xA1,
	0x54, 0xC6, 0xFC, 0x28, 0x9D, 0xBE, 0x97, 0xB9,
	0x54, 0x97, 0x72, 0x3A, 0x61, 0xAF, 0x6F, 0xDE,
	0xF8, 0x0E, 0xBB, 0x6B, 0x96, 0x84, 0xDD, 0x9B,
	0x62, 0xBA, 0x47, 0xB5, 0xC9, 0x3B, 0x4E, 0x8C,
	0x78, 0x2A, 0xCC, 0x0A, 0x69, 0x54, 0x25, 0x5E,
	0x8B, 0xAC, 0x56, 0xD9, 0xFE, 0x48, 0xBA, 0xCE,
	0xA9, 0xCE, 0xA6, 0x1D, 0xBF, 0x3E, 0x3C, 0x66,
	0x40, 0x71, 0x79, 0xAD, 0x5B, 0x26, 0xAD, 0xBE,
	0x58, 0x13, 0x64, 0x60, 0x7C, 0x05, 0xFC, 0xE3,
	0x51, 0x7A, 0xF2, 0xCC, 0x54, 0x16, 0x2C, 0xA4,
	0xCE, 0x5F, 0x59, 0x12, 0x77, 0xEB, 0xD9, 0x23,
	0xE3, 0x86, 0xFB, 0xD7, 0x48, 0x76, 0x9D, 0xE3,
	0x89, 0x87, 0x39, 0xFA, 0x7B, 0x21, 0x0B, 0x76,
	0xB2, 0xED, 0x1C, 0x27, 0x4B, 0xD5, 0x27, 0x05,
	0x8C, 0x7D, 0x58, 0x6C, 0xCA, 0xA5, 0x54, 0x9A,
	0x0F, 0xCB, 0xE9, 0x88, 0x31, 0xAD, 0x49, 0xEE,
	0x38, 0xFB, 0xC9, 0xFB, 0xB4, 0x7A, 0x00, 0x58,
	0x20, 0x32, 0xD3, 0x53, 0x5A, 0xDD, 0x74, 0x95,
	0x60, 0x59, 0x09, 0xAE, 0x7E, 0xEC, 0x74, 0xA3,
	0xB7, 0x1C, 0x6D, 0xF2, 0xAE, 0x79, 0xA4, 0x7C
};

/** AES-256-CBC test vector */
static const struct blockcipher_test_data aes_test_data_11 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A,
			0xD4, 0xC3, 0xA3, 0xAA, 0x33, 0x62, 0x61, 0xE0,
			0x37, 0x07, 0xB8, 0x23, 0xA2, 0xA3, 0xB5, 0x8D
		},
		.len = 32
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes256cbc,
		.len = 512
	}
};

/** AES-128-CBC SHA256 HMAC test vector (160 bytes) */
static const struct blockcipher_test_data aes_test_data_12 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 160
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 160
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA256_HMAC,
	.auth_key = {
		.data = {
			0x42, 0x1A, 0x7D, 0x3D, 0xF5, 0x82, 0x80, 0xF1,
			0xF1, 0x35, 0x5C, 0x3B, 0xDD, 0x9A, 0x65, 0xBA,
			0x58, 0x34, 0x85, 0x61, 0x1C, 0x42, 0x10, 0x76,
			0x9A, 0x4F, 0x88, 0x1B, 0xB6, 0x8F, 0xD8, 0x60
		},
		.len = 32
	},
	.digest = {
		.data = {
			0x92, 0xEC, 0x65, 0x9A, 0x52, 0xCC, 0x50, 0xA5,
			0xEE, 0x0E, 0xDF, 0x1E, 0xA4, 0xC9, 0xC1, 0x04,
			0xD5, 0xDC, 0x78, 0x90, 0xF4, 0xE3, 0x35, 0x62,
			0xAD, 0x95, 0x45, 0x28, 0x5C, 0xF8, 0x8C, 0x0B
		},
		.len = 32,
		.truncated_len = 16
	}
};

/** AES-128-CBC SHA1 HMAC test vector (160 bytes) */
static const struct blockcipher_test_data aes_test_data_13 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 160
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 160
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0x4F, 0x16, 0xEA, 0xF7, 0x4A, 0x88, 0xD3, 0xE0,
			0x0E, 0x12, 0x8B, 0xE7, 0x05, 0xD0, 0x86, 0x48,
			0x22, 0x43, 0x30, 0xA7
		},
		.len = 20,
		.truncated_len = 12
	}
};

/** XTS-AES-128 test vector (2 keys * 128 bit) */
static const uint8_t plaintext_aes128xts_16bytes[] = {
	0xEB, 0xAB, 0xCE, 0x95, 0xB1, 0x4D, 0x3C, 0x8D,
	0x6F, 0xB3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1C
};
static const uint8_t ciphertext_aes128xts_16bytes[] = {
	0x77, 0x8A, 0xE8, 0xB4, 0x3C, 0xB9, 0x8D, 0x5A,
	0x82, 0x50, 0x81, 0xD5, 0xBE, 0x47, 0x1C, 0x63
};
static const struct
blockcipher_test_data aes_test_data_xts_key_32_pt_16 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_XTS,
	.cipher_key = {
		.data = {
			0xA1, 0xB9, 0x0C, 0xBA, 0x3F, 0x06, 0xAC, 0x35,
			0x3B, 0x2C, 0x34, 0x38, 0x76, 0x08, 0x17, 0x62,
			0x09, 0x09, 0x23, 0x02, 0x6E, 0x91, 0x77, 0x18,
			0x15, 0xF2, 0x9D, 0xAB, 0x01, 0x93, 0x2F, 0x2F
		},
		.len = 32
	},
	.iv = {
		.data = {
			0x4F, 0xAE, 0xF7, 0x11, 0x7C, 0xDA, 0x59, 0xC6,
			0x6E, 0x4B, 0x92, 0x01, 0x3E, 0x76, 0x8A, 0xD5
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes128xts_16bytes,
		.len = 16
	},
	.ciphertext = {
		.data = ciphertext_aes128xts_16bytes,
		.len = 16
	}
};
static const uint8_t plaintext_aes128xts_32bytes[] = {
	0xB0, 0x4D, 0x84, 0xDA, 0x85, 0x6B, 0x9A, 0x59,
	0xCE, 0x2D, 0x62, 0x67, 0x46, 0xF6, 0x89, 0xA8,
	0x05, 0x1D, 0xAC, 0xD6, 0xBC, 0xE3, 0xB9, 0x90,
	0xAA, 0x90, 0x1E, 0x40, 0x30, 0x64, 0x88, 0x79
};
static const uint8_t ciphertext_aes128xts_32bytes[] = {
	0xF9, 0x41, 0x03, 0x9E, 0xBA, 0xB8, 0xCA, 0xC3,
	0x9D, 0x59, 0x24, 0x7C, 0xBB, 0xCB, 0x4D, 0x81,
	0x6C, 0x72, 0x6D, 0xAE, 0xD1, 0x15, 0x77, 0x69,
	0x2C, 0x55, 0xE4, 0xAC, 0x6D, 0x3E, 0x68, 0x20
};
static const struct
blockcipher_test_data aes_test_data_xts_key_32_pt_32 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_XTS,
	.cipher_key = {
		.data = {
			0x78, 0x3A, 0x83, 0xEC, 0x52, 0xA2, 0x74, 0x05,
			0xDF, 0xF9, 0xDE, 0x4C, 0x57, 0xF9, 0xC9, 0x79,
			0xB3, 0x60, 0xB6, 0xA5, 0xDF, 0x88, 0xD6, 0x7E,
			0xC1, 0xA0, 0x52, 0xE6, 0xF5, 0x82, 0xA7, 0x17
		},
		.len = 32
	},
	.iv = {
		.data = {
			0x88, 0x6E, 0x97, 0x5B, 0x29, 0xBD, 0xF6, 0xF0,
			0xC0, 0x1B, 0xB4, 0x7F, 0x61, 0xF6, 0xF0, 0xF5
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes128xts_32bytes,
		.len = 32
	},
	.ciphertext = {
		.data = ciphertext_aes128xts_32bytes,
		.len = 32
	}
};

/** XTS-AES-256 test vector (2 keys * 256 bit) */
/* Encryption */
static const uint8_t plaintext_aes256xts_32bytes[] = {
	0x2E, 0xED, 0xEA, 0x52, 0xCD, 0x82, 0x15, 0xE1,
	0xAC, 0xC6, 0x47, 0xE8, 0x10, 0xBB, 0xC3, 0x64,
	0x2E, 0x87, 0x28, 0x7F, 0x8D, 0x2E, 0x57, 0xE3,
	0x6C, 0x0A, 0x24, 0xFB, 0xC1, 0x2A, 0x20, 0x2E
};
static const uint8_t ciphertext_aes256xts_32bytes[] = {
	0xCB, 0xAA, 0xD0, 0xE2, 0xF6, 0xCE, 0xA3, 0xF5,
	0x0B, 0x37, 0xF9, 0x34, 0xD4, 0x6A, 0x9B, 0x13,
	0x0B, 0x9D, 0x54, 0xF0, 0x7E, 0x34, 0xF3, 0x6A,
	0xF7, 0x93, 0xE8, 0x6F, 0x73, 0xC6, 0xD7, 0xDB
};
static const struct
blockcipher_test_data aes_test_data_xts_key_64_pt_32 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_XTS,
	.cipher_key = {
		.data = {
			0x1E, 0xA6, 0x61, 0xC5, 0x8D, 0x94, 0x3A, 0x0E,
			0x48, 0x01, 0xE4, 0x2F, 0x4B, 0x09, 0x47, 0x14,
			0x9E, 0x7F, 0x9F, 0x8E, 0x3E, 0x68, 0xD0, 0xC7,
			0x50, 0x52, 0x10, 0xBD, 0x31, 0x1A, 0x0E, 0x7C,
			0xD6, 0xE1, 0x3F, 0xFD, 0xF2, 0x41, 0x8D, 0x8D,
			0x19, 0x11, 0xC0, 0x04, 0xCD, 0xA5, 0x8D, 0xA3,
			0xD6, 0x19, 0xB7, 0xE2, 0xB9, 0x14, 0x1E, 0x58,
			0x31, 0x8E, 0xEA, 0x39, 0x2C, 0xF4, 0x1B, 0x08
		},
		.len = 64
	},
	.iv = {
		.data = {
			0xAD, 0xF8, 0xD9, 0x26, 0x27, 0x46, 0x4A, 0xD2,
			0xF0, 0x42, 0x8E, 0x84, 0xA9, 0xF8, 0x75, 0x64
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes256xts_32bytes,
		.len = 32
	},
	.ciphertext = {
		.data = ciphertext_aes256xts_32bytes,
		.len = 32
	}
};
static const uint8_t plaintext_aes256xts_48bytes[] = {
	0xBB, 0xB2, 0x34, 0xDB, 0x01, 0xBE, 0x79, 0xD3,
	0xDC, 0x7D, 0xCF, 0x49, 0xBC, 0x53, 0xB8, 0xEF,
	0xC6, 0x2F, 0xE7, 0x17, 0x94, 0x39, 0x06, 0x1D,
	0x73, 0xA6, 0xB2, 0x0E, 0xDB, 0x9B, 0x7D, 0xA0,
	0x45, 0x0B, 0x19, 0xF0, 0x20, 0x82, 0x02, 0x09,
	0xAF, 0xE0, 0x81, 0x12, 0xAE, 0x4A, 0xFD, 0xD3
};
static const uint8_t ciphertext_aes256xts_48bytes[] = {
	0xCB, 0xF6, 0x88, 0x48, 0xC4, 0x20, 0x49, 0xEF,
	0xB1, 0x46, 0xE2, 0xD1, 0xE4, 0x11, 0x5F, 0x62,
	0xE4, 0xFA, 0xA4, 0xEF, 0xFF, 0x19, 0x8F, 0x1A,
	0x0A, 0xA0, 0xC9, 0x8B, 0xA0, 0x2C, 0xC4, 0x4D,
	0xA3, 0x76, 0xD7, 0x62, 0xD6, 0x15, 0x1F, 0x23,
	0x45, 0x87, 0xBF, 0x12, 0x8B, 0x6A, 0x7F, 0xFA
};
static const struct
blockcipher_test_data aes_test_data_xts_key_64_pt_48 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_XTS,
	.cipher_key = {
		.data = {
			0x7F, 0xB0, 0x92, 0x2F, 0xCE, 0x09, 0xED, 0xDD,
			0x36, 0x65, 0xA1, 0x1F, 0x52, 0x35, 0xD5, 0x1E,
			0xF7, 0x72, 0x06, 0xA7, 0xDE, 0x45, 0x47, 0x75,
			0xB6, 0x9D, 0xCC, 0x54, 0x59, 0xAC, 0xDB, 0x24,
			0xCC, 0xF0, 0x5C, 0x41, 0x5A, 0xF5, 0xAB, 0x8A,
			0x06, 0x62, 0x3D, 0x19, 0x8D, 0x5B, 0x91, 0x85,
			0x95, 0xA9, 0xDC, 0xAA, 0xF5, 0x39, 0x2E, 0xE7,
			0x17, 0xC1, 0x04, 0x7F, 0x2F, 0x08, 0xF6, 0x2B
		},
		.len = 64
	},
	.iv = {
		.data = {
			0x8E, 0xA3, 0x63, 0x8B, 0x9D, 0x40, 0x62, 0xF1,
			0x69, 0x19, 0x6F, 0xF4, 0x55, 0x5A, 0xD0, 0xAF
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes256xts_48bytes,
		.len = 48
	},
	.ciphertext = {
		.data = ciphertext_aes256xts_48bytes,
		.len = 48
	}
};

/* AES-DOCSIS-BPI test vectors */

/* Multiple of AES block size */
static const struct blockcipher_test_data aes_test_data_docsis_1 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_DOCSISBPI,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 512
	},
	.ciphertext = {
		.data = ciphertext512_aes128cbc,
		.len = 512
	}
};

/* Less than AES block size */
static const struct blockcipher_test_data aes_test_data_docsis_2 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_DOCSISBPI,
	.cipher_key = {
		.data = {
			0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
			0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A,
			0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_docsis_bpi_cfb,
		.len = 7
	},
	.ciphertext = {
		.data = ciphertext_aes_docsis_bpi_cfb,
		.len = 7
	}
};

/* Not multiple of AES block size */
static const struct blockcipher_test_data aes_test_data_docsis_3 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_DOCSISBPI,
	.cipher_key = {
		.data = {
			0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
			0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A,
			0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_docsis_bpi_cbc_cfb,
		.len = 19
	},
	.ciphertext = {
		.data = ciphertext_aes_docsis_bpi_cbc_cfb,
		.len = 19
	}
};

static const uint8_t
cipher_aescbc_offset_16[] = {
	0x57, 0x68, 0x61, 0x74, 0x20, 0x61, 0x20, 0x6C,
	0x6F, 0x75, 0x73, 0x79, 0x20, 0x65, 0x61, 0x72,
	0x68, 0x2A, 0x6A, 0x82, 0xE0, 0x73, 0xC7, 0x51,
	0x81, 0xF4, 0x47, 0x27, 0x1A, 0xEF, 0x76, 0x15,
	0x1C, 0xE1, 0x38, 0x5F, 0xE1, 0x81, 0x77, 0xC7,
	0x8B, 0xF0, 0x69, 0xC3, 0x3C, 0x45, 0x1C, 0x0A,
	0xA3, 0x93, 0xBF, 0x60, 0x57, 0x88, 0xD2, 0xFF,
	0xE1, 0x8F, 0xC0, 0x64, 0x2C, 0x42, 0xC5, 0x22,
	0xE3, 0x5F, 0x71, 0x1F, 0xF7, 0x62, 0xA2, 0x7E,
	0x0D, 0x42, 0xD9, 0xE7, 0xF3, 0x10, 0xB0, 0xEE,
};

/** AES-128-CBC SHA1 OOP test vector for swapping src/dst */
static const struct blockcipher_test_data aes_test_data_14 = {
	.crypto_algo = RTE_CRYPTO_CIPHER_AES_CBC,
	.cipher_key = {
		.data = {
			0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
			0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A
		},
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
		},
		.len = 16
	},
	.plaintext = {
		.data = plaintext_aes_common,
		.len = 80
	},
	.cipher_offset = 16,
	.auth_offset = 0,
	.ciphertext = {
		.data = cipher_aescbc_offset_16,
		.len = 80
	},
	.auth_algo = RTE_CRYPTO_AUTH_SHA1_HMAC,
	.auth_key = {
		.data = {
			0xF8, 0x2A, 0xC7, 0x54, 0xDB, 0x96, 0x18, 0xAA,
			0xC3, 0xA1, 0x53, 0xF6, 0x1F, 0x17, 0x60, 0xBD,
			0xDE, 0xF4, 0xDE, 0xAD
		},
		.len = 20
	},
	.digest = {
		.data = {
			0xCC, 0x15, 0x83, 0xF7, 0x23, 0x87, 0x96, 0xA7,
			0x29, 0x34, 0x32, 0xE4, 0x4C, 0x06, 0xE8, 0xEB,
			0x70, 0x72, 0x4B, 0xAD
		},
		.len = 20,
		.truncated_len = 12
	}
};

static const struct blockcipher_test_case aes_chain_test_cases[] = {
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Decryption Digest "
			"Verify OOP Offset",
		.test_data = &aes_test_data_14,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX
	},
		{
		.test_descr = "AES-128-CTR HMAC-SHA1 Encryption Digest",
		.test_data = &aes_test_data_1,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP
	},
	{
		.test_descr = "AES-128-CTR HMAC-SHA1 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_1,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP
	},
	{
		.test_descr = "AES-192-CTR XCBC Encryption Digest",
		.test_data = &aes_test_data_2,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER
	},
	{
		.test_descr = "AES-192-CTR XCBC Decryption Digest Verify",
		.test_data = &aes_test_data_2,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER
	},
	{
		.test_descr = "AES-192-CTR XCBC Decryption Digest Verify "
				"Scatter Gather",
		.test_data = &aes_test_data_2,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SG |
			BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask =
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER
	},
	{
		.test_descr = "AES-256-CTR HMAC-SHA1 Encryption Digest",
		.test_data = &aes_test_data_3,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP
	},
	{
		.test_descr = "AES-256-CTR HMAC-SHA1 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_3,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Encryption Digest",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Encryption Digest "
			"(short buffers)",
		.test_data = &aes_test_data_13,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Encryption Digest "
				"Scatter Gather",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SG |
			BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask =
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Decryption Digest "
			"Verify Scatter Gather",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SG,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			    BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			    BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			    BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			    BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Decryption Digest "
			"Verify (short buffers)",
		.test_data = &aes_test_data_13,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA256 Encryption Digest",
		.test_data = &aes_test_data_5,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA256 Encryption Digest "
			"(short buffers)",
		.test_data = &aes_test_data_12,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA256 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_5,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA256 Decryption Digest "
			"Verify (short buffers)",
		.test_data = &aes_test_data_12,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA512 Encryption Digest",
		.test_data = &aes_test_data_6,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA512 Encryption Digest "
			"Sessionless",
		.test_data = &aes_test_data_6,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SESSIONLESS,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA512 Encryption Digest "
				"Scatter Gather Sessionless",
		.test_data = &aes_test_data_6,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SESSIONLESS |
			BLOCKCIPHER_TEST_FEATURE_SG |
			BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask =
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA512 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_6,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA512 Decryption Digest "
			"Verify Scatter Gather",
		.test_data = &aes_test_data_6,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SG |
			BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask =
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC XCBC Encryption Digest",
		.test_data = &aes_test_data_7,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER
	},
	{
		.test_descr = "AES-128-CBC XCBC Decryption Digest Verify",
		.test_data = &aes_test_data_7,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Encryption Digest "
			"OOP",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Encryption Digest "
			"OOP Offset",
		.test_data = &aes_test_data_14,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Decryption Digest "
			"Verify OOP",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA224 Encryption Digest",
		.test_data = &aes_test_data_8,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA224 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_8,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_NITROX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA384 Encryption Digest",
		.test_data = &aes_test_data_9,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA384 Decryption Digest "
			"Verify",
		.test_data = &aes_test_data_9,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP
	},
	{
		.test_descr = "AES-128-CBC HMAC-SHA1 Encryption Digest "
				"Sessionless",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SESSIONLESS,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr =
				"AES-128-CBC HMAC-SHA1 Decryption Digest "
				"Verify Sessionless",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SESSIONLESS,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_ARMV8 |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "NULL-CIPHER-NULL-AUTH encrypt & gen x8byte",
		.test_data = &null_test_data_chain_x8_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-AUTH-NULL-CIPHER verify & decrypt x8byte",
		.test_data = &null_test_data_chain_x8_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-CIPHER-NULL-AUTH encrypt & gen x8byte - OOP",
		.test_data = &null_test_data_chain_x8_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-AUTH-NULL-CIPHER verify & decrypt x8byte - OOP",
		.test_data = &null_test_data_chain_x8_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-CIPHER-NULL-AUTH encrypt & gen x4byte",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-AUTH-NULL-CIPHER verify & decrypt x4byte",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-CIPHER-NULL-AUTH encrypt & gen x4byte - OOP",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-AUTH-NULL-CIPHER verify & decrypt x4byte - OOP",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-CIPHER-NULL-AUTH encrypt & gen x1byte",
		.test_data = &null_test_data_chain_x1_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-AUTH-NULL-CIPHER verify & decrypt x1byte",
		.test_data = &null_test_data_chain_x1_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-CIPHER-NULL-AUTH encrypt & gen x1byte - OOP",
		.test_data = &null_test_data_chain_x1_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENC_AUTH_GEN,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "NULL-AUTH-NULL-CIPHER verify & decrypt x1byte - OOP",
		.test_data = &null_test_data_chain_x1_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_AUTH_VERIFY_DEC,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
};

static const struct blockcipher_test_case aes_cipheronly_test_cases[] = {
	{
		.test_descr = "AES-128-CBC Encryption",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CBC Decryption",
		.test_data = &aes_test_data_4,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-192-CBC Encryption",
		.test_data = &aes_test_data_10,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-192-CBC Encryption Scater gather",
		.test_data = &aes_test_data_10,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SG |
			BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-192-CBC Decryption",
		.test_data = &aes_test_data_10,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-192-CBC Decryption Scatter Gather",
		.test_data = &aes_test_data_10,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_SG,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-256-CBC Encryption",
		.test_data = &aes_test_data_11,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-256-CBC Decryption",
		.test_data = &aes_test_data_11,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-256-CBC OOP Encryption",
		.test_data = &aes_test_data_11,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
	{
		.test_descr = "AES-256-CBC OOP Decryption",
		.test_data = &aes_test_data_11,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_VIRTIO |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
	{
		.test_descr = "AES-128-CTR Encryption",
		.test_data = &aes_test_data_1,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CTR Decryption",
		.test_data = &aes_test_data_1,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-192-CTR Encryption",
		.test_data = &aes_test_data_2,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-192-CTR Decryption",
		.test_data = &aes_test_data_2,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-256-CTR Encryption",
		.test_data = &aes_test_data_3,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-256-CTR Decryption",
		.test_data = &aes_test_data_3,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_OPENSSL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_SCHEDULER |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC |
			BLOCKCIPHER_TEST_TARGET_PMD_CAAM_JR |
			BLOCKCIPHER_TEST_TARGET_PMD_MVSAM |
			BLOCKCIPHER_TEST_TARGET_PMD_CCP |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2
	},
	{
		.test_descr = "AES-128-CTR Encryption (12-byte IV)",
		.test_data = &aes_test_data_1_IV_12_bytes,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2 |
				BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
				BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC
	},
	{
		.test_descr = "AES-192-CTR Encryption (12-byte IV)",
		.test_data = &aes_test_data_2_IV_12_bytes,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2 |
				BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
				BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC
	},
	{
		.test_descr = "AES-256-CTR Encryption (12-byte IV)",
		.test_data = &aes_test_data_3_IV_12_bytes,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX |
			    BLOCKCIPHER_TEST_TARGET_PMD_OCTEONTX2 |
				BLOCKCIPHER_TEST_TARGET_PMD_DPAA2_SEC |
				BLOCKCIPHER_TEST_TARGET_PMD_DPAA_SEC
	},
	{
		.test_descr = "AES-128-XTS Encryption (16-byte plaintext)",
		.test_data = &aes_test_data_xts_key_32_pt_16,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-128-XTS Decryption (16-byte plaintext)",
		.test_data = &aes_test_data_xts_key_32_pt_16,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-128-XTS Encryption (32-byte plaintext)",
		.test_data = &aes_test_data_xts_key_32_pt_32,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-128-XTS Decryption (32-byte plaintext)",
		.test_data = &aes_test_data_xts_key_32_pt_32,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-256-XTS Encryption (32-byte plaintext)",
		.test_data = &aes_test_data_xts_key_64_pt_32,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-256-XTS Decryption (32-byte plaintext)",
		.test_data = &aes_test_data_xts_key_64_pt_32,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-256-XTS Encryption (48-byte plaintext)",
		.test_data = &aes_test_data_xts_key_64_pt_48,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-256-XTS Decryption (48-byte plaintext)",
		.test_data = &aes_test_data_xts_key_64_pt_48,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x8 - encryption",
		.test_data = &null_test_data_chain_x8_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
				BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x8 - decryption",
		.test_data = &null_test_data_chain_x8_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x4 - encryption",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x4 - decryption",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x4 - encryption - OOP",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x4 - decryption - OOP",
		.test_data = &null_test_data_chain_x4_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT
		},
	{
		.test_descr = "cipher-only - NULL algo - x1 - encryption",
		.test_data = &null_test_data_chain_x1_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "cipher-only - NULL algo - x1 - decryption",
		.test_data = &null_test_data_chain_x1_multiple,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_NULL |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
};

static const struct blockcipher_test_case aes_docsis_test_cases[] = {

	{
		.test_descr = "AES-DOCSIS-BPI Full Block Encryption",
		.test_data = &aes_test_data_docsis_1,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-DOCSIS-BPI Runt Block Encryption",
		.test_data = &aes_test_data_docsis_2,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-DOCSIS-BPI Uneven Encryption",
		.test_data = &aes_test_data_docsis_3,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-DOCSIS-BPI Full Block Decryption",
		.test_data = &aes_test_data_docsis_1,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-DOCSIS-BPI Runt Block Decryption",
		.test_data = &aes_test_data_docsis_2,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-DOCSIS-BPI Uneven Decryption",
		.test_data = &aes_test_data_docsis_3,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_MB |
			BLOCKCIPHER_TEST_TARGET_PMD_QAT
	},
	{
		.test_descr = "AES-DOCSIS-BPI OOP Full Block Encryption",
		.test_data = &aes_test_data_docsis_1,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
	{
		.test_descr = "AES-DOCSIS-BPI OOP Runt Block Encryption",
		.test_data = &aes_test_data_docsis_2,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},

	{
		.test_descr = "AES-DOCSIS-BPI OOP Uneven Block Encryption",
		.test_data = &aes_test_data_docsis_3,
		.op_mask = BLOCKCIPHER_TEST_OP_ENCRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
	{
		.test_descr = "AES-DOCSIS-BPI OOP Full Block Decryption",
		.test_data = &aes_test_data_docsis_1,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
	{
		.test_descr = "AES-DOCSIS-BPI OOP Runt Block Decryption",
		.test_data = &aes_test_data_docsis_2,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
	{
		.test_descr = "AES-DOCSIS-BPI OOP Uneven Block Decryption",
		.test_data = &aes_test_data_docsis_3,
		.op_mask = BLOCKCIPHER_TEST_OP_DECRYPT,
		.feature_mask = BLOCKCIPHER_TEST_FEATURE_OOP,
		.pmd_mask = BLOCKCIPHER_TEST_TARGET_PMD_QAT |
			BLOCKCIPHER_TEST_TARGET_PMD_MB
	},
};
#endif /* TEST_CRYPTODEV_AES_TEST_VECTORS_H_ */
