/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "tabs.h"
#include "labelbutton.h"
#include "button.h"
#include "backend.h"
#include "font.h"
#include "skin.h"

#define _XITK_TABS_SINGLE_SKIN 1


typedef struct {
  xitk_widget_t           w;

  int                     all_width, lgap, rgap;
  int                     num_entries, selected, start, stop;

  xitk_image_t           *skin;
  xitk_widget_t          *left, *right, *tabs[MAX_TABS];
  const char             *labels[MAX_TABS];

  xitk_int_callback_t     callback;
} _tabs_private_t;

static void _tabs_paint (_tabs_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;
  xitk_hv_t hv1, hv2, hv3, hv4;
  widget_event_t ne;
  uint32_t d = wp->w.state ^ wp->w.shown_state;

  if (!(wp->w.state & XITK_WIDGET_STATE_VISIBLE))
    return;
  if (!(wp->w.wl->flags & XITK_WL_EXPOSED))
    return;

  if (d & XITK_WIDGET_STATE_FOCUS) {
    if (wp->w.state & XITK_WIDGET_STATE_ENABLE)
      xitk_widgets_state (wp->tabs + wp->selected, 1, XITK_WIDGET_STATE_FOCUS, wp->w.state);
  }

  /* We have up to 2 partly visible tabs here. They are marked invisible
   * to stop engine from painting too much of them. Instead, switch them
   * visible temporarily and relay the clipped group paint event to them
   * manually here. */

  if (wp->lgap > 0) do {
    if (!wp->tabs[wp->start - 1])
      break;
    XITK_HV_H (hv1) = event->x;
    XITK_HV_V (hv1) = event->y;
    XITK_HV_H (hv2) = event->width;
    XITK_HV_V (hv2) = event->height;
    hv2.w += hv1.w;
    hv1.w = xitk_hv_max (hv1.w, wp->w.pos.w);
    hv3.w = wp->w.size.w;
    XITK_HV_H (hv3) = wp->lgap;
    hv3.w += wp->w.pos.w;
    hv2.w = xitk_hv_min (hv2.w, hv3.w);
    hv2.w -= hv1.w;
    if (!XITK_HV_IS_RECT (hv2))
      break;
    ne.x = XITK_HV_H (hv1);
    ne.y = XITK_HV_V (hv1);
    ne.width = XITK_HV_H (hv2);
    ne.height = XITK_HV_V (hv2);
    ne.type = WIDGET_EVENT_PAINT;
    wp->tabs[wp->start - 1]->state |= XITK_WIDGET_STATE_VISIBLE;
    wp->tabs[wp->start - 1]->event (wp->tabs[wp->start - 1], &ne);
    wp->tabs[wp->start - 1]->state &= ~XITK_WIDGET_STATE_VISIBLE;
  } while (0);

  if (wp->rgap > 0) do {
    if (!wp->tabs[wp->stop])
      break;
    XITK_HV_H (hv1) = event->x;
    XITK_HV_V (hv1) = event->y;
    XITK_HV_H (hv2) = event->width;
    XITK_HV_V (hv2) = event->height;
    hv2.w += hv1.w;
    hv3.w = hv4.w = wp->w.pos.w + wp->w.size.w - 40;
    XITK_HV_V (hv3) = XITK_HV_V (wp->w.pos);
    hv3.w -= wp->rgap;
    hv1.w = xitk_hv_max (hv1.w, hv3.w);
    hv2.w = xitk_hv_min (hv2.w, hv4.w);
    hv2.w -= hv1.w;
    if (!XITK_HV_IS_RECT (hv2))
      break;
    ne.x = XITK_HV_H (hv1);
    ne.y = XITK_HV_V (hv1);
    ne.width = XITK_HV_H (hv2);
    ne.height = XITK_HV_V (hv2);
    ne.type = WIDGET_EVENT_PAINT;
    wp->tabs[wp->stop]->state |= XITK_WIDGET_STATE_VISIBLE;
    wp->tabs[wp->stop]->event (wp->tabs[wp->stop], &ne);
    wp->tabs[wp->stop]->state &= ~XITK_WIDGET_STATE_VISIBLE;
  } while (0);
}

static int _tabs_arrange (_tabs_private_t *wp, int item) {
  XITK_HV_INIT;
  int width = XITK_HV_H (wp->w.size);

  if (width >= wp->all_width) {
    if (wp->start == 0)
      return 0;
    xitk_widgets_state (wp->tabs + wp->start, wp->stop - wp->start, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0);
    wp->start = 0;
    wp->stop = wp->num_entries;
  } else if (item >= wp->stop) {
    int i;
    /* shift left. */
    if (item >= wp->num_entries)
      return 0;
    xitk_widgets_state (wp->tabs + wp->start, wp->stop - wp->start, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0);
    wp->stop = item + 1;
    width -= 40;
    /* count left hand tabs that fit completely. */
    for (i = wp->stop - 1; i >= 0; i--) {
      int w = xitk_get_widget_width (wp->tabs[i]);
      width -= w;
      if (width < 0) {
        width += w;
        break;
      }
    }
    wp->start = i + 1;
    /* count possible small right hand tabs that add completely. */
    while (wp->stop < wp->num_entries) {
      int w = xitk_get_widget_width (wp->tabs[wp->stop]);
      width -= w;
      if (width < 0) {
        width += w;
        break;
      }
      wp->stop++;
    }
  } else if (item < wp->start) {
    int i;
    /* shift right. */
    if (item < 0)
      return 0;
    xitk_widgets_state (wp->tabs + wp->start, wp->stop - wp->start, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0);
    wp->start = item;
    width -= 40;
    /* count right hand tabs that fit completely. */
    for (i = wp->start; i < wp->num_entries; i++) {
      int w = xitk_get_widget_width (wp->tabs[i]);
      width -= w;
      if (width < 0) {
        width += w;
        break;
      }
    }
    wp->stop = i;
    /* count ossible small left hand tabs that add completely. */
    while (wp->start > 0) {
      int w = xitk_get_widget_width (wp->tabs[wp->start - 1]);
      width -= w;
      if (width < 0) {
        width += w;
        break;
      }
      wp->start--;
    }
  } else {
    return 0;
  }

  if (wp->start > 0) {
    if (wp->stop < wp->num_entries) { /* middle */
      wp->lgap = width >> 1;
      wp->rgap = width - wp->lgap;
    } else { /* right */
      wp->lgap = width;
      wp->rgap = 0;
    }
  } else { /* left */
    wp->rgap = (wp->stop < wp->num_entries) ? width : 0;
    wp->lgap = 0;
  }
  xitk_widgets_state (&wp->left, 1, XITK_WIDGET_STATE_ENABLE, wp->start <= 0 ? 0 : ~0u);
  xitk_widgets_state (&wp->right, 1, XITK_WIDGET_STATE_ENABLE, wp->stop < wp->num_entries ? ~0u : 0);

  {
    int i, x = XITK_HV_H (wp->w.pos) + wp->lgap;
    if (wp->start > 0) {
      int w = xitk_get_widget_width (wp->tabs[wp->start - 1]);
      xitk_set_widget_pos (wp->tabs[wp->start - 1], x - w, XITK_HV_V (wp->w.pos));
    }
    for (i = wp->start; i < wp->stop; i++) {
      int w = xitk_get_widget_width (wp->tabs[i]);
      xitk_set_widget_pos (wp->tabs[i], x, XITK_HV_V (wp->w.pos));
      x += w;
    }
    xitk_widgets_state (wp->tabs + wp->start, wp->stop - wp->start,
      XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_RECHECK_MOUSE,
      wp->w.state | XITK_WIDGET_STATE_RECHECK_MOUSE);
    if (wp->stop < wp->num_entries)
      xitk_set_widget_pos (wp->tabs[wp->stop], x, XITK_HV_V (wp->w.pos));
  }

  if (wp->w.state & XITK_WIDGET_STATE_VISIBLE) {
    if ((wp->lgap > 0) && wp->tabs[wp->start - 1]) {
      widget_event_t ne;
      ne.type = WIDGET_EVENT_PAINT;
      ne.x = XITK_HV_H (wp->w.pos);
      ne.y = XITK_HV_V (wp->w.pos);
      ne.width = wp->lgap;
      ne.height = XITK_HV_V (wp->w.size);
      wp->tabs[wp->start - 1]->state |= XITK_WIDGET_STATE_VISIBLE;
      wp->tabs[wp->start - 1]->event (wp->tabs[wp->start - 1], &ne);
      wp->tabs[wp->start - 1]->state &= ~XITK_WIDGET_STATE_VISIBLE;
    }
    if ((wp->rgap > 0) && wp->tabs[wp->stop]) {
      widget_event_t ne;
      ne.type = WIDGET_EVENT_PAINT;
      ne.x = XITK_HV_H (wp->w.pos) + XITK_HV_H (wp->w.size) - 40 - wp->rgap;
      ne.y = XITK_HV_V (wp->w.pos);
      ne.width = wp->rgap;
      ne.height = XITK_HV_V (wp->w.size);
      wp->tabs[wp->stop]->state |= XITK_WIDGET_STATE_VISIBLE;
      wp->tabs[wp->stop]->event (wp->tabs[wp->stop], &ne);
      wp->tabs[wp->stop]->state &= ~XITK_WIDGET_STATE_VISIBLE;
    }
  }

  return 1;
}

/*
 *
 */
static void _tabs_shift (xitk_widget_t *w, void *data) {
  _tabs_private_t *wp = (_tabs_private_t *)data;

  _tabs_arrange (wp, (w == wp->left) ? wp->start - 1 : wp->stop);
}

static int _tabs_event (xitk_widget_t *w, const widget_event_t *event) {
  _tabs_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp || !event)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_TABS)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      _tabs_paint (wp, event);
      break;
    case WIDGET_EVENT_KEY:
      {
        int i;
        if (!(wp->w.state & XITK_WIDGET_STATE_ENABLE))
          return 0;
        i = xitk_browser_list_nav (event, wp->labels, -wp->num_entries, wp->selected, 2);
        if (i >= 0) {
          uint32_t mask;
          switch (event->string[1]) {
            case XITK_MOUSE_WHEEL_UP:
            case XITK_MOUSE_WHEEL_DOWN:
              mask = XITK_WIDGET_STATE_ON;
              break;
            default:
              mask = XITK_WIDGET_STATE_ON | XITK_WIDGET_STATE_FOCUS;
          }
          if (i == wp->selected)
            return 1;
          xitk_widgets_state (wp->tabs + wp->selected, 1, mask, 0);
          wp->selected = i;
          _tabs_arrange (wp, wp->selected);
          xitk_widgets_state (wp->tabs + wp->selected, 1, mask, ~0u);
          if (wp->callback)
            wp->callback (&wp->w, wp->w.userdata, wp->selected, event->modifier);
          return 1;
        }
      }
      return 0;
    case WIDGET_EVENT_DESTROY:
#ifdef _XITK_TABS_SINGLE_SKIN
      xitk_widgets_delete (wp->tabs, wp->num_entries);
      xitk_image_free_image (&wp->skin);
#endif
      break;
    case WIDGET_EVENT_ENABLE:
      {
        uint32_t state_left = wp->w.state, state_right = wp->w.state;
        if (wp->start <= 0)
          state_left &= ~XITK_WIDGET_STATE_ENABLE;
        if (wp->stop >= wp->num_entries)
          state_right &= ~XITK_WIDGET_STATE_ENABLE;
        xitk_widgets_state (&wp->left, 1,
          XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, state_left);
        xitk_widgets_state (&wp->right, 1,
          XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, state_right);
        xitk_widgets_state (wp->tabs + wp->start, wp->stop - wp->start,
          XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, wp->w.state);
      }
      break;
    case WIDGET_EVENT_SELECT:
      if (XITK_0_TO_MAX_MINUS_1 (event->button, wp->num_entries) && (event->button != wp->selected)) {
        uint32_t state = wp->tabs[wp->selected] ? wp->tabs[wp->selected]->state : 0;
        xitk_widgets_state (wp->tabs + wp->selected, 1,
          XITK_WIDGET_STATE_FOCUS | XITK_WIDGET_STATE_ON, 0);
        wp->selected = event->button;
        _tabs_arrange (wp, event->button);
        xitk_widgets_state (wp->tabs + wp->selected, 1,
          XITK_WIDGET_STATE_FOCUS | XITK_WIDGET_STATE_ON, state | XITK_WIDGET_STATE_ON);
      }
      return wp->selected;
    default: ;
  }
  return 0;
}

/*
 *
 */
static void _tabs_select (xitk_widget_t *w, void *data, int select, unsigned int modifier) {
  _tabs_private_t *wp = (_tabs_private_t *)data;

  if (select) {
    xitk_widgets_state (wp->tabs + wp->selected, 1, XITK_WIDGET_STATE_FOCUS | XITK_WIDGET_STATE_ON, 0);
    wp->selected = xitk_widget_user_id (w);
    if (wp->callback)
      wp->callback (&wp->w, wp->w.userdata, wp->selected, modifier);
  } else {
    xitk_widgets_state (wp->tabs + wp->selected, 1, XITK_WIDGET_STATE_FOCUS | XITK_WIDGET_STATE_ON, ~0u);
  }
}

/*
 *
 */
xitk_widget_t *xitk_noskin_tabs_create (const xitk_tabs_widget_t *t, int x, int y, int width, const char *fontname) {
  _tabs_private_t *wp;
  const char * const *entries;
  int xa[MAX_TABS + 1];
  static const char *none = "---";
  XITK_HV_INIT;

  wp = (_tabs_private_t *)xitk_widget_new (&t->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  if (t->entries && (t->num_entries > 0)) {
    wp->num_entries = t->num_entries <= MAX_TABS ? t->num_entries : MAX_TABS;
    entries = t->entries;
  } else {
    wp->num_entries = 1;
    entries = &none;
  }
  wp->callback    = t->callback;

  wp->w.state    &= ~XITK_WIDGET_STATE_VISIBLE;
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  XITK_HV_H (wp->w.size) = width;
  wp->w.type      = WIDGET_GROUP | WIDGET_TYPE_TABS | WIDGET_PARTIAL_PAINTABLE | WIDGET_KEYABLE | WIDGET_TABABLE;
  wp->w.event     = _tabs_event;

  {
    xitk_font_t *fs = xitk_font_load_font (wp->w.wl->xitk, fontname);
    int fheight = xitk_font_text_height (fs, " ", 1);
    int xx, i;
    xitk_labelbutton_widget_t  lb = {
      .nw = {
        .wl = wp->w.wl,
        .userdata = wp,
        .group = &wp->w,
        .add_state = XITK_WIDGET_STATE_CLEAR,
        .mode_mask = WIDGET_TABABLE | WIDGET_GROUP_MEMBER | WIDGET_GROUP_TABS,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_TABS,
      },
      .button_type    = TAB_BUTTON,
      .align          = ALIGN_CENTER,
      .callback = _tabs_select
    };
#ifdef _XITK_TABS_SINGLE_SKIN
    xitk_skin_element_info_t info = {
      .x = x,
      .y = y,
      .visibility = 1,
      .enability = 1,
      .pixmap_img = {
        .height = (fheight + 18) * 4,
        .num_states = -4
      },
      .label_alignment = ALIGN_CENTER,
      .label_printable = 1,
      .label_color = XITK_NOSKIN_TEXT_NORM,
      .label_color_focus = XITK_NOSKIN_TEXT_NORM,
      .label_color_click = XITK_NOSKIN_TEXT_NORM,
      .label_fontname = fontname,
    };
#endif
    XITK_HV_V (wp->w.size) = fheight + 18;
#ifdef _XITK_TABS_SINGLE_SKIN
    for (xx = i = 0; i < wp->num_entries; i++)
      xx += (xa[i] = xitk_font_text_width (fs, entries[i], -1) + 20);
    if (wp->w.wl->xitk->verbosity >= 2)
      printf ("xitk.tabs.noskin_image (%d x %d).\n", xx, (int)info.pixmap_img.height);
    info.pixmap_img.image = wp->skin = xitk_image_new (wp->w.wl->xitk, NULL, 0, xx, info.pixmap_img.height);
    for (xx = x, i = 0; i < wp->num_entries; i++) {
      info.x = xx;
      info.pixmap_img.x = xx - x;
      info.pixmap_img.width = xa[i];
      xitk_image_draw_tab (&info.pixmap_img);
      lb.nw.user_id = i;
      lb.label = entries[i];
      wp->tabs[i] = xitk_info_labelbutton_create (&lb, &info);
      wp->labels[i] = xitk_labelbutton_get_label (wp->tabs[i]);
      xa[i] = xx;
      xx += info.pixmap_img.width;
    }
#else
    for (xx = x, i = 0; i < wp->num_entries; i++) {
      int fwidth = xitk_font_text_width (fs, entries[i], -1) + 20;

      xa[i] = xx;
      lb.nw.user_id = i;
      lb.label = entries[i];
      wp->tabs[i] = xitk_noskin_labelbutton_create (&lb,
        xx, y, fwidth, XITK_HV_V (wp->w.size), XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);
      wp->labels[i] = xitk_labelbutton_get_label (wp->tabs[i]);
      xx += fwidth;
    }
#endif
    xa[i] = xx;
    wp->all_width = xx - x;
    xitk_font_unload_font (fs);
    xitk_widgets_state (wp->tabs, i, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0);
  }

  /* Add left/rigth arrows */
  if (wp->all_width > XITK_HV_H (wp->w.size)) {
    int i, right;
    xitk_button_widget_t b = {
      .nw = {
        .wl = wp->w.wl,
        .group = &wp->w,
        .add_state = XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_TABABLE | WIDGET_GROUP_MEMBER | WIDGET_GROUP_TABS,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_TABS,
        .userdata = wp
      },
      .callback = _tabs_shift
    };

    b.symbol = XITK_SYMBOL_LEFT;
    wp->left = xitk_noskin_button_create (&b,
      XITK_HV_H (wp->w.pos) + width - 40, y - 1 + XITK_HV_V (wp->w.size) - 20, 20, 20);

    b.symbol = XITK_SYMBOL_RIGHT;
    b.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
    wp->right = xitk_noskin_button_create (&b,
      XITK_HV_H (wp->w.pos) + width - 20, y - 1 + XITK_HV_V (wp->w.size) - 20, 20, 20);

    right = XITK_HV_H (wp->w.pos) + XITK_HV_H (wp->w.size) - 40;
    for (i = 0; xa[i + 1] <= right; i++) ;
    wp->stop = i;
    wp->rgap = right - xa[i];
  } else {
    if (xitk_init_NULL ()) {
      wp->left = NULL;
      wp->right = NULL;
    }
    wp->stop = wp->num_entries;
#if 0
    wp->rgap = 0;
#endif
  }
#if 0
  wp->selected = 0;
  wp->start = 0;
  wp->lgap = 0;
#endif
  xitk_widgets_state (wp->tabs + wp->selected, 1, XITK_WIDGET_STATE_ON, ~0u);
  return _xitk_new_widget_apply (&t->nw, &wp->w);
}
