<?php
/*
 * tweeper - a Twitter to RSS web scraper
 *
 * Copyright (C) 2013-2015  Antonio Ospite <ao2@ao2.it>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once 'XML/Serializer.php';

date_default_timezone_set('UTC');

class Tweeper {

  private static $USER_AGENT = "Mozilla/5.0 (Windows NT 6.1; rv:22.0) Gecko/20130405 Firefox/22.0";

  public function __construct($generate_enclosure = FALSE) {
    $this->generate_enclosure = $generate_enclosure;
  }

  public static function epoch_to_gmdate($timestamp)
  {
    if (!is_numeric($timestamp) || is_nan($timestamp)) {
      $timestamp = 0;
    }

    return gmdate('D, d M Y H:i:s', $timestamp) . ' GMT';
  }

  public static function str_to_gmdate($date)
  {
    $timestamp = strtotime($date);
    if (FALSE === $timestamp) {
      $timestamp = 0;
    }

    return Tweeper::epoch_to_gmdate($timestamp);
  }

  private static function get_contents($url)
  {
    $ch = curl_init($url);
    curl_setopt_array($ch, array(
      CURLOPT_HEADER => FALSE,
      CURLOPT_FOLLOWLOCATION => TRUE,  // follow http redirects to get the real URL
      CURLOPT_RETURNTRANSFER => TRUE,
      CURLOPT_SSL_VERIFYHOST => FALSE,
      CURLOPT_SSL_VERIFYPEER => FALSE,
      CURLOPT_HTTPHEADER => array('Accept-language: en'),
      CURLOPT_USERAGENT => Tweeper::$USER_AGENT,
    ));
    $contents = curl_exec($ch);
    curl_close($ch);

    return $contents;
  }

  private static function get_info($url)
  {
    $ch = curl_init($url);
    curl_setopt_array($ch, array(
      CURLOPT_HEADER => TRUE,
      CURLOPT_NOBODY => TRUE,
      CURLOPT_FOLLOWLOCATION => TRUE,  // follow http redirects to get the real URL
      CURLOPT_RETURNTRANSFER => TRUE,
      CURLOPT_SSL_VERIFYHOST => FALSE,
      CURLOPT_SSL_VERIFYPEER => FALSE,
      CURLOPT_USERAGENT => Tweeper::$USER_AGENT,
    ));
    curl_exec($ch);
    $url_info = curl_getinfo($ch);
    curl_close($ch);

    return $url_info;
  }

  public static function generate_enclosure($url)
  {
    $supported_content_types = array(
      "application/ogg",
      "audio/aac",
      "audio/mp4",
      "audio/mpeg",
      "audio/ogg",
      "audio/vorbis",
      "audio/wav",
      "audio/webm",
      "audio/x-midi",
      "image/gif",
      "image/jpeg",
      "video/avi",
      "video/mp4",
      "video/mpeg",
      "video/ogg",
    );

    // The RSS specification says that the enclosure element url must be http.
    // See http://sourceforge.net/p/feedvalidator/bugs/72/
    $http_url = preg_replace("/^https/", "http", $url);

    $url_info = Tweeper::get_info($http_url);

    $supported = in_array($url_info['content_type'], $supported_content_types);
    if (!$supported) {
      error_log("Unsupported enclosure content type \"" . $url_info['content_type'] . "\" for URL: " . $url_info['url']);
      return '';
    }

    $dom = new DomDocument();
    $enc = $dom->createElement('enclosure');
    $enc->setAttribute('url', $url_info['url']);
    $enc->setAttribute('length', $url_info['download_content_length']);
    $enc->setAttribute('type', $url_info['content_type']);

    $dom->appendChild($enc);

    return $dom->saveXML($enc);
  }

  /* Mimic the message from libxml.c::php_libxml_ctx_error_level() */
  private function log_xml_error($error) {
    $output = "";

    switch ($error->level) {
    case LIBXML_ERR_WARNING:
      $output .= "Warning $error->code: ";
      break;
    case LIBXML_ERR_ERROR:
      $output .= "Error $error->code: ";
      break;
    case LIBXML_ERR_FATAL:
      $output .= "Fatal Error $error->code: ";
      break;
    }

    $output .= trim($error->message);

    if ($error->file) {
      $output .= " in $error->file";
    } else {
      $output .= " in Entity,";
    }

    $output .=" line $error->line";

    error_log($output);
  }

  private function load_stylesheet($host) {
    $stylesheet = "file://" . __DIR__ . "/rss_converter_" . $host . ".xsl";
    if (FALSE === file_exists($stylesheet)) {
      trigger_error("Conversion to RSS not supported for $host ($stylesheet not found)", E_USER_ERROR);
      return NULL;
    }

    $stylesheet_contents = $this->get_contents($stylesheet);

    $xslDoc = new DOMDocument();
    $xslDoc->loadXML($stylesheet_contents);

    $xsltProcessor = new XSLTProcessor();
    $xsltProcessor->registerPHPFunctions();
    $xsltProcessor->setParameter('', 'generateEnclosure', $this->generate_enclosure);
    $xsltProcessor->importStylesheet($xslDoc);

    return $xsltProcessor;
  }

  private function json_to_xml($html, $json_match_expr, $rootName) {
    // pre-process, convert json to XML
    $ret = preg_match($json_match_expr, $html, $matches);
    if ($ret !== 1) {
      trigger_error("Cannot match expression: $json_match_expr\n", E_USER_ERROR);
      return NULL;
    }

    $data = json_decode($matches[1]);
    if (!$data) {
      return NULL;
    }

    $serializer_options = array (
      'addDecl' => TRUE,
      'encoding' => "UTF-8",
      'indent' => '  ',
      'rootName' => $rootName,
    );

    $serializer = new XML_Serializer($serializer_options);

    $status = $serializer->serialize($data);
    if (PEAR::isError($status)) {
      trigger_error($status->getMessage(), E_USER_ERROR);
      return NULL;
    }

    return $serializer->getSerializedData();
  }

  private function get_xml_instagram_com($html) {
    return $this->json_to_xml($html, '/window._sharedData = (.*);/', 'instagram');
  }

  private function preprocess_html_facebook_com($html) {
    $html = str_replace('<!--', '', $html);
    $html = str_replace('-->', '', $html);
    return $html;
  }

  private function html_to_xml($html, $host) {
    $xmlDoc = new DOMDocument();

    // Handle warnings and errors when loading invalid HTML.
    $xml_errors_value = libxml_use_internal_errors(true);

    // If there is a host-specific method to get the xml data, use it!
    $get_xml_host_method = 'get_xml_' . str_replace(".", "_", $host);
    if (method_exists($this, $get_xml_host_method)) {
      $xml_data = call_user_func_array(array($this, $get_xml_host_method), array($html));
      $xmlDoc->loadXML($xml_data);
    } else {
      $xmlDoc->loadHTML($html);
    }

    foreach (libxml_get_errors() as $xml_error) {
      $this->log_xml_error($xml_error);
    }
    libxml_clear_errors();
    libxml_use_internal_errors($xml_errors_value);

    return $xmlDoc;
  }

  public function tweep($src_url) {
    $url = parse_url($src_url);
    if (FALSE === $url || empty($url["host"])) {
      trigger_error("Invalid url: $src_url", E_USER_ERROR);
      return NULL;
    }

    // Strip the leading www. to be more forgiving on input URLs
    $host = preg_replace('/^www\./', '', $url["host"]);

    $xsltProcessor = $this->load_stylesheet($host);
    if (NULL === $xsltProcessor) {
      return NULL;
    }

    $html = $this->get_contents($src_url);
    if (FALSE === $html) {
      return NULL;
    }

    $preprocess_html_host_method = 'preprocess_html_' . str_replace(".", "_", $host);
    if (method_exists($this, $preprocess_html_host_method)) {
      $html = call_user_func_array(array($this, $preprocess_html_host_method), array($html));
    }

    $xmlDoc = $this->html_to_xml($html, $host);
    if (NULL === $xmlDoc) {
      return NULL;
    }

    $output = $xsltProcessor->transformToXML($xmlDoc);

    if (FALSE === $output) {
      trigger_error('XSL transformation failed.', E_USER_ERROR);
      return NULL;
    }
    return $output;
  }
}

function is_cli()
{
  return (php_sapi_name() === "cli");
}

function usage($argv)
{
  if (is_cli()) {
    $usage = "{$argv[0]} [-e|-h|--help] <src_url>\n";
  } else {
    $usage = htmlentities("{$_SERVER['SCRIPT_NAME']}?src_url=<src_url>&generate_enclosure=<0|1>");
  }

  return "usage: $usage";
}

function parse_options_cli($argv, $argc)
{
  $options = array(
    'generate_enclosure' => FALSE
  );

  if ($argc < 2)
    return $options;

  $cli_options = getopt("eh", array("help"));
  foreach ($cli_options as $opt => $val) {
    switch ($opt) {
    case 'e':
      $options['generate_enclosure'] = TRUE;
      break;
    case 'h':
    case 'help':
      echo usage($argv);
      exit(0);
    default:
      fwrite(STDERR, usage($argv));
      exit(1);
    }
  }

  $options['src_url'] = $argv[count($cli_options) + 1];

  return $options;
}

function parse_options_query_string()
{
  $options = array(
    'generate_enclosure' => FALSE
  );

  if (isset($_GET['src_url']))
    $options['src_url'] = $_GET['src_url'];

  if (isset($_GET['generate_enclosure']))
    $options['generate_enclosure'] = $_GET['generate_enclosure'] == 1;

  return $options;
}


if (is_cli()) {
  $options = parse_options_cli($argv, $argc);
  $ERROR_STREAM = fopen('php://stderr', 'w');
} else {
  $options = parse_options_query_string();
  $ERROR_STREAM = fopen('php://output', 'w');
}

if (!isset($options['src_url'])) {
  fwrite($ERROR_STREAM, usage(is_cli() ? $argv : NULL));
  exit(1);
}

$tweeper = new Tweeper($options['generate_enclosure']);
echo $tweeper->tweep($options['src_url']);
