#include "rheolef.h"
using namespace std;
using namespace rheolef;

// banded level set: on the unit sphere
Float phi (const point& x) { return norm(x) - 1; }

// first test:
point normal_exact (const point& x) { return x/norm(x); }; // normal in X on C(0,1) is vector OX:
// note: div_s(n) = d-1

// second test:
point u_exact (const point& x) { 
       return point (x[0]*x[1], x[1]*x[2], (x[0] + x[1])*x[2]); 
}; 
// exact solution  
Float div_u (const point& x) { 
     return   x[0] + 2*x[1] + x[2]
            - 2*x[1]*sqr(x[0]) 
            - 2*x[2]*sqr(x[1]) 
            - 2*x[0]*sqr(x[2]) 
            - 2*x[1]*sqr(x[2]) ; 
}
int
compute_error_s (
  const geo&                    omega,
  const string&                 approx,
  const quadrature_option_type& qopt,
  const Float&                  tol,
  bool                          test_normal)
{
  space Xh  (omega, approx);
  space Xvh (omega, approx, "vector");
  trial p (Xh); test q (Xh);
  trial u (Xvh); test v (Xvh);
  form m = integrate (omega, p*q, qopt);
  form b = integrate (omega, div_s(u)*q, qopt);

  // compute div_uh = div_s(uh)
  field uh = test_normal ? interpolate(Xvh, normal_exact) : interpolate (Xvh, u_exact);
  field lh = b*uh;
  solver sm (m.uu());
  field div_uh (Xh);
  div_uh.set_u() = sm.solve(lh.u());
  
  size_t d = omega.dimension();
  field pi_h_div_u = test_normal ? field(Xh, d-1) : interpolate (Xh, div_u); 
  field err_h  = div_uh - pi_h_div_u; 
  Float err_l2   = sqrt(m(err_h, err_h));
  Float err_linf = err_h.max_abs();
  dout << "# nelt err_l2 err_linf" << endl
       << omega.dis_size() << " " << err_l2 << " " << err_linf << endl;
  return (err_linf <= tol) ? 0 : 1;
}
int
compute_error_band (
  const geo&                    lambda,
  const string&                 approx,
  const quadrature_option_type& qopt,
  const Float&                  tol,
  bool                          test_normal)
{
#ifndef _RHEOLEF_HAVE_STD_INITIALIZER_LIST
  // compiler cannot support concatenation of csr matrix
  return 0;
#else // _RHEOLEF_HAVE_STD_INITIALIZER_LIST
  space Xh  (lambda, approx);
  field phi_h = interpolate(Xh, phi);
  band gh (phi_h); // TODO: option with quadrangles
  space Bh  (gh.band(), approx);
  Bh.block ("isolated");
  Bh.unblock ("zero");
  trial p (Bh); test q (Bh);
  form m = integrate (gh, p*q, qopt);

  field phi_h_band = phi_h [gh.band()];
  vector<vec<Float> > c (gh.n_connected_component());
  for (size_t i = 0; i < c.size(); i++) {
    const domain& cci = gh.band() ["cc"+itos(i)];
    field phi_h_cci (Bh, 0);
    phi_h_cci [cci] = phi_h_band [cci];
    c[i] = phi_h_cci.u();
  }
  csr<Float> M = { { m.uu(), trans(c)},
                   { c,      Float(0)} };
  M.set_symmetry(true);
  // M.set_pattern_dimension(m.uu().pattern_dimension());
  solver sm = ldlt (M);

  // compute rhs div_uh = div_s(uh)
  space Bvh (gh.band(), approx, "vector");
  trial u (Bvh); test v (Bvh);
  form b = integrate (gh, div_s(u)*q, qopt);
  field uh = test_normal ? interpolate(Bvh, normal_exact) : interpolate (Bvh, u_exact);
  field lh = b*uh;
  vector<Float>       z (gh.n_connected_component(), 0);
  vec<Float> F =   { lh.u(),    z    };
 
  // solve: div_s(uh) on the band
  vec<Float> U = sm.solve (F);
  field div_uh (Bh,0);
  div_uh.set_u() = U [range(0,div_uh.u().size())];

  // compute the L2 error
  size_t d = lambda.dimension();
  field pi_h_div_u = test_normal ? field(Bh, d-1) : interpolate (Bh, div_u); 
  field err_h  = div_uh - pi_h_div_u; 
  Float err_l2   = sqrt(m(err_h, err_h));

  // projet div_s(uh) on Gamma_h for the L^infty error
  space Wh (gh.level_set(), approx);
  field err_h_gamma = interpolate (Wh, err_h);
  Float err_linf = err_h_gamma.max_abs();
  dout << "# nelt err_l2 err_linf" << endl
       << gh.band().dis_size() << " " << err_l2 << " " << err_linf << endl;
  return (err_linf <= tol) ? 0 : 1;
#endif // _RHEOLEF_HAVE_STD_INITIALIZER_LIST
}
// usage: prog mesh.geo [-normal|-polynom] [-band] qorder tol
int main (int argc, char**argv) {
  environment rheolef (argc, argv);
  quadrature_option_type qopt;
  qopt.set_family (quadrature_option_type::gauss);
  geo omega (argv[1]);
  bool test_normal = false;
  int io = 2;
  if (argc > io && (argv[io] == string("-normal") || 
                    argv[io] == string("-polynom")  )) {
    test_normal =  (argv[io++] == string("-normal"));
  }
  string approx = argv[io++];
  qopt.set_order  (atoi(argv[io++]));
  Float tol = (argc > io) ? atof(argv[io++]) : 1e-7;
  derr << "approx      = " << approx << endl
       << "test_normal = " << test_normal << endl
       << "qorder      = " << qopt.get_order() << endl
       << "tol         = " << tol << endl;

  return (omega.map_dimension() < omega.dimension()) ?
    compute_error_s    (omega, approx, qopt, tol, test_normal):
    compute_error_band (omega, approx, qopt, tol, test_normal);
}
