#' Prints the status of a Slurm job and, if completed, its console/error output
#'
#' Prints the status of a Slurm job and, if completed, its console/error output.
#'
#' If the specified Slurm job is still in the queue or running, this function
#' prints its current status (as output by the Slurm \code{squeue} command).
#' The output displays one row by node currently running part of the job ("R" in
#' the "ST" column) and how long it has been running ("TIME"). One row indicates
#' the portions of the job still in queue ("PD" in the "ST" column), if any. 
#' 
#' If all portions of the job have completed or stopped, the function prints the 
#' console and error output, if any, generated by each node.
#' 
#' @param slr_job A \code{slurm_job} object.
#' 
#' @name print_job_status-deprecated
#' @usage print_job_status(slr_job)
#' @seealso \code{\link{rslurm-deprecated}}
#' @keywords internal
NULL

#' @rdname rslurm-deprecated
#' @section \code{print_job_status}:
#' For \code{print_job_status}, use \code{\link{get_job_status}}.
#'
#' @export
print_job_status <- function(slr_job) {
    .Deprecated("get_job_status")
    
    if (!(class(slr_job) == "slurm_job")) stop("input must be a slurm_job")  
    stat <- suppressWarnings(
        system(paste("squeue -n", slr_job$jobname), intern = TRUE))
    if (length(stat) > 1) {
        cat(paste(c("Job running or in queue. Status:", stat), collapse = "\n"))
    } else {
        cat("Job completed or stopped. Printing console output below if any.\n")
        tmpdir <- paste0("_rslurm_", slr_job$jobname)
        out_files <- file.path(tmpdir, 
                               paste0("slurm_", 0:(slr_job$nodes - 1), ".out"))
        for (outf in out_files) {
            cat(paste("\n----", outf, "----\n\n"))
            cat(paste(readLines(outf), collapse = "\n"))
        }
    }
}  
