% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/addExperiments.R
\name{addExperiments}
\alias{addExperiments}
\title{Add Experiments to the Registry}
\usage{
addExperiments(
  prob.designs = NULL,
  algo.designs = NULL,
  repls = 1L,
  combine = "crossprod",
  reg = getDefaultRegistry()
)
}
\arguments{
\item{prob.designs}{[named list of \code{\link[base]{data.frame}}]\cr
Named list of data frames (or \code{\link[data.table]{data.table}}).
The name must match the problem name while the column names correspond to parameters of the problem.
If \code{NULL}, experiments for all defined problems without any parameters are added.}

\item{algo.designs}{[named list of \code{\link[data.table]{data.table}} or \code{\link[base]{data.frame}}]\cr
Named list of data frames (or \code{\link[data.table]{data.table}}).
The name must match the algorithm name while the column names correspond to parameters of the algorithm.
If \code{NULL}, experiments for all defined algorithms without any parameters are added.}

\item{repls}{[\code{integer()}]\cr
Number of replications for each problem design in `prob.designs` (automatically replicated to
the correct length).}

\item{combine}{[\code{character(1)}]\cr
How to combine the rows of a single problem design with the rows of a single algorithm design?
Default is \dQuote{crossprod} which combines each row of the problem design which each row of the algorithm design
in a cross-product fashion. Set to \dQuote{bind} to just \code{\link[base]{cbind}} the tables of
problem and algorithm designs where the shorter table is repeated if necessary.}

\item{reg}{[\code{\link{ExperimentRegistry}}]\cr
Registry. If not explicitly passed, uses the last created registry.}
}
\value{
[\code{\link[data.table]{data.table}}] with ids of added jobs stored in column \dQuote{job.id}.
}
\description{
Adds experiments (parametrized combinations of problems with algorithms) to the registry and thereby defines batch jobs.

If multiple problem designs or algorithm designs are provided, they are combined via the Cartesian product.
E.g., if you have two problems \code{p1} and \code{p2} and three algorithms \code{a1}, \code{a2} and \code{a3},
\code{addExperiments} creates experiments for all parameters for the combinations \code{(p1, a1)}, \code{(p1, a2)},
\code{(p1, a3)}, \code{(p2, a1)}, \code{(p2, a2)} and \code{(p2, a3)}.
}
\note{
R's \code{data.frame} converts character vectors to factors by default in R versions prior to 4.0.0 which frequently resulted in problems using \code{addExperiments}.
Therefore, this function will warn about factor variables if the following conditions hold:
\enumerate{
  \item R version is < 4.0.0
  \item The design is passed as a \code{data.frame}, not a \code{\link[data.table]{data.table}} or \code{\link[tibble]{tibble}}.
  \item The option \dQuote{stringsAsFactors} is not set or set to \code{TRUE}.
}
}
\examples{
\dontshow{ batchtools:::example_push_temp(1) }
tmp = makeExperimentRegistry(file.dir = NA, make.default = FALSE)

# add first problem
fun = function(job, data, n, mean, sd, ...) rnorm(n, mean = mean, sd = sd)
addProblem("rnorm", fun = fun, reg = tmp)

# add second problem
fun = function(job, data, n, lambda, ...) rexp(n, rate = lambda)
addProblem("rexp", fun = fun, reg = tmp)

# add first algorithm
fun = function(instance, method, ...) if (method == "mean") mean(instance) else median(instance)
addAlgorithm("average", fun = fun, reg = tmp)

# add second algorithm
fun = function(instance, ...) sd(instance)
addAlgorithm("deviation", fun = fun, reg = tmp)

# define problem and algorithm designs
library(data.table)
prob.designs = algo.designs = list()
prob.designs$rnorm = CJ(n = 100, mean = -1:1, sd = 1:5)
prob.designs$rexp = data.table(n = 100, lambda = 1:5)
algo.designs$average = data.table(method = c("mean", "median"))
algo.designs$deviation = data.table()

# add experiments and submit
addExperiments(prob.designs, algo.designs, reg = tmp)

# check what has been created
summarizeExperiments(reg = tmp)
unwrap(getJobPars(reg = tmp))
}
\seealso{
Other Experiment: 
\code{\link{removeExperiments}()},
\code{\link{summarizeExperiments}()}
}
\concept{Experiment}
