"""Experimental data for the dam break problem.

The data is extracted from plots in the following references:

Martin and Moyce 1952
"An Experimental Study of the Collapse of Liquid Columns on a Rigid Horizontal
Plane", J. C. Martin and W. J. Moyce, Philososphical Transactions of the Royal
Society of London Series A, 244, 312--324 (1952).
DOI:10.1098/rsta.1952.0007

"Moving-Particle Semi-Implicit Method for Fragmentation of Incompressible
fluid", S. Koshizuka and Y. Oka, Nuclear Science and Engineering, 123,
421--434 (1996).
DOI:10.13182/NSE96-A24205

"B.Buchner, Green Water on Ship - Type Off shore Structures, Ph.D.thesis,
TU Delft, Delft University of Technology (2002)."
URL: https://repository.tudelft.nl/islandora/object/uuid%3Af0c0bd67-d52a-4b79-8451-1279629a5b80

"L.Lobovsky, E.Botia-Vera, F.Castellana, J.Mas-Soler, A.Souto-Iglesias,
Experimental investigation of dynamic pressure loads during dam break,
J. Fluids Struct. 48 (2014) 407-434."
DOI:10.1016/j.jfluidstructs.2014.03.009


"K.M.T. Kleefsman, G.Fekken, A.E.P.Veldman, B.Iwanowski, B.Buchner,
A volume-of-fluid based simulation method for wave impact problems,
J. Com-put.Phys.206 (1)(2005)363-393."
DOI:10.1016/j.jcp.2004.12.007

and

"Peter E. Raad, Razvan Bidoae, The three-dimensional
Eulerian-Lagrangian marker and micro cell method
for the simulation of free surface flows, Journal of Computational Physics,
Volume 203, Issue 2, 2005, Pages 668-699,
ISSN 0021-9991"
DOI: https://doi.org/10.1016/j.jcp.2004.09.013


"""

import numpy as np
from io import StringIO

# This is the data for n^2=2 and a=1.125 from Figure 3.
mm_data_1 = u"""
0.849   1.245
1.212   1.443
1.602   1.884
2.283   2.689
2.950   3.728
3.598   4.528
3.905   4.999
4.592   5.841
4.961   6.271
5.316   6.717
"""

# This is the data for n^2=2 and a=2.25 from Figure 3.
mm_data_2 = u"""
0.832   1.217
1.219   1.474
1.997   2.292
2.547   2.995
3.345   4.134
4.034   4.944
4.418   5.881
5.091   6.980
5.685   7.945
6.306   8.966
6.822   9.986
7.439   10.963
8.031   11.977
8.633   13.005
9.237   13.970
"""

ko_data = u"""
0.0     1.000
0.381   1.111
0.769   1.252
1.153   1.505
1.537   1.892
1.935   2.241
2.323   2.615
2.719   3.003
3.096   3.624
"""

ko_mps_data = u"""
0.000   1.002
0.227   1.019
0.416   1.091
0.591   1.205
0.778   1.351
0.958   1.512
1.095   1.637
1.226   1.771
1.381   1.931
1.536   2.100
1.684   2.268
1.858   2.480
2.043   2.707
2.278   3.004
2.451   3.251
2.604   3.481
2.752   3.700
2.943   3.997
"""

buchner_exp_data = u"""
0.055	0.000
0.394	0.0280
0.466	-0.134
0.506	-0.162
0.553	0.1629
0.675	-0.078
0.674	0.0505
0.765	-0.050
0.844	0.0505
0.994	0.0786
1.055	-0.067
1.113	0.1292
1.156	-0.134
1.295	-0.078
1.323	0.1011
1.525	-0.056
1.594   0.0280
1.835	-0.044
1.864	0.0168
2.084	0.0112
2.095	-0.033
2.285   -0.028
2.244	0.0112
2.455   -0.028
2.494	0.0168
2.514	0.0786
2.493	0.1516
2.512  	0.258
2.611  	0.359
2.590  	0.455
2.599  	0.550
2.628  	0.601
2.678  	0.662
2.728   0.617
2.738  	0.691
2.828  	0.662
2.868  	0.601
2.968  	0.651
2.988  	0.606
3.109  	0.573
3.249  	0.544
3.519  	0.539
3.649  	0.539
3.909  	0.550
4.009  	0.556
4.189  	0.550
4.339  	0.556
4.509  	0.550
4.669   0.528
4.869  	0.522
5.149  	0.511
5.319  	0.539
5.399  	0.561
5.499  	0.589
5.648   0.617
5.708  	0.662
5.697  	0.724
5.767  	0.747
5.777  	0.780
5.806   0.853
5.836  	0.887
5.917   0.735
5.918  	0.685
5.928   0.646
5.989  	0.589
6.089  	0.556
6.199  	0.561
6.219  	0.595
6.308  	0.640
6.339  	0.589
6.349  	0.533
6.370  	0.488
6.460  	0.477
6.550  	0.488
6.699  	0.522
6.749  	0.561
6.769  	0.511
6.800  	0.460
6.860  	0.415
6.970  	0.432
7.020  	0.477
7.059  	0.505
7.100  	0.466
7.160  	0.455
7.210  	0.415
7.360  	0.449
"""

lobovsky_exp_data_p1 = u"""
0.081	0.023
0.247	0.020
0.412	0.017
0.578	0.014
0.743	0.011
0.909	0.008
1.074	0.005
1.239	0.002
1.405	0.000
1.570	0.003
1.736	0.006
1.901	0.009
2.066	0.012
2.232	0.015
2.382	0.004
2.427	0.173
2.427	0.364
2.435	1.612
2.435	1.260
2.435	0.908
2.435	0.556
2.435	1.973
2.450	2.843
2.450	2.499
2.498	2.040
2.564	1.953
2.630	1.725
2.675	1.551
2.758	1.335
2.829	1.202
2.924	1.062
3.089	0.883
3.170	0.832
3.255	0.775
3.420	0.718
3.585	0.689
3.751	0.707
3.916	0.644
4.082	0.599
4.247	0.555
4.413	0.559
4.578	0.532
4.743	0.578
4.909	0.607
5.074	0.573
5.240	0.531
5.405	0.544
5.571	0.579
5.736	0.621
5.901	0.793
5.974	0.909
6.067	0.816
6.232	0.839
6.307	0.794
6.398	0.824
6.495	0.790
6.563	0.766
6.649	0.696
6.729	0.700
6.886	0.651
6.982	0.618
"""

lobovsky_exp_data_p3 = u"""
0.098	0.006
0.200	0.007
0.303	0.007
0.405	0.007
0.508	0.006
0.610	0.007
0.713   0.007
0.816	0.007
0.918	0.007
1.021	0.007
1.123	0.007
1.226	0.007
1.328	0.007
1.431	0.007
1.534	0.007
1.636	0.007
1.739	0.007
1.841	0.007
1.944	0.007
2.046	0.007
2.149	0.007
2.251	0.007
2.354	0.007
2.461	0.006
2.516	0.083
2.524	0.201
2.531	0.286
2.538	0.395
2.545	0.495
2.547	0.568
2.559	0.668
2.564	0.778
2.573	0.891
2.583	0.995
2.592	1.092
2.599	1.193
2.610	1.291
2.624	1.372
2.671	1.495
2.774	1.518
2.848	1.480
2.890	1.395
2.937	1.306
2.988	1.219
3.039	1.131
3.086	1.051
3.119	0.962
3.151	0.872
3.193	0.779
3.254	0.693
3.342	0.625
3.445	0.616
3.548	0.572
3.650	0.572
3.753	0.592
3.858	0.577
3.900	0.538
3.933	0.642
3.954	0.599
4.008	0.564
4.060	0.540
4.133	0.507
4.162	0.606
4.200	0.512
4.250	0.564
4.275	0.681
4.296	0.618
4.358	0.597
4.405	0.581
4.450	0.603
4.491	0.551
4.509	0.605
4.533	0.655
4.566	0.649
4.611	0.642
4.650	0.623
4.666	0.597
4.713	0.568
4.775	0.558
4.816	0.571
4.866	0.590
4.918	0.611
5.021	0.591
5.123	0.597
5.226	0.571
5.329	0.576
5.431	0.592
5.534	0.600
5.636	0.623
5.739	0.644
5.824	0.687
5.884	0.776
5.900	0.824
5.932	0.874
5.953	0.927
5.975	0.863
6.025	0.796
6.083	0.831
6.134	0.853
6.158	0.896
6.229	0.869
6.258	0.902
6.275	0.850
6.300	0.805
6.336	0.837
6.391	0.811
6.440	0.803
6.508	0.798
6.538	0.762
6.558	0.727
6.608	0.696
6.650	0.759
6.684	0.710
6.747	0.647
6.774	0.806
6.814	0.710
6.844	0.626
6.916	0.636
6.966	0.649
"""

lobovsky_exp_data_p4 = u"""
0.086	0.003
0.252	0.002
0.418	0.002
0.584	0.002
0.749	0.002
0.915	0.003
1.081	0.003
1.247	0.002
1.413	0.002
1.579	0.002
1.745	0.002
1.911	0.002
2.077	0.002
2.243	0.002
2.408	0.002
2.574	0.013
2.666	0.000
2.726	0.042
2.774	0.117
2.845	0.192
2.919	0.312
2.949	0.382
3.049	0.447
3.216	0.519
3.383	0.544
3.549	0.577
3.715	0.577
3.805	0.537
3.880	0.534
4.047	0.562
4.213	0.567
4.379	0.576
4.464	0.555
4.543	0.509
4.609	0.481
4.709	0.482
4.875	0.485
5.041	0.481
5.206	0.464
5.372	0.465
5.538	0.461
5.704	0.476
5.848	0.521
5.911	0.574
5.921	0.660
5.973	0.723
5.989	0.827
6.050	0.704
6.162	0.709
6.231	0.716
6.326	0.693
6.393	0.697
6.491	0.639
6.563	0.635
6.656	0.556
6.732	0.530
6.823	0.519
6.941	0.475
"""

kleefsman_exp_data_p1 = u"""
0.38	0.014
0.804	0.012
1.228	0.011
1.575	0.03
1.672	0.447
1.678	0.909
1.684	1.435
1.711	2.022
1.778	1.721
1.948	1.358
2.308	1.08
2.733	0.776
3.157	0.686
3.581	0.641
4.006	0.586
4.43	0.556
4.854	0.538
5.279	0.527
5.703	0.463
6.127	0.501
6.552	0.515
6.976	0.512
7.4	    0.543
7.824	0.547
8.249	0.551
8.673	0.55
9.097	0.544
9.522	0.511
9.946	0.486
10.37	0.472
10.795	0.448
11.219	0.424
11.643	0.388
12.068	0.359
12.492	0.341
12.916	0.314
13.341	0.308
13.765	0.299
14.189	0.295
14.614	0.288
15.038	0.286
15.462	0.265
15.886	0.256
16.311	0.238
16.735	0.224
17.159	0.224
17.584	0.218
"""

kleefsman_exp_data_p3 = u"""
0.291	0.007
0.716	0.005
1.141	0.005
1.565	0.004
1.778	0.596
1.797	0.744
1.816	0.945
1.934	1.217
1.997	1.107
2.079	0.981
2.338	0.92
2.531	0.776
2.84	0.684
3.265	0.693
3.69	0.677
4.115	0.61
4.54	0.579
4.965	0.511
5.223	0.454
5.584	0.412
6.009	0.397
6.434	0.377
6.859	0.4
7.284	0.405
7.709	0.409
8.134	0.42
8.559	0.432
8.984	0.432
9.409	0.42
9.834	0.403
10.259	0.377
10.683	0.349
11.108	0.319
11.533	0.289
11.958	0.263
12.383	0.241
12.808	0.221
13.233	0.215
13.658	0.204
14.083	0.215
14.508	0.205
14.933	0.2
15.358	0.184
15.783	0.159
16.208	0.151
16.633	0.14
17.058	0.137
17.483	0.131
17.773	0.122
"""

# data extracted from Figure 17
yeh_and_petroff_v_data = u"""
0.009	2.127
0.013	2.211
0.013	2.037
0.015	2.112
0.023	2.001
0.027	2.052
0.028	1.923
0.032	1.861
0.036	1.948
0.044	1.925
0.045	1.822
0.049	1.869
0.058	1.784
0.06	1.98
0.071	1.707
0.072	1.802
0.078	1.731
0.084	1.791
0.092	1.746
0.098	1.628
0.105	1.649
0.107	1.707
0.126	1.604
0.127	1.707
0.131	1.567
0.139	1.627
0.145	1.667
0.151	1.59
0.158	1.614
0.161	1.509
0.161	1.41
0.161	1.552
0.163	1.727
0.163	1.658
0.174	1.592
0.19	1.507
0.192	1.552
0.192	1.602
0.2	1.552
0.202	1.515
0.212	1.41
0.216	1.373
0.22	1.455
0.22	1.433
0.226	1.448
0.23	1.498
0.234	1.34
0.239	1.539
0.245	1.366
0.252	1.561
0.252	1.429
0.267	1.432
0.286	1.386
0.289	1.457
0.291	1.299
0.295	1.381
0.313	1.34
0.316	1.428
0.316	1.388
0.324	1.336
0.325	1.231
0.33	1.299
0.344	1.321
0.346	1.198
0.349	1.4
0.359	1.248
0.363	1.572
0.368	1.22
0.372	1.068
0.379	1.231
0.384	1.308
0.393	1.217
0.401	1.201
0.409	1.187
0.424	1.164
0.428	1.231
0.433	1.148
0.442	1.119
0.445	1.188
0.454	1.149
0.462	1.09
0.464	1.037
0.467	1.125
0.475	0.993
0.478	1.119
0.49	1.094
0.502	0.948
0.507	0.978
0.511	1
0.513	1.022
0.532	0.988
0.535	0.918
0.539	0.955
0.545	0.993
0.56	1.022
0.563	0.881
0.57	0.96
0.576	0.933
0.595	0.966
0.608	0.944
0.618	0.828
0.868	0.451
0.884	0.452
0.905	0.373
0.907	0.451
0.922	0.403
0.923	0.322
0.929	0.433
0.947	0.396
0.947	0.435
0.972	0.431
0.982	0.381
0.985	0.414
1.009	0.393
1.018	0.313
1.029	0.415
1.038	0.244
1.048	0.372
1.068	0.372
1.068	0.244
1.069	0.291
1.089	0.306
1.089	0.269
1.108	0.246
1.108	0.427
1.108	0.317
1.128	0.352
1.138	0.351
1.148	0.243
1.178	0.319
1.179	0.269
1.188	0.383
1.189	0.299
1.208	0.287
1.218	0.175
1.228	0.306
1.238	0.239
1.248	0.257
1.259	0.272
1.268	0.09
1.278	0.03
1.318	0.153
1.318	0.187
1.338	0.118
"""

# data extracted from Figure 18
yeh_and_petroff_force_data = u"""
-0.002	-4.375
0.002	-0.446
0.01	-2.768
0.018	-1.161
0.018	-0.982
0.029	-2.411
0.037	-1.339
0.049	-0.446
0.049	-0.268
0.061	-0.268
0.073	0.089
0.084	0.268
0.1	    0.625
0.104	-0.446
0.116	-0.089
0.127	-0.089
0.139	-0.446
0.159	0.446
0.175	0.268
0.19	0.089
0.206	-0.268
0.214	-0.446
0.222	-0.982
0.222	-1.339
0.241	-0.268
0.241	-0.625
0.269	-0.268
0.288	0.446
0.292	-0.268
0.312	0.089
0.312	-0.089
0.327	0.089
0.331	0.089
0.335	1.518
0.343	4.375
0.347	0.804
0.351	2.589
0.355	8.036
0.355	4.732
0.355	11.161
0.363	16.696
0.367	14.196
0.367	25.089
0.371	17.589
0.371	21.518
0.382	26.161
0.382	28.839
0.39	30.804
0.394	31.696
0.396	33.125
0.424	30.982
0.429	32.054
0.429	30.089
0.435	28.125
0.437	32.589
0.441	24.911
0.445	24.911
0.445	24.018
0.451	28.75
0.453	24.554
0.457	22.411
0.473	23.482
0.476	21.696
0.476	21.161
0.484	19.018
0.496	19.911
0.5	    18.125
0.504	19.732
0.52	19.554
0.524	17.768
0.539	19.554
0.539	19.018
0.559	19.554
0.559	18.125
0.571	17.589
0.598	18.661
0.606	17.411
0.618	17.946
0.629	17.589
0.649	17.589
0.657	17.946
0.661	18.125
0.665	16.696
0.673	17.411
0.692	18.304
0.71	17.143
0.733	16.696
0.755	16.25
0.778	15.982
0.786	15.268
0.81	15.268
0.839	16.161
0.849	14.911
0.853	14.554
0.861	13.482
0.865	15.536
0.88	14.375
0.888	13.661
0.908	12.054
0.912	12.589
0.929	11.425
0.947	11.339
0.959	10.625
0.963	9.554
0.967	9.911
0.978	10.446
0.986	9.911
0.99	8.839
0.994	9.375
0.998	8.839
1.018	8.839
1.025	7.946
1.037	6.696
1.053	6.696
1.057	8.482
1.076	7.232
1.076	8.571
1.1	    6.696
1.112	6.161
1.127	5.982
1.139	5.268
1.159	4.375
1.159	5.625
1.167	5.268
1.178	4.732
1.19	5.804
1.202	4.911
1.218	4.732
1.227	4.018
1.249	3.661
1.28	3.839
1.296	3.304
1.316	2.768
1.316	1.696
1.32	3.839
1.331	3.125
1.339	-0.804
1.345	-1.875
1.359	2.946
1.369	-2.143
1.375	2.054
1.376	-3.304
1.386	-3.661
1.398	2.411
1.402	1.161
1.402	-3.839
1.418	0.268
1.418	-5.089
1.429	1.339
1.429	-0.446
1.429	-5.982
1.437	-1.161
1.437	-6.875
1.445	-0.268
1.457	-2.232
1.457	-8.571
1.461	-1.339
1.469	-3.304
1.473	-3.482
1.473	-3.839
1.476	-4.554
1.476	-8.929
1.48	-5.089
1.484	-9.554
1.488	-6.518
1.488	-7.589
1.492	-10.268
1.508	-6.875
1.508	-11.161
1.516	-10.089
1.518	-8.482
1.524	-9.911
1.531	-11.518
1.531	-9.554
1.543	-10.804
1.555	-9.196
1.555	-10.625
1.559	-10.446
1.567	-11.696
1.567	-9.911
1.578	-9.375
1.578	-8.839
1.586	-7.768
1.586	-11.161
1.598	-10.625
1.606	-7.054
1.606	-8.482
1.61	-10.089
1.618	-6.518
1.622	-9.732
1.622	-7.946
1.637	-6.518
1.645	-9.732
1.657	-8.125
1.657	-6.875
1.669	-9.196
1.669	-7.589
1.673	-6.696
1.68	-8.839
1.692	-7.411
1.692	-5.982
1.7 	-5.804
1.704	-7.054
1.712	-8.661
1.712	-6.875
1.716	-8.304
1.716	-5.804
1.727	-8.125
1.731	-5.446
1.735	-6.696
1.743	-7.946
1.743	-6.339
1.763	-6.339
1.771	-7.232
1.786	-6.161
1.79	-7.232
1.798	-5.089
1.802	-4.911
1.81	-6.161
1.818	-4.732
1.822	-5.804
1.825	-4.196
1.829	-5.446
1.837	-5.268
1.849	-3.839
1.849	-4.375
1.865	-4.554
1.865	-3.304
1.873	-2.946
1.876	-5.446
1.892	-2.946
1.892	-3.839
1.896	-4.911
1.9	    -2.411
1.908	-4.732
1.92	-3.304
1.924	-4.464
1.924	-2.232
1.939	-3.125
1.947	-4.196
1.947	-1.696
1.955	-2.768
1.967	-3.839
1.967	-1.518
1.978	-2.768
1.978	-1.161
1.994	-3.125
1.998	-0.982
2.014	-1.875
2.018	-0.804
2.022	-1.518
2.025	-2.768
2.037	-1.518
2.041	-2.054
2.045	-0.268
2.073	-1.696
2.084	-0.089
2.088	-1.518
2.108	-1.339
2.108	-0.089
2.116	-0.089
2.124	-1.339
2.163	-0.982
2.171	0.268
2.194	-0.089
2.214	-0.625
2.229	-0.089
2.241	-0.089
2.257	0.089
2.265	-0.446
2.284	0.089
2.288	-0.089
2.304	-0.625
2.316	-0.982
2.331	-0.625
2.367	-0.982
2.394	-1.339
2.418	-1.339
2.445	-1.696
2.469	-1.875
2.488	-1.875
2.5	    -2.054
2.531	-2.232
2.543	-1.518
2.547	-2.411
2.563	-1.875
2.563	-3.125
2.575	-2.768
2.586	-2.946
2.598	-1.875
2.61	-2.768
2.622	-1.696
2.637	-1.339
2.645	-3.214
2.649	-2.232
2.665	-1.071
2.665	-2.054
2.669	-3.125
2.68	-1.518
2.684	-2.946
2.7	    -2.768
2.712	-1.339
2.727	-3.125
2.735	-3.304
2.744	-1.452
2.755	-2.946
2.786	-2.589
2.786	-1.518
2.814	-2.589
2.814	-1.696
2.818	-1.161
2.829	-1.696
2.841	-2.232
2.853	-1.518
2.861	-0.625
2.869	-2.232
2.873	-0.625
2.884	-1.696
2.892	-1.339
2.908	-0.625
2.927	-0.804
2.935	-0.446
2.939	-1.875
2.943	-1.518
2.951	-1.518
2.959	-0.446
2.963	-0.982
2.982	0.268
2.986	-0.982
"""

def get_martin_moyce_1():
    """Returns t*sqrt(2*g/a), z/a for the case where a = 1.125 inches
    """
    # z/a vs t*np.sqrt(2*g/a)
    t, z = np.loadtxt(StringIO(mm_data_1), unpack=True)
    return t, z


def get_martin_moyce_2():
    """Returns t*sqrt(2*g/a), z/a for the case where a = 2.25 inches
    """
    # z/a vs t*np.sqrt(2*g/a)
    t, z = np.loadtxt(StringIO(mm_data_2), unpack=True)
    return t, z


def get_koshizuka_oka_data():
    # z/L vs t*np.sqrt(2*g/L)
    t, z = np.loadtxt(StringIO(ko_data), unpack=True)
    return t, z


def get_koshizuka_oka_mps_data():
    """These are computational results using the MPS scheme.
    """
    # z/L vs t*np.sqrt(2*g/L)
    t, z = np.loadtxt(StringIO(ko_mps_data), unpack=True)
    return t, z


def get_buchner_data():
    """Pressure probe readings for a probe at (5.366*H, 0.2*H)
    for a 2D dam break problem
    """
    # p/(rho*g*H) v/s t*sqrt(2*g/H)
    t, p0 = np.loadtxt(StringIO(buchner_exp_data), unpack=True)
    return t, p0


def get_lobovsky_data():
    """Pressure probe readings for a probes at (1610, 0, 3),
    (1610, 0, 30), (1610, 0, 80) for a 3D dam break problem.
    Units in mm.
    """
    # p/(rho*g*H) v/s t*sqrt(2*g/H)
    t1, p1 = np.loadtxt(StringIO(lobovsky_exp_data_p1), unpack=True)
    t3, p3 = np.loadtxt(StringIO(lobovsky_exp_data_p3), unpack=True)
    t4, p4 = np.loadtxt(StringIO(lobovsky_exp_data_p4), unpack=True)
    return t1, t3, t4, p1, p3, p4


def get_kleefsman_data():
    """Pressure probe readings for a probes at (2.42, 0, 0.021),
    (2.42, 0, 0.101) for a 3D dam break problem with an obstacle.
    """
    # p/(rho*g*H) v/s t*sqrt(2*g/H)
    t1, p1 = np.loadtxt(StringIO(kleefsman_exp_data_p1), unpack=True)
    t3, p3 = np.loadtxt(StringIO(kleefsman_exp_data_p3), unpack=True)
    return t1, t3, p1, p3


def get_yeh_petroff_data():
    """Velocity probe readings for a probes at (0.814, 0, 0.026),
    for a 3D dam break experiment with a tall obstacle as done by
    Yeh and Petroff in their dam in a bore problem.
    """
    # V v/s t, Force(N) v/s t
    vt, v = np.loadtxt(StringIO(yeh_and_petroff_v_data), unpack=True)
    ft, f = np.loadtxt(StringIO(yeh_and_petroff_force_data), unpack=True)

    return vt, v, ft, f
