/*
 *  PlanFacile (Easy plan, in french) is a small tool to help people to
 *  write a document on a particular subject.
 *  Copyright (C) 2005  Julien BRUGUIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "localisationfichier.h"

static Resultat localisationfichier_copieinterne(TRAVAIL(LocalisationFichier) original , TRAVAIL(LocalisationFichier) copie)
{
	/* Ralise une copie de la liste des positions.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si un allocation choue.
	 */
	DEBUTZONESECURISEE
	if(S_T(original)==NULL)
		return RESULTAT_OK;
	SECURISE(localisationfichier_copieinterne(CHAMP_TRAVAIL(original , inclusion),copie));
	SECURISE(localisationfichier_ajout(copie,T_S_(CHAMP(original , nom)),T_S_(CHAMP(original , nomabsolu)),T_S_(CHAMP(original , descripteur)),T_S_(CHAMP(original , position))));
	FINZONESECURISEE
	return RESULTAT_OK;
}

static Resultat localisationfichier_copieinterne_commande(TRAVAIL(LocalisationFichier) original , TRAVAIL(LocalisationFichier) copie)
{
	/* Ralise une copie de la liste des positions.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si un allocation choue.
	 */
	DEBUTZONESECURISEE
	if(S_T(original)==NULL)
		return RESULTAT_OK;
	SECURISE(localisationfichier_copieinterne(CHAMP_TRAVAIL(original , inclusion),copie));
	SECURISE(localisationfichier_ajout(copie,T_S_(CHAMP(original , nom)),T_S_(NULL),T_S_(NULL),T_S_(CHAMP(original , position))));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_initialisation(TRAVAIL(LocalisationFichier) localisationfichier)
{
	/* Cre une localisation de fichier vide.
	 */
	S_T(localisationfichier)=NULL;
	return RESULTAT_OK;
}

Resultat localisationfichier_ajout(TRAVAIL(LocalisationFichier) localisationfichier , TRAVAIL_SCALAIRE(NomFichier) nom , TRAVAIL_SCALAIRE(NomFichier) nomabsolu , TRAVAIL_SCALAIRE(DescripteurFichier) descripteur , TRAVAIL_SCALAIRE(PositionFichier) position)
{
	/* Ajoute une localisation de fichier.
	 * Utilis lors de l'inclusion d'un fichier.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si l'allocation choue.
	 */
	STOCKAGE(LocalisationFichier) fichierinclus;
	if((fichierinclus=(STOCKAGE(LocalisationFichier))(malloc(SIZEOF(CONTENEUR(LocalisationFichier)))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	CHAMP_STOCKAGE(fichierinclus , descripteur)=S_T_(descripteur);
	CHAMP_STOCKAGE(fichierinclus , position)=S_T_(position);
	if((CHAMP_STOCKAGE(fichierinclus , nom)=(STOCKAGE_SCALAIRE(NomFichier))(strdup((char*)(nom))))==NULL)
	{
		free(fichierinclus);
		return RESULTAT_ERREUR_MEMOIRE;
	}
	if(nomabsolu==T_S_(NULL))
		CHAMP_STOCKAGE(fichierinclus , nomabsolu)=NULL;
	else
		if((CHAMP_STOCKAGE(fichierinclus , nomabsolu)=(STOCKAGE_SCALAIRE(NomFichier))(strdup((char*)(nomabsolu))))==NULL)
		{
			free(CHAMP_STOCKAGE(fichierinclus , nom));
			free(fichierinclus);
			return RESULTAT_ERREUR_MEMOIRE;
		}
	CHAMP_STOCKAGE(fichierinclus , inclusion)=S_T(localisationfichier);
	S_T(localisationfichier)=fichierinclus;
	return RESULTAT_OK;
}

Resultat localisationfichier_modification(TRAVAIL(LocalisationFichier) localisationfichier , TRAVAIL_SCALAIRE(PositionFichier) position)
{
	/* Modifie la position dans le fichier courant.
	 */
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	CHAMP(localisationfichier , position)=S_T_(position);
	return RESULTAT_OK;
}

Resultat localisationfichier_retrait(TRAVAIL(LocalisationFichier) localisationfichier)
{
	/* Retire une localisation de fichier.
	 * Ne referme pas le fichier dsign par le descripteur.
	 * Utilis lors d'un retour de fichier inclus.
	 */
	STOCKAGE(LocalisationFichier) fichierinclus;
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	fichierinclus=S_T(localisationfichier);
	S_T(localisationfichier)=CHAMP_STOCKAGE(fichierinclus , inclusion);
	free(CHAMP_STOCKAGE(fichierinclus , nom));
	if(CHAMP_STOCKAGE(fichierinclus , nomabsolu)!=NULL)
		free(CHAMP_STOCKAGE(fichierinclus , nomabsolu));
	free(fichierinclus);
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_nom(TRAVAIL(LocalisationFichier) localisationfichier , REFERENCE_SCALAIRE(NomFichier) nom)
{
	/* Lit le nom du fichier.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	T_R_(nom)=T_S_(CHAMP(localisationfichier , nom));
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_nomabsolu(TRAVAIL(LocalisationFichier) localisationfichier , REFERENCE_SCALAIRE(NomFichier) nomabsolu)
{
	/* Lit le nom absolu du fichier.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	T_R_(nomabsolu)=T_S_(CHAMP(localisationfichier , nomabsolu));
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_descripteur(TRAVAIL(LocalisationFichier) localisationfichier , REFERENCE_SCALAIRE(DescripteurFichier) descripteur)
{
	/* Lit le descripteur du dernier fichier inclus.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	T_R_(descripteur)=T_S_(CHAMP(localisationfichier , descripteur));
	return RESULTAT_OK;
}
Resultat localisationfichier_lecture_position(TRAVAIL(LocalisationFichier) localisationfichier , REFERENCE_SCALAIRE(PositionFichier) position)
{
	/* Lit la position dans le dernier fichier inclus.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	T_R_(position)=T_S_(CHAMP(localisationfichier , position));
	return RESULTAT_OK;
}

Resultat localisationfichier_lecture_inclusion(TRAVAIL(LocalisationFichier) localisationfichier , REFERENCE(LocalisationFichier) inclusion)
{
	/* Renvoie la position du fichier parent dans l'ordre d'inclusion.
	 * Renvoie RESULTAT_ERREUR si localisationfichier est NULL.
	 */
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_ERREUR;
	T_R(inclusion)=CHAMP_TRAVAIL(localisationfichier , inclusion);
	return RESULTAT_OK;
}

Resultat localisationfichier_vide(TRAVAIL(LocalisationFichier) localisationfichier , REFERENCE_SCALAIRE(Booleen) vide)
{
	/* Indique si la localisationfichier est vide ou non.
	 */
	if(S_T(localisationfichier)==NULL)
		T_R_(vide)=T_S_(VRAI);
	else
		T_R_(vide)=T_S_(FAUX);
	return RESULTAT_OK;
}

Resultat localisationfichier_test_fichier(TRAVAIL(LocalisationFichier) localisationfichier , TRAVAIL_SCALAIRE(NomFichier) nomabsolu , REFERENCE_SCALAIRE(Booleen) present)
{
	/* Indique si un nom de fichier est prsent dans la localisation
	 * de fichier, afin de savoir si on ne ralise pas une inclusion
	 * rcursive. Le rsultat est renvoy dans le boolen pass en
	 * paramtre.
	 */
	DEBUTZONESECURISEE
	if(S_T(localisationfichier)==NULL)
	{
		T_R_(present)=T_S_(FAUX);
		return RESULTAT_OK;
	}
	SECURISE(localisationfichier_test_fichier(CHAMP_TRAVAIL(localisationfichier , inclusion),nomabsolu,present));
	if(T_R_(present)==T_S_(VRAI))
		return RESULTAT_OK;
	if((nomabsolu==T_S_(NULL))||(CHAMP(localisationfichier , nomabsolu)==NULL))
		return RESULTAT_OK;
	if((strcmp((char*)(nomabsolu),(char*)(T_S_(CHAMP(localisationfichier , nomabsolu)))))==0)
		T_R_(present)=T_S_(VRAI);
		return RESULTAT_OK;
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_copie(TRAVAIL(LocalisationFichier) localisationfichier , TRAVAIL(LocalisationFichier) copie)
{
	/* Effectue une copie d'une localisation de fichier.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	DEBUTZONESECURISEE
	SECURISE(localisationfichier_destruction(copie));
	SECURISE(localisationfichier_initialisation(copie));
	SECURISE(localisationfichier_copieinterne(localisationfichier,copie));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_copie_commande(TRAVAIL(LocalisationFichier) localisationfichier , TRAVAIL(LocalisationFichier) copie)
{
	/* Effectue une copie d'une localisation de fichier,
	 * en ne recopiant ni le descripteur de fichier, ni le nom
	 * absolu ventuel.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si une allocation choue.
	 */
	DEBUTZONESECURISEE
	SECURISE(localisationfichier_destruction(copie));
	SECURISE(localisationfichier_initialisation(copie));
	SECURISE(localisationfichier_copieinterne_commande(localisationfichier,copie));
	FINZONESECURISEE
	return RESULTAT_OK;
}

Resultat localisationfichier_destruction(TRAVAIL(LocalisationFichier) localisationfichier)
{
	/* Dtruit une localisation de fichier.
	 */
	DEBUTZONESECURISEE
	if(S_T(localisationfichier)==NULL)
		return RESULTAT_OK;
	SECURISE(localisationfichier_destruction(CHAMP_TRAVAIL(localisationfichier , inclusion)));
	free(CHAMP(localisationfichier , nom));
	if(CHAMP(localisationfichier , nomabsolu)!=NULL)
		free(CHAMP(localisationfichier , nomabsolu));
	free(S_T(localisationfichier));
	S_T(localisationfichier)=NULL;
	FINZONESECURISEE
	return RESULTAT_OK;
}

