/*
 *  PlanFacile (Easy plan, in french) is a small tool to help people to
 *  write a document on a particular subject.
 *  Copyright (C) 2005  Julien BRUGUIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

%{
#include <src/global/global.h>

#include <src/donnees/general/general.h>
#include <src/donnees/commandes/localisationfichier.h>

#include <src/problemes/verbeux/verbeux.h>

#include <src/donnees/environnement/environnement.h>

#include <src/analyseur/donnees/filenomfichier.h>

#include <src/analyseur/donnees/pilechaine.h>
#include <src/analyseur/donnees/pileentier.h>

#include <src/analyseur/analyseur/analyseur.syn.h>

/* Nom de la variable d'environnement
 * indiquant le fichier inclus par
 * #standard.
 */
#define VARENVPLANFACILE	"PLANFACILE"

/* Chemin vers le fichier standard
 * de l'utilisateur courant, inclus
 * par #standard.
 */
#define UTILISATEURPLANFACILE	"~/.planfacile/standard.plf"

/* Chemin vers le fichier standard
 * du systme, inclus par #standard.
 */
#include <src/analyseur/analyseur/standard.h>

/* Numro de la premire ligne d'un fichier
 */
#define PREMIERE_LIGNE_FICHIER	1


extern STOCKAGE(General) general;
extern STOCKAGE_SCALAIRE(Entier) niveauimbricationmacro;
extern Resultat analyseur_verificationcopielocalisation(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(Booleen) copielocalisation);

Resultat analyseur_verificationcheminabsolu(TRAVAIL(Environnement) environnement , REFERENCE_SCALAIRE(Booleen) cheminabsolu)
{
	/* Sert  dterminer si le chemin  afficher pour les problmes lors des inclusions de fichiers
	 * est le chemin absolu (depuis la racine du FS) ou le chemin donn par l'utilisateur.
	 * Renvoie RESULTAT_ERREUR si environnement vaut NULL.
	 */
	TRAVAIL(Options) options;
	DEBUTZONESECURISEE
	if(S_T(environnement)==NULL)
		return RESULTAT_ERREUR;
	SECURISE(environnement_lecture_options(environnement,R_T(options)));
	SECURISE(options_lecture_localisationabsolue(options,cheminabsolu));
	FINZONESECURISEE
	return RESULTAT_OK;
}

/* Pile de buffers pour les changements d'entre
 */
typedef struct pilebuffer CONTENEUR(PileBuffer);

struct pilebuffer
{
	STOCKAGE_SCALAIRE(YY_BUFFER_STATE) buffer;
	STOCKAGE(PileBuffer) suivant;
};

static STOCKAGE(PileBuffer) pile=NULL;

Resultat buffer_empile()
{
	STOCKAGE(PileBuffer) nouveau;
	if((nouveau=(STOCKAGE(PileBuffer))(malloc(SIZEOF(CONTENEUR(PileBuffer)))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	CHAMP_STOCKAGE(nouveau , buffer)=YY_CURRENT_BUFFER;
	CHAMP_STOCKAGE(nouveau , suivant)=pile;
	pile=nouveau;
	return RESULTAT_OK;
}

Resultat buffer_depile()
{
	STOCKAGE(PileBuffer) ancien;
	if(pile==NULL)
		return RESULTAT_ERREUR;
	ancien=pile;
	yy_delete_buffer(T_S_(YY_CURRENT_BUFFER));
	yy_switch_to_buffer(T_S_(CHAMP_STOCKAGE(ancien , buffer)));
	pile=CHAMP_STOCKAGE(ancien , suivant);
	free(ancien);
	return RESULTAT_OK;
}


/* Fonction utilise pour la gestion de #include et #standard
 */

Resultat analyseur_inclusion_fichier(TRAVAIL(General) general , TRAVAIL_SCALAIRE(NomFichier) nomfichierinclusion , TRAVAIL_SCALAIRE(Probleme) problemechemininvalide , TRAVAIL_SCALAIRE(Probleme) problemerecursivite , TRAVAIL_SCALAIRE(Probleme) problemeinexistant , TRAVAIL_SCALAIRE(Probleme) problemetypeincorrect , TRAVAIL_SCALAIRE(Probleme) problemeentree , REFERENCE_SCALAIRE(Booleen) inclusion)
{
	/* Effectue les oprations ncssaires pour la lecture d'un fichier  inclure
	 * dans le fichier courant.
	 * Permet de spcifier le probleme   lever lors d'un echec d'ouverture de fichier.
	 * Retourne RESULTAT_ERREUR si general vaut NULL.
	 * Retourne RESULTAT_ERREUR si nomfichierinclusion vaut NULL.
	 */
	TRAVAIL_SCALAIRE(NomFichier) nomfichierabsolu;
	TRAVAIL_SCALAIRE(Booleen) chemin_valide;
	TRAVAIL_SCALAIRE(Booleen) bouclage;
	TRAVAIL_SCALAIRE(DescripteurFichier) fichierinclus;
	TRAVAIL_SCALAIRE(Booleen) copielocalisation;
	TRAVAIL_SCALAIRE(Booleen) localisationcheminabsolu;
	STOCKAGE_SCALAIRE(struct stat) infosfichier;
	DEBUTZONESECURISEE
	if((S_T(general)==NULL)||(S_T_(nomfichierinclusion)==NULL))
		return RESULTAT_ERREUR;
	SECURISE(verbeux_verbeux(general,T_S_(VERBEUX_ANALYSE_INCLUSION),nomfichierinclusion));
	SECURISE(environnement_ajout_repertoirefichier(CHAMP_TRAVAIL(general , environnement),nomfichierinclusion,R_T_(nomfichierabsolu),R_T_(chemin_valide)));
	/* Ajout du rpertoire courant sur la pile des repertoires et
	 * dplacement dans le repertoire du fichier  inclure.
	 */
	if(chemin_valide==T_S_(FAUX))
	{
		/* Le chemin vers le fichier  inclure n'est pas valide.
		 */
		inclusion=T_S_(FAUX);
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,problemechemininvalide,CHAMP_TRAVAIL(general , localisation),nomfichierinclusion));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,problemechemininvalide,T_S(localisationnulle),nomfichierinclusion));
		}
		return RESULTAT_OK;
	}
	SECURISE(localisationfichier_test_fichier(CHAMP_TRAVAIL(general , localisation),nomfichierabsolu,R_T_(bouclage)));
	if(bouclage==T_S_(VRAI))
	{
		/* Bouclage d'inclusion dtect.
		 */
		T_R_(inclusion)=T_S_(FAUX);
		free(S_T_(nomfichierabsolu));
		SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
		/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
		 * de la pile des rpertoires.
		 */
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,problemerecursivite,CHAMP_TRAVAIL(general , localisation),nomfichierinclusion));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,problemerecursivite,T_S(localisationnulle),nomfichierinclusion));
		}
		return RESULTAT_OK;
	}
	if((stat((char*)(nomfichierabsolu),C_S_(infosfichier))!=T_S_(0)))
	{
		/* Le fichier n'existe pas.
		 */
		inclusion=T_S_(FAUX);
		free(S_T_(nomfichierabsolu));
		SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
		/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
		 * de la pile des rpertoires.
		 */
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,problemeinexistant,CHAMP_TRAVAIL(general , localisation),nomfichierinclusion));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,problemeinexistant,T_S(localisationnulle),nomfichierinclusion));
		}
		return RESULTAT_OK;
	}
	if((S_ISDIR(CHAMP_STOCKAGE_(infosfichier , st_mode)))||(S_ISBLK(CHAMP_STOCKAGE_(infosfichier , st_mode))))
	{
		/* Le nom de fichier dsigne un rpertoire ou un fichier de type block.
		 */
		inclusion=T_S_(FAUX);
		free(S_T_(nomfichierabsolu));
		SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
		/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
		 * de la pile des rpertoires.
		 */
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,problemetypeincorrect,CHAMP_TRAVAIL(general , localisation),nomfichierinclusion));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,problemetypeincorrect,T_S(localisationnulle),nomfichierinclusion));
		}
		return RESULTAT_OK;
	}
	if((S_T_(fichierinclus)=(STOCKAGE_SCALAIRE(DescripteurFichier))(fopen((char*)(nomfichierabsolu),T_S_("r"))))==NULL)
	{
		/* Le fichier  inclure n'a pu tre ouvert en lecture.
		 */
		T_R_(inclusion)=T_S_(FAUX);
		free(S_T_(nomfichierabsolu));
		SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
		/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
		 * de la pile des rpertoires.
		 */
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,problemeentree,CHAMP_TRAVAIL(general , localisation),nomfichierinclusion));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,problemeentree,T_S(localisationnulle),nomfichierinclusion));
		}
		return RESULTAT_OK;
	}
	SECURISE(analyseur_verificationcheminabsolu(CHAMP_TRAVAIL(general , environnement),R_T_(localisationcheminabsolu)));
	if(localisationcheminabsolu==T_S_(VRAI))
	{
		SECURISE(localisationfichier_ajout(CHAMP_TRAVAIL(general , localisation),nomfichierabsolu,nomfichierabsolu,fichierinclus,T_S_(PREMIERE_LIGNE_FICHIER)));
	}
	else
	{
		SECURISE(localisationfichier_ajout(CHAMP_TRAVAIL(general , localisation),nomfichierinclusion,nomfichierabsolu,fichierinclus,T_S_(PREMIERE_LIGNE_FICHIER)));
	}
	/* Le fichier a t ouvert en lecture, ajout de sa localisation.
	 */
	SECURISE(buffer_empile());
	/* Mise sur la pile du buffer de lecture de lex du fichier courant.
	 */
	T_S_(yyin)=fichierinclus;
	yy_switch_to_buffer(yy_create_buffer(T_S_(yyin),T_S_(YY_BUF_SIZE)));
	/* Cration et utilisation du buffer d'entre de lex pour le fichier inclus.
	 */
	T_R_(inclusion)=T_S_(VRAI);
	free(S_T_(nomfichierabsolu));
	FINZONESECURISEE
	return RESULTAT_OK;
}


/* Fonctions utiles pour grer la commande #include
 */

static Resultat analyseur_obtenirnomfichierinclude(TRAVAIL_SCALAIRE(Chaine) chainetoken , REFERENCE_SCALAIRE(NomFichier) nomfichier)
{
	/* Extrait de la chaine reconnue par lex le nom de fichier
	 * entre par l'utilisateur.
	 * Retourne RESULTAT_ERREUR si chainetoken vaut NULL.
	 * Retourne RESULTAT_ERREUR_MEMOIRE si une allocation echoue.
	 */
	STOCKAGE_SCALAIRE(Chaine) copiechainetoken;
	REFERENCE_SCALAIRE(Caractere) debut;
	REFERENCE_SCALAIRE(Caractere) parcours;
	STOCKAGE_SCALAIRE(Entier) decalage;
	if(S_T_(chainetoken)==NULL)
		return RESULTAT_ERREUR;
	if((copiechainetoken=(STOCKAGE_SCALAIRE(Chaine))(strdup((char*)(chainetoken))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	debut=((REFERENCE_SCALAIRE(Caractere))(copiechainetoken))+8;/* on saute le '#include'	*/
	/* on "mange" les blancs entre le #include et le debut de parametre { 			*/
	while((T_R_(debut)==T_S_(' '))||(T_R_(debut)==T_S_('\t'))||(T_R_(debut)==T_S_('\n')))
		debut++;
	debut++;/* on "mange" le debut de parametre { */
	decalage=0;
	for(parcours=debut ; (T_R_(parcours+decalage))!=T_S_('\0') ; parcours++)
	{
		if(((T_R_(parcours+decalage))==T_S_('#'))&&((T_R_(parcours+decalage+1))==T_S_('}')))
		{
			decalage=S_T_(T_S_(decalage)+T_S_(1));
			T_R_(parcours)=T_S_('}');
		}
		else
			T_R_(parcours)=T_R_(parcours+decalage);
	}
	/* on rduit tous les #} en }
	 */
	T_R_(parcours-1)=T_S_('\0');
	/* on marque la fin de la chaine en ecrasant la fin de parametre }
	 */
	if((T_R_(nomfichier)=T_S_((STOCKAGE_SCALAIRE(NomFichier))(strdup((char*)(T_S_((STOCKAGE_SCALAIRE(Chaine))(debut)))))))==T_S_(NULL))
		return RESULTAT_ERREUR_MEMOIRE;
	free(copiechainetoken);
	return RESULTAT_OK;
}

static Resultat analyseur_traitementinclude(TRAVAIL(General) general , TRAVAIL_SCALAIRE(Chaine) chainetokeninclude)
{
	/* Traite une inclusion demande par la commande #include.
	 * Retourne RESULTAT_ERREUR si general vaut NULL.
	 * Retourne RESULTAT_ERREUR si chainetokeninclude vaut NULL.
	 */
	TRAVAIL_SCALAIRE(NomFichier) nomfichierinclude;
	TRAVAIL_SCALAIRE(Booleen) inclusion_ok;
	TRAVAIL_SCALAIRE(Booleen) copielocalisation;
	DEBUTZONESECURISEE
	if((S_T(general)==NULL)||(S_T_(chainetokeninclude)==NULL))
		return RESULTAT_ERREUR;
	SECURISE(analyseur_obtenirnomfichierinclude(chainetokeninclude,R_T_(nomfichierinclude)));
	if(strlen(nomfichierinclude)!=0)
	{
		/* Un nom de fichier non vide a t lu.
		 */
		SECURISE(analyseur_inclusion_fichier(general,nomfichierinclude,T_S_(PROBLEME_OUVERTURE_CHEMIN_INVALIDE),T_S_(PROBLEME_RECURSIVITE_FICHIER),T_S_(PROBLEME_OUVERTURE_INEXISTANT),T_S_(PROBLEME_OUVERTURE_TYPE_INCORRECT),T_S_(PROBLEME_OUVERTURE_ENTREE),R_T_(inclusion_ok)));
	}
	else
	{
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_ENTREE_VIDE),CHAMP_TRAVAIL(general , localisation)));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_ENTREE_VIDE),T_S(localisationnulle)));
		}
	}
	free(S_T_(nomfichierinclude));
	FINZONESECURISEE
	return RESULTAT_OK;
}


/* Fonction pour la gestion de #standard
 */

static Resultat analyseur_traitementstandard(TRAVAIL(General) general)
{
	/* Traite une inclusion demande par la commande #standard.
	 * La priorit d'inclusion est:
	 *	1) le fichier indiqu par la variable d'environnement PLANFACILE
	 *	   si elle existe.
	 *	2) le fichier defini par UTILISATEURPLANFACILE.
	 *	3) le fichier defini par SYSTEMEPLANFACILE.
	 * Si une inclusion choue, le fichier de priorit infrieure est utilis.
	 * Renvoie RESULTAT_ERREUR si general vaut NULL.
	 */
	TRAVAIL_SCALAIRE(NomFichier) nomfichier;
	STOCKAGE_SCALAIRE(Booleen) inclusion_ok;
	TRAVAIL_SCALAIRE(Booleen) copielocalisation;
	DEBUTZONESECURISEE
	if(S_T(general)==NULL)
		return RESULTAT_ERREUR;
	inclusion_ok=T_S_(FAUX);
	if((S_T_(nomfichier)=(STOCKAGE_SCALAIRE(NomFichier))(getenv(T_S_(VARENVPLANFACILE))))!=NULL)
	{
		/* La variable d'environnement de planfacile existe.
		 */
		if(strlen(nomfichier)!=0)
		{
			SECURISE(analyseur_inclusion_fichier(general,nomfichier,T_S_(PROBLEME_OUVERTURE_CHEMIN_INVALIDE_STANDARD),T_S_(PROBLEME_RECURSIVITE_FICHIER_STANDARD),T_S_(PROBLEME_OUVERTURE_INEXISTANT_STANDARD),T_S_(PROBLEME_OUVERTURE_TYPE_INCORRECT_STANDARD),T_S_(PROBLEME_OUVERTURE_ENTREE_STANDARD),C_S_(inclusion_ok)));
			/* Traitement de l'inclusion du fichier dsign par la variable
			 * d'environnement de planfacile.
			 */
		}
		else
		{	/* La variable d'environnement est vide.
			 */
			inclusion_ok=FAUX;
		}
		if(inclusion_ok==FAUX)
		{
			/* L'inclusion du fichier a choue, on appelle le problme relatif au passage au fichier
			 * de priorit infrieure.
			 */
			SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
			if(copielocalisation==T_S_(VRAI))
			{
				SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_STANDARD_SUIVANTE),CHAMP_TRAVAIL(general , localisation),T_S_(UTILISATEURPLANFACILE)));
			}
			else
			{
				STOCKAGE(LocalisationFichier) localisationnulle;
				localisationnulle=NULL;
				SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_STANDARD_SUIVANTE),T_S(localisationnulle),T_S_(UTILISATEURPLANFACILE)));
			}
		}
	}
	if(inclusion_ok==FAUX)
	{
		/* Aucun fichier n'a encore t inclus, on essaie d'inclure le fichier standard utilisateur.
		 */
		nomfichier=T_S_(UTILISATEURPLANFACILE);
		SECURISE(analyseur_inclusion_fichier(general,nomfichier,T_S_(PROBLEME_OUVERTURE_CHEMIN_INVALIDE_STANDARD),T_S_(PROBLEME_RECURSIVITE_FICHIER_STANDARD),T_S_(PROBLEME_OUVERTURE_INEXISTANT_STANDARD),T_S_(PROBLEME_OUVERTURE_TYPE_INCORRECT_STANDARD),T_S_(PROBLEME_OUVERTURE_ENTREE_STANDARD),C_S_(inclusion_ok)));
	}
	if(inclusion_ok==FAUX)
	{
		/* L'inclusion du fichier a choue...
		 */
		nomfichier=T_S_(SYSTEMEPLANFACILE);
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_STANDARD_SUIVANTE),CHAMP_TRAVAIL(general , localisation),T_S_(SYSTEMEPLANFACILE)));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_STANDARD_SUIVANTE),T_S(localisationnulle),T_S_(SYSTEMEPLANFACILE)));
		}
		SECURISE(analyseur_inclusion_fichier(general,nomfichier,T_S_(PROBLEME_OUVERTURE_CHEMIN_INVALIDE_STANDARD),T_S_(PROBLEME_RECURSIVITE_FICHIER_STANDARD),T_S_(PROBLEME_OUVERTURE_INEXISTANT_STANDARD),T_S_(PROBLEME_OUVERTURE_TYPE_INCORRECT_STANDARD),T_S_(PROBLEME_OUVERTURE_ENTREE_STANDARD),C_S_(inclusion_ok)));
	}
	if(inclusion_ok==FAUX)
	{
		/* Aucun fichier n'a pu tre inclus.
		 */
		SECURISE(analyseur_verificationcopielocalisation(CHAMP_TRAVAIL(general , environnement),R_T_(copielocalisation)));
		if(copielocalisation==T_S_(VRAI))
		{
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_STANDARD),CHAMP_TRAVAIL(general , localisation)));
		}
		else
		{
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_STANDARD),T_S(localisationnulle)));
		}
	}
	FINZONESECURISEE
	return RESULTAT_OK;
}


/* Fonctions pour la mise  jour de la position de la localisation de fichier
 */

static Resultat analyseur_modificationposition(TRAVAIL(General) general , TRAVAIL_SCALAIRE(Entier) increment)
{
	/* Ajoute (increment)  la position de genral->localisation.
	 * Retourne RESULTAT_ERREUR si general est NULL.
	 */
	TRAVAIL_SCALAIRE(PositionFichier) position;
	DEBUTZONESECURISEE
	if(S_T(general)==NULL)
		return RESULTAT_ERREUR;
	if(increment==T_S_(0))
		return RESULTAT_OK;
	SECURISE(localisationfichier_lecture_position(CHAMP_TRAVAIL(general , localisation),R_T_(position)));
	position+=increment;
	SECURISE(localisationfichier_modification(CHAMP_TRAVAIL(general , localisation),position));
	/* Sans commentaires... ^_^
	 */
	FINZONESECURISEE
	return RESULTAT_OK;
}
static TRAVAIL_SCALAIRE(Entier) analyseur_nombreretourchariot(TRAVAIL_SCALAIRE(Chaine) chaine)
{
	/* Retourne le nombre de retours charlots (^_^) prsents dans la chaine.
	 * Retourne 0 si la chaine vaut NULL.
	 */
	TRAVAIL_SCALAIRE(Entier) nb_retourchariot;
	REFERENCE_SCALAIRE(Caractere) curseur;
	nb_retourchariot=T_S_(0);
	if(chaine!=T_S_(NULL))
		for(curseur=((REFERENCE_SCALAIRE(Caractere))(chaine)) ; T_R_(curseur)!=T_S_('\0') ; curseur++)
		{
			if(T_R_(curseur)==T_S_('\n'))
				nb_retourchariot++;
		}
	return nb_retourchariot;
}

static Resultat analyseur_modificationerreursyntaxe(TRAVAIL(General) general , TRAVAIL_SCALAIRE(Chaine) chainetokenerreur)
{
	/* Fait pointer general->erreursyntaxe sur une copie de la chaine
	 * passe en paramtre, pour connaitre la derniere chaine lue si
	 * une erreur de syntaxe est dtecte.
	 * Libre la chaine prcdement alloue si besoin est.
	 * Renvoie RESULTAT_ERREUR si general vaut NULL.
	 * Renvoie RESULTAT_ERREUR si chainetokenerreur vaut NULL.
	 * Renvoie RESULTAT_ERREUR_MEMOIRE si l'allocation echoue.
	 */
	if((S_T(general)==NULL)||(S_T_(chainetokenerreur)==NULL))
		return RESULTAT_ERREUR;
	if(CHAMP(general , erreursyntaxe)!=NULL)
		free(CHAMP(general , erreursyntaxe));
	if((CHAMP(general , erreursyntaxe)=(STOCKAGE_SCALAIRE(Chaine))(strdup((char*)(chainetokenerreur))))==NULL)
		return RESULTAT_ERREUR_MEMOIRE;
	return RESULTAT_OK;
}

%}

%option nounput

%%

#include[ \n\t]*"{"([^}]|#"}")*"}"	{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#include"))));
						SECURISE(analyseur_traitementinclude(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext))));
						SECURISE(analyseur_modificationposition(T_S(general),analyseur_nombreretourchariot(T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext)))));
						FINZONESECURISEE
						return TOKEN_INCLUDE;
					}
#standard				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#standard"))));
						SECURISE(analyseur_traitementstandard(T_S(general)));
						FINZONESECURISEE
						return TOKEN_STANDARD;
					}

#option					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#option"))));
						FINZONESECURISEE
						return TOKEN_OPTION;
					}
#options				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#options"))));
						FINZONESECURISEE
						return TOKEN_OPTIONS;
					}
#case					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#case"))));
						FINZONESECURISEE
						return TOKEN_CASE;
					}
#other					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#other"))));
						FINZONESECURISEE
						return TOKEN_OTHER;
					}
#end					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#end"))));
						FINZONESECURISEE
						return TOKEN_END;
					}

#define					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#define"))));
						FINZONESECURISEE
						return TOKEN_DEFINE;
					}

#warning				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#warning"))));
						FINZONESECURISEE
						return TOKEN_WARNING;
					}
#error					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#error"))));
						FINZONESECURISEE
						return TOKEN_ERROR;
					}

#start					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#start"))));
						FINZONESECURISEE
						return TOKEN_START;
					}
#head					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#head"))));
						FINZONESECURISEE
						return TOKEN_HEAD;
					}
#foot					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#foot"))));
						FINZONESECURISEE
						return TOKEN_FOOT;
					}
#section				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#section"))));
						FINZONESECURISEE
						return TOKEN_SECTION;
					}
#reference				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#reference"))));
						FINZONESECURISEE
						return TOKEN_REFERENCE;
					}
#message				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#message"))));
						FINZONESECURISEE
						return TOKEN_MESSAGE;
					}

#title					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#title"))));
						FINZONESECURISEE
						return TOKEN_TITLE;
					}
#ref					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#ref"))));
						FINZONESECURISEE
						return TOKEN_REF;
					}
#txt					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#txt"))));
						FINZONESECURISEE
						return TOKEN_TXT;
					}	
#sec					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_(STOCKAGE_SCALAIRE(Chaine))(("#sec"))));
						FINZONESECURISEE
						return TOKEN_SEC;
					}
#mesg					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#mesg"))));
						FINZONESECURISEE
						return TOKEN_MESG;
					}
#text					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#text"))));
						FINZONESECURISEE
						return TOKEN_TEXT;
					}

#idea					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#idea"))));
						FINZONESECURISEE
						return TOKEN_IDEA;
					}
#missing				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#missing"))));
						FINZONESECURISEE
						return TOKEN_MISSING;
					}
#generic				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#generic"))));
						FINZONESECURISEE
						return TOKEN_GENERIC;
					}

#index					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#index"))));
						FINZONESECURISEE
						return TOKEN_INDEX;
					}
#extref					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#extref"))));
						FINZONESECURISEE
						return TOKEN_EXTREF;
					}
#extrefs				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#extrefs"))));
						FINZONESECURISEE
						return TOKEN_EXTREFS;
					}

#dep					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#dep"))));
						FINZONESECURISEE
						return TOKEN_DEP;
					}
#depref					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#depref"))));
						FINZONESECURISEE
						return TOKEN_DEPREF;
					}

#comment				{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#comment"))));
						FINZONESECURISEE
						return TOKEN_COMMENT;
					}

##					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("##"))));
						FINZONESECURISEE
						return TOKEN_ECHAP_COMMANDE;
					}
#\{					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#{"))));
						FINZONESECURISEE
						return TOKEN_ECHAP_PARAM_DEBUT;
					}
#\}					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#}"))));
						FINZONESECURISEE
						return TOKEN_ECHAP_PARAM_FIN;
					}
#"\n"					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#\\n"))));
						SECURISE(analyseur_modificationposition(T_S(general),T_S_(1)));
						FINZONESECURISEE
						return TOKEN_ECHAP_RETOURLIGNE;
					}
#" "					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("# "))));
						FINZONESECURISEE
						return TOKEN_ECHAP_ESPACE;
					}
#"\t"					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("#\\t"))));
						FINZONESECURISEE
						return TOKEN_ECHAP_TAB;
					}


#[^0-9 \t\n#{}][^ \t\n#{}]*		{	
						DEBUTZONESECURISEE
						SECURISE(pilenommacro_ajout(T_S(CHAMP_STOCKAGE(general , pilenommacro)),T_S_(yytext+1),T_S_(niveauimbricationmacro)));
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext))));
						FINZONESECURISEE
						return TOKEN_MACRO;
					}
#[0-9]+					{	
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(pileentier_ajout(T_S(CHAMP_STOCKAGE(general , pileparametre)),(TRAVAIL_SCALAIRE(Entier))(T_S_(atoi(yytext+1)))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext))));
						FINZONESECURISEE
						return TOKEN_PARAMETRE;
					}

\{					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("{"))));
						FINZONESECURISEE
						return TOKEN_PARAM_DEBUT;
					}
\}					{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))("}"))));
						FINZONESECURISEE
						return TOKEN_PARAM_FIN;
					}

[^#{} \t\n][^#{}\n]*(\n)?		{
						DEBUTZONESECURISEE
						SECURISE(pilelocalisationfichier_ajout(T_S(CHAMP_STOCKAGE(general , pilelocalisation)),T_S(CHAMP_STOCKAGE(general , localisation))));
						SECURISE(pilechaine_ajout(T_S(CHAMP_STOCKAGE(general , piletexte)),T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext))));
						SECURISE(analyseur_modificationerreursyntaxe(T_S(general),T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext))));
						SECURISE(analyseur_modificationposition(T_S(general),analyseur_nombreretourchariot(T_S_((STOCKAGE_SCALAIRE(Chaine))(yytext)))));
						FINZONESECURISEE
						return TOKEN_TEXTE;
					}

[\n]        				{
						DEBUTZONESECURISEE
						SECURISE(analyseur_modificationposition(T_S(general),T_S_((STOCKAGE_SCALAIRE(Entier))(1))));
						FINZONESECURISEE
					}
[ \t]					{
						/* RIEN */
					}

%%

Resultat analyseur_ouverture_fichier(TRAVAIL(General) general , REFERENCE_SCALAIRE(Booleen) finanalyse)
{
	/* Prpare la structure 'general' pour l'analyse du prochain fichier valide.
	 * Renvoie RESULTAT_ERREUR si general vaut NULL.
	 */
	TRAVAIL_SCALAIRE(NomFichier) nomfichierfile;
	TRAVAIL_SCALAIRE(NomFichier) nomfichierabsolu;
	TRAVAIL_SCALAIRE(Booleen) cheminvalide;
	TRAVAIL_SCALAIRE(Booleen) filenomfichiervide;
	TRAVAIL_SCALAIRE(Booleen) localisationcheminabsolu;
	TRAVAIL_SCALAIRE(DescripteurFichier) fichier;
	STOCKAGE_SCALAIRE(struct stat) infosfichier;
	DEBUTZONESECURISEE
	if(S_T(general)==NULL)
		return RESULTAT_ERREUR;
	T_R_(finanalyse)=T_S_(FAUX);
	for(;;)
	{
		SECURISE(filenomfichier_vide(CHAMP_TRAVAIL(general , sources),R_T_(filenomfichiervide)));
		if(filenomfichiervide==T_S_(VRAI))
		{
			/* La file des noms de fichier est vide : il n'y a plus de fichier  traiter.
			 */
			T_R_(finanalyse)=T_S_(VRAI);
			break;
		}
		SECURISE(filenomfichier_retrait_nomfichier(CHAMP_TRAVAIL(general , sources),R_T_(nomfichierfile)));
		/* Retrait d'un nom de fichier de la file.
		 */
		SECURISE(verbeux_verbeux(general,T_S_(VERBEUX_ANALYSE_SOURCE),nomfichierfile));
		if(strcmp((char*)(nomfichierfile),T_S_("-"))==0)
		{
			/* L'entre est l'entre standard.
			 */
			free(S_T_(nomfichierfile));
			SECURISE(environnement_ajout_repertoirefichier(CHAMP_TRAVAIL(general , environnement),T_S_("./"),R_T_(nomfichierabsolu),R_T_(cheminvalide)));
			/* On ajoute le repertoire courant dans la pile des repertoires.
			 * (Le rpertoire courant tant retir  chaque fin d'analyse de fichier,
			 *  il convient d'ajouter galement un repertoire pour l'entre standard)
			 * Le nomfichierabsolu retourn n'est que le chemin absolu vers le rpertoire
			 * courant : il sera rapidement dtruit.
			 */
			if(cheminvalide==T_S_(FAUX))
			{
				/* Le rpertoire courant n'a pu tre ajout  la pile des rpertoires.
				 */
				STOCKAGE(LocalisationFichier) localisationnulle;
				localisationnulle=NULL;
				SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_REPERTOIRE_COURANT),T_S(localisationnulle)));
				continue;
			}
			free(S_T_(nomfichierabsolu));
			SECURISE(localisationfichier_ajout(CHAMP_TRAVAIL(general , localisation),T_S_("-"),T_S_("-"),T_S_(stdin),T_S_(PREMIERE_LIGNE_FICHIER)));
			/* Ajout de la localisation approprie pour l'analyse.
			 */
			break;
		}
		/* L'entre n'est pas l'entre standard.
		 */
		SECURISE(environnement_ajout_repertoirefichier(CHAMP_TRAVAIL(general , environnement),nomfichierfile,R_T_(nomfichierabsolu),R_T_(cheminvalide)));
		/* On se place dans le repertoire du fichier  traiter.
		 */
		if(cheminvalide==T_S_(FAUX))
		{
			/* Le chemin vers le rpertoire du fichier n'a pu tre rsolu.
			 */
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_CHEMIN_INVALIDE),T_S(localisationnulle),nomfichierfile));
			/* Memory Leak sur nomfichierfile en cas d'erreur */
			free(S_T_(nomfichierfile));
			continue;
		}
		if((stat((char*)(nomfichierabsolu),C_S_(infosfichier))!=T_S_(0)))
		{
			/* Le fichier n'existe pas.
			 */
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			free(S_T_(nomfichierabsolu));
			SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
			/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
			 * de la pile des rpertoires.
			 */
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_INEXISTANT),T_S(localisationnulle),nomfichierfile));
			continue;
		}
		if((S_ISDIR(CHAMP_STOCKAGE_(infosfichier , st_mode)))||(S_ISBLK(CHAMP_STOCKAGE_(infosfichier , st_mode))))
		{
			/* Le nom de fichier dsigne un rpertoire, ou un fichier de type block.
			 */
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			free(S_T_(nomfichierabsolu));
			SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
			/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
			 * de la pile des rpertoires.
			 */
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_TYPE_INCORRECT),T_S(localisationnulle),nomfichierfile));
			continue;
		}
		if((S_T_(fichier)=(STOCKAGE_SCALAIRE(DescripteurFichier))(fopen((char*)(nomfichierabsolu),T_S_("r"))))==NULL)
		{
			/* Le fichier n'a pas pu tre ouvert en lecture.
			 */
			STOCKAGE(LocalisationFichier) localisationnulle;
			localisationnulle=NULL;
			free(S_T_(nomfichierabsolu));
			SECURISE(environnement_retrait_repertoirefichier(CHAMP_TRAVAIL(general , environnement)));
			/* Retour dans le repertoire prcdent avec retrait du dernier rpertoire
			 * de la pile des rpertoires.
			 */
			SECURISE(probleme_probleme(general,T_S_(PROBLEME_OUVERTURE_ENTREE),T_S(localisationnulle),nomfichierfile));
			/* Memory Leak sur nomfichierfile en cas d'erreur.
			 */
			free(S_T_(nomfichierfile));
			continue;
		}
		SECURISE(analyseur_verificationcheminabsolu(CHAMP_TRAVAIL(general , environnement),R_T_(localisationcheminabsolu)));
		if(localisationcheminabsolu==T_S_(VRAI))
		{
			SECURISE(localisationfichier_ajout(CHAMP_TRAVAIL(general , localisation),nomfichierabsolu,nomfichierabsolu,fichier,T_S_(PREMIERE_LIGNE_FICHIER)));
		}
		else
		{
			SECURISE(localisationfichier_ajout(CHAMP_TRAVAIL(general , localisation),nomfichierfile,nomfichierabsolu,fichier,T_S_(PREMIERE_LIGNE_FICHIER)));
		}
		/* Ajout de la localisation approprie pour l'analyse.
		 */
		free(S_T_(nomfichierfile));
		free(S_T_(nomfichierabsolu));
		break;
	}
	FINZONESECURISEE
		return RESULTAT_OK;
}

int yywrap( void )
{
	TRAVAIL_SCALAIRE(Booleen) fin_fichier;
	TRAVAIL_SCALAIRE(Booleen) fin_analyse;
	DEBUTZONESECURISEE
	fclose(T_S_(yyin));
	/* Fermeture du descripteur de fichier de l'entre analyse.
	 */
	SECURISE(environnement_retrait_repertoirefichier(T_S(CHAMP_STOCKAGE(general , environnement))));
	/* Retrait du repertoire de l'environnement.
	 */
	SECURISE(localisationfichier_retrait(T_S(CHAMP_STOCKAGE(general , localisation))));
	/* Retrait de la localisation du fichier analys.
	 */
	SECURISE(localisationfichier_vide(T_S(CHAMP_STOCKAGE(general , localisation)),R_T_(fin_fichier)));
	if(fin_fichier==T_S_(VRAI))
	{
		/* Traitement de l'entre termine.
		 */
		SECURISE(analyseur_ouverture_fichier(T_S(general),R_T_(fin_analyse)));
		/* Initialisation du traitement du fichier suivant  traiter.
		 */
		if(fin_analyse==T_S_(VRAI))
		{
			/* Fin de l'analyse. */
			yylex_destroy();
			return 1;
		}
	}
	else
	{
		SECURISE(verbeux_verbeux(T_S(general),T_S_(VERBEUX_ANALYSE_FININCLUSION)));
		SECURISE(buffer_depile());
		/* Retour d'un fichier inclus, recupration du buffer prcdent
		 */
	}
	SECURISE(localisationfichier_lecture_descripteur(T_S(CHAMP_STOCKAGE(general , localisation)),R_S_(yyin)));
	/* Branchement de yyin sur le descripteur de fichier appropri.
	 */
	return 0;/* Poursuite de l'analyse. */
	FINZONESECURISEE
}

