/*	Polar_Stereographic_Spherical_Projection

PIRL CVS ID: Polar_Stereographic_Spherical_Projection.java,v 1.3 2012/04/16 06:10:20 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Image_Tools;

import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.PVL_Exception;

import	java.awt.Point;
import	java.awt.geom.Point2D;


public class Polar_Stereographic_Spherical_Projection
	extends Projection
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Image_Tools.Polar_Stereographic_Spherical_Projection (1.3 2012/04/16 06:10:20)";

private static final String
	PROJECTION_NAME					= "Polar Stereographic Spherical";

private double
	Center_Latitude_Sign,
	Center_Latitude_Sine,
	Center_Latitude_Cosine,
	Polar_Radius_times_2;

/*==============================================================================
	Constructors
*/
public Polar_Stereographic_Spherical_Projection
	(
	Parameter	parameters
	)
	throws PVL_Exception
{
//	Initialize the base projection parameters.
super (parameters);

if (Not_Identity)
	{
	//	Precalculated coefficients.
	if (Center_Latitude < 0.0)
		Center_Latitude_Sign = -1.0;
	else
		Center_Latitude_Sign = 1.0;
	Center_Latitude_Sine = StrictMath.sin (Center_Latitude);
	Center_Latitude_Cosine = StrictMath.cos (Center_Latitude);
	Polar_Radius_times_2 = Polar_Radius * 2.0;
	}
}

/*==============================================================================
	Accessors
*/
public String Name ()
{return PROJECTION_NAME;}

/*==============================================================================
	Converters
*/
/**	Get the world longitude,latitude coordinate for an image sample,line
	coordinate.
<p>
	The conversion algorithm is:
<p><blockquote>

</blockquote>
<p>
	@param	image_coordinate	The image sample,line coordinate.
	@return	The world longitude,latitude coordinate. The {@link
		Point2D#getX() x} value of the coordinate Point2D is the
		longitude; the {@link Point2D#getY() y} value is the latitude.
		Values are in degrees within the range 0-360.
*/
public Point2D.Double to_World
	(
	Point2D		image_coordinate
	)
	throws IllegalArgumentException, ArithmeticException
{
if (image_coordinate == null)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to project image coordinates ("
			+ (int)image_coordinate.getX () + ','
			+ (int)image_coordinate.getX () + ") to world coordinates\n"
		+ "because the image coordinates point is null.");
Point2D.Double
	world_coordinate = null;
if (Not_Identity)
	{
	double
		x = Sample_to_Projection_X (image_coordinate.getX ()),
		y = Line_to_Projection_Y   (image_coordinate.getY ()),
		P = StrictMath.sqrt ((x * x) + (y * y)),
		C = 2.0 * StrictMath.atan (P / Polar_Radius_times_2);

	double
		latitude = StrictMath.asin (
			(StrictMath.cos (C) * Center_Latitude_Sine)
			+ (y * StrictMath.sin (C) * Center_Latitude_Cosine / P));
	if (Math.abs (latitude) > PI_OVER_2)
		throw new ArithmeticException (ID + '\n'
			+ "Projection of image coordinates ("
				+ (int)image_coordinate.getX () + ','
				+ (int)image_coordinate.getX () + ") to world coordinates\n"
			+ "resulted in an invalid latitude of "
				+ Math.toDegrees (latitude) + " degrees.");
	if (Planetocentric)
		latitude = Planetographic_to_Planetocentric (latitude);
	latitude = StrictMath.toDegrees (latitude);

	double
		longitude = Center_Longitude
			+ StrictMath.atan (x / -Center_Latitude_Sign * y);
	if (Positive_West)
		longitude *= -1.0;
	longitude = To_360 (StrictMath.toDegrees (longitude));

	world_coordinate = new Point2D.Double (longitude, latitude);
	}
else
	world_coordinate = new Point2D.Double
		(image_coordinate.getX (), image_coordinate.getY ());
return world_coordinate;
}

/**	Get the image sample,line coordinate for a world longitude,latitude
	coordinate.
<p>
	The conversion algorithm is:
<p><blockquote>

</blockquote>
<p>
	@param	world_coordinate	The world longitude,latitude coordinate.
		The {@link Point2D#getX() x} value of the coordinate Point2D is
		the longitude; the {@link Point2D#getY() y} value is the
		latitude. Values are in degrees.
	@return	The image sample,line coordinate.
*/
public Point to_Image
	(
	Point2D		world_coordinate
	)
	throws IllegalArgumentException
{
if (world_coordinate == null)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to project world coordinates to image coordinates\n"
		+ "because the world coordinates point is null.");
Point
	image_coordinate = null;
if (Not_Identity)
	{
	double
		longitude = StrictMath.toRadians (world_coordinate.getX ()),
		latitude  = StrictMath.toRadians (world_coordinate.getY ());

	if (Planetocentric)
		latitude = Planetographic_to_Planetocentric (latitude);
	if (Positive_West)
		longitude *= -1.0;

	double
		rho = Polar_Radius_times_2
			* StrictMath.tan (PI_OVER_4
				- (Center_Latitude_Sign * 0.5 * latitude)),
		distance = longitude - Center_Longitude;

	image_coordinate = new Point
		(
		(int)(Projection_X_to_Sample (rho * StrictMath.sin (distance))
			+ 0.5),		//	Round to nearest pixel.
		(int)(Projection_Y_to_Line   (rho * StrictMath.cos (distance)
				* -Center_Latitude_Sign)
			+ 0.5)		//	Round to nearest pixel.
		);
	}
else
	image_coordinate = new Point
		((int)image_coordinate.getX (), (int)image_coordinate.getY ());
return image_coordinate;
}


}
