/*	Data_View

PIRL CVS ID: Data_View.java,v 1.14 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2002-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.PVL.*;
import PIRL.Viewers.*;
import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;

import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.Iterator;
import java.io.File;
import java.io.FileWriter;
import java.io.BufferedWriter;
import java.io.IOException;
import java.net.URL;

/**	<I>Data_View</I> manages access to <I>Database_View</I>s
	as mediated by its Configuration.
<P>
	@author	Bradford Castalia, UA/PIRL
	@version 1.14
	@see	Configuration
*/
public class Data_View
	extends JFrame
{
private static final String
	ID = "PIRL.Database.Data_View (1.14 2012/04/16 06:08:57)";

private static final Point		DEFAULT_INITIAL_LOCATION
									= new Point (50, 25);
private boolean					Tooltips_Enabled
									= true;
private JMenuItem				Tooltips_Checkbox;
private static final String		TOOLTIPS_PARAMETER
									= "/Tooltips";

//	Configuration.
private static final String		DEFAULT_CONFIGURATION_FILENAME
									= "Database.conf";
private String					CWD	// Current Working Directory.
									= System.getProperty ("user.dir"),
								Configuration_Filename
									= null;
private JMenuItem				Save_Menu_Item;
private Configuration			Default_Configuration
									= null,
								Current_Configuration;
private JComboBox				Server_Names;
private Vector					Servers,
								Removed_Parameters
									= new Vector ();
private static final String		REMOVE_PARAMETER
									= "/remove",
								MASKED_VALUE
									= "*******";
private static ImageIcon		OPEN_ICON
									= null;
private static final String		OPEN_ICON_NAME
									= "Open_File16.gif";
private static ImageIcon		SAVE_ICON;
private static final String		SAVE_ICON_NAME
									= "Save16.gif";
private static ImageIcon		SAVE_AS_ICON;
private static final String		SAVE_AS_ICON_NAME
									= "SaveAs16.gif";
private static ImageIcon		ABOUT_ICON;
private static final String		ABOUT_ICON_NAME
									= "About16.gif";
private static ImageIcon		PIRL_ICON;
private static final String		PIRL_ICON_NAME
									= "PIRL_Logo.gif";
private static ImageIcon		LPL_ICON;
private static final String		LPL_ICON_NAME
									= "LPL_Logo.gif";
private static ImageIcon		UA_ICON;
private static final String		UA_ICON_NAME
									= "UofA_Logo.gif";

//	Database.
private Vector					Database_Views
									= new Vector ();
private View_Locator			Database_View_Locator
									= new View_Locator ();
private static final String		DATABASE_TYPE_PARAMETER
									= Database.TYPE;

//	Connection.
private JButton					Connect_Button;
private boolean					Confirm_Connect
									= false;
private JMenuItem				Confirm_Connect_Checkbox;
private static final String		CONFIRM_CONNECT_PARAMETER
									= "/Confirm_Connect";
private static final Dimension	PASSWORD_DIALOG_SIZE
									= new Dimension (250, 100);
private static final String		HOSTNAME_PARAMETER
									= Configuration.HOST,
								USERNAME_PARAMETER
									= Configuration.USER,
								PASSWORD_PARAMETER
									= "password",
								CLASSPATH_PARAMETER
									= Configuration.CLASSPATH;

private Data_View				This_Data_View;

private static final char
	File_Separator = System.getProperty ("file.separator").charAt (0);

private static final String		NL = Database.NL;


//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_SETUP		= 1 << 0,
	DEBUG_CONNECT	= 1 << 1,
	DEBUG_CONFIG	= 1 << 2,
	DEBUG_HELPERS	= 1 << 3,
	DEBUG_ALL		= -1,

	DEBUG       	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Data_View
	(
	Configuration	configuration
	)
	throws Configuration_Exception
{
super("Data_View");
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> Data_View");
This_Data_View = this;
Load_Icons (this);

Font
	Default_Font = new Font ("SansSerif", Font.PLAIN, 12);

setJMenuBar (Menu_Bar ());
Configure (configuration);
getContentPane ().add (Panels (), BorderLayout.CENTER);
pack ();
}

private Data_View ()
	throws Database_Exception
{
throw new Database_Exception
	(
	getClass().getName() +
	": Hey! You shouldn't be using the default constructor."
	);
}

/*==============================================================================
	Accessors
*/
public Vector Servers ()
{return Servers;}

/*==============================================================================
	Menus
*/
private JMenuBar Menu_Bar ()
{
JMenuBar
	menu_bar = new JMenuBar ();
JMenu
	menu;
JMenuItem
	menu_item;

menu = new JMenu ("File");

menu_item = new JMenuItem ("Open", OPEN_ICON);
menu_item.setMnemonic ('O');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('O', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Open_Configuration ();}});
menu.add (menu_item);

Save_Menu_Item = new JMenuItem ("Save", SAVE_ICON);
Save_Menu_Item.setMnemonic ('O');
Save_Menu_Item.setAccelerator (KeyStroke.getKeyStroke ('S', Event.CTRL_MASK));
Save_Menu_Item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Save_Configuration (false);}});
menu.add (Save_Menu_Item);

menu_item = new JMenuItem ("Save As ...", SAVE_AS_ICON);
menu_item.setMnemonic ('A');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('O', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Save_Configuration (true);}});
menu.add (menu_item);

menu.addSeparator ();

menu_item = new JMenuItem ("Exit");
menu_item.setMnemonic ('X');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('Q', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener()
	{public void actionPerformed (ActionEvent event)
		{System.exit (0);}});
menu.add (menu_item);

menu_bar.add (menu);

menu = new JMenu ("Preferences");

Confirm_Connect_Checkbox
	= new JCheckBoxMenuItem ("Confirm Connection", Confirm_Connect);
Confirm_Connect_Checkbox.setMnemonic ('C');
Confirm_Connect_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Flip_Confirm_Connect ();}});
menu.add (Confirm_Connect_Checkbox);

Tooltips_Checkbox
	= new JCheckBoxMenuItem ("Tooltips", Tooltips_Enabled);
Tooltips_Checkbox.setMnemonic ('T');
Tooltips_Checkbox.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Flip_Tooltips_Enabled ();}});
menu.add (Tooltips_Checkbox);

menu_bar.add (menu);

menu = new JMenu ("Help");

menu_item = new JMenuItem ("About ...", ABOUT_ICON);
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{About (This_Data_View);}});
menu.add (menu_item);

menu_bar.add (menu);

return menu_bar;
}

/*==============================================================================
	Panels
*/
private JPanel Panels ()
{
JPanel panel		= new JPanel ();
GridBagLayout grid	= new GridBagLayout ();
GridBagConstraints location = new GridBagConstraints ();
panel.setLayout (grid);

//	Connect button:
Connect_Button		= new JButton ("Connect");
Connect_Button.setMnemonic ('C');
Connect_Button.setToolTipText ("Open a connection to the database.");
Connect_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
		{Connect_to_Database ();}});
Connect_Button.setDefaultCapable (true);
getRootPane ().setDefaultButton (Connect_Button);
location.insets		= new Insets (10, 10, 10, 5);
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
panel.add (Connect_Button, location);

//	Type:
location.insets		= new Insets (10, 0, 10, 5);
location.fill		= GridBagConstraints.NONE;
panel.add (new JLabel("to server:"), location);

Server_Names		= new JComboBox (Servers);
Server_Names.setEnabled (! Servers.isEmpty ());
Server_Names.setMinimumSize (new Dimension (150, 25));
Server_Names.setPreferredSize (new Dimension (150, 25));
Server_Names.setToolTipText ("The name of a database server.");
location.insets		= new Insets (10, 0, 10, 10);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
panel.add (Server_Names, location);

return panel;
}

/*==============================================================================
	Configuration
*/
private void Configure
	(
	Configuration	configuration
	)
	throws Configuration_Exception
{
Configuration
	default_configuration;
if (Default_Configuration == null)
	{
	//	First time: Load the class's default configuration.
	try
		{
		Default_Configuration = new Configuration
			(ClassLoader.getSystemResource
			(getClass ().getPackage ().getName ().replace ('.', File_Separator)
			+ File_Separator + DEFAULT_CONFIGURATION_FILENAME));
		if ((DEBUG & DEBUG_CONFIG) != 0)
			System.out.println ("    Default Configuration -" + NL
				+ Default_Configuration.Description ());
		}
	catch (Configuration_Exception exception)
		{
		Dialog_Box.Error ("Unable to load the default Configuration." + NL
			+ NL
			+ ID + NL
			+ exception.getMessage ());
		throw exception;
		}

	//	Get any list of parameters to be removed.
	Removed_Parameters = Default_Configuration.Get_Specific (REMOVE_PARAMETER);
	//	Remove passwords.
	if (! Removed_Parameters.contains (PASSWORD_PARAMETER))
		Removed_Parameters.add (PASSWORD_PARAMETER);
	}
	
if (configuration == null)
	Current_Configuration = new Configuration (Default_Configuration);
else
	Current_Configuration = configuration;

//	Configuration flags.
boolean
	case_sensitive = Current_Configuration.Case_Sensitive (false);
Tooltips_Checkbox.setSelected
	(Tooltips_Enabled = get_flag (Current_Configuration,
		TOOLTIPS_PARAMETER, Tooltips_Enabled));
Confirm_Connect_Checkbox.setSelected 
	(Confirm_Connect = get_flag (Current_Configuration,
		CONFIRM_CONNECT_PARAMETER, Confirm_Connect));

//	Server names.
Servers = Current_Configuration.Get
	(Configuration.Absolute_Pathname (null, Database.SERVER));
if (Servers.isEmpty ())
	Servers = Current_Configuration.Get
	(Configuration.Absolute_Pathname (null, Database.TYPE));
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println ("    Server names: " + Servers);


//	Check for a local filename.
String
	Configuration_Filename = Current_Configuration.Filename ();
if (Configuration_Filename == null ||
	Configuration_Filename.equals (Configuration.DEFAULTS))
	//	No filename.
	Configuration_Filename = null;
else if (Configuration_Filename.startsWith ("file:" + File_Separator))
	//	file:/ URL; use the pathname portion.
	Configuration_Filename = Configuration_Filename.substring (5);
else
	{
	int
		index = Configuration_Filename.indexOf (':');
	if (index > 0 &&
		Configuration_Filename.indexOf (File_Separator) == (index + 1))
		//	URL
		Configuration_Filename = null;
	}
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println ("    Configuration_Filename - "
		+ Configuration_Filename);
Set_Savable (Configuration_Filename != null);

Current_Configuration.Case_Sensitive (case_sensitive);
}

public static boolean get_flag
	(
	Configuration	configuration,
	String			parameter,
	boolean			default_flag
	)
{
String
	value = configuration.Get_One (parameter);
if (value == null)
	return default_flag;
if (value.equalsIgnoreCase ("on") ||
	value.equalsIgnoreCase ("true") ||
	value.equalsIgnoreCase ("enabled"))
	return true;
else
	return false;
}

/*==============================================================================
	Actions
*/
//	OPEN a new Configuration file.
private void Open_Configuration ()
{
JFileChooser
	file_chooser = new JFileChooser (CWD);
file_chooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
if (file_chooser.showOpenDialog (rootPane)
	== JFileChooser.APPROVE_OPTION)
	{
	File
		file = file_chooser.getSelectedFile ();
	CWD = file.getParent ();
	try
		{
		Configure (new Configuration (file.getPath ()));

		//	Reset the server names.
		Server_Names.removeAllItems ();
		if (Servers.isEmpty ())
			Server_Names.setEnabled (false);
		else
			for (Iterator
					names = Servers.iterator ();
					names.hasNext ();
				 Server_Names.addItem (names.next ()));
		}
	catch (Exception exception)
		{
		Dialog_Box.Error ("Unable to Open the Configuration -" + NL
			+ file.getPath () + NL
			+ NL
			+ exception.getMessage (),
			this);
		}
	}
}

//	SAVE the Configuration to a file.
private void Save_Configuration
	(
	boolean		interactive
	)
{
//	Make sure the configuration has all the settings.
try
	{
	Current_Configuration.Set (TOOLTIPS_PARAMETER,
		(Tooltips_Enabled ? "true" : "false"));
	Current_Configuration.Set (CONFIRM_CONNECT_PARAMETER,
		(Confirm_Connect ? "true" : "false"));
	Current_Configuration.Remove (CLASSPATH_PARAMETER);
	}
catch (Configuration_Exception exception)
	{
	Dialog_Box.Error ("Unable to set the configuration values!" + NL
		+ NL
		+ ID + NL
		+ exception.getMessage ());
	return;
	}

File
	file;
if (! interactive &&
	Configuration_Filename == null)
	//	No filename; force interactive operation.
	interactive = true;
if (interactive)
	{
	JFileChooser
		file_chooser = new JFileChooser (CWD);
	file_chooser.setFileSelectionMode (JFileChooser.FILES_ONLY);
	if (file_chooser.showSaveDialog (this)
			!= JFileChooser.APPROVE_OPTION)
		return;
	file = file_chooser.getSelectedFile ();
	CWD = file.getParent ();
	}
else
	file = new File (Configuration_Filename);
if ((DEBUG & DEBUG_CONFIG) != 0)
	System.out.println ("    Data_View.Save_File_As: " + file.getPath ());

//	Write the configuration file.
BufferedWriter
	writer = null;
String
	name = Current_Configuration.Name ();
try
	{
	writer = new BufferedWriter (new FileWriter (file));
	writer.write (Parser.CROSSHATCH);
	writer.write (' ');
	writer.write (name + NL);
	writer.write (Parser.CROSSHATCH);
	writer.write (' ');
	writer.write (ID + NL);
	}
catch (IOException exception)
	{
	Dialog_Box.Error ("Unable to save the configuration file -" + NL
		+ file.getPath () + NL
		+ NL
		+ ID + NL
		+ exception.getMessage ());
	}
Lister
	lister = new Lister (writer).Strict (false);
//	Prevent the top level from being written.
Current_Configuration.Name (Parser.CONTAINER_NAME);
try
	{
	lister.Write (Current_Configuration);
	Configuration_Filename = file.getPath ();
	Set_Savable (false);
	}
catch (PVL_Exception exception)
	{
	Dialog_Box.Error ("The configuration contains an invalid parameter!" + NL
		+ NL
		+ ID + NL
		+ exception.getMessage ());
	Current_Configuration.Name (name);	//	Restore the original name.
	}
catch (IOException exception)
	{
	Dialog_Box.Error ("Unable to write the configuration to the file -" + NL
		+ file.getPath () + NL
		+ NL
		+ ID + NL
		+ exception.getMessage ());
	Current_Configuration.Name (name);	//	Restore the original name.
	}

if (lister.Warning () != null)
	{
	Dialog_Box.Warning ("Potential syntax problem in the configuration file -" + NL
		+ file.getPath () + NL
		+ NL
		+ ID + NL
		+ lister.Warning ().getMessage () + NL);
	}
}

//	Enable/Disable the Save menu item in the File menu.
private void Set_Savable
	(
	boolean		savable
	)
{
if (savable && Configuration_Filename == null)
	savable = false;
Save_Menu_Item.setEnabled (savable);
}

/*------------------------------------------------------------------------------
	CONNECT to a Database and provide a new Database_View.
*/
private void Connect_to_Database ()
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println (">>> Connect_to_Database");
Database
	database = null;
try
	{
	Configuration
		configuration = new Configuration (Current_Configuration);
	database = new Database (configuration);

	//	The the Configuration to Open the database connection.
	if ((configuration = Get_Connection_Info (database)) == null)
		return;	//	User cancelled.

	database.Open (configuration);
	}
catch (Configuration_Exception exception)
	{
	Dialog_Box.Error ("Unable to work with the application's configuration." + NL
		+ NL
		+ ID + NL
		+ Masked_Message (Removed_Parameters, exception.getMessage ()));
	return;
	}
catch (Database_Exception exception)
	{
	Dialog_Box.Error ("Unable to Connect to the Database." + NL
		+ NL
		+ ID + NL
		+ Masked_Message (Removed_Parameters, exception.getMessage ()));
	return;
	}

//	Construct a new Database_View.
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println ("    Connect_to_Database: Creating a new Database_View");
final Database
	DB = database;
final Blinker
	blinking = new Blinker (Connect_Button);
blinking.start ();
final Cursor
	cursor = getCursor ();
setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));
Thread
	construct_Database_View = new Thread ()
		{
		public void run ()
		{
		try
			{
			Database_View
				database_view = new Database_View (DB);
			database_view.addWindowListener (new WindowAdapter ()
				{public void windowClosing (WindowEvent event)
				{
				Window window = event.getWindow ();
				Database_Views.remove (window);
				window.dispose ();
        		}});				
			//	Set the new screen location.
			if (Database_Views.isEmpty ())
				{
				//	First Database_View.
				Database_View_Locator
					.Offsets    (0, 0)
					.Horizontal (View_Locator.LEFT   | View_Locator.INWARD)
					.Vertical   (View_Locator.BOTTOM | View_Locator.OUTWARD);
				//	Locate relative to this view.
				Database_View_Locator.Relocate (database_view, This_Data_View);
				}
			else
				{
				Database_View_Locator.Policies (null);	//	Defaults.
				//	Locate relative to the last Database_View.
				Database_View_Locator.Relocate
					(database_view, (JFrame)Database_Views.lastElement ());
				}
			//	Remember it and view it.
			Database_Views.add (database_view);
			database_view.setVisible (true);
			Set_Savable (true);
			}
		catch (Exception exception)
			{
			Dialog_Box.Error ("Unable to create the new Database_View." + NL
				+ NL
				+ ID + NL
				+ Masked_Message (Removed_Parameters, exception.getMessage ()));
			}
		SwingUtilities.invokeLater (new Runnable ()
			{
			public void run () 
			{
			blinking.stop ();
			setCursor (cursor);
			}});
		}};
construct_Database_View.start ();
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println ("<<< Connect_to_Database");
}

private void Flip_Tooltips_Enabled ()
{
ToolTipManager.sharedInstance ().setEnabled
	(Tooltips_Enabled = ! Tooltips_Enabled);
Set_Savable (true);
}

private void Flip_Confirm_Connect ()
{
Confirm_Connect = ! Confirm_Connect;
Set_Savable (true);
}

/*==============================================================================
	Helpers
*/
private Configuration Get_Connection_Info
	(
	Database		database
	)
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> Get_Connection_Info");
String
	server = null;
if (! Servers.isEmpty ())
	server = (String)Server_Names.getSelectedItem ();
Configuration
	database_configuration = null;
try {database_configuration = database.Server_Configuration (server);}
catch (Configuration_Exception e) {/* Shouldn't happen */}
if (database_configuration == null)
	database_configuration = database.Configuration ();
if (server == null)
	server = "";
String
	hostname = database_configuration.Get_One (HOSTNAME_PARAMETER),
	username = database_configuration.Get_One (USERNAME_PARAMETER),
	password = database_configuration.Get_One (PASSWORD_PARAMETER);
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("    Get_Connection_Info:" + NL
		+"    Database Configuration -" + NL
		+ database_configuration.Description ());

Connect_Info:
if (Confirm_Connect ||
	(hostname == null) ||
	(username == null) ||
	(password == null))
	{
	JPanel panel		= new JPanel ();
	GridBagLayout grid	= new GridBagLayout ();
	GridBagConstraints location = new GridBagConstraints ();
	panel.setLayout (grid);

	//	Server hostname.
	JTextField hostname_field	= new JTextField (hostname);
	location.gridwidth	= 1;
	location.anchor		= GridBagConstraints.EAST;
	location.fill		= GridBagConstraints.NONE;
	location.weightx	= 0.0;
	panel.add (new JLabel("Server name:"), location);
	panel.add (Box.createHorizontalStrut (5), location);
	location.gridwidth	= GridBagConstraints.REMAINDER;
	location.anchor		= GridBagConstraints.WEST;
	location.fill		= GridBagConstraints.HORIZONTAL;
	location.weightx	= 1.0;
	panel.add (hostname_field, location);

	//	User name.
	JTextField username_field	= new JTextField (username);
	location.gridwidth	= 1;
	location.anchor		= GridBagConstraints.EAST;
	location.fill		= GridBagConstraints.NONE;
	location.weightx	= 0.0;
	panel.add (new JLabel("User name:"), location);
	panel.add (Box.createHorizontalStrut (5), location);
	location.gridwidth	= GridBagConstraints.REMAINDER;
	location.anchor		= GridBagConstraints.WEST;
	location.fill		= GridBagConstraints.HORIZONTAL;
	location.weightx	= 1.0;
	panel.add (username_field, location);

	//	Password.
	JPasswordField password_field = new JPasswordField (password);
	location.gridwidth	= 1;
	location.anchor		= GridBagConstraints.EAST;
	location.fill		= GridBagConstraints.NONE;
	location.weightx	= 0.0;
	panel.add (new JLabel("Password:"), location);
	panel.add (Box.createHorizontalStrut (5), location);
	location.gridwidth	= GridBagConstraints.REMAINDER;
	location.anchor		= GridBagConstraints.WEST;
	location.fill		= GridBagConstraints.HORIZONTAL;
	location.weightx	= 1.0;
	panel.add (password_field, location);
	JLabel warning		= new JLabel ("For the database, NOT the system");
	warning.setFont (new Font ("SansSerif", Font.PLAIN, 10));
	location.anchor		= GridBagConstraints.EAST;
	location.fill		= GridBagConstraints.NONE;
	location.weightx	= 0.0;
	panel.add (warning, location);

	panel.setPreferredSize (PASSWORD_DIALOG_SIZE);

	if (JOptionPane.showConfirmDialog
			(
			null,
			panel,
			("Connect to Server: " + server),
			JOptionPane.OK_CANCEL_OPTION
			)
		== JOptionPane.OK_OPTION)
		{
		hostname = hostname_field.getText ().trim ();
		if (hostname.length () == 0)
			{
			Dialog_Box.Notice ("A server host name to connect to is required.");
			break Connect_Info;
			}
		username = username_field.getText ().trim ();
		if (username.length () == 0)
			{
			Dialog_Box.Notice ("A database user name is required.");
			break Connect_Info;
			}
		password = new String (password_field.getPassword ());
		if (password.length () == 0)
			{
			Dialog_Box.Notice ("A database password is required.");
			break Connect_Info;
			}
		try
			{
			database_configuration.Set (HOSTNAME_PARAMETER, hostname);
			database_configuration.Set (USERNAME_PARAMETER, username);
			database_configuration.Set (PASSWORD_PARAMETER, password);
			if ((DEBUG & DEBUG_CONNECT) != 0)
				System.out.println
					("<<< Get_Connection_Info:" + NL
					+"    hostname: " + hostname + NL
					+"    username: " + username + NL
					+"    password: " + password + NL
					+"    Configuration -" + NL
					+ database_configuration.Description ());
			}
		catch (Configuration_Exception exception)
			{
			Dialog_Box.Error ("Unable to set the configuration parameters." + NL
				+ NL
				+ ID + NL
				+ Masked_Message (Removed_Parameters, exception.getMessage ()));
			}
		}
	else
		{
		database_configuration = null;
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("<<< Get_Connection_Info: cancelled.");
		}
	}
return database_configuration;
}

public static String Masked_Message
	(
	Vector	removed_parameters,
	String	message
	)
{
if (message == null)
	return "";
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println (">>> Masked_Message:" + NL
		+ message + NL);
StringBuffer
	masked_message = new StringBuffer (message);
int
	patch_size = MASKED_VALUE.length (),
	stop = masked_message.length ();
Iterator
	parameters = removed_parameters.iterator ();
while (parameters.hasNext ())
	{
	String
		word = (String)parameters.next ();
	if ((DEBUG & DEBUG_HELPERS) != 0)
		System.out.println ("    Removed parameter " + word);
	int
		length = word.length (),
		start = 0,
		end;
	while ((start = masked_message.indexOf (word, start)) != -1)
		{
		/*	Found the name of a value to be masked.
			Skip it and following whitespace and '=' characters.
		*/
		for (start += length, end = 0;
			 start != stop &&
				(masked_message.charAt (start) == ' ' ||
				 masked_message.charAt (start) == '=');
			 start++)
			{
			if (masked_message.charAt (start) == '=')
				{
				if (end == 1)
					break;
				end = 1;
				}
			}
		if (start == stop)
			break;
		if (end == 0)
			continue;
		//	Find the end of the value to be obscured.
		for (end = start + 1;
			 end != stop &&
			 	(masked_message.charAt (end) != ' ' &&
				masked_message.charAt (end) != '\n' &&
				masked_message.charAt (end) != ')');
			 end++);
		//	Obscure it.
		if ((DEBUG & DEBUG_HELPERS) != 0)
			System.out.println ("    Masked: " + start + " - " + end);
		masked_message.replace (start, end, MASKED_VALUE);

		//	Adjust the known message length.
		stop += patch_size - (end - start);
		start = end;
		}
	}
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println ("<<< Masked_Message:" + NL
		+ masked_message);
return masked_message.toString ();
}

private static void Load_Icons
	(
	Object	object
	)
{
if (OPEN_ICON != null)
	return;
String
	package_name = object.getClass ().getName ();
package_name = package_name.substring (0, package_name.lastIndexOf ('.'))
		.replace ('.', File_Separator)
		+ File_Separator + "Icons" + File_Separator;
URL
	iconURL;
iconURL = ClassLoader.getSystemResource (package_name + OPEN_ICON_NAME);
if (iconURL != null)
	OPEN_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + SAVE_ICON_NAME);
if (iconURL != null)
	SAVE_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + SAVE_AS_ICON_NAME);
if (iconURL != null)
	SAVE_AS_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + ABOUT_ICON_NAME);
if (iconURL != null)
	ABOUT_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + PIRL_ICON_NAME);
if (iconURL != null)
	PIRL_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + LPL_ICON_NAME);
if (iconURL != null)
	LPL_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + UA_ICON_NAME);
if (iconURL != null)
	UA_ICON = new ImageIcon (iconURL);
}

public static void About
	(
	JFrame	parent
	)
{
final JFrame
	frame = new JFrame ("PIRL Data_View");
frame.setDefaultCloseOperation (WindowConstants.DISPOSE_ON_CLOSE);
JPanel
	content = new JPanel (new GridBagLayout ());
GridBagConstraints location = new GridBagConstraints ();

location.insets		= new Insets (10, 10, 20, 10);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
content.add (new JLabel
	("<HTML>This is a <FONT COLOR=RED>Beta Release</FONT></B>: <CODE>" +
	ID + "</CODE>"),
	location);

location.insets		= new Insets (0, 10, 5, 0);
location.gridwidth	= 1;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
content.add (new JLabel (PIRL_ICON), location);
location.insets		= new Insets (0, 10, 5, 10);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
content.add (new JLabel
	("<HTML><B><FONT face=\"timesroman, serif\" size=+1>"
	+"Planetary Image<P>Research Laboratory"),
	location);

location.insets		= new Insets (0, 10, 5, 0);
location.gridwidth	= 1;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
content.add (new JLabel (LPL_ICON), location);
location.insets		= new Insets (0, 10, 5, 10);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
content.add (new JLabel
	("<HTML><B><FONT face=\"timesroman, serif\" size=+1>"
	+"Department of<P>Planetary Sciences"),
	location);

location.insets		= new Insets (0, 10, 5, 0);
location.gridwidth	= 1;
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
content.add (new JLabel (UA_ICON), location);
location.insets		= new Insets (0, 10, 5, 10);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
content.add (new JLabel
	("<HTML><B><FONT face=\"timesroman, serif\" size=+1>"
	+"University of Arizona<P>Tucson, Arizona"),
	location);

location.insets		= new Insets (20, 10, 10, 10);
location.fill		= GridBagConstraints.BOTH;
location.anchor		= GridBagConstraints.NORTHWEST;
location.weightx	= 1.0;
location.weighty	= 1.0;
content.add (new JLabel
	("<HTML>"
	+"Comments - including recommendations for changes," + NL
	+"error reports (please include the full text of any error messages)" + NL
	+"or any other communication -" + NL
	+"can be sent to the author, Bradford Castalia, at Castalia@Arizona.edu." + NL),
	location);

content.setPreferredSize (new Dimension (375, 400));
frame.setContentPane (content);
frame.pack ();

View_Locator
	locator = new View_Locator ()
		.Orientation (
			View_Locator.RIGHT | View_Locator.OUTWARD,
			View_Locator.TOP | View_Locator.INWARD)
		.Offsets (0, 0);
locator.Relocate (frame, parent);
frame.setVisible (true);
}

/*==============================================================================
	Application
*/
public static void main (String[] arguments) 
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("*** Data_View ***");
try
	{
	Configuration
		configuration;
	if (arguments.length > 0)
		configuration = new Configuration (arguments[0]);
	else
		configuration = new Configuration (DEFAULT_CONFIGURATION_FILENAME);
	if ((DEBUG & DEBUG_CONFIG) != 0)
		System.out.println ("    User Configuration -" + NL
			+ configuration.Description ());
	if (configuration.Filename ().equals (Configuration.DEFAULTS))
		//	Only got defaults; leave Data_View to load its defaults.
		configuration = null;
	Data_View
		data_view = new Data_View (configuration);
	data_view.setDefaultCloseOperation (JFrame.EXIT_ON_CLOSE);
	//	Set the new screen location.
	data_view.setLocation (DEFAULT_INITIAL_LOCATION);
	data_view.setVisible (true);

	if (! data_view.Servers ().isEmpty ())
		{
		//	Open the default Database.
		if ((DEBUG & DEBUG_SETUP) != 0)
			System.out.println ("    Connecting to the Database...");
		data_view.Connect_to_Database ();
		}
	}
catch (Exception exception)
	{
	System.out.println (exception.getMessage ());
	System.exit (1);
	}
}

}	//	End of Data_View class.

