#!/usr/local/bin/perl5
#
# I wrote a simple copying script:
#
# pi-ldif [-p port] fromfile tofile
#
# where fromfile or tofile can be "pilot" (the box), "anything.pl" (a perl
# structure dump) or "anything.ldif" (an ldif file suitable for Netscape,
# best with Netscape 4.5).
#
# It is state-preserving:  if you upload the ldif file into netscape 4.5,
# and download it again, the pilot will not lose any info (it tries to
# find a place for every netscape datum on the pilot, and every pilot
# datum on netscape.  If all else fails, it stuffs perl text into the
# Notes field).
#
# I use it a lot, but now I don't like it so much.  I'm pretty sure the
# right thing to do is to sync directly to a writable subtree of an LDAP
# server.  This would (a) allow Netscape and Outlook and every other
# LDAP-aware application to share the addressbook, (b) make the address
# book network-accessible, (c) allow the schema flexibility to stow
# whatever the pilot needs on the server.  Real Soon Now (tm).
#
# For the same reason, I like the SynCM calendar solution (talk to a
# calendar server), and prefer the proposed IMAP solution for mail (talk
# to a mail server, rather than local mail file(s)).  I'm always happier,
# I find, if I can sync to a network accessible representation of my data,
# rather than to a file.
#
# Brad Rubenstein, rubenb@jany.gs.com, 19990122
#

BEGIN {
    # PDA is installed here:
    unshift @INC, 
    "/sw/external/pilot-981209/lib/perl5/site_perl/5.005/sun4-solaris";
};

eval 'use PDA::Pilot';
if ( $@ )
{
    die "
Could not find PDA::Pilot package.
Set your PERLLIB variable, or edit this script to its installed
location.
";
}

use Carp;
use Getopt::Std;
use Data::Dumper;
use sigtrap qw(die normal-signals);

use strict;
use integer;

#
# The basic mapping from Pilot (neutral) to LDIF
#

my %LDIFToPilot=(
	'dn'			=> undef,
	'cn'			=> undef,
	'sn'			=> 'last name',
	'givenname'		=> 'first name',
	'title'			=> 'title',
	'xmozillanickname'	=> undef,
	'gender'		=> undef,
	'pilotid'		=> 'id',
	'category'		=> 'category',
	'private'		=> 'secret',
	'carphone'		=> undef,
	'cellphone'		=> 'mobile',
	'facsimiletelephonenumber'=> 'fax',
	'homephone'		=> 'home',
	'msgphone'		=> 'main',
	'pagerphone'		=> 'pager',
	'otherphone'		=> 'other',
	'parentphone'		=> undef,
	'telephonenumber'	=> 'work',
	'xmozillaanyphone'	=> undef, 
	'mail'			=> 'e-mail',
	'pageremail'		=> undef,
	'xmozillausehtmlmail'	=> undef,
	'homeurl'		=> undef,
	'workurl'		=> undef,
	'o'			=> 'company',
	'ou'			=> undef,
	'departmentnumber'	=> undef,
	'postofficebox'		=> undef,
	'streetaddress'		=> 'address',
	'locality'		=> 'city',
	'st'			=> 'state',
	'countryname'		=> 'country',
	'postalcode'		=> 'zip code',
	'bday'			=> 'birthdate',
	'manager'		=> undef,
	'secretary'		=> undef,
	'carlicense'		=> undef,
	'custom1'		=> 'custom 1',
	'custom2'		=> 'custom 2',
	'custom3'		=> 'custom 3',
	'custom4'		=> 'custom 4',
	'custom5'		=> 'custom 5',
	'description'		=> 'note',
	'modifytimestamp'	=> undef, 
	'objectclass'		=> undef, 
);

# an ldif written by NS will never contain these fields
# (they have to be pushed into the description for safe keeping)
my %NSUnsupported = (
	category => 1,
	otherphone => 1,
	private => 1,
	pilotid => 1,
	);


my %PilotToLDIF;
my $AppBlockDN = 'cn="~ Pilot AddressDB Data"';

sub PilotToNeutral
{
    my ( $pi_rec, $pdb ) = @_;
    my %rec;
    my ( $pi_label, $label, $i );

    # print Data::Dumper->Dumpxs( [ $pi_rec ], [ 'PilotToNeutral_in' ]), "\n";

    for $pi_label ( qw( id category secret ) )
	{
		$label = $PilotToLDIF{lc($pi_label)};
		if ( ! defined $label )
		{
			warn "NOTE: no mapping for field $pi_label\n";
			$label = lc($pi_label);
		}
		$rec{$label} = $pi_rec->{$pi_label};
	}

    for( $i = $#{$pi_rec->{entry}}; $i >= 0; $i-- )
    {
	next unless defined $pi_rec->{entry}[$i];

	# use phoneLabel for these
	if ( $i >= 3 && $i <= 7 )
	{
	    $pi_label = $pdb->{$AppBlockDN}->{phoneLabel}[$pi_rec->{phoneLabel}[$i-3]];
	}
	else
	{
	    $pi_label = $pdb->{$AppBlockDN}->{label}[$i];
	}

	$label = $PilotToLDIF{lc($pi_label)};

	if ( ! defined $label )
	{
	    warn "NOTE: no mapping for field $pi_label";
	    $label = lc($pi_label);
	}

	$rec{$label} = $pi_rec->{entry}[$i];
	$rec{$label} =~ s/^\s*//;
	$rec{$label} =~ s/\s*$//;
    }

    if( defined $rec{description}
    && ($rec{description} =~ s/\s*(\$EXTRA = [{][^}]*[}];)\s*//o ))
    {
	my $EXTRA;
	eval $1;
	warn "$@ $1" if $@;
	map($rec{$_} = $EXTRA->{$_}, keys %$EXTRA);
    }

    # we extract a few more things from the pilot

    # final parenthesized givenname is moved to nickname
    if ( $rec{givenname} && ($rec{givenname} =~ s/\s*\((.*)\)\s*$// ))
    {
	$rec{xmozillanickname} = $1;
    }

    $rec{xmozillaanyphone} =
	$pi_rec->{entry}[$pi_rec->{showPhone}+3];


    # URL in E-mail is moved to homeurl
    my $show;
    if ( defined $rec{mail} && ($rec{mail} =~ s,\s*(http://\S+)\s*,,io ))
    {
	$rec{homeurl} = $1;
    }

    # final parenthesized "(html)" in E-mail is moved to 
    # xmozillausehtmlmail 
    if ( defined $rec{mail} && ($rec{mail} =~ s,\s*\(html\)\s*$,,io ))
    {
	$rec{xmozillausehtmlmail} = 'TRUE';
    }

    # 2nd and subsequent lines of Company are moved to department
    if ( defined $rec{o} && ( $rec{o} =~ s,\s*\n\s*(.+)\s*,,o ))
    {
	$rec{ou} = $1;
    }

    # 1st line of address is moved to PostOfficeBox
    if ( defined $rec{streetaddress} && ($rec{streetaddress} =~ s/^(.*)\n//i ))
    {
	$rec{postofficebox} = $1;
    }

    if ( $rec{givenname} && $rec{sn} )
    {
	$rec{cn} = $rec{givenname} . ' ' . $rec{sn};
    }
    elsif( $rec{sn} )
    {
	$rec{cn} = $rec{sn};
    }
    elsif( $rec{o} )
    {
	$rec{cn} = $rec{o};
    }
    else
    {
	warn "ERROR: entry with no name discarded: " .
	    Data::Dumper->Dumpxs( [ \%rec ], [ 'rec' ]) . "\n";
	next;
    }

    if ($rec{mail})
    {
	$rec{dn} = sprintf "cn=\"%s\",mail=\"%s\"", $rec{cn}, $rec{mail};
    }
    else
    {
	$rec{dn} = sprintf "cn=\"%s\"", $rec{cn};
    }

    # print Data::Dumper->Dumpxs( [ \%rec ], [ 'PilotToNeutral_out' ]), "\n";

    return \%rec;
}

sub NeutralToPilot
{
    my ( $rec, $pdb, $addrdb ) = @_;
    my ( $pi_rec, $i, $j, %obj, %entry );
    my ( $label, $pi_label );

    # print Data::Dumper->Dumpxs( [ $rec ], [ 'rec' ]), "\n";

    $pi_rec = $addrdb->newRecord;

    return undef unless defined $rec->{pilotid};

    %obj = %$rec; 

    # we reverse the process of PilotToNeutral, putting things
    # into $pi_rec, and deleting them from $obj.  whatever is left,
    # we stuff onto Note EXTRA.
    for $label ( qw( pilotid category private ))
    {
	$pi_label = $LDIFToPilot{$label};
	$pi_rec->{$pi_label} = $obj{$label};
	delete $obj{$label};
    }

    for( $i = 18; $i >= 0; $i-- )
    {
	# use phoneLabel for these
	if ( $i >= 3 && $i <= 7 )
	{
	    $pi_label = $pdb->{$AppBlockDN}->{phoneLabel}[$i-3];
	}
	else
	{
	    $pi_label = $pdb->{$AppBlockDN}->{label}[$i];
	}

	$label = $PilotToLDIF{lc($pi_label)};
	$label = lc($pi_label) unless defined $label;

	# for later, when we want to stuff stuff back in
	$entry{$label} = $i;

	$pi_rec->{entry}[$i] = $obj{$label};
	delete $obj{$label};
    }

    # we put first 5 phones in their "natural" slots (above)
    $pi_rec->{phoneLabel} = [ 0,1,2,3,4 ];
    # now place any remaining phone numbers in spare entries
    for ( $j = 7; $j >= 5; $j-- )
    {
	$pi_label = $pdb->{$AppBlockDN}->{phoneLabel}[$j];
	$label = $PilotToLDIF{lc($pi_label)};
	$label = lc($pi_label) unless defined $label;
	next unless defined $obj{$label};

	for ( $i = 7; $i >= 3; $i-- )
	{
	    last unless defined $pi_rec->{entry}[$i];
	}
	last if $i < 3; # no empty phone slots left

	# put phone $j into entry $i
	$pi_rec->{entry}[$i] = $obj{$label};
	delete $obj{$label};
	$pi_rec->{phoneLabel}[ $i - 3 ] = $j;
    }

    $pi_rec->{showPhone} = 0;
    for ( $i = 3; $i <= 7; $i++ )
    {
	$pi_rec->{showPhone} = $i-3
	    if $obj{xmozillaanyphone} eq $pi_rec->{entry}[$i];
    }
    delete $obj{xmozillaanyphone};
    delete $obj{cn};
    delete $obj{dn};
    delete $obj{objectclass};

    # combine pob and street
    if ( defined $obj{postofficebox} && defined $entry{streetaddress} )
    {
	$pi_rec->{entry}[$entry{streetaddress}] =
	    $obj{postofficebox}
	    . "\n" .
	    $pi_rec->{entry}[$entry{streetaddress}];
	delete $obj{postofficebox};
    }

    # combine o and ou
    if ( defined $obj{ou} && defined $entry{o} )
    {
	$pi_rec->{entry}[$entry{o}] .= "\n" . $obj{ou};
	delete $obj{ou};
    }

    if ( defined $obj{xmozillausehtmlmail} && defined $entry{mail} )
    {
	$pi_rec->{entry}[$entry{mail}] .= " (html)"
	    if lc($obj{xmozillausehtmlmail}) eq "true";
	delete $obj{xmozillausehtmlmail};
    }

    if ( defined $obj{homeurl} && defined $entry{mail}
    && $obj{homeurl} =~ m,^http://\S+, )
    {
	$pi_rec->{entry}[$entry{mail}] .= "\n" . $obj{homeurl};
	delete $obj{homeurl};
    }

    if ( defined $obj{xmozillanickname} && defined $entry{givenname} )
    {
	$pi_rec->{entry}[$entry{givenname}] .= " ("
	. $obj{xmozillanickname}
	. ")";
	delete $obj{xmozillanickname};
    }

    if ( %obj && $entry{description} )
    {
	$pi_rec->{entry}[$entry{description}] .= "\n" .
	    Data::Dumper->Dumpxs( [ \%obj ], [ 'EXTRA' ]);
    }

    # print Data::Dumper->Dumpxs( [ $pi_rec ], [ 'pi_rec' ]), "\n";

    return $pi_rec;
}


sub ReadLDIF
{
    my ($filename) = @_;
    my $pdb;
    my %obj;
    my $key;
    my $value;
    my $munged;
    my %LDIFToPilot;

    print "Reading $filename...\n";

    open( F, $filename ) || return undef;

    while(<F>)
    {
	chop;
	if ( $value && s/^ // )
	{
	    $value .= $_;
	}
	elsif ( /^([^: ]+)\s*(:+)\s*(.*)/ || /^$/ )
	{
	    if( $key )
	    {
		if(exists $obj{$key})
		{
		    $obj{$key} = [$obj{$key}]
			unless (ref $obj{$key} eq 'ARRAY');
		    push @{$obj{$key}}, $value;
		}
		else
		{
		    if( $munged )
		    {
			$value = from_base64($value);
		    }
		    $obj{$key} = $value;
		}
	    }
	    if( $1 )
	    {
		$key = lc($1);
		$value = $3;
		$munged = ($2 eq '::');
	    }
	    elsif( $obj{dn} )
	    {

		if( defined $obj{description}
		&& ($obj{description} =~ s/\s*(\$EXTRA = [{][^}]*[}];)\s*//o ))
		{
		    my $EXTRA;
		    eval $1;
		    warn "$@ $1" if $@;
		    map($obj{$_} = $EXTRA->{$_}, keys %$EXTRA);
		}

		for $key ( keys %obj )
		{
		    $pdb->{$obj{dn}}{$key} = $obj{$key};
		}
		undef $key, $value;
		undef %obj;
	    }
	}
    }
    close(F);

    # calculate appblock

    my $dn = 'cn="~ Pilot AddressDB Data"';
    if ( defined $pdb->{$dn} )
    {
	# print Data::Dumper->Dumpxs([$pdb->{$dn}],['x']),"\n";
	eval $pdb->{$dn}{description};
	warn "ERROR: Corrupt AppBlock:\n$@\n" if $@;
	delete $pdb->{$dn};
    }

    if( 0 )
    {
	my $CategoryName;
	my $member;

	my %CategoryMap;
	my $i = 0;
	for $CategoryName ( @{$pdb->{$AppBlockDN}{categoryName}} )
	{
	    $CategoryMap{ $CategoryName } = $i++;
	}

	for $dn ( keys %$pdb )
	{
	    next unless $dn =~ m/ Category ([^"]+)/;
	    $CategoryName = $1;
	    if( ! defined $CategoryMap{ $CategoryName } )
	    {
		warn "Unknown Category: $dn\n" unless defined $CategoryMap{ $CategoryName };
		next;
	    }
	    $pdb->{$dn}{member} = [ $pdb->{$dn}{member} ]
		unless ref $pdb->{$dn}{member} eq 'ARRAY';
	    for $member ( @{$pdb->{$dn}{member}} )
	    {
		#printf "setting category of $member to $CategoryName\n";
		$pdb->{$member}{category} = $CategoryMap{ $CategoryName };
	    }
	}
    }

    return $pdb;
}

#
# best reference for LDIF schema is RFC2256
# LDAP value syntax is RFC2252
# phone syntax is that ITU-T Rec. E.123
# netscape address book ref is
# http://developer1.netscape.com:80/docs/manuals/communicator/addrapi.htm
#

# neutral fields not supported by the platform. get appended
# to the description field of that platform.

sub NeutralToLDIF
{
    my ( $rec, $pdb ) = @_;
    my %obj;
    my %EXTRA;
    my $key;

    %obj = %$rec;

    $obj{objectclass} = [ 'top', 'person' ];

    for $key ( keys %NSUnsupported )
    {
	next unless defined $obj{$key};
	$EXTRA{$key} = $obj{$key};
	delete $obj{key};
    }

    $obj{description} .= "\n" .
	Data::Dumper->Dumpxs( [ \%EXTRA ], [ 'EXTRA' ]);

    # print Data::Dumper->Dumpxs( [ \%obj ], [ 'ldif_rec' ]), "\n";

    return \%obj;
}

sub AppBlockToLDIF
{
    my ( $rec ) = @_;
    my %obj;

    $obj{description} = Data::Dumper->Dumpxs( [ $rec ], [ "pdb->{'$AppBlockDN'}" ] );
    $obj{dn}= $AppBlockDN;
    ( $obj{cn} ) = ($obj{dn} =~ m/cn="(.*)"/);
    $obj{objectclass} = [ 'top', 'person' ];
    return \%obj;
}

sub CategoryToLDIF
{
    my ( $group, $category ) = @_;
    my %obj;

    $obj{cn}='~ Category '.$category;
    $obj{dn}= sprintf 'cn="%s"', $obj{cn};
    $obj{member}=[ sort {lc($a) cmp lc($b)} @{$group->{$category}} ];
    $obj{objectclass}=[ 'top', 'groupOfNames' ];
    $obj{description}='PalmPilot Category';
    return \%obj;
}

sub WriteLDIF
{
    my ($filename, $pdb) = @_;

    printf STDERR "Writing output file $filename...\n";

    my ($db, $key, $value, $member, $category);
    my %group;
    my $obj;

    open( F, ">$filename" ) || return undef;

    my $dn;

    for $dn ( sort {lc($a) cmp lc($b)} keys %{$pdb} )
    {
	next if $dn eq 'UserInfo';
	next if $dn eq 'DBInfo';
	
	if ( $dn eq $AppBlockDN )
	{
	    $obj = &AppBlockToLDIF( $pdb->{$dn} );
	}
	else
	{
	    $obj = &NeutralToLDIF( $pdb->{$dn} );
	}

	for $key ( sort keys %$obj )
	{
	    next unless defined $obj->{$key};

	    $value = $obj->{$key};

	    if ( ref $value eq 'ARRAY' )
	    {
		for $member ( @$value )
		{
		    printf F "%s: %s\n", $key, $member;
		}
	    }
	    else
	    {
		if ( $value =~ m/[\000-\037\177-\377]/ )
		{
		    printf F "%s:: %s\n", $key, to_base64($value);
		}
		else
		{
		    printf F "%s: %s\n", $key, $value;
		}

	    }
	}

	print F "\n";
    }

    close(F);

    1;
}

sub status
{
	my ($msg,$perc) = @_;
	print STDERR $msg if $perc == 0;
	print STDERR ".";
	print STDERR "\n" if $perc == 100;
}


#
# This is defined in rfc2045, section 6.8
#
sub to_base64
{
    use integer;

    my $in = shift;
    my ( @a, @b, @map, $x, $y, $i );

    @map = unpack("C64", "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/");

    while(length($in))
    {
	@a = unpack("C3", $in);
	$in = substr($in,3);
	
	@b = ( $map[$a[0] >> 2],
	 $map[(($a[0]<<4)|($a[1]>>4))&0x3f],
	 $map[(($a[1]<<2)|($a[2]>>6))&0x3f],
	 $map[$a[2] & 0x3f] );
	$x .= pack("C4", @b);
    }

    # replace null final characters with pad
    if ( $#a == 0 )
    {
	$x =~ s/..$/==/;
    }
    elsif ( $#a == 1 )
    {
	$x =~ s/.$/=/;
    }

    #
    # first line is 63, remainder are 76
    #
    $y = substr($x,0,63);
    if( length($x) > 63 )
    {
	$y .= "\n " . substr($x,63,76);
    }
    for($i = 63+76; $i < length($x); $i += 76)
    {
	$y .= "\n " . substr($x,$i,76);
    }

    return $y;
};

sub from_base64
{
    use integer;

    my $in = shift;
    my ( @a, @b, $x );
    my $i;
    my $pad;

    # remove non base64 chars
    $in =~ s,[^A-Za-z0-9+/=],,g;

    while(length($in))
    {
	@a = unpack("C4", $in);
	$in = substr($in,4);
	$pad = 0;
	for( $i = 0; $i < 4; $i++)
	{
	    if( $a[$i] >= ord("0") && $a[$i] <= ord("9"))
	    {
		$a[$i] = $a[$i] - ord("0") + 52;
	    }
	    elsif( $a[$i] >= ord("a") && $a[$i] <= ord("z"))
	    {
		$a[$i] = $a[$i] - ord("a") + 26;
	    }
	    elsif( $a[$i] >= ord("A") && $a[$i] <= ord("Z"))
	    {
		$a[$i] = $a[$i] - ord("A") + 0;
	    }
	    elsif( $a[$i] == ord("+"))
	    {
		$a[$i] = 62;
	    }
	    elsif( $a[$i] == ord("/"))
	    {
		$a[$i] = 63;
	    }
	    elsif( $a[$i] == ord("="))
	    {
		$a[$i] = 0;
		$pad++;
	    }
	}
	@b =((($a[0]<<2)&0xff) | ($a[1]>>4),
	     (($a[1]<<4)&0xff) | ($a[2]>>2),
	     (($a[2]<<6)&0xff) | ($a[3]));
	splice @b, -$pad if $pad;
	$x .= pack("C*", @b );
    }
    return $x;
};


sub ReadPilot
{
    my ($dlp, $dbName ) = @_;
    my $addrdb;
    my $pdb;
    my ($pi_rec, $i, $id);
    my ($rec );
    my ($max);

    return undef unless $addrdb = $dlp->open( $dbName );


    $i = 0;
    $max = $addrdb->getRecords();

    if ($max == 0)
    {
	$addrdb->close;
	return undef;
    }

    $pdb->{$AppBlockDN} = $addrdb->getAppBlock();
    delete $pdb->{$AppBlockDN}{raw};
    delete $pdb->{$AppBlockDN}{labelRenamed};
    delete $pdb->{$AppBlockDN}{categoryRenamed};
    $pdb->{DBInfo} = $dlp->findDBInfo( 0, $dbName, 0, 0 );
    $pdb->{UserInfo} = $dlp->getUserInfo;
    $pdb->{UserInfo}{thisSyncDate} = time;

    printf STDERR "Reading $max records from AddressDB on pilot...\n";

    for( $i=0;; $i++ )
    {
	&status("Loading Pilot AddressBook ", int(100 * $i/$max))
	    unless $max > 20 && ($i % int($max/20));

	$pi_rec = $addrdb->getRecord($i);
	last unless defined $pi_rec;

	next if $pi_rec->{"deleted"};
	next if $pi_rec->{"archived"};
	next if $pi_rec->{"busy"}; # This should never happen

	$rec = &PilotToNeutral( $pi_rec, $pdb );
	$pdb->{$rec->{dn}} = $rec;
    }

    &status("Loading Pilot AddressBook ", 100);
    $addrdb->close;
    return $pdb;
}

sub WritePilot
{
    my ($dlp, $dbname, $pdb) = @_;
	my ($id, $max, $key, $addrdb, $i, $pi_rec );

    $dlp->delete( $dbname );
    $addrdb = $dlp->create( $dbname, 'addr', 'DATA', 0, 0);
    $dlp->getStatus;
    if( $pdb->{$AppBlockDN} )
    {
	$id = $addrdb->setAppBlock( $pdb->{$AppBlockDN} );
	croak "ERROR: setAppBlock failed with error $id\n"
	    if $id < 0;
    }
    else
    {
	$pdb->{$AppBlockDN} = $addrdb->getAppBlock();
    }

    $max = scalar keys %$pdb;
    $i = 0;

    for $key ( keys %$pdb )
    {
	&status("Saving Pilot AddressBook ", int(100 * $i/$max))
	    unless ($max >= 20 ) && ($i % int($max/20));
	$i++;

	next if $key eq $AppBlockDN;
	next if $key eq 'UserInfo';
	next if $key eq 'DBInfo';

	next unless $pi_rec = &NeutralToPilot( $pdb->{$key}, $pdb, $addrdb );
	$id = $addrdb->setRecord($pi_rec);
	warn "ERROR: setRecord failed with error $id"
	    if $id < 0;
    }
    &status("Saving Pilot AddressBook ", 100);
	$addrdb->close;
    1;
}

my $dlp;
my $socket;
my $db;
my %opts;
my $pdb;

my $usage = <<EOQ

Usage: pi-ldif [-p port] fromfile tofile

Move Address Book on PalmPilot to/from a file, where it is stored
in a LDIF format, suitable for importing into Netscape.

fromfile and tofile are filenames, (like foo.ldif for netscape LDIF
format, or foo.pl for a perl dump), or the special name "pilot" which
means to/from the pilot connected to the serial port.

-p: pilot port (default \$PILOTPORT)

EOQ
;

$opts{p} = $ENV{PILOTPORT} if length $ENV{PILOTPORT};

croak $usage if ! getopts('p:', \%opts)
|| !exists $opts{p};

my $fromfile = shift;
my $tofile = shift;

croak $usage unless $fromfile && $tofile;

$Data::Dumper::Purity = 1;
$Data::Dumper::Deepcopy = 1;
$Data::Dumper::Indent = 1;

for $_ ( keys %LDIFToPilot ) 
{
    $PilotToLDIF{$LDIFToPilot{$_}} = $_ if defined $LDIFToPilot{$_};
}

eval {
    if ( "\L$fromfile" eq 'pilot' )
    {
	croak unless $socket = PDA::Pilot::openPort($opts{p});
	print "Please start HotSync on port $opts{p} now.\n";
	croak unless $dlp = PDA::Pilot::accept( $socket );
	$dlp->getStatus;
	$pdb = &ReadPilot( $dlp, 'AddressDB' );
    }
    elsif ( $fromfile =~ /\.pl$/ )
    {
	printf "Reading $fromfile...\n";
	eval `cat $fromfile`;
    }
    elsif ( $fromfile =~ /\.ldif$/ )
    {
	$pdb = &ReadLDIF( $fromfile );
    }
};

croak "$fromfile : $@ $!" if $@ || !$pdb;

eval {
    if ( "\L$tofile" eq 'pilot' )
    {
	if( !$dlp )
	{
	    croak unless $socket = PDA::Pilot::openPort($opts{p});
	    print "Please start HotSync on port $opts{p} now.\n";
	    croak unless $dlp = PDA::Pilot::accept( $socket );
	    $dlp->getStatus;
	}
	croak unless &WritePilot( $dlp, 'AddressDB', $pdb );
    }
    elsif ( $tofile =~ /\.pl$/ )
    {
	croak unless open(FD, ">$tofile");
	print "Writing $tofile...\n";
	print FD Data::Dumper->Dumpxs([$pdb], ['pdb']);
	close(FD);
    }
    elsif ( $tofile =~ /\.ldif$/ )
    {
	croak unless &WriteLDIF( $tofile, $pdb );
    }
};

croak "$tofile : $@ $!" if $@;

END {
    $dlp->close if $dlp;
    PDA::Pilot::close($socket) if $socket;
}
