///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file UndoManager.h 
 * \brief Contains the definition of the Core::UndoManager class. 
 */

#ifndef __OVITO_UNDO_MANAGER_H
#define __OVITO_UNDO_MANAGER_H

#include <core/Core.h>
#include <core/plugins/PluginClass.h>

namespace Core {

/**
 * \brief Abstract base class for all undoable operation records.
 * 
 * All atomic operations or functions that modify the scene in same way
 * should register an UndoableOperation with the UndoManager using UndoManager::addOperation().
 * 
 * For each specific operation a sub-class of UndoableOperation should be defined that
 * allows the UndoManager to undo or to redo the operation at a later time.
 * 
 * Multiple atomic operations can be combined into a CompoundOperation. They can then be undone
 * or redone at once.
 * 
 * \author Alexander Stukowski
 * \sa UndoManager
 */
class CORE_DLLEXPORT UndoableOperation 
{
public:

	/// \brief A virtual destructor.
	///
	/// The default implementation does nothing. 
	virtual ~UndoableOperation() {}

	/// \brief Provides a localized, human readable description of this operation.
	/// \return A localized string that describes the operation. It is shown in the
	///         edit menu of the application. 
	///
	/// The default implementation returns a default string but it should be overriden
	/// by all sub-classes.
	virtual QString displayName() const { return "Undoable Operation"; }

	/// \brief Undoes the operation encapsulated by this object.
	/// 
	/// This method is called by the UndoManager to undo the operation.
	virtual void undo() = 0;

	/// \brief Re-apply the change, assuming that it had been undone before.
	/// 
	/// This method is called by the UndoManager to redo the operation.
	virtual void redo() = 0;

	/// \brief Indicates whether this is a compound operation.
	/// \return \c true if this is an instance of the CompoundOperation class; \c false otherwise.
	virtual bool isCompoundOperation() { return false; }
};

/**
 * \brief This class is used to assemble little UndoableOperation objects into great big ones.
 * 
 * \author Alexander Stukowski
 * \sa UndoManager::beginCompoundOperation()
 */
class CORE_DLLEXPORT CompoundOperation : public UndoableOperation 
{
public:

	/// \brief Creates an empty compound operation with the given display name.
	/// \param name The localized and human-readable display name for this compound operation.
	///             It will be returned by the displayName() method.
	///             It can later be changed using the setDisplayName() method. 
	/// \sa displayName()
	CompoundOperation(const QString& name) : _displayName(name) {}

	/// \brief Deletes all sub-operations of this compound operation.
	virtual ~CompoundOperation() {
		Q_FOREACH(UndoableOperation* op, subOperations)
			delete op;
	}

	/// \brief Provides a localized, human readable description of this operation.
	/// \return A localized string that describes the operation. It is shown in the
	///         edit menu of the application.
	/// \sa setDisplayName()  
	virtual QString displayName() const { return _displayName; }

	/// \brief Sets this operation's display name to a new string.
	/// \param newName The localized and human-readable display name for this compound operation.
	/// \sa displayName()
	void setDisplayName(const QString& newName) { _displayName = newName; }

	/// Undo the edit operation that was made.
	virtual void undo();

	/// Re-apply the change, assuming that it has been undone. 
	virtual void redo();

	/// \brief Adds a sub-record to this compound operation.
	/// \param operation An instance of a UndoableOperation derived class that encapsulates
	///                  the operation. The CompoundOperation becomes the owner of  
	///                  this object and is responsible for its deletion.
	void addOperation(UndoableOperation* operation) { subOperations.push_back(operation); }

	/// \brief Indicates whether this UndoableOperation is significant or can be ignored.
	/// \return \c true if the CompoundOperation contains at least one sub-operation; \c false it is empty.
	bool isSignificant() const { return subOperations.empty() == false; }

	/// \brief Removes all sub-records from this compound after optionally undoing all sub-operation.
	/// \param undo Controls whether all sub-operations should undone in reverse order before they are removed from
	///             the CompoundOperation.
	///
	/// This method calls UndoableOperation::undo() on all sub-operations contained in this compound.
	void clear(bool undo = true) { 
		if(undo) 
			this->undo(); 
		Q_FOREACH(UndoableOperation* op, subOperations)
			delete op;
		subOperations.clear();
	}

	/// \brief Indicates whether this is a compound operation.
	/// \return Always \c true.
	virtual bool isCompoundOperation() { return true; }

private:

	/// List of contained operations.
	QVector<UndoableOperation*> subOperations;

	/// Stores the display name of this compound passed to the constructor.
	QString _displayName;
};

/**
 * \brief This class records a change to a Qt property to a QObject derived class.
 * 
 * This predefined subclass of UndoableOperation can be readily used to record
 * a change to a Qt property of an object. The property must defined through the
 * standard Qt mechanism using the \c Q_PROPERTY macro.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SimplePropertyChangeOperation : public UndoableOperation
{
public:
	/// \brief Constructor.
	/// \param obj The object whose property is being changed.
	/// \param propName The identifier of the property that is changed. This is the identifier
	///                 name given to the property in the \c Q_PROPERTY macro.  
	/// \note This class does not make a copy of the property name parameter.
	///       So the caller should only pass constant string literals to this constructor.
	SimplePropertyChangeOperation(PluginClass* obj, const char* propName) : 
		object(obj), propertyName(propName) 
	{
		// Make a copy of the current property value.
		oldValue = object->property(propertyName);
		OVITO_ASSERT_MSG(oldValue.isValid(), "SimplePropertyChangeOperation", "The object does not have a property with the given name.");  
	}

	///	\brief Provides a localized, human readable description of this operation.
	virtual QString displayName() const;

	/// \brief Restores the old property value.
	virtual void undo() {
		// Swap old value and current property value.
		QVariant temp = object->property(propertyName);
		object->setProperty(propertyName, oldValue);
		oldValue = temp;
	}

	/// \brief Re-apply the change, assuming that it has been undone. 
	virtual void redo() { undo(); }
	
private:

	/// The object whose property has been changed.
	PluginClass::SmartPtr object;
	
	/// The name of the property that has been changed.
	const char* propertyName;
	
	/// The old value of the property.
	QVariant oldValue;
};


/// \def UNDO_MANAGER
/// \brief The predefined instance of the Core::UndoManager class.
/// 
/// Always use this macro to access the Core::UndoManager class instance.
#define UNDO_MANAGER		(*UndoManager::getSingletonInstance())

/**
 * \brief Manages the global undo stack.
 * 
 * The UndoManager records all user operations. Operations can be undone or reversed
 * one by one.
 * 
 * This is a singleton class with only one predefined instance of this class. 
 * You can access the instance of this class using the UNDO_MANAGER macro.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT UndoManager : public QObject
{
	Q_OBJECT
	
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the UndoManager singleton class.
	/// \note You should use the UNDO_MANAGER macro to access the UndoManager instance instead
	///       of this method.
	inline static UndoManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "UndoManager::getSingletonInstance", "UndoManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns whether the manager is currently recording undoable operations.
	/// \return \c true if the UndoManager currently records any changes made to the scene on its stack.
	///         \c false if changes to the scene are ignored by the UndoManager.
	///
	/// The recording state can be controlled via the suspend() and resume() methods.
	/// Or it can be temporarly suspended using the UndoSuspender helper class.
	/// 
	/// \sa suspend(), resume()
	/// \sa UndoSuspender
	bool isRecording() const { return (suspendCount == 0 && compoundStack.empty() == false); }

	/// \brief Gets the maximum number of undo steps to hold in memory.
	/// \return The maximum number of steps the UndoManager maintains on its stack.
	///         A negative value means infinite number of undo steps.
	///
	/// If the maxmimum number of undo steps is reached then the oldest operation at the bottom of the
	/// stack are removed.
	///
	/// \sa setMaximumUndoSteps()
	int maximumUndoSteps() const { return maxUndoSteps; }

	/// \brief Sets the maximum number of undo steps to hold in memory.
	/// \param steps The maximum height of the undo stack.
	///              A negative value means infinite number of undo steps.
	/// 
	/// \sa maximumUndoSteps(), limitUndoStack()
	void setMaximumUndoSteps(int steps) { maxUndoSteps = steps; }

	/// \brief Opens a new compound operation and assigns it the given display name.
	/// \param displayName A human-readable name that is shown in the edit menu and that describes
	///                    the operation.
	/// \return A pointer to the newly created compoung operation entry that has been put on top
	///         of the undo stack. It can be used to restore the old application state when the current
	///         operation has been aborted.
	///
	/// The current compound operation on top of the stack can also be retrieved via
	/// currentCompoundOperation(). 
	///
	/// \note Every call to beginCompoundOperation() must be followed by exact one call to 
	///       endCompoundOperation() to commit the operation. Multiple compound operations
	///       can be nested by multiple calls to beginCompoundOperation() followed by calls to
	///       endCompoundOperation().
	///
	/// \sa endCompoundOperation()
	CompoundOperation* beginCompoundOperation(const QString& displayName);

	/// \brief Closes the current compound operation.
	/// \sa beginCompoundOperation()
	void endCompoundOperation();

	/// \brief Gets the current compound record on the stack that is being
	///        filled with undoable operation records.
	/// \return The last compound operation opened using beginCompoundOperation() or \c NULL
	///         If the there is currently no compound operation open.
	CompoundOperation* currentCompoundOperation() {
        if(compoundStack.empty()) return NULL;
		return compoundStack.back();
	}

	/// \brief Recoards a single atomic operation.
	/// \param operation An instance of a UndoableOperation derived class that encapsulates
	///                  the operation. The UndoManager becomes the owner of  
	///                  this object and is responsible for its deletion.
	void addOperation(UndoableOperation* operation);

	/// Returns the operation record that is to be undone next.
	/// \return The current user operation or \c NULL if the undo stack is empty.
	UndoableOperation* currentOperation() { 
        if(currentIndex < 0) return NULL;
		return operations[currentIndex];
	}
	
	/// \brief Suspends the recording of undoable operations.
	///
	/// Recording of operations is suspended by this method until a call to resume().
	/// If suspend() is called multiple times then resume() must be called the same number of
	/// times until recording is enabled again.
	///
	/// It is recommended to use the UndoSuspender helper class to suspend recording because
	/// this is more exception save than the suspend()/resume() combination.
	///
	/// \sa resume()
	/// \sa UndoSuspender
	void suspend() { suspendCount++; }

	/// \brief Resumes the recording of undoable operations.
	/// 
	/// This re-enables recording of undoable operations after it has 
	/// been suspended by a call to suspend().
	///
	/// \sa suspend()
	/// \sa UndoSuspender
	void resume() { 
		OVITO_ASSERT_MSG(suspendCount > 0, "UndoManager::resume()", "resume() has been called more often than suspend()."); 
		suspendCount--; 
	}

	/// \brief Indicates whether the undo manager is currently undoing a recorded operation.
	/// \return \c true if the UndoManager is currently restoring the application state before a user
	///         operation. This is usually due to a call to undo();
	///         \c false otherwise.
	/// \sa isRedoing(), isUndoingOrRedoing()
	bool isUndoing() const { return _isUndoing; }

	/// \brief Indicates whether the undo manager is currently redoing a breviously undone operation.
	/// \return \c true if the UndoManager is currently replaying a recorded operation. 
	///         This is usually due to a call to redo();
	///         \c false otherwise.
	/// \sa isUndoing(), isUndoingOrRedoing()
	bool isRedoing() const { return _isRedoing; }

	/// \brief Indicates whether the undo manager is currently undoing or redoing a recorded operation.
	/// \return \c true if currently an operation from the undo stack is being undone or redone, i.e.
	///         isUndoing() or isRedoing() returns \c true; \c false otherwise.
	bool isUndoingOrRedoing() const { return isUndoing() || isRedoing(); }
   
	/// \brief Shrinks the undo stack to maximum number of undo steps.
	///
	/// If the current stack is higher then the maximum number of steps then the oldest entries
	/// from the bottom of the stack are removed.
	///
	/// \sa maximumUndoSteps(), setMaximumUndoSteps()
	void limitUndoStack();
	
public Q_SLOTS:

	/// \brief Resets the undo stack. 
	///
	/// The undo stack is cleared by this method.
	void reset();

	/// \brief Undoes the last operation in the undo stack.
	/// \sa redo()
	void undo();

	/// \brief Redoes the last undone operation in the undo stack.
	/// \sa undo()
	void redo();
	
Q_SIGNALS:

	/// \brief This signal is emmited when undo stack has been changed or the current pointer
	///        into the stack.
	void statusChanged();

private:

	/// Stack with records of undoable operations.
    QVector<UndoableOperation*> operations;

	/// Current position in the undo stack. This is where
	/// new undoable edits will be inserted.
	int currentIndex;

	/// A call to suspend() increaes this value by one.
	/// A call to resume() decreases it.
	/// addOperation() will do nothing is this value is greater than zero.
	int suspendCount;

	/// The stack of open compound records.
	QVector<CompoundOperation*> compoundStack;

	/// Maximum number of records in the undo stack.
	int maxUndoSteps;

	/// Indicates if we are currently undoing an operation.
	bool _isUndoing;

	/// Indicates if we are currently redoing an operation.
	bool _isRedoing;

	/// Updates the Undo/Redo menu items.
	void updateUI();

private:
    
	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	UndoManager();

	/// Initializes the UndoManager.
	/// This is called at program startup.
	static void initialize() { 
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new UndoManager();
	}
	
	/// UndoManager shutdown.
	/// This is called at program shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}
	
	/// The singleton instance of this class.
	static UndoManager* _singletonInstance;

	friend class ApplicationManager;
};

/**
 * \brief A small helper object that suspends recording of undoable operations while it
 *        exists. It can be used to make your code exception-safe.
 * 
 * The constructor of this class calls UndoManager::suspend() and
 * the destructor calls UndoManager::resume().
 * 
 * Just create an instance of this class on the stack to suspend recording of operations on the undo stack
 * during the lifetime of the class instance.
 * 
 * \author Alexander Stukowski
 * \sa UndoManager
 */
struct UndoSuspender {
	UndoSuspender() { UNDO_MANAGER.suspend(); }
	~UndoSuspender() { UNDO_MANAGER.resume(); }
};

};

#endif // __OVITO_UNDO_MANAGER_H
