import arc, os, re, posix, time
from datetime import datetime
from subprocess import Popen, PIPE
from utils import map_option
try:
    from subprocess import CalledProcessError
except ImportError:
    class CalledProcessError(OSError):
	def __init__(self, exitcode, cmd):
	    OSError.__init__(self, '%s exited with %d'%(cmd, exitcode))

class ParseError(Exception):
    """Exception raised on unrecognized command output."""
    pass


# Job States
#

S_UNKNOWN = 0
S_ENTRY = 1
S_PRERUN = 2
S_INLRMS = 3
S_POSTRUN = 4
S_FINAL = 5

class Jobstate(object):
    def __init__(self, name, stage):
	self.name = name
	self.stage = stage
    def __str__(self):
	return self.name
    def is_final(self):
	return self.stage == S_FINAL

class InlrmsJobstate(Jobstate):
    def __init__(self, name):
	Jobstate.__init__(self, name, S_INLRMS)

class PendingJobstate(Jobstate):
    def __init__(self, name, stage, pending):
	Jobstate.__init__(self, name, stage)
	self.pending = pending

def _jobstate(name, stage = S_UNKNOWN):
    if name.startswith('INLRMS:'):
	assert stage == S_UNKNOWN or stage == S_INLRMS
	js = InlrmsJobstate(name)
    elif name.startswith('PENDING:'):
	pending = jobstate_of_str(name[8:])
	js = PendingJobstate(name, stage, pending)
    else:
	js = Jobstate(name, stage)
    _jobstate_of_str[name] = js
    return js

_jobstate_of_str = {}
def jobstate_of_str(name):
    if not name in _jobstate_of_str:
	_jobstate_of_str[name] = _jobstate(name)
    return _jobstate_of_str[name]

J_NOT_SEEN	= _jobstate("NOT_SEEN",		stage = S_ENTRY)
J_ACCEPTED	= _jobstate("Accepted",		stage = S_ENTRY)
J_PREPARING	= _jobstate("Preparing",	stage = S_PRERUN)
J_SUBMITTING	= _jobstate("Submitting",	stage = S_PRERUN)
J_HOLD		= _jobstate("Hold",		stage = S_PRERUN)
J_QUEUING	= _jobstate("Queuing",		stage = S_INLRMS)
J_RUNNING	= _jobstate("Running",		stage = S_INLRMS)
J_FINISHING	= _jobstate("Finishing",	stage = S_POSTRUN)
J_FINISHED	= _jobstate("Finished",		stage = S_FINAL)
J_KILLED	= _jobstate("Killed",		stage = S_FINAL)
J_FAILED	= _jobstate("Failed",		stage = S_FINAL)
J_DELETED	= _jobstate("Deleted",		stage = S_FINAL)
J_OTHER		= _jobstate("Other",		stage = S_UNKNOWN)

# Also added to _jobstate_of_str for backwards compatibility.  We used
# "Specific state" earlier, and it may be saved in active.map files.
SJ_ACCEPTING		= _jobstate("ACCEPTING",	stage = S_ENTRY)
SJ_PENDING_ACCEPTED	= _jobstate("PENDING:ACCEPTED",	stage = S_ENTRY)
SJ_ACCEPTED		= _jobstate("ACCEPTED",		stage = S_ENTRY)
SJ_PENDING_PREPARING	= _jobstate("PENDING:PREPARING",stage = S_PRERUN)
SJ_PREPARING		= _jobstate("PREPARING",	stage = S_PRERUN)
SJ_SUBMIT		= _jobstate("SUBMIT",		stage = S_PRERUN)
SJ_SUBMITTING		= _jobstate("SUBMITTING",	stage = S_PRERUN)
SJ_PENDING_INLRMS	= _jobstate("PENDING:INLRMS",	stage = S_INLRMS)
SJ_INLRMS		= _jobstate("INLRMS",		stage = S_INLRMS)
SJ_INLRMS_Q		= _jobstate("INLRMS:Q",		stage = S_INLRMS)
SJ_INLRMS_R		= _jobstate("INLRMS:R",		stage = S_INLRMS)
SJ_INLRMS_EXECUTED	= _jobstate("INLRMS:EXECUTED",	stage = S_INLRMS)
SJ_INLRMS_S		= _jobstate("INLRMS:S",		stage = S_INLRMS)
SJ_INLRMS_E		= _jobstate("INLRMS:E",		stage = S_INLRMS)
SJ_INLRMS_O		= _jobstate("INLRMS:O",		stage = S_INLRMS)
SJ_FINISHING		= _jobstate("FINISHING",	stage = S_POSTRUN)
SJ_CANCELING		= _jobstate("CANCELING",	stage = S_POSTRUN)
SJ_FAILED		= _jobstate("FAILED",		stage = S_FINAL)
SJ_KILLED		= _jobstate("KILLED",		stage = S_FINAL)
SJ_FINISHED		= _jobstate("FINISHED",		stage = S_FINAL)
SJ_DELETED		= _jobstate("DELETED",		stage = S_FINAL)



# Utilities
#

def explain_wait_status(status):
    if posix.WIFEXITED(status):
	exitcode = posix.WEXITSTATUS(status)
	if exitcode:
	    msg = 'exited with %d'%exitcode
	else:
	    msg = 'exited normally'
    elif posix.WIFSTOPPED(status):
	signo = posix.WSTOPSIG(status)
	msg = 'stopped by signal %d'%signo
    elif posix.WIFSIGNALED(status):
	signo = posix.WTERMSIG(status)
	msg = 'terminated by signal %d'%signo
    elif posix.WIFCONTINUED(status):
	msg = 'continued'
    if posix.WCOREDUMP(status):
	return '%s (core dumped)' % msg
    else:
	return msg

def _simple_cmd(argv, log = None):
    p = Popen(argv, stdout = PIPE, stderr = PIPE)
    output, errors = p.communicate()
    if log:
	if output:
	    for ln in output.strip().split('\n'):
		log.info('%s: %s'%(argv[0], ln))
	if errors:
	    for ln in errors.strip().split('\n'):
		log.error('%s: %s'%(argv[0], ln))
    if p.returncode != 0:
	raise CalledProcessError(p.returncode, ' '.join(argv))

def roundarg(t):
    return str(int(t + 0.5))


# ARC Commands
#

def arccp(src_url, dst_url, log = None, timeout = 20):
    return _simple_cmd(['arccp', '-t', roundarg(timeout), src_url, dst_url],
		       log = log)

def arccp_T(src_url, dst_url, log = None, timeout = 20):
    return _simple_cmd(['arccp', '-T', '-t', roundarg(timeout),
			src_url, dst_url], log = log)

def arcrm(dst_url, log = None, timeout = 20):
    return _simple_cmd(['arcrm', '-t', roundarg(timeout), dst_url], log = log)

def arcls(url, log = None, timeout = 20):
    p = Popen(['arcls', '-t', roundarg(timeout), '-l', url],
	      stdout = PIPE, stderr = PIPE)
    output, errors = p.communicate()
    entries = []
    for ln in output.split('\n')[1:]:
	if ln:
	    comps = ln.rsplit(None, 7)
	    entries.append(ArclsEntry(*comps))
    if p.returncode != 0:
	if log:
	    log.error('Errors from arcls:\n%s'%errors)
	raise CalledProcessError(p.returncode, 'arcls')
    return entries

def arcls_L(url, log = None, timeout = 20):
    p = Popen(['arcls', '-L', '-t', roundarg(timeout), '-l', url],
	      stdout = PIPE, stderr = PIPE)
    output, errors = p.communicate()
    if p.returncode != 0:
	if log:
	    log.error('Errors from arcls -L:\n%s'%errors)
	raise CalledProcessError(p.returncode, 'arcls')
    return [s.strip() for s in output.split('\n')[1:]]

def arcclean(jobids, force = False, timeout = 20, log = None):
    argv = ['arcclean', '-t', roundarg(timeout)]
    if force:
	argv.append('-f')
    argv += jobids
    _simple_cmd(argv, log = log)

_arcstat_state_re = re.compile(r'(\w+)\s+\(([^()]+)\)')
def parse_old_arcstat_state(s):
    mo = re.match(_arcstat_state_re, s)
    if not mo:
	raise ParseError('Malformed arcstat state %s.'%s)
    return jobstate_of_str(mo.group(1)), mo.group(2)

class Arcstat(object):
    def __init__(self,
		 state = None, specific_state = None,
		 submitted = None,
		 job_error = None,
		 exit_code = None):
	self.state = state
	self.specific_state = state
	self.submitted = submitted
	self.job_error = job_error
	self.exit_code = exit_code

def arcstat(jobids = None, log = None):
    cmd = ['arcstat', '-l']
    if jobids is None:
	cmd.append('-a')
    else:
	cmd += jobids
    fd = Popen(cmd, stdout = PIPE).stdout
    jobstats = {}
    lnno = 0

    def parse_error(msg):
	if log:
	    log.error('Unexpected output from arcstat '
		      'at line %d: %s'%(lnno, msg))
	else:
	    fd.close()
	    raise ParseError('Unexpected output from arcstat '
			     'at line %d: %s'%(lnno, msg))
    def convert(jobid, jobstat):
	if 'Specific state' in jobstat:
	    state = jobstate_of_str(jobstat['State'])
	    specific_state = jobstat['Specific state']
	elif 'State' in jobstat:
	    # Compatibility with old arcstat.
	    state, specific_state = parse_old_arcstat_state(jobstat['State'])
	else:
	    raise ParseError('Missing "State" or "Specific state" for %s.'
			     % jobid)
	return Arcstat(state = state, specific_state = specific_state,
		       exit_code = map_option(int, jobstat.get('Exit code')),
		       submitted = jobstat.get('Submitted'),
		       job_error = jobstat.get('Job Error'))

    jobid, jobstat = None, None
    for ln in fd:
	ln = ln.strip()
	lnno += 1

	if ln == '':
	    if not jobid is None:
		jobstats[jobid] = convert(jobid, jobstat)
		jobid, jobstat = None, None
	elif ln.startswith('No jobs') or ln.startswith('Status of '):
	    break
	elif ln.startswith('Job: '):
	    if not jobid is None:
		parse_error('Missing empty line before "Job: ..."')
		continue
	    jobid = ln[5:].strip()
	    jobstat = {}
	elif ln.startswith('Warning:'):
	    pass
	else:
	    if jobid is None:
		 parse_error('Missing "Job: ..." header before %r'%ln)
		 continue
	    try:
		k, v = ln.split(':', 1)
		jobstat[k] = v.strip()
	    except ValueError:
		 parse_error('Expecting "<key>: <value>", got %r'%ln)
		 continue
    fd.close()
    if not jobid is None:
	jobstats[jobid] = convert(jobid, jobstat)
    return jobstats

def load_jobs_xml(timeout = 20, log = None):
    deadline = time.time() + timeout
    try:
	import xml.etree.ElementTree as et	# Python 2.6
    except ImportError:
	import elementtree.ElementTree as et	# Python 2.4
    uc = arc.UserConfig()
    jobstats = {}
    jp = os.path.join(uc.ARCUSERDIRECTORY, 'jobs.xml')
    # Should create a lock file, but better not make assumptions about ARC
    # internals.  This code should be oboselete when NG-3240 is closed.
    while os.path.exists(jp + '.lock'):
	if deadline - time.time() <= 1:
	    if log: log.warn('Timout while waiting for jobs.xml lock.')
	    return {}
	time.sleep(1)
    jx = et.parse(jp)
    for job in jx.findall('Job'):
	t_sub = job.findtext('LocalSubmissionTime')
	if t_sub:
	    t_sub = t_sub.replace('T', ' ')
	    if t_sub[-1] == 'Z':
		    t_sub = t_sub[0:-1]
	    jobstats[job.findtext('JobID')] = Arcstat(submitted = t_sub)
    return jobstats

def arcprune(max_age = 604800, log = None, timeout = 20,
	     use_jobs_xml = False, force = True):
    t_start = time.time()
    pruned_count = 0
    failed_count = 0
    if use_jobs_xml:
	try:
	    jobstats = load_jobs_xml(timeout = 0.8*timeout, log = log)
	except ImportError, xc:
	    if log:
		log.warn('Missing dependency for custom jobs.xml loader: %s'%xc)
		log.warn('Falling back to arcstat.')
	    jobstats = arcstat(log = log)
	except Exception, xc:
	    if log:
		log.warn('Cannot read jobs.xml: %s'%xc)
		log.warn('Falling back to arcstat.')
	    jobstats = arcstat(log = log)
    else:
	jobstats = arcstat(log = log)
    for jobid, jobstat in jobstats.iteritems():
	submitted = jobstat.submitted
	if not submitted:
	    continue
	tm_sub = time.strptime(submitted, '%Y-%m-%d %H:%M:%S')
	t_sub = time.mktime(tm_sub)
	t_left = timeout - time.time() + t_start
	if t_left < 1:
	    if log: log.warn('Timout while pruning jobs.')
	    break
	if t_start - t_sub > max_age:
	    try:
		if log: log.info('Cleaning %s from %s.'%(jobid, submitted))
		arcclean([jobid], log = log, timeout = t_left, force = force)
		pruned_count += 1
	    except Exception, xc:
		failed_count = 0
		if log: log.warn('Failed to prune %s: %s'%(jobid, xc))
    return len(jobstats), pruned_count, failed_count

class ArclsEntry(object):
    DIR = 0
    FILE = 1

    def __init__(self, name, typ, size, cdate, ctod, validity, checksum, latency = ''):
	self.filename = name
	if typ == 'dir':
	    self.entry_type = self.DIR
	elif typ == 'file':
	    self.entry_type = self.FILE
	else:
	    self.entry_type = None
	self.size = size
#	Not in Python 2.4:
#	self.ctime = datetime.strptime(cdate + 'T' + ctod, '%Y-%m-%dT%H:%M:%S')
	def drop_NA(s):
	    if s != '(n/a)':
		return s

	if cdate == '(n/a)':
	    self.validity = drop_NA(validity)
	    self.checksum = drop_NA(checksum)
	    self.latency = drop_NA(latency)
	else:
	    self.validity = drop_NA(ctod)
	    self.checksum = drop_NA(validity)
	    self.latency = drop_NA(checksum)
