# config.py
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Handles configuration of musiclibrarian.  This has a structure that
# allows distributed registration of options with defaults.
#
# It is not an error for an unknown option to exist in the
# configuration file; however, it is an error for an unknown option to
# be *requested*.


import configfile
import exceptions
import listenable
import os
import sys
from warnings import warn

# This can be externally modified: if true, every call to set_option
# rewrites the configuration file.
sync=True

# Configuration problems due to input errors
#
# Not currently used -- instead I print a message to stdout and
# fall back to the default value.
class ConfigError(exceptions.Exception):
    pass

# Configuration problems due to internal program errors.
class ConfigInternalError(exceptions.Exception):
    pass

# The known configuration options for a section are stored in a
# dictionary mapping the option name to information about it.  The
# information is currently just a tuple (default, validator).  The
# validator is a function which takes a string and returns a boolean
# value.
#
# This is a dictionary mapping section names to dictionaries mapping
# configuration names to configuration options
sections={}

# This is the stored configuration itself: a dictionary mapping
# section names to a dictionary mapping option names to option values.
#
# This is separate from the logical configuration because it's a
# different beast: it represents what was found in the configuration
# file, rather than the list of "known" configuration options.

config={}

for fn in [os.path.expanduser('~/.musiclibrarian/config')]:
    try:
        f=open(fn)
    except:
        continue

    configfile.read_config_file(f, config)

# A dictionary mapping pairs (section name, option value) to
# Listenable objects.
listenables={}

# Add a new configuration option.
def add_option(section, name, default=None, validator=lambda x:True):
    sect=sections.setdefault(section, {})

    if sect.has_key(name):
        warn('Option %s/%s was multiply defined, overriding old definition'%(section,name))

    if not validator(default):
        raise ConfigInternalError,'Inconsistent option definition: %s does not pass the validator for %s/%s'%(default,section,name)

    sect[name]=(default,validator)

def lookup_option(section, name):
    if not sections.has_key(section):
        raise ConfigInternalError,'Configuration section %s does not exist'%section

    sect=sections[section]

    if not sect.has_key(name):
        raise ConfigInternalError,'No configuration option %s in section %s'%(name,section)

    return sect[name]

def add_listener(section, name, listener, as_weak=True):
    """ Add a listener on a configuration option.  Returns an opaque
    token associated with the listener.

    Listeners are called with this signature:
    listener(section, name, old_val, new_val)"""
    l = listenables.get((section, name), None)

    if l == None:
        l = listenable.Listenable()
        listenables[(section, name)] = l

    return l.add_listener(listener, as_weak)

def remove_listener(section, name, listener):
    """Remove a listener, given the id returned by add_listener or a
    listener reference (as for Listenable.remove_listener)."""

    l = listenables.get((section, name), None)

    if l == None:
        warn('No listener object for configuration item %s:%s'%(section,name))
    else:
        l.remove_listener(listener)

def get_option(section, name):
    """Return the option 'name' in the section 'section'.  If the
    value has not been set by the user, the default value for this
    option is returned."""
    default,validator=lookup_option(section, name)

    optval=config.get(section, {}).get(name, default)

    if not validator(optval):
        sys.stderr.write('%s is not a valid setting for %s/%s.\nReverting to default %s\n'%(optval, section, name, str(default)))
        optval=default

    return optval

def set_option(section, name, val):
    """Set the option 'name' in the section 'section' to 'val',
    calling listeners as appropriate."""
    default,validator=lookup_option(section, name)

    oldval=config.get(section, {}).get(name, default)

    if not validator(val):
        raise ConfigInternalError,'%s is not a valid setting for %s/%s'%(val, section, name)

    config.setdefault(section, {})[name]=val

    if sync:
        save_options()

    # To save time here, we don't compare oldval and val...
    l = listenables.get((section, name), None)
    if l <> None:
        l.call_listeners(section, name, oldval, val)

# Saves options.
#
# Changes to defaults are only preserved if set_option has never been
# called for that setting.  Simply being equal to the default isn't
# enough.  (should I change this?)

def save_options():
    progdir=os.path.expanduser(os.path.join('~','.musiclibrarian'))

    if not os.path.isdir(progdir):
        os.mkdir(progdir)

    tmpname=os.path.join(progdir, 'config.%s'%os.getpid())

    configfile.write_config_file(open(os.path.join(progdir, tmpname), 'w'),
                                 config)

    os.rename(tmpname, os.path.join(progdir, 'config'))
