!
!  FITS I/O for photometric calibration
!
!  Copyright © 2014-5 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!  
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!  
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.

module fits_fotran

  use fitsio

  implicit none

  integer, parameter, private :: dbl = selected_real_kind(15)

  private :: idcol
  
contains

  subroutine traload(nametable,photosys_ref,photosys_instr, &
       tr,trerr,tr1,tr1err,status)

    character(len=*), intent(in) :: nametable
    character(len=*), intent(out) :: photosys_ref,photosys_instr
    real(dbl), dimension(:,:), allocatable, intent(out) :: tr,trerr,tr1,tr1err
    integer, intent(out) :: status

    character(len=FLEN_KEYWORD) :: key
    character(len=FLEN_VALUE) :: extname
    character(len=FLEN_CARD) :: buf
    integer :: ncols,nrows,blocksize,i
    integer, parameter :: frow = 1, felem = 1, extver = 0
    real(dbl), parameter :: nullval = 0.0_dbl
    logical :: anyf

    status = 0

    ! open and move to first table extension
    call ftopen(15,nametable,0,blocksize,status)
    if( status /= 0 ) goto 666

    ! check reference frame and identification of this table
    call ftkeyn(FITS_KEY_PHOTSYS,1,key,status)
    call ftgkys(15,key,photosys_instr,buf,status)
    call ftkeyn(FITS_KEY_PHOTSYS,2,key,status)
    call ftgkys(15,key,photosys_ref,buf,status)

    extname = trim(FTHDUNAME) // '_FORWARD'
    call ftmnhd(15, ANY_HDU, extname, extver,status)
    if( status /= 0 ) goto 666
    
    call ftgnrw(15,nrows,status)
    call ftgncl(15,ncols,status)
    
    allocate(tr(nrows,ncols))
    do i = 1, ncols
       call ftgcvd(15,i,frow,felem,nrows,nullval,tr(:,i),anyf,status)
    end do

    extname = trim(FTHDUNAME) // '_BACKWARD'
    call ftmnhd(15, ANY_HDU, extname, extver,status)
    if( status /= 0 ) goto 666    
    call ftgnrw(15,nrows,status)
    call ftgncl(15,ncols,status)
    allocate(tr1(nrows,ncols))
    do i = 1, ncols
       call ftgcvd(15,i,frow,felem,nrows,nullval,tr1(:,i),anyf,status)
    end do

    allocate(trerr(nrows,ncols),tr1err(nrows,ncols))
    trerr = -1
    tr1err = -1
    
666 continue

    call ftrprt('STDERR',status)
    call ftclos(15,status)

    if( status /= 0 ) then
       if( allocated(tr) ) deallocate(tr)
       if( allocated(tr1) ) deallocate(tr1)
       if( allocated(trerr) ) deallocate(trerr)
       if( allocated(tr1err) ) deallocate(tr1err)
    end if

  end subroutine traload


  subroutine trawrite(output,backup,photosys_ref,photosys_instr,filters_ref, &
       filters_instr,ra,dec,ci0,ph,dph,cts,dcts, &
       tr,trerr,tr1,tr1err,filenames,airmass,ctph,extin)

    character(len=*), intent(in) :: output, backup,photosys_ref, photosys_instr
    character(len=*), dimension(:), intent(in) :: filters_ref, filters_instr
    character(len=*), dimension(:), intent(in), optional :: filenames
    real(dbl), dimension(:), intent(in) :: ra,dec
    real(dbl), dimension(:), intent(in), optional :: airmass,ctph,extin
    real(dbl), intent(in) :: ci0
    real(dbl), dimension(:,:), intent(in) :: tr,trerr,tr1,tr1err, &
         ph,dph,cts,dcts

    integer, parameter :: frow = 1, felem = 1
    character(len=FLEN_VALUE), dimension(:), allocatable :: ttype, tform,tunit
    character(len=FLEN_VALUE) :: extname
    character(len=FLEN_KEYWORD) :: key
    character(len=FLEN_CARD) :: buf,a
    real(dbl) :: x
    integer :: n,nrows,ncols, nbands,status, blocksize

    status = 0
    blocksize = 0
    call fitsbackup(output,backup,status)
    call ftinit(15,output,blocksize,status)
    call ftiimg(15,8,0,(/0/),status)
    call ftpcom(15,'Photometry system conversion table.',status)
    call ftpkys(15,FITS_KEY_CREATOR,FITS_VALUE_CREATOR,FITS_COM_CREATOR,status)
    call ftpcom(15,MUNIPACK_VERSION,status)
    call ftpcom(15,'Description: http://munipack.physics.muni.cz/dataform_phfotran.html',status)

    call ftkeyn(FITS_KEY_PHOTSYS,1,key,status)
    call ftpkys(15,key,photosys_instr,'instrumental photometry system',status)

    call ftkeyn(FITS_KEY_PHOTSYS,2,key,status)
    call ftpkys(15,key,photosys_ref,'reference photometry system (standard)',status)

    if( present(airmass) ) then
       x = sum(airmass)/size(airmass)
       call ftpkyd(15,FITS_KEY_AIRMASS,x,-6,'average air-mass',status)
    end if
       
    ! mean colour index
    call ftpkyd(15,'REFCI',ci0,-6,'reference colour index',status)

    if( present(filenames) ) then
       call ftpcom(15,'Photometry transformation has been derived by processing..',status)
       call ftpcom(15,' filename, filter, airmass, ctph, extinction:',status)
       do n = 1,size(filenames)
          if( size(airmass) > 0 ) then          
             write(buf,'(f8.5)') airmass(n)
          else
             buf = "-"
          end if
          if( size(ctph) > 0 ) then
             write(a,'(1p,g0.5)') ctph(n)
          else
             a = "-"
          end if
          buf = trim(buf) // " " // trim(a)
          if( size(extin) > 0 ) then
             write(a,'(1p,g0.5)') extin(n)
          else
             a = "-"
          end if
          buf = trim(buf) // " " // trim(a)
          call ftpcom(15,"'"//trim(filenames(n))//"' '"//trim(filters_instr(n)) &
            //"' "//trim(buf),status)
       end do
       call ftpcom(15,'All elements are normalized for both unit area and time.',status)
    end if

    
    ! extension - FOTRAN
    nrows = size(tr,1)
    ncols = size(tr,2)
    allocate(ttype(ncols),tform(ncols),tunit(ncols))
    tform = '1D'
    tunit = '' !'an instrumental filter'
    ttype = filters_ref

    extname = trim(FTHDUNAME) // '_FORWARD'
    call ftibin(15,nrows,ncols,ttype,tform,tunit,extname,0,status)
    call ftpcom(15,'Forward matrix (photon rate to counts rate).',status)
    
    do n = 1, ncols
       call ftpcld(15,n,frow,felem,nrows,tr(:,n),status)
    end do

    extname = trim(FTHDUNAME) // '_BACKWARD'
    call ftibin(15,nrows,ncols,ttype,tform,tunit,extname,0,status)
    call ftpcom(15,'Backward matrix (counts rate to photon rate).',status)
    
    do n = 1, ncols
       call ftpcld(15,n,frow,felem,nrows,tr1(:,n),status)
    end do

    extname = trim(FTHDUNAME) // '_FORWARD_ERR'
    call ftibin(15,nrows,ncols,ttype,tform,tunit,extname,0,status)
    do n = 1, ncols
       call ftpcld(15,n,frow,felem,nrows,trerr(:,n),status)
    end do

    extname = trim(FTHDUNAME) // '_BACKWARD_ERR'
    call ftibin(15,nrows,ncols,ttype,tform,tunit,extname,0,status)
    do n = 1, ncols
       call ftpcld(15,n,frow,felem,nrows,tr1err(:,n),status)
    end do
    
    deallocate(ttype,tform,tunit)

    
    ! data info extension
    nrows = size(ra)
    nbands = size(ph,2)
    ncols = (2*2)*nbands + 2
    allocate(ttype(ncols),tform(ncols),tunit(ncols))
    tform = '1D'
    tunit(1:2) = 'deg'
    tunit(3:2+2*nbands) = 'ph/s/m2'
    tunit(3+2*nbands:2+4*nbands) = 'W/m2'
    tunit(3+4*nbands:) = 'cts/s/m2'
    ttype(1) = FITS_COL_RA
    ttype(2) = FITS_COL_DEC
    ttype(3:2+1*nbands) = 'PH_'//filters_ref
    ttype(3+1*nbands:2+2*nbands) = 'ePH_'//filters_ref
    ttype(3+2*nbands:2+3*nbands) = 'CTS_'//filters_instr
    ttype(3+3*nbands:) = 'eCTS_'//filters_instr

    call ftibin(15,nrows,ncols,ttype,tform,tunit,trim(FTHDUNAME)//'data',0,status)
    call ftpcld(15,1,frow,felem,nrows,ra,status)
    call ftpcld(15,2,frow,felem,nrows,dec,status)
    do n = 1,nbands
       call ftpcld(15,2+n,frow,felem,nrows,ph(:,n),status)
    end do
    do n = 1,nbands
       call ftpcld(15,2+nbands+n,frow,felem,nrows,dph(:,n),status)
    end do
    do n = 1,nbands
       call ftpcld(15,2+2*nbands+n,frow,felem,nrows,cts(:,n),status)
    end do
    do n = 1,nbands
       call ftpcld(15,2+3*nbands+n,frow,felem,nrows,dcts(:,n),status)
    end do

    deallocate(ttype,tform,tunit)

    call ftclos(15,status)
    call ftrprt('STDERR',status)

  end subroutine trawrite

  subroutine tradata(nametable,ra,dec,ph,dph,cts,dcts)

    character(len=*), intent(in) :: nametable
    real(dbl), dimension(:), allocatable, intent(out) :: ra,dec
    real(dbl), dimension(:,:), allocatable, intent(out) :: ph,dph,cts,dcts

    integer :: ncols,nrows,blocksize,i,n,rcol,dcol,status
    integer, parameter :: frow = 1, felem = 1, extver = 0
    real(dbl), parameter :: nullval = 0.0_dbl
    integer, dimension(:), allocatable :: col_cts, col_ph, col_dcts, col_dph
    logical :: anyf

    status = 0

    ! open and move to first table extension
    call ftopen(15,nametable,0,blocksize,status)
    call ftmnhd(15,BINARY_TBL,trim(FTHDUNAME)//'data', extver,status)
    if( status /= 0 ) goto 666

    call ftgnrw(15,nrows,status)
    call ftgncl(15,ncols,status)
    
    call ftgcno(15,.true.,FITS_COL_RA,rcol,status)
    call ftgcno(15,.true.,FITS_COL_DEC,dcol,status)
    if( status /= 0 ) goto 666

    allocate(col_cts(ncols),col_ph(ncols),col_dph(ncols),col_dcts(ncols))

    call idcol(15,'CTS_*',col_cts,n,status)
    call idcol(15,'eCTS_*',col_dcts,n,status)
    call idcol(15,'PH_*',col_ph,n,status)
    call idcol(15,'ePH_*',col_dph,n,status)
    
    if( size(col_ph) /= size(col_cts) .or. n == 0 ) &
         stop 'CTS or PH columns badly identified.'
    
    allocate(cts(nrows,n),ph(nrows,n),dcts(nrows,n),dph(nrows,n),ra(nrows),dec(nrows))
    call ftgcvd(15,rcol,frow,felem,nrows,nullval,ra,anyf,status)
    call ftgcvd(15,dcol,frow,felem,nrows,nullval,dec,anyf,status)
    do i = 1,n
       call ftgcvd(15,col_cts(i),frow,felem,nrows,nullval,cts(:,i),anyf,status)
    end do
    do i = 1,n
       call ftgcvd(15,col_dcts(i),frow,felem,nrows,nullval,dcts(:,i),anyf,status)
    end do
    do i = 1,n
       call ftgcvd(15,col_ph(i),frow,felem,nrows,nullval,ph(:,i),anyf,status)
    end do
    do i = 1,n
       call ftgcvd(15,col_dph(i),frow,felem,nrows,nullval,dph(:,i),anyf,status)
    end do
    deallocate(col_cts,col_ph)    

666 continue

    call ftrprt('STDERR',status)
    call ftclos(15,status)

    if( status /= 0 ) then
       continue
    end if

  end subroutine tradata

  subroutine idcol(un,temp,cols,n,status)

    integer, intent(in) :: un
    character(len=*), intent(in) :: temp
    integer, dimension(:), intent(out) :: cols
    integer, intent(out) :: n,status
    integer :: l
    
    n = 0
    do
       call ftgcno(un,.true.,temp,l,status)
       if( status == MULTIPLE_MATCH ) then
          n = n + 1
          cols(n) = l
       else
          exit
       end if
    end do
    status = 0

  end subroutine idcol
    
end module fits_fotran
