/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QDebug>
#include <QRegularExpressionMatch>
#include <QScriptEngine>
#include <QScriptString>
#include <QScriptClassPropertyIterator>


/////////////////////// Local includes
#include <globals/globals.hpp>

#include <libmass/MassSpectrum.hpp>
#include <libmass/DataPointJs.hpp>
#include <libmass/MassSpectrumJs.hpp>
#include <libmass/MassSpectrumJsPrototype.hpp>
#include <libmass/MassSpectrumJsPropertyIterator.hpp>


#include <stdlib.h>


Q_DECLARE_METATYPE(msXpSlibmass::MassSpectrumJs*);
Q_DECLARE_METATYPE(msXpSlibmass::DataPoint*);


namespace msXpSlibmass
{


	MassSpectrumJs::MassSpectrumJs(QScriptEngine *engine)
		: 
			QObject(engine), QScriptClass(engine)
	{
		qScriptRegisterMetaType<MassSpectrum>(engine, toScriptValue, fromScriptValue);

		title = engine->toStringHandle(QLatin1String("title"));
		length = engine->toStringHandle(QLatin1String("length"));
		rt = engine->toStringHandle(QLatin1String("rt"));
		dt = engine->toStringHandle(QLatin1String("dt"));

		proto = engine->newQObject(new MassSpectrumJsPrototype(this),
				QScriptEngine::QtOwnership,
				QScriptEngine::SkipMethodsInEnumeration
				| QScriptEngine::ExcludeSuperClassMethods
				| QScriptEngine::ExcludeSuperClassProperties);

		QScriptValue global = engine->globalObject();

		proto.setPrototype(global.property("Object").property("prototype"));

		ctor = engine->newFunction(construct, proto);
		ctor.setData(engine->toScriptValue(this));
	}


	MassSpectrumJs::~MassSpectrumJs()
	{
	}


	QString
		MassSpectrumJs::name() const
		{
			return QLatin1String("MassSpectrum");
		}


	QScriptValue 
		MassSpectrumJs::prototype() const
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__;
			return proto;
		}


	QScriptValue 
		MassSpectrumJs::constructor()
		{
			return ctor;
		}


	QScriptValue 
		MassSpectrumJs::newInstance()
		{
			MassSpectrum massSpectrum;

			return newInstance(massSpectrum);
		}


	QScriptValue 
		MassSpectrumJs::newInstance(const QString &title)
		{
			MassSpectrum massSpectrum(title);

			return newInstance(massSpectrum);
		}


	QScriptValue 
		MassSpectrumJs::newInstance(const MassSpectrum &other)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before getting QScriptValue as newVariant(QVariant::fromValue(MassSpectrum))";

			QScriptValue data = engine()->newVariant(QVariant::fromValue(other));

			// 'this', below, is because the JS object (data) needs to be linked to
			// the corresponding JS class that must match the class of the object.
			// 'data' is an object of MassSpectrum class and thus needs to be
			// associated to the MassSpectrumJs class (that is, this).

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before returning QScriptValue as newObject(this, data);";

			return engine()->newObject(this, data);
		}


	QScriptValue::PropertyFlags MassSpectrumJs::propertyFlags(const QScriptValue &/*object*/,
			const QScriptString &name, uint /*id*/)
	{
		if (name == title) 
			return QScriptValue::Undeletable;

		return QScriptValue::Undeletable;
	}


	QScriptClass::QueryFlags 
		MassSpectrumJs::queryProperty(const QScriptValue &object,
				const QScriptString &name, QueryFlags flags, uint *id)
		{
			MassSpectrum *massSpectrum = qscriptvalue_cast<MassSpectrum *>(object.data());

			if (massSpectrum == Q_NULLPTR)
				return 0;

			if(name == title) 
			{
				return flags;
			}

			if(name == length) 
			{
				return flags;
			}

			if(name == rt)
			{
				return flags;
			}

			if(name == dt)
			{
				return flags;
			}

			bool isArrayIndex;

			qint32 index = name.toArrayIndex(&isArrayIndex);

			if(!isArrayIndex)
				return 0;

			*id = index;

			if((flags & HandlesReadAccess) && (index >= massSpectrum->size()))
				flags &= ~HandlesReadAccess;

			return flags;
		}


	QScriptValue 
		MassSpectrumJs::property(const QScriptValue &object,
				const QScriptString &name, uint id)
		{
			MassSpectrum *massSpectrum = qscriptvalue_cast<MassSpectrum *>(object.data());

			if(!massSpectrum)
				return QScriptValue();

			if(name == title)
			{
				return QScriptValue(massSpectrum->m_title);
			}
			else if(name == length) 
			{
				return massSpectrum->size();
			}
			else if(name == rt) 
			{
				return massSpectrum->m_rt;
			}
			else if(name == dt) 
			{
				return massSpectrum->m_dt;
			}
			else
			{
				qint32 index = id;

				if((index < 0) || (index >= massSpectrum->size()))
					return QScriptValue();

				//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
					//<< "The index is:" << index
					//<< "and the variant:" << QVariant::fromValue(*(massSpectrum->at(index)));

				QScriptValue data = engine()->newVariant(QVariant::fromValue(massSpectrum->at(index)));

				//MassPeak *massPeak = qscriptvalue_cast<MassPeak *>(data);
				//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
				//<< "The mass peak:" << massPeak->mz() << "/" << massPeak->i();

				// We need to allocate the proper JS proxy for the object class: the
				// class of the native object is MassPeak, so we need to allocate a
				// MassPeakJs instance to feed to newObject() below. Only if we do this,
				// will we have a properly behaving MassPeak JS object when accessing
				// the individual members of the mass spectrum.

				DataPointJs *dataPointJs = new DataPointJs(engine());

				return engine()->newObject(dataPointJs, data);
			}

			return QScriptValue();
		}


	void 
		MassSpectrumJs::setProperty(QScriptValue &object, const QScriptString &name,
				uint id, const QScriptValue &value)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()";

			MassSpectrum *massSpectrum = qscriptvalue_cast<MassSpectrum *>(object.data());

			if(name == title)
			{
				massSpectrum->m_title = qscriptvalue_cast<QString>(value);
			}
			if(name == rt)
			{
				massSpectrum->m_rt = qscriptvalue_cast<double>(value);
			}
			if(name == dt)
			{
				massSpectrum->m_dt = qscriptvalue_cast<double>(value);
			}

			// We do not handle setting MassPeak yet.

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
				//<< "Right before returning void";
			return ;
		}


	QScriptClassPropertyIterator *
		MassSpectrumJs::newIterator(const QScriptValue &object)
		{
			return new MassSpectrumJsPropertyIterator(object);
		}



	QScriptValue
		MassSpectrumJs::construct(QScriptContext *ctx, QScriptEngine *)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" ;

			MassSpectrumJs *msJs = qscriptvalue_cast<MassSpectrumJs*>(ctx->callee().data());

			if (!msJs)
				return QScriptValue();

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
				//<< "Found the MassSpectrumJs class.";

			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{
				QScriptValue arg = ctx->argument(0);

				if(arg.instanceOf(ctx->callee()))
				{
					//qDebug() << __FILE__ << __LINE__ << __FUNCTION__<< "()"
						//<< "The argument passed to the constructor is also a MassSpectrum."
						//<< "Right before returning newInstance(MassSpectrum-C++ casted from QScritValue)";

					return msJs->newInstance(qscriptvalue_cast<MassSpectrum>(arg));
				}

				if(arg.isString())
				{
					// The string is the title of the mass spectrum.
					return msJs->newInstance(MassSpectrum(qscriptvalue_cast<QString>(arg)));
				}
			}

			// By default return an empty mass spectrum.
			return msJs->newInstance();
		}


	QScriptValue 
		MassSpectrumJs::toScriptValue(QScriptEngine *eng, const MassSpectrum &massSpectrum)
		{
			QScriptValue ctor = eng->globalObject().property("MassSpectrum");

			MassSpectrumJs *msJs = qscriptvalue_cast<MassSpectrumJs*>(ctor.data());

			if (!msJs)
				return eng->newVariant(QVariant::fromValue(massSpectrum));

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before newInstance(MassSpectrum-C++)";

			return msJs->newInstance(massSpectrum);
		}


	void 
		MassSpectrumJs::fromScriptValue(const QScriptValue &obj, MassSpectrum &massSpectrum)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before massSpectrum = qvariant_cast<MassSpectrum>(obj.data().toVariant());"
				//<< "to fill-in MassSpectrum" << &massSpectrum;

			QVariant variant = obj.data().toVariant();

			qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
			<< "Variant type:" << variant.userType();

			massSpectrum = qvariant_cast<MassSpectrum>(variant);

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right after massSpectrum = qvariant_cast<MassSpectrum>(obj.data().toVariant());";
		}


} // namespace msXpSlibmass
