
"""This plugin provides test results in Mago 1.0 Compatible XML format.

It's needed while transitioning from the previous version of mago until
the tests have been converted to mago-ng. The output is simplified compared
to mago itself because the changes with the xml file are incompatible with
mago-ng

Add this shell command to your builder ::

    nosetests --with-magoxml

And by default a file named mago.xml will be written to the
working directory.

If you need to change the name or location of the file, you can set the
``--magoxml-file`` option.

Here is an example output

<suite name="gedit chains">
  <class>gedit_chains.GEditChain</class>
  <description>
    Tests which verify gedit's save file functionality.
  </description>
  <case name="ASCII Test">
    <method>testChain</method>
    <description>Test ASCII text saving.</description>
  <result><time>6.2365219593</time><pass>1</pass></result></case>
</suite>

"""

import doctest
import os
import traceback
import re
import inspect
from nose.plugins.base import Plugin
from nose.exc import SkipTest
from time import time
from xml.sax import saxutils
from subprocess import Popen, PIPE

# Invalid XML characters, control characters 0-31 sans \t, \n and \r
CONTROL_CHARACTERS = re.compile(r"[\000-\010\013\014\016-\037]")
REPORT_XSL = os.path.join(os.path.dirname(__file__), "magoreport.xsl")

def xml_safe(value):
    """Replaces invalid XML characters with '?'."""
    return CONTROL_CHARACTERS.sub('?', value)

def escape_cdata(cdata):
    """Escape a string for an XML CDATA section."""
    return xml_safe(cdata).replace(']]>', ']]>]]&gt;<![CDATA[')

def nice_classname(obj):
    """Returns a nice name for class object or class instance.

        >>> nice_classname(Exception()) # doctest: +ELLIPSIS
        '...Exception'
        >>> nice_classname(Exception)
        'exceptions.Exception'

    """
    if inspect.isclass(obj):
        cls_name = obj.__name__
    else:
        cls_name = obj.__class__.__name__
    mod = inspect.getmodule(obj)
    if mod:
        name = mod.__name__
        # jython
        if name.startswith('org.python.core.'):
            name = name[len('org.python.core.'):]
        return "%s.%s" % (name, cls_name)
    else:
        return cls_name

def exc_message(exc_info):
    """Return the exception's message."""
    exc = exc_info[1]
    if exc is None:
        # str exception
        result = exc_info[0]
    else:
        try:
            result = str(exc)
        except UnicodeEncodeError:
            try:
                result = unicode(exc)
            except UnicodeError:
                # Fallback to args as neither str nor
                # unicode(Exception(u'\xe6')) work in Python < 2.6
                result = exc.args[0]
    return xml_safe(result)

class MagoXML(Plugin):
    """This plugin provides test results in the standard magoxml XML format."""
    name = 'magoxml'
    encoding = 'UTF-8'
    error_report_file = None
    error_report_html = None

    def _timeTaken(self):
        if hasattr(self, '_timer'):
            taken = time() - self._timer
        else:
            # test died before it ran (probably error in setup())
            # or success/failure added before test started probably 
            # due to custom TestResult munging
            taken = 0.0
        return taken

    def _quoteattr(self, attr):
        """Escape an XML attribute. Value can be unicode."""
        attr = xml_safe(attr)
        if isinstance(attr, unicode):
            attr = attr.encode(self.encoding)
        return saxutils.quoteattr(attr)

    def options(self, parser, env):
        """Sets additional command line options."""
        Plugin.options(self, parser, env)
        parser.add_option(
            '--magoxml-file', action='store',
            dest='magoxml_file', metavar="FILE",
            default=env.get('NOSE_MAGOXML_FILE', 'mago_result.xml'),
            help=("Path to xml file to store the mago report in. "
                  "Default is mago_result.xml in the working directory "
                  "[NOSE_MAGOXML_FILE]"))

        parser.add_option(
            '--magoxml-html', action='store',
            dest='magoxml_html', metavar="FILE",
            default=env.get('NOSE_MAGOXML_HTML'),
            help=("Path to html file to store the mago report in. If this"
                  "option is not set only the xml report is generated"
                  "[NOSE_MAGOXML_HTML]"))

    def configure(self, options, config):
        """Configures the magoxml plugin."""
        Plugin.configure(self, options, config)
        self.config = config
        if self.enabled:
            self.stats = {'errors': 0,
                          'failures': 0,
                          'passes': 0,
                          'skipped': 0
                          }
            self.errorlist = []
            self.error_report_file = open(options.magoxml_file, 'w')
            self.error_report_html = options.magoxml_html

    def report(self, stream):
        """Writes an magoxml-formatted XML file

        The file includes a report of test errors and failures.

        """
        self.stats['encoding'] = self.encoding
        self.stats['total'] = (self.stats['errors'] + self.stats['failures']
                               + self.stats['passes'] + self.stats['skipped'])
        self.stats['suitename'] = self._suitename
        self.error_report_file.write(
            '<?xml version="1.0" encoding="%(encoding)s"?>'
            '<suite name=%(suitename)s tests="%(total)d" '
            'errors="%(errors)d" failures="%(failures)d" '
            'skip="%(skipped)d">' % self.stats)
        self.error_report_file.write(''.join(self.errorlist))
        self.error_report_file.write('</suite>')
        self.error_report_file.close()
        if self.config.verbosity > 1:
            stream.writeln("-" * 70)
            stream.writeln("XML: %s" % self.error_report_file.name)

        if self.error_report_html:
            cmd = ["xsltproc", "-o", self.error_report_html, REPORT_XSL,
                   self.error_report_file.name]
            p = Popen(cmd)


    def startTest(self, test):
        """Initializes a timer before starting a test."""
        self._timer = time()
        self._suitename = self._quoteattr(test.id().split('.')[0])

    def addError(self, test, err, capt=None):
        """Add error output to magoxml report.
        """
        taken = self._timeTaken()

        if issubclass(err[0], SkipTest):
            type = 'skipped'
            self.stats['skipped'] += 1
        else:
            type = 'error'
            self.stats['errors'] += 1
        tb = ''.join(traceback.format_exception(*err))
        id = test.id()
        desc = test.shortDescription()

        self.errorlist.append(
            '<case name=%(name)s>'
            '<method>%(cls)s</method>'
            '<description>%(desc)s</description>'
            '<result><stacktrace><![CDATA[%(tb)s]]></stacktrace><message><![CDATA[%(message)s]]></message><time>%(taken).2f</time><pass>0</pass></result></case>' %
            {'name': self._quoteattr(id.split('.')[-1]),
             'cls': self._quoteattr('.'.join(id.split('.')[-2:])),
             'desc': desc,
             'tb': escape_cdata(tb),
             'message': self._quoteattr(exc_message(err)),
             'taken': taken,
             })

    def addFailure(self, test, err, capt=None, tb_info=None):
        """Add failure output to magoxml report.
        """
        taken = self._timeTaken()
        tb = ''.join(traceback.format_exception(*err))
        self.stats['failures'] += 1
        id = test.id()
        desc = test.shortDescription()

        self.errorlist.append(
            '<case name=%(name)s>'
            '<method>%(cls)s</method>'
            '<description>%(desc)s</description>'
            '<result><stacktrace><![CDATA[%(tb)s]]></stacktrace><message><![CDATA[%(message)s]]></message><time>%(taken).2f</time><pass>0</pass></result></case>' %
            {'name': self._quoteattr(id.split('.')[-1]),
             'cls': self._quoteattr('.'.join(id.split('.')[-2:])),
             'desc': desc,
             'tb': escape_cdata(tb),
             'message': self._quoteattr(exc_message(err)),
             'taken': taken,
             })


    def addSuccess(self, test, capt=None):
        """Add success output to magoxml report.
        """
        taken = self._timeTaken()
        self.stats['passes'] += 1
        id = test.id()
        desc = test.shortDescription()
        self.errorlist.append(
            '<case name=%(name)s>'
            '<method>%(cls)s</method>'
            '<description>%(desc)s</description>'
            '<result><time>%(taken).2f</time><pass>1</pass></result></case>' %
            {'name': self._quoteattr(id.split('.')[-1]),
             'cls': self._quoteattr('.'.join(id.split('.')[-2:])),
             'desc': desc,
             'taken': taken,
             })
