/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_return_poolindex_without_gsi.c
    \brief  This interface returns the uid, gids and poolindex, i.e. leaseid,
            using LCMAPS
    \author Martijn Steenbakkers for the EU DataGrid.

    This interface returns the uid, gids and poolindex, i.e. leaseid, using LCMAPS.
    As input it requires the DN and FQANs asserted to the user
    -# lcmaps_return_poolindex_without_gsi_va
        Returns the poolindex based on a variable argument list
    -# lcmaps_return_poolindex_without_gsi: interface function
        Returns the poolindex based on fixed arguments:
                security context, buffer and buffer length
*/

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>

/* LCMAPS includes */
#include "lcmaps.h"
#include "lcmaps_account.h"
#include "lcmaps_log.h"
#include "lcmaps_return_poolindex_without_gsi.h"

#include "pluginmanager/_lcmaps_log.h"
#include "pluginmanager/_lcmaps_utils.h"

/******************************************************************************
                             Module specific defines
******************************************************************************/
/* Default settings */
#define LCMAPS_MAX_POLICIES 10

/* Flags to influence behaviour of the interface */
#define DONT_REQUIRE_POOLINDEX ((unsigned short)0x0000)
#define REQUIRE_POOLINDEX      ((unsigned short)0x0001)


/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int           lcmaps_return_poolindex_without_gsi_va(int , ...);

/******************************************************************************
Function:   lcmaps_return_poolindex_without_gsi_va
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The arguments of this function are passed as a variable argument list.

Parameters:
    narg:  the number of arguments that follow
    ap:    vararg list, that consists of
        - input:  The DN of the user
        - input:  The list of FQANs (type char**)
        - input:  The number of FQANs in the list
        - output: a structure that contains account information:
                uid, gids (primary and secondary) and the poolindex
        - input, optional: the mapcounter
Returns:
    0: success
    1: failure
******************************************************************************/
/*!
    \fn lcmaps_return_poolindex_without_gsi_va(
        int        narg,
        va_list    ap
    )
    \brief LCMAPS returns the poolindex and user account information

    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The arguments of this function are passed as a variable argument list.

    \param narg  the number of arguments that follow
    \param ap    va_arg list. This list currently contains:
        -# input:  The DN of the user
        -# input:  The list of FQANs (type char **)
        -# input:  The number of FQANs in the list
        -# output: A structure that contains account information:
                uid, gids (primary and secondary) and the poolindex
        -# input, optional: the mapcounter
    \retval 0 success.
    \retval 1 failure.
    \internal
*/
static int lcmaps_return_poolindex_without_gsi_va(
    int        narg,
    ...
)
{
    va_list                             ap;
    char *                              user_dn = NULL;          /* input variable */
    char **                             fqan_list = NULL;        /* input variable */
    int                                 nfqan = -1;              /* input variable */
    int                                 mapcounter = -1;         /* input variable */
    unsigned short                      flags = 0;               /* input variable */
    lcmaps_account_info_t *             plcmaps_account = NULL;  /* output variable */
    char *                              lcmaps_policy_string = NULL;
    char *                              lcmaps_policies[LCMAPS_MAX_POLICIES];
    int                                 lcmaps_npols = LCMAPS_MAX_POLICIES;
    int                                 rc = -1;
    uid_t                               uid = (uid_t)-1;
    gid_t *                             pgid_list = NULL;
    int                                 npgid = 0;
    gid_t *                             sgid_list = NULL;
    int                                 nsgid = 0;
    char *                              poolindex = NULL;
    int                                 jj = 0;

    /* First initialize LCMAPS, initialize without file pointer or name which
     * will result in getting env var LCMAPS_LOG_FILE. Specify DO_USRLOG to try
     * that first, if that fails we will go automatically to syslog */
    if (lcmaps_init_and_logfile(NULL, NULL, (unsigned short)(DO_USRLOG)))
    {
        lcmaps_log(LOG_ERR, "%s: LCMAPS initialization failure\n", __func__);
        return 1;
    }


    /* Parse arguments from va_list */
    va_start(ap, narg);
    if (narg >= 4)
    {
        user_dn = va_arg(ap, char *);
        fqan_list = va_arg(ap, char **);
        nfqan = va_arg(ap, int);
        plcmaps_account = va_arg(ap, lcmaps_account_info_t *);
        if (narg >=5)
        {
            mapcounter = va_arg(ap, int);
        }
        if (narg >=6)
        {
            flags = (unsigned short)va_arg(ap, unsigned int);
        }
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: The number of arguments (%d) should be in the range %d-%d\n", __func__, narg, 4, 6);
        return 1;
    }
    va_end(ap);

    /* Parse policy environment variable */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++) lcmaps_policies[jj] = 0;
    lcmaps_policy_string = getenv("LCMAPS_POLICY_NAME");
    if ((rc = lcmaps_tokenize(lcmaps_policy_string, lcmaps_policies, &lcmaps_npols, ":")) != 0)
    {
        lcmaps_log(LOG_ERR, "%s: Cannot parse LCMAPS_POLICY_NAME environment variable, because\n", __func__);
        switch (rc)
        {
            case -1:
                lcmaps_log(LOG_ERR, "%s: of a malloc error\n", __func__);
                break;
            case -2:
                lcmaps_log(LOG_ERR, "%s: the policy list is too large (max = %d)\n",
                    __func__, LCMAPS_MAX_POLICIES);
                break;
            case -3:
                lcmaps_log(LOG_ERR, "%s: of a non-matching quote\n", __func__);
                break;
            case -4:
                lcmaps_log(LOG_ERR, "%s: of invalid input\n", __func__);
                break;
            default:
                lcmaps_log(LOG_ERR, "%s: of an unknown error\n", __func__);
                break;
        }
        goto return_poolindex_without_gsi_error;
    }

    /*
     * Now that we have the credential let's run (and terminate) LCMAPS !
     */
    if (narg == 4)
    {
        rc = lcmaps_run_with_fqans_and_return_account(
            user_dn,
            fqan_list,
            nfqan,
            NULL,
            lcmaps_npols,
            lcmaps_policies,
            &uid,
            &pgid_list,
            &npgid,
            &sgid_list,
            &nsgid,
            &poolindex
        );
    }
    else if ((narg == 5) || (narg == 6))
    {
        rc = lcmaps_run_with_fqans_mapcounter_and_return_account(
            user_dn,
            fqan_list,
            nfqan,
            mapcounter,
            NULL,
            lcmaps_npols,
            lcmaps_policies,
            &uid,
            &pgid_list,
            &npgid,
            &sgid_list,
            &nsgid,
            &poolindex
        );
    }
    if (rc != 0)
    {
        lcmaps_log (LOG_ERR, "LCMAPS failed to do mapping and return account information\n");
        if (lcmaps_term())
        {
            lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
            goto return_poolindex_without_gsi_error;
        }
        goto return_poolindex_without_gsi_error;
    }
    /*
     *  Check if a poolindex has been found.
     *  If this is not the case and the flag REQUIRE_POOLINDEX has been set:
     *  Error.
     */
    if ( (poolindex == NULL) && ((flags & REQUIRE_POOLINDEX) == REQUIRE_POOLINDEX) )
    {
        lcmaps_log (LOG_NOTICE, "LCMAPS failed to find a poolindex, error.\n");
        if (lcmaps_term())
        {
            lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
            goto return_poolindex_without_gsi_error;
        }
        goto return_poolindex_without_gsi_error;
    }

    /*
     * Fill the lcmaps account information
     */
    rc = lcmaps_account_info_fill(
        &uid,
        &pgid_list,
        &npgid,
        &sgid_list,
        &nsgid,
        &poolindex,
        plcmaps_account
    );
    if (rc != 0)
    {
        lcmaps_log (LOG_NOTICE, "LCMAPS failed to do mapping and return account information\n");
        if (lcmaps_term())
        {
            lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
            goto return_poolindex_without_gsi_error;
        }
        goto return_poolindex_without_gsi_error;
    }

    rc = lcmaps_term();
    if (rc)
    {
        lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
        goto return_poolindex_without_gsi_error;
    }


    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }

    return 0;

 return_poolindex_without_gsi_error:
    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }

    return 1;
}

/******************************************************************************
Function:   lcmaps_return_poolindex_without_gsi
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The call will result in a failure if no poolindex is found (e.g. in the case of
    statically assigned accounts).

Parameters:
    user_dn:          The DN of the user (input)
    fqan_list:        The list of FQANs (type char**) (input)
    nfqan:            The number of FQANs in the list (input)
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure (if no poolindex is found)
******************************************************************************/
int lcmaps_return_poolindex_without_gsi(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        lcmaps_account_info_t * plcmaps_account
)
{
    return (lcmaps_return_poolindex_without_gsi_va(
                6,
                user_dn,
                fqan_list,
                nfqan,
                plcmaps_account,
                -1,
                REQUIRE_POOLINDEX));
}

/******************************************************************************
Function:   lcmaps_return_poolindex_with_mapcounter
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The call will result in a failure if no poolindex is found (e.g. in the case of
    statically assigned accounts).
    An additional counter may be added to enable multiple acccounts per set of user
    credentials.

Parameters:
    user_dn:          The DN of the user (input)
    fqan_list:        The list of FQANs (type char**) (input)
    nfqan:            The number of FQANs in the list (input)
    mapcounter:       The counter which will be added to the poolindex, effectively enabling
                      multiple account mappings
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure (if no poolindex is found)
******************************************************************************/
int lcmaps_return_poolindex_with_mapcounter(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
)
{
    return (lcmaps_return_poolindex_without_gsi_va(
                6,
                user_dn,
                fqan_list,
                nfqan,
                plcmaps_account,
                mapcounter,
                REQUIRE_POOLINDEX));
}

/******************************************************************************
Function:   lcmaps_return_account_without_gsi
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex (if available), alternatively named leaseid, to the calling application.
    The call will not result in a failure if no poolindex is found, so it supports
    poolaccounts and statically assigned accounts simultaneously.
    An additional counter may be added to enable multiple acccounts per set of user
    credentials.

Parameters:
    user_dn:          The DN of the user (input)
    fqan_list:        The list of FQANs (type char**) (input)
    nfqan:            The number of FQANs in the list (input)
    mapcounter:       The counter which will be added to the poolindex, effectively enabling
                      multiple account mappings
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_return_account_without_gsi(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
)
{
    return (lcmaps_return_poolindex_without_gsi_va(
                6,
                user_dn,
                fqan_list,
                nfqan,
                plcmaps_account,
                mapcounter,
                DONT_REQUIRE_POOLINDEX));
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps_return_poolindex_without_gsi.c,v $
    $Date: 2014-03-13 11:56:06 +0100 (Thu, 13 Mar 2014) $
    $Revision: 17647 $
    $Author: msalle $
******************************************************************************/
