/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "connection.h"
#include "tool.h"

#include "sharedfolder_record.h"

using namespace Form;

Entry SharedFolderRecord::loadEntry( const QString &id, PagePolicy::State )
{
  QString filter( "objectClass=*" );
  QLdapResponse response = Connection::self()->search( id, QLdap::Base, filter );

  if ( !response.isValid() || response.entries().isEmpty() )
    return Entry();

  return Entry::fromLdapEntry( response.entries().first() );
}

bool SharedFolderRecord::saveEntry( const Entry &entry, PagePolicy::State state, QString &errorMsg )
{
  if ( state == PagePolicy::Add || state == PagePolicy::Modify ) {
    QString dnRoot = Connection::self()->baseDn();

    QLdapEntry ldapEntry;
    ldapEntry.addValue( "objectClass", "kolabSharedFolder" );
    ldapEntry.setValue( "cn", entry.value( "cn" ).toLower() );
    ldapEntry.setValue( "kolabHomeServer", entry.value( "kolabHomeServer" ) );
    ldapEntry.setValue( "cyrus-userquota", entry.value( "cyrus-userquota" ) );

    QStringList acls = entry.values( "acl" );
    for ( int i = 0; i < acls.count(); ++i )
      ldapEntry.addValue( "acl", acls[ i ] );

    if ( state == PagePolicy::Modify ) {
      QString newDn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), dnRoot );

      /**
       * The 'cn' of the shared folder has changed, so we have to perform some
       * extra checks.
       */
      if ( entry.id() != newDn ) {
        if ( !Connection::self()->add( newDn, ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not modify shared folder '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

        if ( !Connection::self()->remove( entry.id() ) ) {
          errorMsg = QObject::tr( "Could not modify shared folder '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

      } else {
        if ( !Connection::self()->modifyAttributes( entry.id(), ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not modify shared folder '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }
      }
    } else if ( state == PagePolicy::Add ) {
      QString dn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), dnRoot );

      if ( !Connection::self()->add( dn, ldapEntry ) ) {
        errorMsg = QObject::tr( "Could not add shared folder '%1': %2" )
                     .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
        return false;
      }
    }
  } else if ( state == PagePolicy::Delete ) {
    QString ldapError;
    if ( !Tool::deleteSharedFolder( Connection::self(), entry.id(), ldapError ) ) {
      errorMsg = QObject::tr( "Could not delete shared folder '%1': %2" )
                            .arg( entry.value( "cn" ), ldapError );
      return false;
    }
  }

  return true;
}
