/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtGui>
//#include <qobject.h>
//#include <qlistview.h>
//#include <qstring.h>
//#include <qfile.h>
//#include <qfileinfo.h>
//#include <qtimer.h>
//#include <qdir.h>
//#include <qmessagebox.h>
//#include <qapplication.h>
//#include <qdeepcopy.h>
#include <errno.h>

#include "ifpguiview.h"
#include "createdirdlg.h"
#include "backend/iointerface.h"
#include "backend/ifpinterface.h"
#include "ifp_gui.h"

int IOInterface::internal_norm_ProtocolStr(QString& fileName, QString protoStr)
{
    int pat_pos;
    QString filestr;

    pat_pos = fileName.indexOf("://");
    if (pat_pos >= 0)
    {
        filestr = fileName.left(pat_pos + 3);
        if (filestr != protoStr)
        {
            return(-1);
        }
        fileName = fileName.remove(0, pat_pos + 3);
    }
    return(IO_INTF_OK);
}

QString IOInterface::internal_get_ProtocolStr(QString& fileName)
{
    int pat_pos;
    QString filestr;

    filestr = "";
    pat_pos = fileName.indexOf("://");
    if (pat_pos >= 0)
    {
        filestr = fileName.left(pat_pos + 3);
    }
    return(filestr);
}

int IOInterface::internal_remove_ProtocolStr(QString& fileName)
{
    int pat_pos, ret;

    ret = IO_INTF_OK;
    pat_pos = fileName.indexOf("://");
    if (pat_pos >= 0)
    {
        fileName = fileName.remove(0, pat_pos + 3);
    }
    return(ret);
}

/**
 * Upload the files in local_file_list from the local harddrive to the iRiver directory remoteDest
 * @param local_file_list the list of filenames on the local harddrive.
 * @param remoteDest the iRiver directory; the place to put the uploaded file
 * @return IFP_INTF_OK on success (0).  Otherwise return an error code.
 */
int IOInterface::internal_exec_TransferToiFP(QString& remoteDest, QList<FileInfos> *local_file_list)
{
    int ret, tret, mbret, freespace;
    int filecount, totalsize, tmpcnt, dir_size;
    QFileInfo finfo;
    ifp_progress_info pginfo;
    QString Message, fileName;
    QList<FileInfos>::iterator fi_iter;
    TiFPgetDirSize t_dirSize;
    TiFPreadFileTree t_ifpTree;
    TiFPuploadDirectory t_uploadDir;
    TiFPuploadFile t_uploadFile;

    ret = IO_INTF_OK;

    if (ifp_dh == NULL)
    {
        lastErrString = QObject::tr("iRiver device not connected.");
        return(-1);
    }

    if (local_file_list == NULL)
    {
        return(IO_INTF_OK);
    }

    if (local_file_list->isEmpty())
    {
        return(IO_INTF_OK);
    }

    // remove the protocol info
    ret = internal_norm_ProtocolStr(remoteDest, "ifp://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Remote destination protocol not supported \"%1\"").arg(internal_get_ProtocolStr(remoteDest));
        return(-1);
    }

    filecount = 0;
    totalsize = 0;
    // get the size of the files/directories and the number of files
    for (fi_iter = local_file_list->begin(); fi_iter != local_file_list->end(); ++fi_iter)
    {
        // remove the protocol info
        fileName = fi_iter->FileName();
        ret = internal_norm_ProtocolStr(fileName, "file://");
        if (ret != IO_INTF_OK)
        {
            lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(fileName));
            return(-1);
        }
        // get the file info
        if (QFileInfo(fileName).isFile())
        {
            finfo.setFile(fileName);
            fi_iter->setValues(fileName, IO_INTF_FILE, finfo.size());
            totalsize += finfo.size();
            filecount++;
        }
        else if (QFileInfo(fileName).isDir())
        {
            tmpcnt = 0;
            dir_size = local_get_dir_size(fileName, &tmpcnt);
            fi_iter->setValues(fileName, IO_INTF_DIR, dir_size);
            totalsize += dir_size;
            // this only counts as one file for case where
            //this is the only thing being uploaded
            filecount++;// += tmpcnt;
        }
        else
        {
            fi_iter->setValues(fileName, -1, 0);
        }
    }

    freespace = ifpgui_ifp_freespace(&ifp_dev);
    if (freespace < totalsize)
    {
        lastErrString = QObject::tr("There is not enough free space to upload the requested files.");
        showInfoMsgBox(QObject::tr("iRiver Upload File Status"), lastErrString);
        return -ENOMEM;
    }

    // do the transfer
    Message = QObject::tr("Uploading Files to iRiver device  %1").arg(remoteDest);
    pginfo.total_to_transfer = totalsize;
    pginfo.total_transfered  = 0;
    if (filecount > 1)
    {
        pginfo.is_batch = true;
        Message = QObject::tr("Upload files to iRiver device  %1  ?").arg(remoteDest);
        mbret = showYesNoAllMsgBox(QObject::tr("Uploading files to iRiver"), Message);
    }
    else
    {
        pginfo.is_batch = false;
        Message = QObject::tr("Upload file to iRiver device  %1  ?").arg(remoteDest);
        mbret = showYesNoMsgBox(QObject::tr("Uploading files to iRiver"), Message);
    }

    if (mbret == QMessageBox::No)
        return(IO_INTF_USER_CANCEL);

    ret = IO_INTF_OK;
    for (fi_iter = local_file_list->begin(); fi_iter != local_file_list->end(); ++fi_iter)
    {
        processingFiles = TRUE;
        fi_iter->setProcessed(FALSE);
        fileName = fi_iter->FileName();
        if ((mbret != QMessageBox::YesAll) && ((mbret != QMessageBox::Yes) || (filecount != 1)))
        {
            Message = QObject::tr("Upload  %1  to iRiver device  %2  ?").arg(fileName).arg(remoteDest);
            mbret = showYesNoAllMsgBox(QObject::tr("Uploading files to iRiver"), Message);
        }
        lastErrString = "";
        if (fi_iter->FileType() == IO_INTF_DIR)
        {
            if ((mbret != QMessageBox::Yes) || (mbret != QMessageBox::YesAll))
            {
                ret = t_uploadDir.iFPuploadDirectory(fileName, remoteDest, &pginfo);
                fi_iter->setProcessed(TRUE);
                tmpcnt = 0;
                pginfo.total_transfered += t_dirSize.iFPgetDirSize(fileName, &tmpcnt);
            }
        }
        else
        {
            if ((mbret != QMessageBox::Yes) || (mbret != QMessageBox::YesAll))
            {
                ret = t_uploadFile.iFPuploadFile(fileName, remoteDest, &pginfo);
                fi_iter->setProcessed(TRUE);
                pginfo.total_transfered += fi_iter->FileSize();
            }
        }
        if ((ret != IO_INTF_OK) && (ret != IO_INTF_USER_CANCEL))
        {
            MainIfpGuiWindow->hideTransferProgress();
            lastErrString = QString(ifpgui_ifp_error_message(ret));
            showInfoMsgBox(QObject::tr("iRiver Upload File Status"), lastErrString);
            break;
        }
        if (ret == IO_INTF_USER_CANCEL)
        {
            break;
        }
        MainIfpGuiWindow->updateFileAndStatus();
    }
    MainIfpGuiWindow->hideTransferProgress();
    tret = t_ifpTree.iFPreadFileTree();
    if (tret != IO_INTF_OK)
    {
        Message = QObject::tr("Could not get the file list.  Check your battery.  Try replugging the USB cable.");
        showInfoMsgBox(QObject::tr("iRiver Upload File Status"), Message);
    }
    processingFiles = FALSE;
    return(ret);
}

/**
 * Download the files in remote_file_list from the iFP device to the local directory localDest
 * @param remote_file_list the list of filenames on the iFP device.
 * @param localDest the local directory; the place to save the downloaded file
 * @return IFP_INTF_OK on success (0).  Otherwise return an error code.
 */
int IOInterface::internal_exec_TransferFromiFP(QString& localDest, QList<FileInfos> *remote_file_list)
{
    int ret, mbret;
    FileInfos fi_info;
    int filecount, totalsize, tmpcnt;
    ifp_progress_info pginfo;
    QString Message, fileName, filestr;
    QList<FileInfos>::iterator fi_iter;
    TiFPgetInfo t_info;
    TiFPgetDirSize t_dirSize;
    TiFPdownloadDirectory t_downloadDir;
    TiFPdownloadFile t_downloadFile;

    ret = IO_INTF_OK;
    if (remote_file_list == NULL)
    {
        return(IO_INTF_OK);
    }

    if (remote_file_list->isEmpty())
    {
        return(IO_INTF_OK);
    }

    // remove the protocol info
    ret = internal_norm_ProtocolStr(localDest, "ifp://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Local destination protocol not supported \"%1\"").arg(internal_get_ProtocolStr(localDest));
        return(-1);
    }

    filecount = 0;
    totalsize = 0;
    // get the size of the files/directories and the number of files
    for (fi_iter = remote_file_list->begin(); fi_iter != remote_file_list->end(); ++fi_iter)
    {
        // remove the protocol info
        fileName = fi_iter->FileName();
        ret = internal_norm_ProtocolStr(fileName, "ifp://");
        if (ret != IO_INTF_OK)
        {
            lastErrString = QObject::tr("Remote file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(fileName));
            return(-1);
        }
        // get the file info
        if (t_info.iFPgetInfo(fileName, fi_info) == IO_INTF_OK)
        {
            // copy data from fi_info into fi_iter
            fi_iter->setValues(fileName, fi_info.FileType(), fi_info.FileSize());

            if (fi_iter->FileType() == IO_INTF_FILE)
            {
                filecount++;
                totalsize += fi_iter->FileSize();
            }
            else
            {
                tmpcnt = 0;
                totalsize += t_dirSize.iFPgetDirSize(fi_iter->FileName(), &tmpcnt);
                // this only counts as one file for case where
                //this is the only thing being uploaded
                filecount++;// += tmpcnt;
            }
        }
        else
        {
            // copy data from fi_info into fi_iter
            fi_iter->setValues(fileName, -1, 0);
        }
    }
    // do the transfer
    Message = QObject::tr("Downloading Files from ifp device to  %1").arg(localDest);
    pginfo.total_to_transfer = totalsize;
    pginfo.total_transfered  = 0;
    if (filecount > 1)
    {
        pginfo.is_batch = true;
        Message = QObject::tr("Download files from iRiver to local directory  %1  ?").arg(localDest);
        mbret = showYesNoAllMsgBox(QObject::tr("Downloading files from iRiver"), Message);
    }
    else
    {
        pginfo.is_batch = false;
        Message = QObject::tr("Download file from iRiver to local directory  %1  ?").arg(localDest);
        mbret = showYesNoMsgBox(QObject::tr("Downloading files from iRiver"), Message);
    }

    if (mbret == QMessageBox::No)
        return(IO_INTF_USER_CANCEL);

    ret = IO_INTF_OK;
    for (fi_iter = remote_file_list->begin(); fi_iter != remote_file_list->end(); ++fi_iter)
    {
        processingFiles = TRUE;
        fi_iter->setProcessed(FALSE);
        if ((mbret != QMessageBox::YesAll) && ((mbret != QMessageBox::Yes) || (filecount != 1)))
        {
            Message = QObject::tr("Download  %1  from iRiver device to  %2  ?").arg(fi_iter->FileName()).arg(localDest);
            mbret = showYesNoAllMsgBox(QObject::tr("Downloading files from iRiver"), Message);
        }
        lastErrString = "";
        if (fi_iter->FileType() == IO_INTF_DIR)
        {
            if ((mbret != QMessageBox::Yes) || (mbret != QMessageBox::YesAll))
            {
                fileName = fi_iter->FileName();
                ret = t_downloadDir.iFPdownloadDirectory(fileName, localDest, &pginfo);
                fi_iter->setProcessed(TRUE);
                tmpcnt = 0;
                pginfo.total_transfered += t_dirSize.iFPgetDirSize(fi_iter->FileName(), &tmpcnt);
            }
        }
        else
        {
            if ((mbret != QMessageBox::Yes) || (mbret != QMessageBox::YesAll))
            {
                fileName = fi_iter->FileName();
                ret = t_downloadFile.iFPdownloadFile(fileName, localDest, &pginfo);
                fi_iter->setProcessed(TRUE);
                pginfo.total_transfered += fi_iter->FileSize();
            }
        }
        if ((ret != IO_INTF_OK) && (ret != IO_INTF_USER_CANCEL))
        {
            lastErrString = QString(ifpgui_ifp_error_message(ret));
            showInfoMsgBox(QObject::tr("Download File Status"), lastErrString);
            break;
        }
        if (ret == IO_INTF_USER_CANCEL)
        {
            break;
        }
    }
    MainIfpGuiWindow->hideTransferProgress();
    processingFiles = FALSE;
    return(ret);
}

/**
 * FileInterface
 * Functions to implement the local file io routines.
 */
/**
 * Upload the files from the local drive to the iFP device
 * @param file_info_list the list of filenames on the local hard drive.
 * @param destination the iFP destination directory.
 * @return IFP_INTF_OK on success (0).  Otherwise return an error code.
 */
int FileInterface::exec_TransferUpload(QString& destination, QList<FileInfos> *file_info_list)
{
    return(internal_exec_TransferToiFP(destination, file_info_list));
}
/**
 * Download the files from iFP to local drive
 * @param file_info_list the list of filenames on the iFP device.
 * @param destination the local destination directory.
 * @return IFP_INTF_OK on success (0).  Otherwise return an error code.
 */
int FileInterface::exec_TransferDownload(QString& destination, QList<FileInfos> *file_info_list)
{
    return(internal_exec_TransferFromiFP(destination, file_info_list));
}

int FileInterface::exec_Mkdir(QString& baseDir, QString& newDir)
{
    int ret;
    QDir dirObj;

    ret = internal_norm_ProtocolStr(baseDir, "file://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(baseDir));
        return(-1);
    }

    CreateDirWindow *mkdirdlg;
    lastErrString = "";
    mkdirdlg = new CreateDirWindow(QObject::tr("Create a local directory"));
    mkdirdlg->setBaseDir(baseDir);

    ret = mkdirdlg->exec();
    newDir = mkdirdlg->getNewDirName();
    delete mkdirdlg;

    if (ret == QDialog::Accepted)
    {
        ret = dirObj.mkdir(newDir);
        if (ret != true)
        {
            lastErrString = QObject::tr("Could not make directory [%1]").arg(newDir);
            return(-1);
        }
        return(IO_INTF_OK);
    }
    return(IO_INTF_USER_CANCEL);
}

int FileInterface::exec_Rename(const QString& oldName, const QString& newName)
{
    int ret;
    QString tmp_oldName, tmp_newName;

    tmp_oldName = oldName;
    tmp_newName = newName;

    ret = internal_norm_ProtocolStr(tmp_oldName, "file://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(tmp_oldName));
        return(-1);
    }
    ret = internal_norm_ProtocolStr(tmp_newName, "file://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(tmp_newName));
        return(-1);
    }

    QFile file(oldName);
    if (!file.exists())
    {
        lastErrString = QObject::tr("File does not exist:  %1.").arg(oldName);
        return(-ENOENT);
    }
    QDir dir;
    ret = dir.rename(oldName, newName);
    if (ret != TRUE)
    {
        lastErrString = QObject::tr("Could not rename  %1  to  %2").arg(oldName).arg(newName);
        return(-1);
    }
    return(IO_INTF_OK);
}

int FileInterface::exec_Delete(QList<FileInfos> *file_info_list)
{
    int mbret, ret;
    QString Message, filename;
    QList<FileInfos>::iterator fi_iter;

    if (file_info_list == NULL)
    {
        return(IO_INTF_OK);
    }
    if (file_info_list->isEmpty())
    {
        return(IO_INTF_OK);
    }

    Message = QObject::tr("Delete selected files?");
    mbret = showYesNoAllMsgBox(QObject::tr("Delete selected local files and directories?"), Message);
    if (mbret == QMessageBox::No)
        return(IO_INTF_USER_CANCEL);

    for (fi_iter = file_info_list->begin(); fi_iter != file_info_list->end(); ++fi_iter)
    {
        lastErrString = "";
        fi_iter->setProcessed(FALSE);
        filename = fi_iter->FileName();
        ret = internal_norm_ProtocolStr(filename, "file://");
        if (ret != IO_INTF_OK)
        {
            lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(filename));
            return(-1);
        }
        if (fi_iter->FileType() == IO_INTF_DIR)
        {
            // Delete Directory
            QDir dir(filename);
            if (dir.count() > 2)
            {
                lastErrString = QObject::tr("Directory not empty  %1").arg(filename);
            }
            else
            {
                if (mbret != QMessageBox::YesAll)
                {
                    Message = QObject::tr("Delete local directory  %1?").arg(filename);
                    mbret = showYesNoAllMsgBox(QObject::tr("Delete local directory"), Message);
                }
                if ((mbret == QMessageBox::Yes) || (mbret == QMessageBox::YesAll))
                {
                    if (!dir.rmdir(filename))
                    {
                        lastErrString = QObject::tr("Could not delete directory  %1").arg(filename);
                    }
                    else
                    {
                        fi_iter->setProcessed(TRUE);
                    }
                }
                else if (mbret != QMessageBox::NoAll)
                {
                    break;
                }
            }
        }
        else // IO_INTF_FILE, IO_INTF_SYMLINK, IO_INTF_SPECIAL
        {
            // Delete File
            if (mbret != QMessageBox::YesAll)
            {
                Message = QObject::tr("Delete local file  %1?").arg(filename);
                mbret = showYesNoAllMsgBox(QObject::tr("Delete local file"), Message);
            }
            if ((mbret == QMessageBox::Yes) || (mbret == QMessageBox::YesAll))
            {
                QFile file(filename);
                if (!file.remove())
                {
                    lastErrString = QObject::tr("Could not delete file  %1").arg(filename);
                    //lastErrString = file.errorString();
                }
                else
                {
                    fi_iter->setProcessed(TRUE);
                }
            }
            else if (mbret != QMessageBox::NoAll)
            {
                break;
            }
        }
        if (lastErrString != "")
        {
            // dispaly error
            showInfoMsgBox(QObject::tr("Delete Status"), lastErrString);
        }
    }
    return(IO_INTF_OK);
}

int FileInterface::exec_ShowProperties(const QString& fileName)
{
    if (fileName == "")
    {
        // satisfy compilier
    }
    return(IO_INTF_OK);
}

int FileInterface::exec_GetDirectoryListing(QString dirName, QList<FileInfos> *file_info_list)
{
    int ret, cnt;
    QFileInfoList files;

    if (file_info_list == NULL)
    {
        return(IO_INTF_OK);
    }

    file_info_list->clear();

    // remove the protocol info
    ret = internal_norm_ProtocolStr(dirName, "file://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(dirName));
        return(-1);
    }

    QDir thisDir(dirName);
    if (!thisDir.isReadable())
    {
        return(IO_INTF_OK);
    }

    processingFiles = TRUE;
    files = thisDir.entryInfoList();

    for(cnt = 0; cnt < files.size(); cnt++)
    {
        QFileInfo fi = files.at(cnt);
        if (fi.fileName() == "." || fi.fileName() == "..")
            continue; // nothing

        // fill in info
        // check for directories before symlinks, so we catch symlinks to directories
        if (fi.isDir())
        {
            if (fi.isReadable())
            {
                file_info_list->append(FileInfos(fi.fileName(), IO_INTF_DIR, 0, NULL));
            }
            else
            {
                file_info_list->append(FileInfos(fi.fileName(), IO_INTF_DIR_LOCKED, 0, NULL));
            }
        }
        else if (fi.isSymLink())
        {
            file_info_list->append(FileInfos(fi.fileName(), IO_INTF_SYMLINK, 0, NULL));
        }
        else
        {
            if (fi.isFile())
            {
                file_info_list->append(FileInfos(fi.fileName(), IO_INTF_FILE, fi.size(), NULL));
            }
            else
            {
                file_info_list->append(FileInfos(fi.fileName(), IO_INTF_SPECIAL, fi.size(), NULL));
            }
        }
    }
    processingFiles = FALSE;
    return(IO_INTF_OK);
}

bool FileInterface::get_isReadable(QString& dirName)
{
    int ret;
    // remove the protocol info
    ret = internal_norm_ProtocolStr(dirName, "file://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Local file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(dirName));
        return(false);
    }
    return (QDir(dirName).isReadable());
}

QString FileInterface::get_LastError()
{
    return(lastErrString);
}


/**
 * iFPInterface
 * Functions to implement the file io routines to interact with the iFP device.
 */
int iFPInterface::ifp_connect(bool showError)
{
    int ret;
    QString message;
    TiFPreadFileTree t_ifpTree;

    ret = -1;
    if (ifp_dh == NULL)
    {
        ret = initialize_ifp_device();
        if (errorInLibrary)
        {
            message = QObject::tr("The ifplib Library is missing one or more functions.\n"
                                  "Please get the latest version from http://ifp-driver.sourcefourge.net.\n\n"
                                  "USING THE INTERNAL STATIC VERSIONS OF THE FUNCTIONS.");
            showInfoMsgBox(QObject::tr("Error"), message);
        }
    }
    else
    {
        ret = IO_INTF_OK;
    }

    if (ret == IO_INTF_OK)
    {
        processingFiles = TRUE;
        if ((ret = t_ifpTree.iFPreadFileTree()) != IO_INTF_OK)
        {
            showInfoMsgBox(QObject::tr("Error"), iFPNotConnectedMsg);
            fprintf(stderr, "Could not communicate with iRiver device.\n");
            fprintf(stderr, "Try unplugging the USB link, powering off the device, and plugging it back in.\n");
            processingFiles = FALSE;
            return(ret);
        }
        processingFiles = FALSE;
    }
    else
    {
        if (showError)
        {
            showInfoMsgBox(QObject::tr("Error"), iFPNotConnectedMsg);
            fprintf(stderr, "ifp Device NOT found\n");
        }
    }
    return(ret);
}


/**
 * Upload the files from iFP to local drive
 * @param file_info_list the list of filenames on the iFP device.
 * @param destination the local destination directory.
 * @return IFP_INTF_OK on success (0).  Otherwise return an error code.
 */
int iFPInterface::exec_TransferUpload(QString& destination, QList<FileInfos> *file_info_list)
{
    return(internal_exec_TransferFromiFP(destination, file_info_list));
}
/**
 * Download the files from the local drive to the iFP device
 * @param file_info_list the list of filenames on the local hard drive.
 * @param destination the iFP destination directory.
 * @return IFP_INTF_OK on success (0).  Otherwise return an error code.
 */
int iFPInterface::exec_TransferDownload(QString& destination, QList<FileInfos> *file_info_list)
{
    return(internal_exec_TransferToiFP(destination, file_info_list));
}

int iFPInterface::exec_Mkdir(QString& baseDir, QString& newDir)
{
    int ret, tret;
    QString Message;
    QDir dirObj;
    QByteArray qcfilename;
    TiFPreadFileTree t_ifpTree;

    if (ifp_dh == NULL)
    {
        lastErrString = QObject::tr("iRiver device not connected.");
        return(-1);
    }
    ret = internal_norm_ProtocolStr(baseDir, "ifp://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Remote file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(baseDir));
        return(-1);
    }

    CreateDirWindow *mkdirdlg;

    lastErrString = "";
    mkdirdlg = new CreateDirWindow(QObject::tr("Create a directory on iRiver device."));
    mkdirdlg->setBaseDir(baseDir);

    ret = mkdirdlg->exec();
    newDir = mkdirdlg->getNewDirName();
    delete mkdirdlg;

    if (ret == QDialog::Accepted)
    {
        processingFiles = TRUE;
        slash2backslash(newDir);
        qcfilename = newDir.toUtf8();
        ret = ifpgui_ifp_mkdir(&ifp_dev, qcfilename);
        if (ret != IO_INTF_OK)
        {
            lastErrString = QString("Could not make directory [%1] on iRiver device.").arg(newDir);
            ret = -1;
        }
        else
        {
            processingFiles = TRUE;
            tret = t_ifpTree.iFPreadFileTree();
            if (tret != IO_INTF_OK)
            {
                Message = QObject::tr("Could not get the file list.  Check your battery.  Try replugging the USB cable.");
                showInfoMsgBox(QObject::tr("iRiver Upload File Status"), Message);
            }
            ret = IO_INTF_OK;
        }
        processingFiles = FALSE;
    }
    else
    {
        ret = IO_INTF_USER_CANCEL;
    }
    return(ret);
}

int iFPInterface::exec_Rename(const QString& oldName, const QString& newName)
{
    int ret, tret;
    QString Message;
    QByteArray qcfilename_old, qcfilename_new;
    QString tmp_oldName, tmp_newName;
    TiFPreadFileTree t_ifpTree;

    if (ifp_dh == NULL)
    {
        lastErrString = QObject::tr("iRiver device not connected.");
        return(-1);
    }

    tmp_oldName = oldName;
    tmp_newName = newName;

    ret = internal_norm_ProtocolStr(tmp_oldName, "ifp://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Remote file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(tmp_oldName));
        return(-1);
    }
    ret = internal_norm_ProtocolStr(tmp_newName, "ifp://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Remote file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(tmp_newName));
        return(-1);
    }

    processingFiles = TRUE;
    slash2backslash(tmp_oldName);
    slash2backslash(tmp_newName);
    qcfilename_old = tmp_oldName.toUtf8();
    qcfilename_new = tmp_newName.toUtf8();
    ret = ifpgui_ifp_rename(&ifp_dev, qcfilename_old, qcfilename_new);
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Could not rename  %1  to  %2  on iRiver device.").arg(oldName).arg(newName);
        ret = -1;
    }
    else
    {
        processingFiles = TRUE;
        tret = t_ifpTree.iFPreadFileTree();
        if (tret != IO_INTF_OK)
        {
            Message = QObject::tr("Could not get the file list.  Check your battery.  Try replugging the USB cable.");
            showInfoMsgBox(QObject::tr("iRiver Upload File Status"), Message);
        }
        ret = IO_INTF_OK;
    }
    processingFiles = FALSE;
    return(ret);
}

int iFPInterface::exec_Delete(QList<FileInfos> *file_info_list)
{
    int mbret, ret;
    QByteArray qcfilename;
    QString Message, filename;
    QList<FileInfos>::iterator fi_iter;
    TiFPreadFileTree t_ifpTree;

    if (ifp_dh == NULL)
    {
        lastErrString = QObject::tr("iRiver device not connected.");
        return(-1);
    }

    if (file_info_list == NULL)
    {
        return(IO_INTF_OK);
    }
    if (file_info_list->isEmpty())
    {
        return(IO_INTF_OK);
    }

    Message = QObject::tr("Delete selected files on iRiver device?");
    mbret = showYesNoAllMsgBox(QObject::tr("Delete selected iRiver files and directories?"), Message);
    if (mbret == QMessageBox::No)
        return(IO_INTF_USER_CANCEL);

    processingFiles = TRUE;
    for (fi_iter = file_info_list->begin(); fi_iter != file_info_list->end(); ++fi_iter)
    {
        lastErrString = "";
        fi_iter->setProcessed(FALSE);
        // remove the protocol info
        filename = fi_iter->FileName();
        ret = internal_norm_ProtocolStr(filename, "ifp://");
        if (ret != IO_INTF_OK)
        {
            lastErrString = QObject::tr("Remote file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(filename));
            processingFiles = FALSE;
            return(-1);
        }
        slash2backslash(filename);
        // get the file info
        if (fi_iter->FileType() == IO_INTF_DIR)
        {
            // Delete Directory
            if (mbret != QMessageBox::YesAll)
            {
                Message = QObject::tr("Delete iRiver directory  %1?").arg(filename);
                mbret = showYesNoAllMsgBox(QObject::tr("Delete iRiver directory"), Message);
            }
            if ((mbret == QMessageBox::Yes) || (mbret == QMessageBox::YesAll))
            {
                qcfilename = filename.toUtf8();
                ret = ifpgui_ifp_delete_dir_recursive(&ifp_dev, qcfilename);
                if (ret != IO_INTF_OK)
                {
                    lastErrString = QString(ifpgui_ifp_error_message(ret));
                }
                else
                {
                    fi_iter->setProcessed(TRUE);
                }
            }
            else if (mbret != QMessageBox::NoAll)
            {
                break;
            }
        }
        else // IO_INTF_FILE, IO_INTF_SYMLINK, IO_INTF_SPECIAL
        {
            // Delete File
            if (mbret != QMessageBox::YesAll)
            {
                Message = QObject::tr("Delete iRiver file?   %1").arg(filename);
                mbret = showYesNoAllMsgBox(QObject::tr("Delete iRiver file"), Message);
            }
            if ((mbret == QMessageBox::Yes) || (mbret == QMessageBox::YesAll))
            {
                qcfilename = filename.toUtf8();
                ret = ifpgui_ifp_delete(&ifp_dev, qcfilename);
                if (ret != IO_INTF_OK)
                {
                    lastErrString = QString(ifpgui_ifp_error_message(ret));
                }
                else
                {
                    fi_iter->setProcessed(TRUE);
                }
            }
            else if (mbret != QMessageBox::NoAll)
            {
                break;
            }
        }
        if (lastErrString != "")
        {
            // dispaly error
            showInfoMsgBox(QObject::tr("iRiver Delete Status"), lastErrString);
        }
    }
    t_ifpTree.iFPreadFileTree();
    processingFiles = FALSE;
    return(IO_INTF_OK);
}

int iFPInterface::exec_ShowProperties(const QString& fileName)
{
    if (ifp_dh == NULL)
    {
        lastErrString = QObject::tr("iRiver device not connected.");
        return(-1);
    }

    if (fileName == "")
    {
        // satisfy compilier
    }
    return(IO_INTF_OK);
}

int iFPInterface::exec_GetDirectoryListing(QString dirName, QList<FileInfos> *file_info_list)
{
    int ret;
    TiFPgetFileList t_fileList;

    if (ifp_dh == NULL)
    {
        ret = ifp_connect(TRUE);
        if (ret != IO_INTF_OK)
        {
            lastErrString = QObject::tr("iRiver device not connected.");
            return(ret);
        }
    }

    if (file_info_list == NULL)
    {
        return(IO_INTF_OK);
    }

    file_info_list->clear();

    // remove the protocol info
    ret = internal_norm_ProtocolStr(dirName, "ifp://");
    if (ret != IO_INTF_OK)
    {
        lastErrString = QObject::tr("Remote file protocol not supported \"%1\"").arg(internal_get_ProtocolStr(dirName));
        return(-1);
    }

    slash2backslash(dirName);
    processingFiles = TRUE;
    ret = t_fileList.iFPgetFileList(dirName, file_info_list);
    processingFiles = FALSE;

    return(ret);
}

bool iFPInterface::get_isReadable(QString& dirName)
{
    if (dirName == "")
    {
        // satisfy compilier
    }
    return(true);
}

QString iFPInterface::get_LastError()
{
    return(lastErrString);
}

