/*
Copyright (C) 2002-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "stack.h"

void ** stack_base(stack *s)
{
  return (void **) ((unsigned long)s + (unsigned long)s->offset);
}

void stack_init(stack *s, int size, void **data)
{
  s->offset=(unsigned long)data-(unsigned long)s;
  *data=NULL;
  s->n=0;
  s->alloc=0;
  s->size=size;
}

void stack_alloc(stack *s, int nb)
{
  void **sdat=(void **)((char *)s+s->offset);
  if (s->n+nb <= s->alloc)
    return;
  while (s->n+nb > s->alloc)
    s->alloc=s->alloc?s->alloc<<1:1;
  if (debug_stack)
    fprintf(stderr,"data=%lx (%d/%ld)\n",
        (unsigned long) sdat,s->n,(long) s->alloc);
  *sdat=realloc(*sdat,s->alloc*s->size);
  if (*sdat==NULL)
    fprintf(stderr,"Cannot alloc memory (%ld bytes)\n",(long)s->alloc*s->size);
}

int stack_new(stack *s)
{
  stack_alloc(s, 1);
  return s->n++;
}

void stack_push_int(stack *s, int val)
{
  int *sdat;
  stack_alloc(s, 1);
  sdat = * (int **) ((char *)s+s->offset);
  sdat[s->n++] = val;
}

int stack_has_int(stack *s, int val)
{
  int *sdat= * (int **) ((char *)s+s->offset);
  int i;
  for (i=0; i<s->n; i++)
    if (sdat[i]==val)
      return 1;
  return 0;
}

void stack_remove_int(stack *s, int val)
{
  int *sdat= * (int **) ((char *)s+s->offset);
  int i, k;
  for (i=0, k=0; i<s->n; i++)
  {
    if (sdat[i]==val) k++;
    else sdat[i-k]=sdat[i];
  }
  s->n-=k;
}

static int cmpss(const void *a, const void *b)
{
  return *(int*)a - *(int*)b;
}

void stack_int_sort(stack *s)
{
  int *sdat= * (int **) ((char *)s+s->offset);
  qsort(sdat, s->n, s->size, cmpss);
}

void stack_int_merge(stack *s, stack *t)
{
  int *sdat= * (int **) ((char *)s+s->offset);
  int *tdat= * (int **) ((char *)t+t->offset);
  int *var;
  int i,j;
  int sn = s->n, tn = t->n;
  stack v;
  if (t->n==0) return;
  stack_init(&v,sizeof(*var),(void *)&var);
  for(i=0,j=0; i<sn && j<tn;)
  {
    if (sdat[i] < tdat[j])
      stack_push_int(&v,sdat[i++]);
    else if (sdat[i] > tdat[j])
      stack_push_int(&v,tdat[j++]);
    else
    {
      stack_push_int(&v,sdat[i]);
      i++; j++;
    }
  }
  for (;i<sn;)
    stack_push_int(&v,sdat[i++]);
  for (;j<tn;)
    stack_push_int(&v,tdat[j++]);
  stack_replace(s,&v);
}

void stack_pop(stack *s)
{
  if (s->n)
    s->n--;
  else
    fprintf(stderr,"stack_pop: stack is already empty\n");
}

void stack_pop_safe(stack *s, int n)
{
  if (n == s->n-1)
    stack_pop(s);
  else
    fprintf(stderr,"stack_pop: unexpected stack level\n");
}

/*Push stack t on stack s*/
void stack_push(stack *s, stack *t)
{
  if ( s->size!=t->size )
    fprintf(stderr,"Incompatible stack size in stack_push: %ld!=%ld\n",
            (long)s->size,(long)t->size);
  if ( t->n>0 )
  {
    void **sdat=(void **) ((char *)s+s->offset);
    void **tdat=(void **) ((char *)t+t->offset);
    stack_alloc(s, t->n);
    memcpy((char *)*(sdat)+s->n*s->size,*tdat,t->n*t->size);
    s->n+=t->n;
  }
}

void stack_replace(stack *s, stack *t)
{
  void **sdat=(void **) ((char *)s+s->offset);
  void **tdat=(void **) ((char *)t+t->offset);
  *sdat=realloc(*sdat,0);
  *sdat=*tdat;
  s->n=t->n;
  s->alloc=t->alloc;
}

