//go:build go1.23
// +build go1.23

package pkcs7

import (
	"encoding/base64"
	"testing"
)

func TestParseWindowsSCEPCertificateRequest(t *testing.T) {
	// base64 encoded SCEP requests from a Windows device
	b64 := `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`
	data, err := base64.StdEncoding.DecodeString(b64)
	if err != nil {
		t.Errorf("failed decoding base64 SCEP request data: %v", err)
	}

	p7, err := Parse(data)
	if err == nil {
		t.Errorf("expected error while parsing SCEP request data")
	}
	if err.Error() != "x509: authority key identifier incorrectly marked critical" {
		t.Errorf(`expected "x509: authority key identifier incorrectly marked critical"; got "%s"`, err.Error())
	}
	if p7 != nil {
		t.Errorf("expected p7 to be nil; got %v", p7)
	}

	// enable the legacy parser when Go 1.23 or newer is used, and parse it again
	SetFallbackLegacyX509CertificateParserEnabled(true)
	t.Cleanup(func() {
		SetFallbackLegacyX509CertificateParserEnabled(false)
	})
	p7, err = Parse(data)
	if err != nil {
		t.Errorf("failed parsing SCEP request data with legacy X509 certificate parser enabled: %v", err)
	}

	if len(p7.Certificates) != 1 {
		t.Errorf("expected a single certificate; got %d", len(p7.Certificates))
	}

	if cn := p7.Certificates[0].Subject.CommonName; cn != "SCEP Protocol Certificate" {
		t.Errorf("expected certificate subject common name to be SCEP Protocol Certificate; got %v", cn)
	}
}
