------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick.adb,v 1.25 2008/07/03 06:22:48 vagul Exp $

with
   Ada.Exceptions,
   Ada.Text_IO,
   OCI.Thread,
   OCI.Environments;

package body OCI.Thick is

   use type SWord;
   use type Ub4;
   use type OCIHandle;

   procedure Print_Error (Msg : String);

   procedure Check_Error
     (Code        : in SWord;
      Handle      : in OCIError;
      Htype       : in Integer;
      Raise_Error : in Boolean);

   ----------------------
   -- Alloc_Descriptor --
   ----------------------

   function Alloc_Descriptor
     (Parent : in OCIEnv; Htype : in Ub4) return OCIHandle
   is
      Result : aliased OCIHandle := Empty_Handle;
   begin
      if OCIDescriptorAlloc
           (Parenth => Parent,
            Descpp  => Result'Access,
            Htype   => Htype) /= OCI_SUCCESS
      then
         raise Invalid_Handle;
      end if;

      return Result;
   end Alloc_Descriptor;

   ------------------
   -- Alloc_Handle --
   ------------------

   function Alloc_Handle
     (Parent : in OCIEnv;
      Htype  : in Ub4)
      return OCIHandle
   is
      Result : aliased OCIHandle := Empty_Handle;
   begin
      if OCIHandleAlloc
           (Parenth => OCIHandle (Parent),
            Hndlpp  => Result'Access,
            Htype   => Htype) /= OCI_SUCCESS
      then
         raise Invalid_Handle;
      end if;

      return Result;
   end Alloc_Handle;

   -----------------
   -- Check_Error --
   -----------------

   procedure Check_Error
     (Code        : in SWord;
      Handle      : in OCIError;
      Htype       : in Integer;
      Raise_Error : in Boolean)
   is
      use Ada.Exceptions;
      Bufp : aliased C.char_array := (0 .. 2047 => C.nul);
      Errcodep : aliased Sb4 := 0;
      Rc : SWord;

      procedure Error (Except : Exception_Id; Msg : String);

      -----------
      -- Error --
      -----------

      procedure Error (Except : Exception_Id; Msg : String) is
      begin
         if Raise_Error then
            Raise_Exception (Except, Msg);
         else
            Print_Error (Msg);
         end if;
      end Error;

   begin
      case Code is
         when OCI_ERROR | OCI_NO_DATA | OCI_SUCCESS_WITH_INFO =>
            Rc := OCIErrorGet
              (Hndlp    => Handle,
               Errcodep => Errcodep'Access,
               Bufp     => CStr.To_Chars_Ptr (Bufp'Unchecked_Access),
               Bufsiz   => Bufp'Length - 1,
               Htype    => Ub4 (Htype));

            if Rc = OCI_SUCCESS then
               Error (Lib_Error'Identity, C.To_Ada (Bufp));
            else
               Error
                 (Lib_Error'Identity,
                  "Error code:" & SWord'Image (Code) & ASCII.LF &
                  "Return error code:" & SWord'Image (Rc) & ASCII.LF &
                  "Output error code:" & Sb4'Image (Errcodep) & ASCII.LF &
                  "Message:" & C.To_Ada (Bufp));
            end if;
         when OCI_INVALID_HANDLE =>
            Error (Invalid_Handle'Identity, "Invalid handle");
         when OCI_SUCCESS => null;
         when others =>
            Error
              (Constraint_Error'Identity, "Error code" & SWord'Image (Code));
      end case;
   end Check_Error;

   procedure Check_Error
     (Code            : in SWord;
      Raise_Exception : in Boolean := True) is
   begin
      Check_Error (Code, Thread.Error, OCI_HTYPE_ERROR, Raise_Exception);
   end Check_Error;

   -----------------
   -- Clear_Value --
   -----------------

   procedure Clear_Value (Var : in out Limited_Variable) is
   begin
      Var.Indicator := Null_Indicator;
   end Clear_Value;

   --------------------
   -- Client_Version --
   --------------------

   function Client_Version return String is
      Major_Version   : aliased SWord;
      Minor_Version   : aliased SWord;
      Update_Num      : aliased SWord;
      Patch_Num       : aliased SWord;
      Port_Update_Num : aliased SWord;

      function Image (Item : in SWord) return String;

      function Image (Item : in SWord) return String is
         Img : constant String := SWord'Image (Item);
      begin
         if Img (1) = ' ' then
            return Img (2 .. Img'Last);
         else
            return Img;
         end if;
      end Image;

   begin
      OCIClientVersion
        (Major_Version   => Major_Version'Access,
         Minor_Version   => Minor_Version'Access,
         Update_Num      => Update_Num'Access,
         Patch_Num       => Patch_Num'Access,
         Port_Update_Num => Port_Update_Num'Access);

      return Image (Major_Version)
         & '.' & Image (Minor_Version)
         & '.' & Image (Update_Num)
         & '.' & Image (Patch_Num)
         & '.' & Image (Port_Update_Num);
   end Client_Version;

   ----------
   -- Free --
   ----------

   procedure Free (H : in out OCIHandle; HType : Ub4) is
      Rc : SWord;
   begin
      if H = Empty_Handle then
         return;
      end if;

      Rc := OCIHandleFree (H, HType);
      H  := Empty_Handle;
      Check_Error (Rc, Raise_Exception => False);
   end Free;

   --------------
   -- Get_Attr --
   --------------

   function Get_Attr
     (Param : in OCIHandle;
      HType : in Ub4;
      Attr  : in Ub4) return String
   is
      Buff  : aliased CStr.chars_ptr;
      Rsize : aliased Ub4;
   begin
      Check_Error (Lib.OCIAttrGet
                      (Trgthndlp  => Param,
                       Trghndltyp => HType,
                       Attributep => Buff'Access,
                       Sizep      => Rsize'Access,
                       Attrtype   => Attr,
                       Errhp      => Thread.Error));

      if CStr."=" (Buff, CStr.Null_Ptr) then
         return "";
      else
         return C.To_Ada (CStr.Value (Buff, C.size_t (Rsize)), False);
      end if;
   end Get_Attr;

   function Get_Attr_G
     (H     : in OCIHandle;
      HType : in Ub4;
      Attr  : in Ub4) return Result_Type
   is
      Result : aliased Result_Type;
      Size   : aliased Ub4 := Result_Type'Size / Ub1'Size;
   begin
      Check_Error (OCIAttrGet
                     (Trgthndlp  => H,
                      Trghndltyp => HType,
                      Attributep => Result'Address,
                      Sizep      => Size'Unchecked_Access,
                      Attrtype   => Attr,
                      Errhp      => Thread.Error));

      --  Looks like Oracle does not get/set the attribute sizes for scalar,
      --  but anyway we use to check. So we do everithing possible for
      --  safety programming.

      if Size /= Result_Type'Size / Ub1'Size then
         raise Program_Error with
           "Attribute size" & Ub4'Image (Size) & " /="
           & Ub4'Image (Result_Type'Size / Ub1'Size);
      end if;

      return Result;
   end Get_Attr_G;

   ------------
   -- Handle --
   ------------

   function Handle (Ref : Handle_Reference'Class) return OCIHandle is
   begin
      return Ref.Handle;
   end Handle;

   ------------------
   -- Ignore_Error --
   ------------------

   procedure Ignore_Error (Code : in SWord) is
      pragma Unreferenced (Code);
   begin
      null;
   end Ignore_Error;

   -----------------
   -- Is_Attached --
   -----------------

   function Is_Attached (Var : Limited_Variable) return Boolean is
   begin
      return Is_Binded (Var) or Is_Defined (Var);
   end Is_Attached;

   ---------------
   -- Is_Binded --
   ---------------

   function Is_Binded (Var : in Limited_Variable) return Boolean is
   begin
      return Var.Bind /= OCIBind (Empty_Handle);
   end Is_Binded;

   ----------------
   -- Is_Defined --
   ----------------

   function Is_Defined (Var : in Limited_Variable) return Boolean is
   begin
      return Var.Define /= OCIDefine (Empty_Handle);
   end Is_Defined;

   -------------
   -- Is_Null --
   -------------

   function Is_Null (Var : Limited_Variable) return Boolean is
      use type Sb2;
   begin
      if not Is_Attached (Var) then
         raise Not_Attached;
      end if;

      return Var.Indicator = Null_Indicator;
   end Is_Null;

   ------------------------
   -- Is_Objects_Support --
   ------------------------

   function Is_Objects_Support return Boolean is
   begin
      return Environments.Is_Objects;
   end Is_Objects_Support;

   ---------------------
   -- Last_Error_Code --
   ---------------------

   function Last_Error_Code return Integer is
      Errcodep : aliased Sb4 := 0;
   begin
      if OCIErrorGet
           (Hndlp    => Thread.Error,
            Errcodep => Errcodep'Access,
            Bufp     => CStr.Null_Ptr,
            Bufsiz   => 0,
            Htype    => OCI_HTYPE_ERROR) /= OCI_SUCCESS
      then
         null; -- ???
      end if;

      return Integer (Errcodep);
   end Last_Error_Code;

   ------------------------
   -- Last_Error_Message --
   ------------------------

   function Last_Error_Message (Record_No : in Integer := 1) return String is
      Errcodep : aliased Sb4 := 0;
      Bufp : aliased C.char_array := (0 .. 4095 => C.nul);
   begin
      if Lib.OCIErrorGet
           (Hndlp    => Thread.Error,
            Errcodep => Errcodep'Access,
            Recordno => Ub4 (Record_No),
            Bufp     => CStr.To_Chars_Ptr (Bufp'Unchecked_Access),
            Bufsiz   => Bufp'Length - 1,
            Htype    => OCI_HTYPE_ERROR) /= OCI_SUCCESS
      then
         null; -- ???
      end if;

      return C.To_Ada (Bufp);
   end Last_Error_Message;

   -----------------------
   -- Last_Error_Offset --
   -----------------------

   function Last_Error_Offset return Natural is
      function Get_Attrib is new Get_Attr_G (Ub4);
   begin
      return Natural (Get_Attrib
               (OCIHandle (Thread.Error),
                OCI_HTYPE_ERROR,
                OCI_ATTR_DML_ROW_OFFSET));
   end Last_Error_Offset;

   -----------------
   -- Print_Error --
   -----------------

   procedure Print_Error (Msg : String) is
      use Ada.Text_IO;
   begin
      Put_Line (Current_Error, Msg);
   end Print_Error;

   --------------
   -- Set_Attr --
   --------------

   procedure Set_Attr
     (H     : in OCIHandle;
      HType : in Ub4;
      Attr  : in Ub4;
      Value : in String) is
   begin
      Check_Error (OCIAttrSet
                      (Trgthndlp  => H,
                       Trghndltyp => HType,
                       Attributep => C.To_C (Value),
                       Size       => Value'Length,
                       Attrtype   => Attr,
                       Errhp      => Thread.Error));
   end Set_Attr;

   procedure Set_Attr
     (H     : in OCIHandle;
      HType : in Ub4;
      Attr  : in Ub4;
      Value : in OCIHandle) is
   begin
      Check_Error (OCIAttrSet
                     (Trgthndlp  => H,
                      Trghndltyp => HType,
                      Attributep => Value,
                      Size       => 0,
                      Attrtype   => Attr,
                      Errhp      => Thread.Error));
   end Set_Attr;

   ------------------------
   -- Set_Events_Support --
   ------------------------

   procedure Set_Events_Support is
   begin
      OCI.Environments.Set_Create_Mode_Flag (OCI_EVENTS);
   end Set_Events_Support;

   -------------------------
   -- Set_Objects_Support --
   -------------------------

   procedure Set_Objects_Support is
   begin
      OCI.Environments.Set_Create_Mode_Flag (OCI_OBJECT);
   end Set_Objects_Support;

   --------------------------------
   -- Task_Dedicated_Environment --
   --------------------------------

   procedure Task_Dedicated_Environment is
   begin
      if not Thread.Synch.Task_Dedicated_Environment then
         raise Too_Late_For_Dedicated_Environment;
      end if;
   end Task_Dedicated_Environment;

   ---------------
   -- Task_Done --
   ---------------

   procedure Task_Done (T : Task_Id := Current_Task) is
   begin
      Thread.Attributes.Reinitialize (T);
   end Task_Done;

end OCI.Thick;
