------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-lobs.ads,v 1.17 2006/12/11 04:31:47 vagul Exp $

with Ada.Streams;
with Ada.Strings.Unbounded;
with OCI.Thick.Connections;
with System.Address_To_Access_Conversions;

package OCI.Thick.Lobs is

   use Ada.Streams;
   use Connections;

   type Lob_Type is (Bin, File, Char, NChar);

   type Locator is new Handle_Reference with private;

   -------------------------------
   -- Ada like stream interface --
   -------------------------------

   type Lob_Stream is new Root_Stream_Type with private;

   procedure Stream (Stream : in out Lob_Stream'Class; Loc : in Locator);

   procedure Read
     (Stream : in out Lob_Stream;
      Item   :    out Stream_Element_Array;
      Last   :    out Stream_Element_Offset);

   procedure Write
     (Stream : in out Lob_Stream;
      Item   : in     Stream_Element_Array);

   function End_Of_File (Stream : in Lob_Stream) return Boolean;
   --  Use Last < Item'Last condition after Read call for better performance.

   --------------------------------
   -- Oracle like lob interface. --
   --------------------------------

   function Create
     (Connect : in Connection; Kind : in Lob_Type := Char) return Locator;
   --  For backward compartibility.

   function Create_Temporary
     (Connect : in Connection; Kind : in Lob_Type := Char) return Locator;

   procedure Create_Temporary
     (Loc : in out Locator; Cache : in Boolean := True);

   procedure Free_Temporary (Loc : in out Locator);

   type Count is new Natural;

   subtype Positive_Count is Count range 1 .. Count'Last;

   subtype Raw is Ada.Streams.Stream_Element_Array;
   subtype Raw_Offset is Ada.Streams.Stream_Element_Offset;

   procedure Read
     (Loc    :  in     Locator;
      Offset :  in     Positive_Count;
      Buffer :     out Raw;
      Last   :     out Raw_Offset);

   procedure Read
     (Loc    : in     Locator;
      Offset : in     Positive_Count;
      Buffer :    out String;
      Last   :    out Integer);

   procedure Write
     (Loc    : in out Locator;
      Offset : in     Positive_Count;
      Buffer : in     Raw);

   procedure Write
     (Loc    : in out Locator;
      Offset : in     Positive_Count;
      Buffer : in     String);

   procedure Trim (Loc : in out Locator; Length : in Count := 0);

   function Length (Loc : in Locator) return Count;

   function To_String (Loc : in Locator) return String;

   function To_Unbounded_String
     (Loc : in Locator)
      return Ada.Strings.Unbounded.Unbounded_String;

   function Is_Created   (Value : in Locator) return Boolean;
   function Is_Init      (Value : in Locator) return Boolean;
   function Is_Open      (Value : in Locator) return Boolean;
   function Is_Temporary (Value : in Locator) return Boolean;

   procedure Create_Temporary
      (Stream  : in out Lob_Stream;
       Connect : in     Connection;
       Kind    : in     Lob_Type := Char);

   procedure Free_Temporary (Stream : in out Lob_Stream);

   function To_String (Loc : in Lob_Stream) return String;

   procedure Reset (Stream : in out Lob_Stream);

   procedure Set_Index
     (Stream : in out Lob_Stream; Offset : in Positive_Count);

   function Get_Lob_Type (Loc : Locator) return Lob_Type;

private

   use Reference_Finalization;

   pragma Inline (To_String);

   package AC is new System.Address_To_Access_Conversions (OCIHandle);

   type Locator is new Handle_Reference with record
      Kind    : Lob_Type;
      Connect : Connection;
   end record;

   procedure Destroy (Object : in out Locator);

   type Lob_Stream is new Root_Stream_Type with record
     Loc : Locator;
     Position : Positive_Count := Positive_Count'First;
   end record;

end OCI.Thick.Lobs;
