# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).

from textwrap import dedent

from gi.repository import Gtk

from gdp import (
    config,
    Config,
    format,
    )
from gdp.format import (
    CheckerWorker,
    FilteredReporter,
    Formatter,
    )
from testing import (
    GeditTestCase,
    MockMethod,
    )


class TestFilteredReporter(FilteredReporter):

    def _message_console(self, line_no, message, icon=None,
                         base_dir=None, file_name=None):
        # Suppress printing to stdout.
        source = (base_dir, file_name)
        if file_name is not None and source != self._last_file_name:
            self._last_file_name = source


class FilteredReporterTestCase(GeditTestCase):

    def test_init(self):
        reporter = TestFilteredReporter(FilteredReporter.CONSOLE)
        self.assertIs(FilteredReporter.CONSOLE, reporter.report_type)

    def test_call(self):
        reporter = TestFilteredReporter(FilteredReporter.CONSOLE)
        original_value = config.getboolean('formatter', 'report_only_errors')
        self.addCleanup(
            config.set, 'formatter', 'report_only_errors',
            str(original_value))
        config.set('formatter', 'report_only_errors', str(False))
        reporter(12, "test", icon='info', base_dir='./lib', file_name='eg.py')
        self.assertIs(1, reporter.call_count)
        self.assertEqual(('./lib', 'eg.py'), reporter._last_file_name)

    def test_call_error_only(self):
        reporter = TestFilteredReporter(FilteredReporter.CONSOLE)
        original_value = config.getboolean('formatter', 'report_only_errors')
        self.addCleanup(
            config.set, 'formatter', 'report_only_errors',
            str(original_value))
        config.set('formatter', 'report_only_errors', str(True))
        reporter(12, "test", icon='info', base_dir='./lib', file_name='eg.py')
        self.assertIs(0, reporter.call_count)
        reporter(9, "test", icon='error', base_dir='./lib', file_name='eg.py')
        self.assertIs(1, reporter.call_count)

    def test_error_only(self):
        reporter = TestFilteredReporter(FilteredReporter.CONSOLE)
        original_value = config.getboolean('formatter', 'report_only_errors')
        self.addCleanup(
            config.set, 'formatter', 'report_only_errors',
            str(original_value))
        config.set('formatter', 'report_only_errors', str(True))
        self.assertIs(True, reporter.error_only)
        config.set('formatter', 'report_only_errors', str(False))
        self.assertIs(False, reporter.error_only)

    def test_append_issue(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        formatter = Formatter(window)
        model = formatter.file_lines_view.get_model()
        model.clear()
        reporter = TestFilteredReporter(
            FilteredReporter.FILE_LINES, treeview=formatter.file_lines_view)
        result = reporter.append_issue(
            None, 'eg.txt', 'error', 1, 'message', './')
        self.assertIs(False, result)
        return
        # This is slow but interesting.
        while Gtk.events_pending():
            Gtk.main_iteration()
        lines = []
        for line_match in model:
            lines.append((line_match[2], line_match[3]))
        self.assertEqual([(1, 'message')], lines)


class CheckerWorkerTestCase(GeditTestCase):

    def test_init(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        formatter = Formatter(window)
        formatter.reporter = FilteredReporter(
            FilteredReporter.COLLECTOR, treeview=formatter.file_lines_view)
        checker_worker = CheckerWorker(
            [document], formatter.reporter,
            formatter.on_check_style_complete, True)
        self.assertEqual([document], checker_worker.documents)
        self.assertIs(formatter.reporter, checker_worker.reporter)
        self.assertEqual(
            formatter.on_check_style_complete, checker_worker.callback)
        self.assertIs(True, checker_worker.quiet)
        self.assertIs(
            formatter.file_lines_view.get_model(), checker_worker.model)

    def test_check_style(self):
        # A general test of check_style.
        self.make_config(format, Config)
        css = '\n'.join([
            'div {',
            '    margins: 5px;',
            '    }',
            ])
        css_file = self.make_file(css, suffix='.css')
        window, view, document = self.make_gedit(css_file.name)
        formatter = Formatter(window)
        formatter.reporter = FilteredReporter(
            FilteredReporter.COLLECTOR, treeview=formatter.file_lines_view)
        checker_worker = CheckerWorker(
            [document], formatter.reporter,
            formatter.on_check_style_complete, True)
        checker_worker.run()
        lines = [message for message in formatter.reporter.messages]
        self.assertEqual(
            [(2, 'Unknown Property name.: margins')], lines)

    def test_check_style_show_errors_only(self):
        # A general test of check_style.
        self.make_config(format, Config)
        css = '\n'.join([
            'div {',
            '    margins: 5px;',
            '    }',
            ])
        css_file = self.make_file(css, suffix='.css')
        window, view, document = self.make_gedit(css_file.name)
        formatter = Formatter(window)
        menu_item = Gtk.CheckMenuItem(label='test')
        menu_item.set_active(True)
        formatter.on_show_syntax_errors_only_toggled(menu_item)
        formatter.reporter = FilteredReporter(
            FilteredReporter.COLLECTOR, treeview=formatter.file_lines_view)
        checker_worker = CheckerWorker(
            [document], formatter.reporter,
            formatter.on_check_style_complete, True)
        checker_worker.run()

        lines = [message for message in formatter.reporter.messages]
        self.assertEqual([], lines)

    def test_check_style_without_file(self):
        self.make_config(format, Config)
        python_file = self.make_file('import os; os\n', suffix='.py')
        window, view, document = self.make_gedit(python_file.name)
        # The temp file was removed after the helper loaded the file.
        formatter = Formatter(window)
        formatter.reporter = FilteredReporter(
            FilteredReporter.COLLECTOR, treeview=formatter.file_lines_view)
        checker_worker = CheckerWorker(
            [document], formatter.reporter,
            formatter.on_check_style_complete, True)
        checker_worker.run()
        lines = [message for message in formatter.reporter.messages]
        self.assertEqual(
            [(1, 'E702 multiple statements on one line (semicolon)')], lines)


class FormatterTestCase(GeditTestCase):

    def test_tabs_to_spaces(self):
        # Convert tabs to spaces.
        self.make_gsettings()
        text = '\n'.join([
            "line with end tab\t",
            "\tline with start tab",
            "\t\tline with two start tabs",
            ])
        text_file = self.make_file(text)
        window, view, document = self.make_gedit(text_file.name)
        formatter = Formatter(window)
        formatter.tabs_to_spaces(None)
        tab_size = {'tab_size': ' ' * formatter.get_tab_size()}
        expected = '\n'.join([
            "line with end tab%(tab_size)s",
            "%(tab_size)sline with start tab",
            "%(tab_size)s%(tab_size)sline with two start tabs",
            ]) % tab_size
        self.assertEqual(expected, formatter.text)

    def test_reformat_css(self):
        window, view, document = self.make_gedit('plugins/gdp/data/snark.css')
        formatter = Formatter(window)
        formatter.reformat_css(None)
        expected = dedent("""\
            .snark {
                font-style: italic;
                color: #ff0000;
                }
            .baker {
                font-style: italic;
                color: #3333ff;
                }
            .banker {
                font-style: italic;
                color: #666;
                }
            .captain {
                font-crack: italic;
                color: none;
                }""")
        self.assertEqual(expected, formatter.text)

    def test_reformat_doctest(self):
        text = '\n'.join([
            "narrative",
            ">>> 1 == 2",
            "False",
            ])
        text_file = self.make_file(text)
        window, view, document = self.make_gedit(text_file.name)
        formatter = Formatter(window)
        formatter.reformat_doctest(None)
        expected = dedent("""\
            narrative

                >>> 1 == 2
                False

            """)
        self.assertEqual(expected, formatter.text)

    def test_rewrap_text_language_78(self):
        # Known formats/languages are wrapped at 78 characters.
        long_text = (
            "If-and the thing is wildly possible-the charge of writing "
            "nonsense were ever brought against the author of this\nbrief "
            "but instructive poem")
        text_file = self.make_file(long_text, suffix='.py')
        window, view, document = self.make_gedit(text_file.name)
        formatter = Formatter(window)
        formatter.rewrap_text(None)
        expected = (
            "If-and the thing is wildly possible-the charge of writing "
            "nonsense were ever\nbrought against the author of this brief "
            "but instructive poem")
        self.assertEqual(expected, formatter.text)

    def test_rewrap_text_text_72(self):
        # Unnown formats and text are wrapped at 72 characters for email.
        long_text = (
            "If-and the thing is wildly possible-the charge of writing "
            "nonsense were ever brought against the author of this\nbrief "
            "but instructive poem")
        text_file = self.make_file(long_text)
        window, view, document = self.make_gedit(text_file.name)
        formatter = Formatter(window)
        formatter.rewrap_text(None)
        expected = (
            "If-and the thing is wildly possible-the charge of writing "
            "nonsense were\never brought against the author of this brief "
            "but instructive poem")
        self.assertEqual(expected, formatter.text)

    def test_rewrap_text_text_hyphens(self):
        # Hyphens are not split for wrapping.
        long_text = (
            "If-and the thing is wildly possible-the charge of writing "
            "nonsense were-ever brought against the author of this\nbrief "
            "but instructive poem")
        text_file = self.make_file(long_text)
        window, view, document = self.make_gedit(text_file.name)
        formatter = Formatter(window)
        formatter.rewrap_text(None)
        expected = (
            "If-and the thing is wildly possible-the charge of writing "
            "nonsense\nwere-ever brought against the author of this brief "
            "but instructive poem")
        self.assertEqual(expected, formatter.text)

    def test_on_show_syntax_errors_only_toggled(self):
        # Setup a test config so that the singleton is not mutated.
        config_file = self.make_config(format, Config)
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        formatter = Formatter(window)
        menu_item = Gtk.CheckMenuItem(label='test')
        menu_item.set_active(True)
        formatter.on_show_syntax_errors_only_toggled(menu_item)
        self.assertIs(
            True, format.config.getboolean('formatter', 'report_only_errors'))
        self.assertIn('report_only_errors = True', config_file.read())
        menu_item.set_active(False)
        formatter.on_show_syntax_errors_only_toggled(menu_item)
        self.assertIs(
            False,
            format.config.getboolean('formatter', 'report_only_errors'))

    def test_check_style_already_running(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        formatter = Formatter(window)
        formatter.checker_id = 1
        formatter.checker_worker = 'Bogus'
        formatter.check_style(None)
        self.assertEqual('Bogus', formatter.checker_worker)

    def test_check_style_background(self):
        # Background style checking does not show the panel when there
        # are no issues.
        css_file = self.make_file("", suffix='.css')
        window, view, document = self.make_gedit(css_file.name)
        formatter = Formatter(window)
        MockMethod.bind(self, formatter, 'check_style')
        formatter.check_style_background(None)
        self.assertIs(True, formatter.check_style.called)
        self.assertEqual({'quiet': True}, formatter.check_style.kwargs)

    def test_do_check_style_complete_quiet_show(self):
        # Background style checking does not show the panel when there
        # are no issues.
        css_file = self.make_file("", suffix='.css')
        window, view, document = self.make_gedit(css_file.name)
        formatter = Formatter(window)
        MockMethod.bind(self, formatter, 'show')
        formatter.file_lines_view.get_model().clear()
        formatter.do_check_style_complete(True)
        self.assertIs(False, formatter.show.called)

    def test_do_check_style_complete_is_quiet(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        panel = window.get_side_panel()
        panel.props.visible = False
        formatter = Formatter(window)
        model = formatter.file_lines_view.get_model()
        model.clear()
        formatter.checker_worker = 'worker'
        formatter.checker_id = 1
        formatter.do_check_style_complete(True)
        self.assertIs(None, formatter.checker_worker)
        self.assertIs(None, formatter.checker_id)
        lines = []
        for line_match in model:
            lines.append((line_match[0], line_match[1]))
        self.assertEqual([('No problems found', 'emblem-default')], lines)
        self.assertIs(False, panel.props.visible)

    def test_do_check_style_complete_is_not_quiet(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        panel = window.get_side_panel()
        panel.props.visible = False
        formatter = Formatter(window)
        formatter.do_check_style_complete(True)
        self.assertIs(True, panel.props.visible)
