#!/usr/bin/perl

=head1 NAME

grip-dumpsingle.pl - dump details of a single package as a control file

=cut

use strict;
use warnings;
use File::Basename;
use POSIX qw(locale_h);
use Locale::gettext;
use Emdebian::Grip;
use Debian::Packages::Compare;

use vars qw/ $deb $suite $dir $ctrl $prog $our_version $base @sequence
 $grip_name $file @archlist $package $arch $name $cmpnt @suites $pkg /;

=head1 Synopsis

 emgrip-dumpsingle.pl -b|--base-path PATH [-s|--suite STRING] [-c|--component COMPONENT] [--grip-name STRING] PACKAGE ...
 emgrip-dumpsingle.pl -?|-h|--help|--version

 Commands:
 -b|--base-path PATH:      path to the top level repository directory [required]

 -?|-h|--help|--version:   print this help message and exit

 Options:
 -s|--suite STRING:        Name of the distribution to override [required]
    --grip-name STRING:    alternative name for the grip repository
 -c|--component COMPONENT: Section override from the Debian Packages file.

Prints data from the local grip repository for a single package in a
format similar to a standard debian/control file or apt-cache show
output.

Also adds the component in which the package was found.

Use C<--architecture source> to see details for the (unchanged) source
package.

The default suite is unstable.
 
Multiple package listings are separated by a blank line.
 
=cut

setlocale(LC_MESSAGES, "");
textdomain("emdebian-grip");

$prog = basename ($0);
$our_version = &scripts_version();
$grip_name = "grip";
$suite = "unstable";
$cmpnt = "main";
@sequence = qw/ Package Priority Section Installed-Size Maintainer
Architecture Source Version Replaces Provides Depends Recommends
Suggests Conflicts Breaks Filename Size MD5sum  SHA1 SHA256
Description Origin Bugs Homepage Build-Essential Python-Version /;

while( @ARGV ) {
	$_= shift( @ARGV );
	last if m/^--$/;
	if (!/^-/) {
		unshift(@ARGV,$_);
		last;
	}
	elsif (/^(-\?|-h|--help|--version)$/) {
		&usageversion();
		exit (0);
	}
	elsif (/^(-a|--architecture)$/) {
		$arch = shift;
	}
	elsif (/^(-b|--base-path)$/) {
		$base = shift;
	}
	elsif (/^(-s|--suite)$/) {
		$suite = shift;
	}
	elsif (/^(-c|--component)$/) {
		$cmpnt = shift;
	}
	elsif (/^(--grip-name)$/) {
		$grip_name = shift;
	}
	else {
		die "$prog: "._g("Unknown option")." $_.\n";
	}
}

die ("$prog: "._g("ERR: Please specify an existing directory for the base-path.\n"))
	if (not defined $base);

$base .= '/' if ("$base" !~ m:/$:);
if (not -d $base)
{
	my $msg = sprintf (_g("ERR: Please specify an existing directory for the base-path: %s\n"),$base);
	die ("$prog: ".$msg);
}

&set_base($base);
&set_repo_names ('filter', $grip_name);
my $s = &get_suite_names($grip_name);
@suites = @$s;
$arch = `dpkg-architecture -qDEB_HOST_ARCH 2>/dev/null`
	if ((not defined $arch) or ($arch eq "all"));
chomp ($arch);

die ("$prog: "._g("ERR: Specify a distribution name, not a codename. e.g. testing, not lenny.\n"))
	if ((not defined $suite) or 
	(scalar (grep (/$suite/, @suites)) == 0));

die ("$prog: "._g("ERR: Cannot find Grip configuration directory.\n"))
	if (not -d "${base}${grip_name}/conf/");
foreach $package (@ARGV)
{
	$pkg = &get_single_package($suite, $grip_name, $package, $arch);
	if (exists $$pkg{$cmpnt} and ($cmpnt ne "main"))
	{
		my $p = $$pkg{$cmpnt};
		$pkg = $p;
	}
	elsif ($cmpnt ne "main")
	{
		my $msg = sprintf (_g("%s not found in %s in %s.\n"),
			$package, $cmpnt, $suite);
		warn ("$prog: ".$msg);
		$cmpnt = "main";
	}
	if (not exists $$pkg{'Package'})
	{
		my $msg = sprintf (_g("Cannot find %s in %s {%s}.\n"),
			$package, $suite, $cmpnt);
		die ($msg);
	}
	foreach my $k (@sequence)
	{
		print "$k: ".$$pkg{$k}."\n" if (exists $$pkg{$k});
		if (($k eq "Description") and exists ($$pkg{$k}))
		{
			my @l = split ("\n", $$pkg{'body'});
			foreach my $l (@l)
			{
				print " $l\n";
			}
		}
	}
	print "Component: $cmpnt\n\n";
}
exit 0;

sub usageversion
{
	printf STDERR (_g("
%s - dump details of a single package as a control file
version %s

%s -b|--base-path PATH [-s|--suite STRING] [-c|--component COMPONENT] [--grip-name STRING] PACKAGE ...
%s -?|-h|--help|--version

Commands:
-b|--base-path PATH:       path to the top level repository directory [required]

-?|-h|--help|--version:    print this help message and exit

Options:
 -s|--suite STRING:         Name of the distribution to override [required]
    --grip-name STRING:    alternative name for the grip repository
 -c|--component COMPONENT: Section override from the Debian Packages file.

Prints data from the local grip repository for a single package in a
format similar to a standard debian/control file or apt-cache show
output.

Also adds the component in which the package was found.

Use '--architecture source' to see details for the (unchanged) source
package.

The default suite is unstable.

Multiple package listings are separated by a blank line.

"), $prog, $our_version, $prog, $prog)
	or die ("$0: "._g("failed to write usage").": $!\n");
}

=head1 Copyright and Licence

 Copyright (C) 2009  Neil Williams <codehelp@debian.org>

 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

=cut
