/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/name_separators.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref
 * diagnostics::unittest::name_adheres_convention and @ref
 * diagnostics::unittest::path_adheres_convention
 *
 * $Id: name_separators.t.cpp,v 1.8 2005/06/23 09:54:24 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/name_separators.hpp>


#define TEST_COMPONENT_NAMESPACE diagnostics::unittest
#define TEST_COMPONENT_NAME name_seperators

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

/**
 * @brief [PRO] fct/normal: usage of @ref name_adheres_convention
 */
void test_name_adheres_convention(Test_Data & test_data)
{
    ::std::string const cs(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR);
    ::std::string const ds(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR);
    ::std::string const a(DIAGNOSTICS_UNITTEST_TEST_CASE_ASTERISK);

    TEST_ASSERT(name_adheres_convention(""));

    TEST_ASSERT(name_adheres_convention("bla"));

    TEST_ASSERT(!name_adheres_convention(cs));
    TEST_ASSERT(!name_adheres_convention(ds));

    TEST_ASSERT(!name_adheres_convention("bla" + cs));
    TEST_ASSERT(!name_adheres_convention("bla" + ds));
    TEST_ASSERT(!name_adheres_convention("bla" + a));

    TEST_ASSERT(!name_adheres_convention(cs + "bla"));
    TEST_ASSERT(!name_adheres_convention(ds + "bla"));
    TEST_ASSERT(!name_adheres_convention(a + "bla"));

    TEST_ASSERT(!name_adheres_convention("xyz" + cs + "bla"));
    TEST_ASSERT(!name_adheres_convention("xyz" + ds + "bla"));
    TEST_ASSERT(!name_adheres_convention("xyz" + a + "bla"));

    // if this test fails, the implementation of name_adheres_convention must be changed!
    TEST_ASSERT(cs+cs==ds);
}


/**
 * @brief [PRO] fct/normal: usage of @ref path_adheres_convention
 */
void test_path_adheres_convention(Test_Data & test_data)
{
    ::std::string const cs(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR);
    ::std::string const ds(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR);
    ::std::string const a(DIAGNOSTICS_UNITTEST_TEST_CASE_ASTERISK);


    // correct
    TEST_ASSERT(path_adheres_convention(cs + "x"));
    TEST_ASSERT(path_adheres_convention(cs + "x" + cs + "x"));


    // ends with a DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
    TEST_ASSERT(!path_adheres_convention(cs));
    TEST_ASSERT(!path_adheres_convention(cs + "x" + cs));


    // no leading DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
    TEST_ASSERT(!path_adheres_convention(""));
    TEST_ASSERT(!path_adheres_convention("bla"));


    // two times DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
    TEST_ASSERT(!path_adheres_convention(cs + cs));
    
    // placement of DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR
    TEST_ASSERT(!path_adheres_convention(ds));
    TEST_ASSERT(!path_adheres_convention("bla" + ds));
    TEST_ASSERT(!path_adheres_convention(ds + "bla"));
    TEST_ASSERT(!path_adheres_convention("xyz" + ds + "bla"));
    TEST_ASSERT(!path_adheres_convention(cs + "bla" + ds));
    TEST_ASSERT(!path_adheres_convention(cs + ds + "bla"));
    TEST_ASSERT(!path_adheres_convention(cs + "xyz" + ds + "bla"));

    // placement of DIAGNOSTICS_UNITTEST_TEST_CASE_ASTERISK
    TEST_ASSERT(!path_adheres_convention(a));
    TEST_ASSERT(!path_adheres_convention("bla" + a));
    TEST_ASSERT(!path_adheres_convention(a + "bla"));
    TEST_ASSERT(!path_adheres_convention("xyz" + a + "bla"));
    TEST_ASSERT(!path_adheres_convention(cs + "bla" + a));
    TEST_ASSERT(!path_adheres_convention(cs + a + "bla"));
    TEST_ASSERT(!path_adheres_convention(cs + "xyz" + a + "bla"));

}

/**
 * @brief [PRO] fct/normal: usage of @ref mask_adheres_convention
 */
void test_mask_adheres_convention(Test_Data & test_data)
{
    ::std::string const cs(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR);
    ::std::string const ds(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR);
    ::std::string const a(DIAGNOSTICS_UNITTEST_TEST_CASE_ASTERISK);


    // correct
    TEST_ASSERT(mask_adheres_convention(cs + "x"));
    TEST_ASSERT(mask_adheres_convention(cs + "x" + cs + "x"));
    TEST_ASSERT(mask_adheres_convention(cs + a));
    TEST_ASSERT(mask_adheres_convention(cs + a + cs + "x"));
    TEST_ASSERT(mask_adheres_convention(cs + a + cs + a));
    TEST_ASSERT(mask_adheres_convention(cs + "x" + cs + a));


    // ends with a DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
    TEST_ASSERT(!mask_adheres_convention(cs));
    TEST_ASSERT(!mask_adheres_convention(cs + "x" + cs));
    TEST_ASSERT(!mask_adheres_convention(cs + a + cs));
    TEST_ASSERT(!mask_adheres_convention(cs + a + cs + "x" + cs));


    // no leading DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
    TEST_ASSERT(!mask_adheres_convention(""));
    TEST_ASSERT(!mask_adheres_convention("bla"));

    // two times DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
    TEST_ASSERT(!mask_adheres_convention(cs + cs));
    
    // placement of DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR
    TEST_ASSERT(!mask_adheres_convention(ds));
    TEST_ASSERT(!mask_adheres_convention("bla" + ds));
    TEST_ASSERT(!mask_adheres_convention(ds + "bla"));
    TEST_ASSERT(!mask_adheres_convention("xyz" + ds + "bla"));
    TEST_ASSERT(!mask_adheres_convention(cs + "bla" + ds));
    TEST_ASSERT(!mask_adheres_convention(cs + ds + "bla"));
    TEST_ASSERT(!mask_adheres_convention(cs + "xyz" + ds + "bla"));

    // placement of DIAGNOSTICS_UNITTEST_TEST_CASE_ASTERISK
    TEST_ASSERT(!mask_adheres_convention(a));
    TEST_ASSERT(!mask_adheres_convention("bla" + a));
    TEST_ASSERT(!mask_adheres_convention(a + "bla"));
    TEST_ASSERT(!mask_adheres_convention("xyz" + a + "bla"));
    TEST_ASSERT(!mask_adheres_convention(cs + "bla" + a));
    TEST_ASSERT(!mask_adheres_convention(cs + a + "bla"));
    TEST_ASSERT(!mask_adheres_convention(cs + "xyz" + a + "bla"));
    TEST_ASSERT(!mask_adheres_convention(cs + a + cs + a + "x"));

}


/**
 * @brief [PRO] impl
 */
void case_seperator_twice_is_data_seperator(Test_Data & test_data)
{
    ::std::string const cs(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR);
    ::std::string const ds(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR);
    ::std::string const a(DIAGNOSTICS_UNITTEST_TEST_CASE_ASTERISK);

    // if this test fails, the implementation of all three functions has to change
    TEST_ASSERT(cs+cs==ds);

    // if this test fails, the implementation of all three functions has to change
    TEST_ASSERT(cs.size()==1);

    // if this test fails, the implementation of all three functions has to change
    TEST_ASSERT(a.size()==1);
}



TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&test_name_adheres_convention,LEVEL_PROD);
TEST_NORMAL_CASE(&test_path_adheres_convention,LEVEL_PROD);
TEST_NORMAL_CASE(&test_mask_adheres_convention,LEVEL_PROD);
TEST_IMPL_CASE(&case_seperator_twice_is_data_seperator);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
