/*
 * Copyright (C) 2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <diagnostics/extensions/stacktrace/frame_visitor.hpp>
#include <diagnostics/extensions/stacktrace/posix.hpp>

#include <cstdlib>
#include <stdint.h>

#include <link.h>

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

namespace {
  enum {
    NO_DATA = 0,
    STOP_WALK = 1,
    CONTINUE_WALK = 2
  };

  struct state {
    uintptr_t original;
    Traceback::FrameVisitor *visitor;
  };
}

static std::string
get_executable_name_once()
{
  return POSIX::RealPath("/proc/self/exe");
}

static const char *
executable_name()
{
  static std::string name(get_executable_name_once());
  return name.c_str();
}

static int
callback(struct dl_phdr_info *info, size_t, void *data)
{
  state *pst = static_cast<state *>(data);
  uintptr_t original = pst->original;
  for (unsigned i =0; i < info->dlpi_phnum; ++i) {
    uintptr_t start = info->dlpi_addr + info->dlpi_phdr[i].p_vaddr;
    uintptr_t end = start + info->dlpi_phdr[i].p_memsz;
    if (start <= original && original < end) {
      uintptr_t mapped = original - info->dlpi_addr;
      const char *name = info->dlpi_name;
      if (name == 0 || *name == 0) {
        name = executable_name();
      }
      if (pst->visitor->Frame(reinterpret_cast<void *>(original), name,
                              reinterpret_cast<void *>(mapped))) {
        return CONTINUE_WALK;
      } else {
        return STOP_WALK;
      }
    }
  }
  // Keep scanning for a matching DSO.
  return NO_DATA;
}

bool
Traceback::FrameVisitor::frame(void *original)
{
  state st;
  st.original = reinterpret_cast<uintptr_t>(original);
  st.visitor = this;
  switch (dl_iterate_phdr(callback, &st)) {
  case NO_DATA:
    // Resolution failed.
    return Frame(original, 0, 0);
  case STOP_WALK:
    return false;
  case CONTINUE_WALK:
    return true;
  }
  abort();
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

