#include "TicTacToeGame.h"

// Constructor
TicTacToeGame::TicTacToeGame() {

	// Game is currently running
	alive = true;

	// Connect the mouse press signal to our handleMousePress function
	mousePress = CL_Mouse::sig_key_down().connect(this, &TicTacToeGame::handleMousePress);

	loadGraphics();
	resetGame();
}

// Destructor
TicTacToeGame::~TicTacToeGame() {

	// Delete the surfaces
	delete board;
	delete o;
	delete x;
	delete menu;
}

void TicTacToeGame::loadGraphics() {

	// create loads the file with the given name and stores it for us; we only
	// need a pointer to access it.
	board = new CL_Surface("board.tga");
	o = new CL_Surface("o.tga");
	x = new CL_Surface("x.tga");
	menu = new CL_Surface("menu.tga");
}

void TicTacToeGame::run() {

	// Loop while the game is running and the user hasn't pressed escape
	while (alive && !CL_Keyboard::get_keycode(CL_KEY_ESCAPE)) {

		// Game update function
		paint();

		CL_Display::flip();
		CL_System::sleep(10);
		CL_System::keep_alive();
	}
}

void TicTacToeGame::paint() {

	// First we want to clear the screen to a certain R, G, B color
	// 0, 0, 0, is black (255 is the alpha paramater - we want it full)
	// Black is the default clear color, so you could just use CL_Display::clear() here
	CL_Display::clear(CL_Color(0, 0, 0, 255));

	// Paint the board to the screen
	board->draw(75, 25);

	// Paint the 'restart / quit' menu below the board
	menu->draw(75, 400);

	// Loop through the cells
	for (int i=0; i < 3; i++) {
		for (int j=0; j < 3; j++) {

			// Draw the pieces based on which player owns the cell
			if (cells[i][j] == PLAYER_CELL)
				o->draw(100+i*CELLSIDE, 50+j*CELLSIDE);
			else if (cells[i][j] == COMPUTER_CELL)
				x->draw(100+i*CELLSIDE, 50+j*CELLSIDE);
		}
	}
}

void TicTacToeGame::handleMousePress(const CL_InputEvent &key) {

	// Since a clicked happened, get the screen coordinates of the mouse
	int x = key.mouse_pos.x, y = key.mouse_pos.y;

	int boardX = 100, boardY = 50;

	// See if the player has clicked on the board
	if (playing && x-boardX > 0 && x-boardX < 300 && y-boardY > 0 && y-boardY < 300) {

		// If the user has clicked, find out which cell
		int cellX = (x-boardX)/CELLSIDE, cellY = (y-boardY)/CELLSIDE;

		// If there's no piece there, claim it for the player
		if (cells[cellX][cellY] == BLANK_CELL)
			cells[cellX][cellY] = PLAYER_CELL;

		// Check to see if player won
		checkWinner();

		// Computer's turn
		if (playing) computerMove();

		// Check to see if computer won
		checkWinner();
	}

	// See if the user has clicked 'Restart'

	// 4 points forming a box around the 'Restart' label
	int restartX = 95, restartY = 400, restartWidth = 135, restartHeight = 20;

	// See if the x and y mouse coordinates lie inside that box
	if (x > restartX && x < restartX+restartWidth && y > restartY && y < restartY+restartHeight) {
		resetGame();
	}

	// See if the user has clicked 'Quit'

	// 4 points forming a box around the 'Quit' label
	int quitX = 300, quitY = 400, quitWidth = 85, quitHeight = 20;

	// See if the x and y mouse coordinates lie inside that box
	if (x > quitX && x < quitX+quitWidth && y > quitY && y < quitY+quitHeight) {
		alive = false;
	}
}

void TicTacToeGame::resetGame() {

	playing = true;

	// Zero the playing cells
	for (int i=0; i < 3; i++)
		for (int j=0; j < 3; j++)
			cells[i][j] = BLANK_CELL;
}

void TicTacToeGame::checkWinner() {

	// This function checks to see if there is a winner
	// If there is, it fills the field with that player's pieces
	int winner = findWinner();

	if (winner == PLAYER_CELL)
		for (int i=0; i < 3; i++)
			for (int j=0; j < 3; j++)
				cells[i][j] = PLAYER_CELL;
	else if (winner == COMPUTER_CELL)
		for (int i=0; i < 3; i++)
			for (int j=0; j < 3; j++)
				cells[i][j] = COMPUTER_CELL;

	// Someone has won, play is over
	if (winner != 0) playing = false;
}

int TicTacToeGame::findWinner() {

	// Similar to the computer's move function without piece placement
	// We search for runs of three for either player

	// Search vertical
	for (int i=0; i < 3; i++) {

		int playerPieceCount = 0;
		int computerPieceCount = 0;

		for (int j=0; j < 3; j++) {
			if (cells[i][j] == PLAYER_CELL)	playerPieceCount++;
			if (cells[i][j] == COMPUTER_CELL) computerPieceCount++;
		}

		if (playerPieceCount == 3) return PLAYER_CELL;
		if (computerPieceCount == 3) return COMPUTER_CELL;
	}

	// Search horizontal
	for (int j=0; j < 3; j++) {

		int playerPieceCount = 0;
		int computerPieceCount = 0;

		for (int i=0; i < 3; i++) {
			if (cells[i][j] == PLAYER_CELL)	playerPieceCount++;
			if (cells[i][j] == COMPUTER_CELL) computerPieceCount++;
		}

		if (playerPieceCount == 3) return PLAYER_CELL;
		if (computerPieceCount == 3) return COMPUTER_CELL;
	}

	// Search the two diagonals
	
	// Top left to bottom right
	int playerPieceCount = 0;
	int computerPieceCount = 0;

	if (cells[0][0] == PLAYER_CELL) playerPieceCount++;
	if (cells[1][1] == PLAYER_CELL) playerPieceCount++;
	if (cells[2][2] == PLAYER_CELL) playerPieceCount++;

	if (cells[0][0] == COMPUTER_CELL) computerPieceCount++;
	if (cells[1][1] == COMPUTER_CELL) computerPieceCount++;
	if (cells[2][2] == COMPUTER_CELL) computerPieceCount++;

	if (playerPieceCount == 3) return PLAYER_CELL;
	if (computerPieceCount == 3) return COMPUTER_CELL;

	// Top right to bottom left
	playerPieceCount = 0;
	computerPieceCount = 0;

	if (cells[0][2] == PLAYER_CELL) playerPieceCount++;
	if (cells[1][1] == PLAYER_CELL) playerPieceCount++;
	if (cells[2][0] == PLAYER_CELL) playerPieceCount++;

	if (cells[0][2] == COMPUTER_CELL) computerPieceCount++;
	if (cells[1][1] == COMPUTER_CELL) computerPieceCount++;
	if (cells[2][0] == COMPUTER_CELL) computerPieceCount++;

	if (playerPieceCount == 3) return PLAYER_CELL;
	if (computerPieceCount == 3) return COMPUTER_CELL;

	return 0;
}

void TicTacToeGame::computerMove() {

	// This computer move algorithim is very dumb but works pretty well
	// It looks for spots where the player is one move away from winning
	// and puts a blocking move in place. At the same time, it looks for 
	// move where it is one away and wins it if it can. If it can find 
	// neither, it moves to the first blank space.

	// Search columns
	for (int i=0; i < 3; i++) {

		int playerPieceCount = 0;
		int computerPieceCount = 0;

		// Count player && computer pieces
		for (int j=0; j < 3; j++) {
			if (cells[i][j] == PLAYER_CELL) playerPieceCount++;
			if (cells[i][j] == COMPUTER_CELL) computerPieceCount++;
		}

		// If player or computer has two pieces, locate the blocking or winning space and move there
		if (playerPieceCount == 2 || computerPieceCount == 2) 
			for (int j=0; j < 3; j++) 
				if (cells[i][j] == BLANK_CELL) {
					cells[i][j] = COMPUTER_CELL;
					return;
				}		
	}

	// Search rows
	for (int j=0; j < 3; j++) {

		int playerPieceCount = 0;
		int computerPieceCount = 0;

		for (int j=0; j < 3; j++) {
			if (cells[i][j] == PLAYER_CELL) playerPieceCount++;
			if (cells[i][j] == COMPUTER_CELL) computerPieceCount++;
		}

		if (playerPieceCount == 2 || computerPieceCount == 2) 
			for (int i=0; i < 3; i++) 
				if (cells[i][j] == BLANK_CELL) {
					cells[i][j] = COMPUTER_CELL;
					return;
				}		
	}

	// Search the two diagonals - no way to loop this, unfortunately
	
	// Top left to bottom right
	int playerPieceCount = 0;
	int computerPieceCount = 0;

	if (cells[0][0] == PLAYER_CELL) playerPieceCount++;
	if (cells[1][1] == PLAYER_CELL) playerPieceCount++;
	if (cells[2][2] == PLAYER_CELL) playerPieceCount++;

	if (cells[0][0] == COMPUTER_CELL) computerPieceCount++;
	if (cells[1][1] == COMPUTER_CELL) computerPieceCount++;
	if (cells[2][2] == COMPUTER_CELL) computerPieceCount++;

	// If either piece has two and the row isn't full already, find cell and block or win
	if ((playerPieceCount == 2 || computerPieceCount == 2) && playerPieceCount+computerPieceCount != 3)  {
		if (cells[0][0] == BLANK_CELL) cells[0][0] = COMPUTER_CELL;
		if (cells[1][1] == BLANK_CELL) cells[1][1] = COMPUTER_CELL;
		if (cells[2][2] == BLANK_CELL) cells[2][2] = COMPUTER_CELL;
		return;
	}

	// Top right to bottom left
	playerPieceCount = 0;
	computerPieceCount = 0;

	if (cells[0][2] == PLAYER_CELL) playerPieceCount++;
	if (cells[1][1] == PLAYER_CELL) playerPieceCount++;
	if (cells[2][0] == PLAYER_CELL) playerPieceCount++;

	if (cells[0][2] == COMPUTER_CELL) computerPieceCount++;
	if (cells[1][1] == COMPUTER_CELL) computerPieceCount++;
	if (cells[2][0] == COMPUTER_CELL) computerPieceCount++;

	if ((playerPieceCount == 2 || computerPieceCount == 2) && playerPieceCount+computerPieceCount != 3) {
		if (cells[0][2] == BLANK_CELL) cells[0][2] = COMPUTER_CELL;
		if (cells[1][1] == BLANK_CELL) cells[1][1] = COMPUTER_CELL;
		if (cells[2][0] == BLANK_CELL) cells[2][0] = COMPUTER_CELL;
		return;
	}

	// Otherwise, choose first blank cell
	for (i=0; i < 3; i++)
		for (int j=0; j < 3; j++)
			if (cells[i][j] == BLANK_CELL) {
				cells[i][j] = COMPUTER_CELL;
				return;
			}
}
