#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <fstream>
#include <cstring>
#include <stdexcept>

#include <openssl/asn1t.h>
#include <openssl/ssl.h>
#include <openssl/ui.h>
#include <openssl/safestack.h>

#include "opensslutil.h"
#include "opensslgenericutil.h"
#include "namespacespolicy.h"
#include "datetime.h"
#include "canlxx.h"
#include "fileutil.h"

/*
#ifndef X509_EXTENSIONS
typedef STACK_OF(X509_EXTENSION) X509_EXTENSIONS;

DECLARE_ASN1_ENCODE_FUNCTIONS(X509_EXTENSIONS, X509_EXTENSIONS, X509_EXTENSIONS)

ASN1_ITEM_TEMPLATE(X509_EXTENSIONS) = 
  ASN1_EX_TEMPLATE_TYPE(ASN1_TFLG_SEQUENCE_OF, 0, X509_EXTENSIONS, X509_EXTENSION)
ASN1_ITEM_TEMPLATE_END(X509_EXTENSIONS)

IMPLEMENT_ASN1_ENCODE_FUNCTIONS_fname(X509_EXTENSIONS, X509_EXTENSIONS, X509_EXTENSIONS)
#endif
*/

namespace AuthN {
namespace OpenSSL {


  // TODO: Convert most of error log messages into returned Status
  // Currently this code is a mess. It uses log messages, exceptions and returns bool.
  // It MUST be unified.

  CredentialError::CredentialError(const std::string& what) : Status(-1,what) { }

  static const EVP_MD* get_digest(Context& context, const std::string& digest_name, const EVP_PKEY* privkey);


  static void bio2string(BIO* in, std::string& out) {
    out.clear();
    for(;;) {
      char s[256];
      int l = BIO_read(in,s,sizeof(s));
      if(l <= 0) break;
      out.append(s,l);
    }
  }

  static void read_stream(std::istream& in, std::string& str) {
    int length;
    char* buffer;
    if(!in || !in.good()) return;
    in.seekg (0, std::ios::end);
    length = in.tellg();
    if(length <0) return;
    in.seekg (0, std::ios::beg);
    buffer = new char [length];
    in.read (buffer,length);
    str.assign(buffer,length);
    in.seekg (0, std::ios::beg);
    delete[] buffer;
  }

  static std::string get_hash_value(X509_NAME* name) {
    std::string hash_str;
    char hash[32];
    memset(hash, 0, 32);
    snprintf(hash, 32, "%08lx", X509_NAME_hash(name));
    hash_str = hash;
    return hash_str;
  }

  static std::string get_issuer_name(X509* cert) {
    if(cert == NULL) return std::string();
    char buf[256];
    std::string issuer;
    X509_NAME_oneline(X509_get_issuer_name(cert),buf, sizeof buf);
    issuer = buf;
    return issuer;
  }

  static std::string get_subject_name(X509* cert) {
    if(cert==NULL) return std::string();
    char buf[256];
    std::string subject;
    X509_NAME_oneline(X509_get_subject_name(cert),buf, sizeof buf);
    subject = buf;
    return subject;
  }

  static std::string get_subject_name(X509_REQ* req) {
    if(req==NULL) return std::string();
    char buf[256];
    std::string subject;
    X509_NAME_oneline(X509_REQ_get_subject_name(req),buf, sizeof buf);
    subject = buf;
    return subject;
  }
  
  static std::string get_identity_name(X509* cert) {
    // TODO: it is more correct to go through chain till first non-proxy cert
    // TODO: different format must be implemented
    X509_NAME* subject = NULL;
    X509_NAME_ENTRY* ne = NULL;
    if(!cert) return "";
    subject = X509_NAME_dup(X509_get_subject_name(cert));

    ASN1_STRING* entry;
    std::string entry_str;
    for(;;) {
      ne = X509_NAME_get_entry(subject, X509_NAME_entry_count(subject)-1);
      if(!ne) break;
      if (!OBJ_cmp(ne->object,OBJ_nid2obj(NID_commonName))) {
        entry = X509_NAME_ENTRY_get_data(ne);
        entry_str.assign((const char*)(entry->data), (std::size_t)(entry->length));
        if(entry_str == "proxy" || entry_str == "limited proxy" ||
          entry_str.find_first_not_of("0123456789") == std::string::npos) {
          //Drop the name entry "proxy", "limited proxy", or the random digital(RFC)
          ne = X509_NAME_delete_entry(subject, X509_NAME_entry_count(subject)-1);
          X509_NAME_ENTRY_free(ne);
          ne = NULL;
        }
        else break;
      }
      else break;
    }
    char buf[256];
    if(subject!=NULL) {
      X509_NAME_oneline(subject,buf,sizeof(buf)-1);
      X509_NAME_free(subject);
    }
    buf[sizeof(buf)-1] = 0;
    std::string str(buf);
    return str;
  }

/*
  static ASN1_UTCTIME* utc_to_asn1time(const AuthN::Utils::Time& t) {
    std::string t_str = t.str(AuthN::Utils::MDSTime);
    if(t_str.length() < 2) return NULL;
    ASN1_UTCTIME* s = ASN1_UTCTIME_new();
    if(!s) return NULL;
    if(ASN1_UTCTIME_set_string(s,(char*)(t_str.c_str() +2))) return s;
    ASN1_UTCTIME_free(s);
    return NULL;
  }
*/

  AuthN::Utils::Time asn1_to_utctime(const ASN1_UTCTIME *s) {
    if(s == NULL) return -1;
    std::string t_str;
    if(s->type == V_ASN1_UTCTIME) {
      t_str.append("20");
      t_str.append((char*)(s->data));
    }
    else {//V_ASN1_GENERALIZEDTIME
      t_str.append((char*)(s->data));
    }
    return AuthN::Utils::Time(t_str);
  }

  static ASN1_INTEGER* long_to_asn1int(const long val) {
    ASN1_INTEGER* asn1=NULL;
    if (!(asn1 = ASN1_INTEGER_new())) {
      return NULL;
    }
    if(!ASN1_INTEGER_set(asn1, val)) {
      ASN1_INTEGER_free(asn1); return NULL;
    }
    return asn1;
  }

#define SERIAL_RAND_BITS        64
  int rand_serial(BIGNUM *b, ASN1_INTEGER *ai) {
    BIGNUM *btmp;
    int ret = 0;
    if (b) btmp = b;
    else btmp = BN_new();
    if (!btmp) return 0;
    if (!BN_pseudo_rand(btmp, SERIAL_RAND_BITS, 0, 0)) goto error;
    if (ai && !BN_to_ASN1_INTEGER(btmp, ai)) goto error;
    ret = 1;
error:
    if (!b) BN_free(btmp);
    return ret;
  }


#undef BSIZE
#define BSIZE 256
  BIGNUM *load_serial(Context& context, const std::string& serialfile, ASN1_INTEGER **retai) {
    BIO *in=NULL;
    BIGNUM *ret=NULL;
    char buf[1024];
    ASN1_INTEGER *ai=NULL;
    ai=ASN1_INTEGER_new();
    if (ai == NULL) goto err;
    if ((in=BIO_new(BIO_s_file())) == NULL) {
      context.Log(Context::LogError, GetOpenSSLError());
      goto err;
    }
    if (BIO_read_filename(in,serialfile.c_str()) > 0) {
      if (!a2i_ASN1_INTEGER(in,ai,buf,1024)) {
        context.LogFormat(Context::LogError,"unable to load number from: %s",serialfile.c_str());
        goto err;
      }
      ret=ASN1_INTEGER_to_BN(ai,NULL);
      if (ret == NULL) {
        context.Log(Context::LogError, "error converting number from bin to BIGNUM");
        goto err;
      }
    }

    if (ret && retai) {
      *retai = ai;
      ai = NULL;
    }
err:
    if (in != NULL) BIO_free(in);
    if (ai != NULL) ASN1_INTEGER_free(ai);
    return(ret);
  }

  int save_serial(Context& context, const std::string& serialfile, 
      char *suffix, BIGNUM *serial, ASN1_INTEGER **retai) {
    char buf[1][BSIZE];
    BIO *out = NULL;
    int ret=0;
    ASN1_INTEGER *ai=NULL;
    int j;

    if (suffix == NULL) j = strlen(serialfile.c_str());
    else j = strlen(serialfile.c_str()) + strlen(suffix) + 1;
    if (j >= BSIZE) {
      context.Log(Context::LogError,"file name too long");
      goto err;
    }

    if (suffix == NULL)
      BUF_strlcpy(buf[0], serialfile.c_str(), BSIZE);
    else {
#ifndef OPENSSL_SYS_VMS
      j = BIO_snprintf(buf[0], sizeof buf[0], "%s.%s", serialfile.c_str(), suffix);
#else
      j = BIO_snprintf(buf[0], sizeof buf[0], "%s-%s", serialfile.c_str(), suffix);
#endif
    }
    out=BIO_new(BIO_s_file());
    if (out == NULL) {
      context.Log(Context::LogError, GetOpenSSLError());
      goto err;
    }
    if (BIO_write_filename(out,buf[0]) <= 0) {
      perror(serialfile.c_str());
      goto err;
    }
    if ((ai=BN_to_ASN1_INTEGER(serial,NULL)) == NULL) {
      context.Log(Context::LogError,"error converting serial to ASN.1 format");
      goto err;
    }
    i2a_ASN1_INTEGER(out,ai);
    BIO_puts(out,"\n");
    ret=1;
    if (retai) {
      *retai = ai;
      ai = NULL;
    }
err:
    if (out != NULL) BIO_free_all(out);
    if (ai != NULL) ASN1_INTEGER_free(ai);
    return(ret);
  }

#ifdef POSTFIX
#undef POSTFIX
#endif
#define POSTFIX ".srl"

  static ASN1_INTEGER *x509_load_serial(Context& context, 
      const std::string& CAfile, const std::string& serialfile) {
    ASN1_INTEGER *bs = NULL;
    BIGNUM *serial = NULL;

    std::string serial_f;
    if(!serialfile.empty()) serial_f = serialfile;
    else if(!CAfile.empty()){
      std::size_t pos; pos = CAfile.rfind(".");
      if(pos != std::string::npos) serial_f = CAfile.substr(0, pos);
      serial_f.append(".srl");
    }
    else{ return bs;}

    serial = load_serial(context, serial_f, NULL);
    if (serial == NULL) {
      context.LogFormat(Context::LogError,"load serial from %s failure: %s",serial_f.c_str(),GetOpenSSLError().c_str());
      return bs;
    }

    if (!BN_add_word(serial,1)) {
      context.Log(Context::LogError,"add_word failure");
      BN_free(serial); return bs;
    }

    if(!save_serial(context, serial_f, NULL, serial, &bs)) {
      context.LogFormat(Context::LogError,"save serial to %s failure",serial_f.c_str());
      BN_free(serial); return bs;
    }
    BN_free(serial);
    return bs;
  }


  //subject is expected to be in the format /type0=value0/type1=value1/type2=...
  //where characters may be escaped by '\'
  static X509_NAME* parse_name(Context* context, char *subject, long chtype, int multirdn) {
    size_t buflen = strlen(subject)+1;
    /* to copy the types and values into. due to escaping, the copy can only become shorter */
    char *buf = (char*)(OPENSSL_malloc(buflen));
    size_t max_ne = buflen / 2 + 1; /* maximum number of name elements */
    char **ne_types =  (char **)(OPENSSL_malloc(max_ne * sizeof (char *)));
    char **ne_values = (char **)(OPENSSL_malloc(max_ne * sizeof (char *)));
    int *mval = (int*)(OPENSSL_malloc (max_ne * sizeof (int)));

    char *sp = subject, *bp = buf;
    int i, ne_num = 0;

    X509_NAME *n = NULL;
    int nid;

    if (!buf || !ne_types || !ne_values) {
      context->Log(Context::LogError, "OpenSSL malloc error");
      goto error;
    }
    if (*subject != '/') {
      context->Log(Context::LogError,"Subject does not start with '/'");
      goto error;
    }
    sp++; // skip leading /

    // no multivalued RDN by default
    mval[ne_num] = 0;
    while (*sp) {
      // collect type
      ne_types[ne_num] = bp;
      while (*sp) {
        if (*sp == '\\') /* is there anything to escape in the type...? */
        {
          if (*++sp) *bp++ = *sp++;
          else {
            context->Log(Context::LogError,"escape character at end of string");
            goto error;
          }
        }
        else if (*sp == '=') {
          sp++;
          *bp++ = '\0';
          break;
        }
        else  *bp++ = *sp++;
      }
      if (!*sp) {
        context->LogFormat(Context::LogError,"end of string encountered while processing type of subject name element #%d",ne_num);
        goto error;
      }
      ne_values[ne_num] = bp;
      while (*sp) {
        if (*sp == '\\') {
          if (*++sp)
            *bp++ = *sp++;
          else {
            context->Log(Context::LogError,"escape character at end of string");
            goto error;
          }
        }
        else if (*sp == '/') {
          sp++;
          // no multivalued RDN by default
          mval[ne_num+1] = 0;
          break;
        }
        else if (*sp == '+' && multirdn) {
          // a not escaped + signals a mutlivalued RDN
          sp++;
          mval[ne_num+1] = -1;
          break;
        }
        else
          *bp++ = *sp++;
      }
      *bp++ = '\0';
      ne_num++;
    }

    if (!(n = X509_NAME_new()))
      goto error;

    for (i = 0; i < ne_num; i++) {
      if ((nid=OBJ_txt2nid(ne_types[i])) == NID_undef) {
        context->LogFormat(Context::LogError,"Subject Attribute %s has no known NID, skipped",ne_types[i]);
        //continue;
        goto error;
      }
      if (!*ne_values[i]) {
        context->LogFormat(Context::LogError,"No value provided for Subject Attribute %s skipped",ne_types[i]);
        //continue;
        goto error;
      }

      if (!X509_NAME_add_entry_by_NID(n, nid, chtype, (unsigned char*)ne_values[i], -1,-1,mval[i])) {
        goto error;
      }
    }

    OPENSSL_free(mval);
    OPENSSL_free(ne_values);
    OPENSSL_free(ne_types);
    OPENSSL_free(buf);
    return n;

error:
    X509_NAME_free(n);
    if (ne_values) OPENSSL_free(ne_values);
    if (ne_types) OPENSSL_free(ne_types);
    if (buf) OPENSSL_free(buf);
    return NULL;
  }


  //Parse the string for certificate and get the format of it
  OpenSSLUtil::Credformat OpenSSLUtil::getFormat_str(const std::string& source) {
    Credformat format = CRED_UNKNOWN;
    AutoBIO bio(BIO_new_mem_buf((void*)(source.c_str()), source.length()));
    if(!bio) return format;

    unsigned char* bio_str;
    int len;
    len = BIO_get_mem_data(bio, (unsigned char *) &bio_str);
    char firstbyte;
    if(len>0) {
      firstbyte = bio_str[0];
      if(firstbyte==48)  {
        //DER-encoded, PKCS12 or DER? firstly parse it as PKCS12 ASN.1,
        //if can not parse it, then it is DER format
        PKCS12* pkcs12 = NULL;
        unsigned char* source_chr = (unsigned char*)(source.c_str());
#ifdef HAVE_OPENSSL_OLDRSA
        if((pkcs12 = d2i_PKCS12(NULL, (unsigned char**)&source_chr, source.length())) != NULL){ format=CRED_PKCS12; PKCS12_free(pkcs12); }
#else
        if((pkcs12 = d2i_PKCS12(NULL, (const unsigned char**)&source_chr, source.length())) != NULL){ format=CRED_PKCS12; PKCS12_free(pkcs12); }
#endif
        else {
          format = CRED_DER;
        }
      }
      else { format = CRED_PEM; }
    }
    return format;
  }

  static bool insert_proxy_cn(X509* cert_to_issue, X509* issuer_cert, X509_REQ* req) { 
    X509_NAME* name = X509_get_subject_name(issuer_cert); 
    if(!name) return false;
    AutoX509NAME subject_name(X509_NAME_dup(name));
    if(!subject_name) return false;
    if(!X509_set_issuer_name(cert_to_issue, subject_name)) return false;

    EVP_PKEY* req_pubkey = X509_REQ_get_pubkey(req);
    if(!req_pubkey) return false;

    X509_NAME_ENTRY* name_entry = NULL;
    char* CN_name = NULL;

    unsigned char md[SHA_DIGEST_LENGTH];
    long  sub_hash;
    unsigned int len;
#if (OPENSSL_VERSION_NUMBER >= 0x0090800fL)
    ASN1_digest((int (*)(void*, unsigned char**))i2d_PUBKEY, EVP_sha1(), (char*)req_pubkey,md,&len);
#else
    ASN1_digest((int(*)())i2d_PUBKEY, EVP_sha1(), (char*)req_pubkey,md,&len);
#endif
    sub_hash = md[0] + (md[1] + (md[2] + (md[3] >> 1) * 256) * 256) * 256;
    CN_name = (char*)malloc(sizeof(long)*4 + 1);
    snprintf(CN_name, sizeof(long)*4 + 1, "%ld", sub_hash);

    if((name_entry = X509_NAME_ENTRY_create_by_NID(&name_entry, NID_commonName, V_ASN1_APP_CHOOSE,
       (unsigned char *) CN_name, -1)) == NULL) { free(CN_name); return false; }

    if(!X509_NAME_add_entry(subject_name, name_entry, X509_NAME_entry_count(subject_name), 0) ||
       !X509_set_subject_name(cert_to_issue, subject_name)) {
      free(CN_name); X509_NAME_ENTRY_free(name_entry); return false;
    }
    free(CN_name); 
    X509_NAME_ENTRY_free(name_entry); 
    return true; 
  }

  static Status read_pkcs12(Context* context, const std::string& pkcs12_str, 
      pem_password_cb* pem_cb, PKCS12** pkcs12, EVP_PKEY** pkey, 
      X509** cert, STACK_OF(X509)** cert_chain) {
    const char* pass;
    char tpass[PEM_BUFSIZE];
    int len;
    Status stat;

    unsigned char* pkcs_chr = (unsigned char*)(pkcs12_str.c_str());
#ifdef HAVE_OPENSSL_OLDRSA
    *pkcs12 = d2i_PKCS12(NULL, (unsigned char**)&pkcs_chr, pkcs12_str.length());
#else
    *pkcs12 = d2i_PKCS12(NULL, (const unsigned char**)&pkcs_chr, pkcs12_str.length());
#endif

    if(*pkcs12 == NULL) {
      context->Log(Context::LogError,"Error loading PKCS12, d2i_PKCS12 failed");
      stat = Status(-1, "Error loading PKCS12, d2i_PKCS12 failed");
      goto err;
    }
    // Check if an empty password works
    if(PKCS12_verify_mac(*pkcs12, "", 0) || PKCS12_verify_mac(*pkcs12, NULL, 0)) pass = "";
    else {
      if(!pem_cb) { 
        context->Log(Context::LogError,"Passphrase callback for loading PKCS12 is empty"); 
        stat = Status(-1, "Passphrase callback for loading PKCS12 is empty"); goto err; 
      }
      len = pem_cb(tpass, PEM_BUFSIZE, 0, (void*)(context));
      if(len < 0) {
        context->Log(Context::LogError,"Passphrase callback fails");
        stat = Status(-1, "Passphrase callback fails"); goto err;
      }
      if(len < PEM_BUFSIZE) tpass[len] = 0;
      if(!PKCS12_verify_mac(*pkcs12, tpass, len)) {
        context->Log(Context::LogError,"Mac verification fails, the password is wrong");
        stat = Status(-1, "Mac verification fails, the password is wrong"); goto err;
      }
      pass = tpass;
    }
    if(PKCS12_parse(*pkcs12, pass, pkey, cert, cert_chain)) return Status(0);

    //Any failure occurs, empty the PKCS12 object
  err:
    if(*pkcs12) PKCS12_free(*pkcs12);
    return stat;
  }

  static Status write_pkcs12(Context* context, std::string& pkcs12_str/*output*/, 
      pem_password_cb* pem_cb, EVP_PKEY* pkey, X509* cert, 
      STACK_OF(X509)* cert_chain) {
    AutoBIO bio(BIO_new(BIO_s_mem()));

    PKCS12* p12 = NULL;
    int cert_pbe = NID_pbe_WithSHA1And40BitRC2_CBC;
    int key_pbe = NID_pbe_WithSHA1And3_Key_TripleDES_CBC;
    int enc_iter = PKCS12_DEFAULT_ITER; //encryption interation
    int mac_iter = PKCS12_DEFAULT_ITER; //mac interation
    char* pass;
    char enc_pass[PEM_BUFSIZE];
    char mac_pass[PEM_BUFSIZE];
    int len;

    if(!pem_cb) { 
      context->Log(Context::LogError,"Passphrase callback for loading PKCS12 is empty");
      return Status(-1, "Passphrase callback for loading PKCS12 is empty");
    }

    // Encryption passphrase
    len = pem_cb(enc_pass, PEM_BUFSIZE, 0, (void*)(context));
    if(len < 0) {
      context->Log(Context::LogError,"Passphrase callback for encryption fails");
      return Status(-1, "Passphrase callback for encryption fails");
    }
    if(len < PEM_BUFSIZE) enc_pass[len] = 0;
    pass = enc_pass;
    p12 = PKCS12_create(pass, NULL, pkey, cert, cert_chain,
                        key_pbe, cert_pbe, enc_iter, -1, 0);

    if(!p12) { 
      context->Log(Context::LogError, "Failed to create PKCS12");
      return Status(-1, "Failed to create PKCS12");
    }

    // MAC passphrase
    len = pem_cb(mac_pass, PEM_BUFSIZE, 0, (void*)(context));
    if(len < 0) {
      context->Log(Context::LogInfo,"Passphrase callback for MAC fails, use the same passphrase as encryption");
    }
    else {
      if(len < PEM_BUFSIZE) mac_pass[len] = 0;
      pass = mac_pass;
    }
    if(p12 != NULL) {
      PKCS12_set_mac(p12, pass, -1, NULL, 0, mac_iter, NULL);
      i2d_PKCS12_bio(bio, p12);
      PKCS12_free(p12);
      bio2string(bio, pkcs12_str);
    }
    return Status(0);
  }

//----------
// OSSLItem Implementation
//----------

  OSSLItem::OSSLItem() : cert(NULL), req(NULL), crl(NULL), 
    pkey(NULL), cert_chain(NULL), pkcs12(NULL), logctx(NULL) { }

  OSSLItem::OSSLItem(const OSSLItem& other) : cert(NULL), req(NULL), 
    crl(NULL), pkey(NULL), cert_chain(NULL), pkcs12(NULL), logctx(NULL) {
    *this = other;
  }

  OSSLItem::~OSSLItem() { reset(); }

  OSSLItem& OSSLItem::operator=(const OSSLItem& other) {
    if(this != &other) {
      reset();
      if(other.cert) cert = X509_dup(other.cert);
      if(other.req) req = X509_REQ_dup(other.req);
      if(other.crl) crl = X509_CRL_dup(other.crl);
      if(other.pkey) { 
        RSA* rsa = EVP_PKEY_get1_RSA(other.pkey); 
        if(rsa) {
          pkey = EVP_PKEY_new();
          EVP_PKEY_set1_RSA(pkey, rsa);
          RSA_free(rsa);
        }
      }
      if(other.cert_chain) {
        X509* tmp;
        if(sk_X509_num(other.cert_chain) > 0) cert_chain = sk_X509_new_null();
        for(int n = 0; n < sk_X509_num(other.cert_chain); n++) {
          tmp = X509_dup(sk_X509_value(other.cert_chain, n));
          sk_X509_insert(cert_chain, tmp, n);
        }
      }
      if(other.pkcs12) {
        std::string str;
        other.tostrPKCS12(str);
        fromstrPKCS12(str);
      }
    }
    return *this;
  }

  void OSSLItem::reset() {
    if(cert) { X509_free(cert); cert = NULL; }
    if(req) { X509_REQ_free(req); req = NULL; }
    if(crl) { X509_CRL_free(crl); crl = NULL; }
    if(pkey) { EVP_PKEY_free(pkey); pkey = NULL; }
    if(cert_chain) { sk_X509_pop_free(cert_chain, X509_free); cert_chain = NULL; }
    if(pkcs12) { PKCS12_free(pkcs12); pkcs12 = NULL; }
  }
     
  Status OSSLItem::tostrDER(std::string& out) const {
    AutoBIO bio(BIO_new(BIO_s_mem()));
    if(cert) i2d_X509_bio(bio, cert);
    else if(req) i2d_X509_REQ_bio(bio, req);
    else if(crl) i2d_X509_CRL_bio(bio, crl);
    else if(pkey) {
      if(!i2d_PrivateKey_bio(bio, pkey))
        i2d_PUBKEY_bio(bio, pkey);
    }
    if(!bio) return Status(-1, std::string("Failed to export openssl object to ASN1: ") + GetOpenSSLError());
    bio2string(bio, out);
    return Status(0);
  }

  Status OSSLItem::tostrPEM(std::string& out, bool enc, const std::string& passphrase) const {
    AutoBIO bio(BIO_new(BIO_s_mem()));
    if(cert) { 
      PEM_write_bio_X509(bio, cert);
      if((cert_chain != NULL) && (sk_X509_num(cert_chain) > 0)) {
        for(int n = 0; n < sk_X509_num(cert_chain) ; n++) 
          PEM_write_bio_X509(bio, sk_X509_value(cert_chain,n));
      }
    }
    else if(req) PEM_write_bio_X509_REQ(bio, req);
    else if(crl) PEM_write_bio_X509_CRL(bio, crl);
    else if(pkey) {
      int i;
      if(!enc) i = PEM_write_bio_PrivateKey(bio, pkey, NULL, NULL, 0, NULL, NULL);
      else i = PEM_write_bio_PrivateKey(bio, pkey, EVP_des_ede3_cbc(), 
        (unsigned char*)(passphrase.c_str()), passphrase.size(), 
        &ContextOpenSSLUI::passwordcb, logctx);
      if(!i) {
        int reason = ERR_GET_REASON(ERR_peek_error());
        if(reason == PEM_R_PROBLEMS_GETTING_PASSWORD)
          logctx->Message("Can not read PEM private key: failed to obtain password"); 
        else {
          // The EVP_PKEY object could only include public key information
          if(!PEM_write_bio_PUBKEY(bio, pkey))
        	logctx->Message("Can not read PEM EVP_PKEY");
        }
      }
    }
    if(!bio) return Status(-1, std::string("Failed to export openssl object to PEM: ") + GetOpenSSLError());
    bio2string(bio, out);
    return Status(0);
  }

  Status OSSLItem::tostrPKCS12(std::string& out) const {
    AutoBIO bio(BIO_new(BIO_s_mem()));
    if(pkcs12) { 
      i2d_PKCS12_bio(bio, pkcs12);
      if(!bio) return Status(-1, std::string("Failed to export openssl object to PKCS12: ") + GetOpenSSLError());
      bio2string(bio, out); 
      return Status(0);
    }
    return write_pkcs12(logctx, out, &ContextOpenSSLUI::passwordcb, pkey, cert, cert_chain);
  }

  Status OSSLItem::tofileDER(const char* filename) {
    std::ofstream fp;
    fp.open(filename);
    std::string str; 
    Status stat = tostrDER(str);
    fp.write(str.c_str(), str.length());
    fp.close();
    return stat;
  }

  Status OSSLItem::tofilePEM(const char* filename) {
    std::ofstream fp;
    fp.open(filename);
    std::string str; 
    Status stat = tostrPEM(str);
    fp.write(str.c_str(), str.length());
    fp.close();
    return stat;
  }

  Status OSSLItem::tofilePKCS12(const char* filename) {
    std::ofstream fp;
    fp.open(filename);
    std::string str; 
    Status stat = tostrPKCS12(str);
    fp.write(str.c_str(), str.length());
    fp.close();
    return stat;
  }

  Status OSSLItem::fromstrDER(const std::string& in, Type t) {
    reset();
    unsigned char* der_chr = (unsigned char*)(in.c_str());
#ifdef HAVE_OPENSSL_OLDRSA
    if(t == TypeCert) cert = d2i_X509(NULL, (unsigned char**)&der_chr, in.length());
    else if(t == TypeReq) req = d2i_X509_REQ(NULL, (unsigned char**)&der_chr, in.length());
    else if(t == TypeCRL) crl = d2i_X509_CRL(NULL, (unsigned char**)&der_chr, in.length());
    else if(t == TypeKey) {
      pkey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, (unsigned char**)&der_chr, in.length());
      if(!pkey) pkey = d2i_PublicKey(EVP_PKEY_RSA, NULL, (unsigned char**)&der_chr, in.length());
    }
#else
    if(t == TypeCert) cert = d2i_X509(NULL, (const unsigned char**)&der_chr, in.length());
    else if(t == TypeReq) req = d2i_X509_REQ(NULL, (const unsigned char**)&der_chr, in.length());
    else if(t == TypeCRL) crl = d2i_X509_CRL(NULL, (const unsigned char**)&der_chr, in.length());
    else if(t == TypeKey) {
      pkey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, (const unsigned char**)&der_chr, in.length());
      if(!pkey) pkey = d2i_PublicKey(EVP_PKEY_RSA, NULL, (const unsigned char**)&der_chr, in.length());
    }
#endif
    if(invalid()) return Status(-1, std::string("Failed to parse ASN1 string: ")+ GetOpenSSLError());
    return Status(0);
  }

  Status OSSLItem::fromstrPEM(const std::string& in, Type t) {
    reset();
    AutoBIO bio(BIO_new(BIO_s_mem()));
    BIO_write(bio, in.c_str(), in.size());
    if(t == TypeCert) { 
      cert = PEM_read_bio_X509(bio, NULL, NULL, NULL); 
      if(!BIO_eof(bio)) cert_chain = sk_X509_new_null();
      while(!BIO_eof(bio)){
        X509* tmp = NULL;
        if(!(PEM_read_bio_X509(bio, &tmp, NULL, NULL))){
         ERR_clear_error(); break;
        }
        if(!sk_X509_push(cert_chain, tmp)) {
          //std::string str(X509_NAME_oneline(X509_get_subject_name(tmp),0,0));
          X509_free(tmp);
        }
      }
    }
    else if(t == TypeReq) req = PEM_read_bio_X509_REQ(bio, NULL, NULL, NULL); 
    else if(t == TypeCRL) crl = PEM_read_bio_X509_CRL(bio, NULL, NULL, NULL);
    else if(t == TypeKey) { 
      if(!(pkey = PEM_read_bio_PrivateKey(bio, NULL, &ContextOpenSSLUI::passwordcb, logctx))) {
        int reason = ERR_GET_REASON(ERR_peek_error());
        if(reason == PEM_R_BAD_BASE64_DECODE) {
          logctx->Message("Can not read PEM private key: probably bad password");
        } else if(reason == PEM_R_BAD_DECRYPT) {
          logctx->Message("Can not read PEM private key: failed to decrypt");
        } else if(reason == PEM_R_BAD_PASSWORD_READ) {
          logctx->Message("Can not read PEM private key: failed to obtain password");
        } else if(reason == PEM_R_PROBLEMS_GETTING_PASSWORD) {
          logctx->Message("Can not read PEM private key: failed to obtain password");
        } else {
          // The EVP_PKEY object could only include public key information
          if(!PEM_read_bio_PUBKEY(bio, NULL, NULL, NULL))
          	logctx->Message("Can not read PEM EVP_PKEY");
        }
      }
    }
    if(invalid()) return Status(-1, std::string("Failed to parse ASN1 string: ")+ GetOpenSSLError());
    return Status(0);
  }

  Status OSSLItem::fromstrPKCS12(const std::string& in) {
    reset();
    return read_pkcs12(logctx, in, &ContextOpenSSLUI::passwordcb, &pkcs12, &pkey, &cert, &cert_chain);
/*
    unsigned char* pkcs_chr = (unsigned char*)(in.c_str());
#ifdef HAVE_OPENSSL_OLDRSA
    pkcs12 = d2i_PKCS12(NULL, (unsigned char**)&pkcs_chr, in.length());
#else
    pkcs12 = d2i_PKCS12(NULL, (const unsigned char**)&pkcs_chr, in.length());
#endif
    if(invalid()) return false;

    // Parse the pkcs12 object
    char password[100];
    EVP_read_pw_string(password, 100, "Enter Password for PKCS12 credential:", 0);
    PKCS12_parse(pkcs12, password, &pkey, &cert, &cert_chain);
    return true;
*/
  }

  Status OSSLItem::fromstr(const std::string& in, Type t) {
    if(in.empty()) return false;
    OpenSSLUtil::Credformat fmt = OpenSSLUtil::getFormat_str(in);
    if(fmt == OpenSSLUtil::CRED_PEM) return fromstrPEM(in, t);
    else if(fmt == OpenSSLUtil::CRED_DER) return fromstrDER(in, t); 
    else if(fmt == OpenSSLUtil::CRED_PKCS12) return fromstrPKCS12(in);
    return Status(-1, "Format can not be recognized");
  }

  Status OSSLItem::fromfile(const char* filename, Type t) {
    if(!filename) return false;
    bool ret;
    std::ifstream fp;
    fp.open(filename);
    std::string str;
    read_stream(fp, str);
    ret = fromstr(str, t);
    fp.close();
    return ret;
  }
 
  bool OSSLItem::invalid() const {
    if(!cert && !req && !crl && !pkey && !pkcs12) return true;

    if(req) {
      EVP_PKEY* req_pubkey = NULL;
      req_pubkey = X509_REQ_get_pubkey((X509_REQ*)req);
      if(!req_pubkey) {
        return false;
      }
      if(!X509_REQ_verify((X509_REQ*)req, req_pubkey)){
        EVP_PKEY_free(req_pubkey); return true;
      }
    }
    return false;
  }

//Some variable definition for openssl.cnf 
#define REQ_SECTION     "req"
#define BITS            "default_bits"
#define KEYFILE         "default_keyfile"
#define PROMPT          "prompt"
#define DISTINGUISHED_NAME      "distinguished_name"
#define ATTRIBUTES      "attributes"
#define X509_EXTENSIONS_CONF   "x509_extensions"
#define REQ_EXTENSIONS_CONF  "req_extensions"

#define STRING_MASK     "string_mask"
#define UTF8_IN         "utf8"

#define CA_SECTION      "ca"
#define DEFAULT_CA      "default_ca"
#define ENV_DIR                 "dir"
#define ENV_CERTS               "certs"
#define ENV_CRL_DIR             "crl_dir"
#define ENV_CA_DB               "CA_DB"
#define ENV_NEW_CERTS_DIR       "new_certs_dir"
#define ENV_CERTIFICATE         "certificate"
#define ENV_SERIAL              "serial"
#define ENV_CRLNUMBER           "crlnumber"
#define ENV_CRL                 "crl"
#define ENV_PRIVATE_KEY         "private_key"
#define ENV_RANDFILE            "RANDFILE"
#define ENV_DEFAULT_DAYS        "default_days"
#define ENV_DEFAULT_STARTDATE   "default_startdate"
#define ENV_DEFAULT_ENDDATE     "default_enddate"
#define ENV_DEFAULT_CRL_DAYS    "default_crl_days"
#define ENV_DEFAULT_CRL_HOURS   "default_crl_hours"
#define ENV_DEFAULT_MD          "default_md"
#define ENV_DEFAULT_EMAIL_DN    "email_in_dn"
#define ENV_PRESERVE            "preserve"
#define ENV_POLICY              "policy"
#define ENV_EXTENSIONS          "x509_extensions"
#define ENV_CRLEXT              "crl_extensions"
#define ENV_MSIE_HACK           "msie_hack"
#define ENV_NAMEOPT             "name_opt"
#define ENV_CERTOPT             "cert_opt"
#define ENV_EXTCOPY             "copy_extensions"
#define ENV_UNIQUE_SUBJECT      "unique_subject"
#define ENV_DATABASE            "database"

#define PROXY_EXT_SECTION   "proxy_cert_ext"
#define PROXY_CERTINFO_EXTENSION_CONF     "proxyCertInfo"

//----------
// OSSLConf --- for parsing information from openssl.cnf
//----------
  class OSSLConf {
    private:
      std::string conf_file;
      CONF* conf;
      Context* logctx; 
    public:
      OSSLConf() : conf(NULL), logctx(NULL) { };
      ~OSSLConf() { if(conf != NULL) NCONF_free(conf); };
      OSSLConf(Context* context) : logctx(context) { };
      OSSLConf(const std::string& file, Context* context) : conf_file(file), 
        conf(NULL), logctx(context) {
        load_conf_file(conf_file);
      };
      bool operator!(void) const { return !conf; };
      operator bool(void) const { return (conf != NULL); };

      bool load_conf_file(const std::string& conf_file) {
        if(!conf_file.empty()) {
          long errorline = -1;
          conf = NCONF_new(NULL);
          if(!NCONF_load(conf, conf_file.c_str(), &errorline)) {
            if (errorline <= 0) {
              logctx->LogFormat(Context::LogError,"Error when loading the config file: %s",
                  conf_file.c_str());
            }
            else {
              logctx->LogFormat(Context::LogError,"Error when loading the config file: %s on line: %ld",
                  conf_file.c_str(), errorline);
            }
            return false;
          }
        }
        return true;
      }

      char* get_item_string(const char* section_name, const char* item_name) {
        char* item = NULL;
        item = NCONF_get_string(conf, section_name, item_name);
        if(item == NULL) {
          logctx->LogFormat(Context::LogError, "Failed to find %s from the section %s", item_name, section_name);
        }
        return item;
      }

      char* get_subitem_string(const char* section_name, 
          const char* item_name, const char* sub_item_name) {
        char* sub_section_name = NULL;
        char* subitem = NULL;
        sub_section_name = get_item_string(section_name, item_name);
        if(sub_section_name != NULL) {
          subitem = get_item_string(sub_section_name, sub_item_name);
          if(subitem == NULL) {
            logctx->LogFormat(Context::LogError, "Failed to find %s from the section %s", sub_item_name, item_name);
          }
        }
        else logctx->LogFormat(Context::LogError, "Failed to find %s from the section %s", item_name, section_name);
        return subitem;
      }

      long get_item_long(const char* section_name, const char* item_name) {
        long ret = -1;
        if (!NCONF_get_number(conf, section_name, item_name, &ret))
          logctx->LogFormat(Context::LogError, "Failed to find %s from the section %s", item_name, section_name);
        return ret;
      }

      // This method applies to the case that the item point to another (second level) section
      STACK_OF(CONF_VALUE)* get_section(const char* section_name, const char* item_name) {
        char* conf_sect = NULL;
        STACK_OF(CONF_VALUE)* conf_sk = NULL;
        conf_sect = NCONF_get_string(conf, section_name, item_name);
        if(conf_sect == NULL) {
          logctx->LogFormat(Context::LogError, "Failed to find %s from the section %s", item_name, section_name);
          return NULL;
        }
        conf_sk = NCONF_get_section(conf, conf_sect);
        if(conf_sk == NULL) {
          logctx->LogFormat(Context::LogError, "Failed to get %s section", conf_sect);
          return NULL;
        }
        return conf_sk;
      }

      // This method is specific for parsing the "distinguished_name" in openssl.cnf
      // There are two seprate format for distinguished name, here only the format without 
      // prompt option is supported
      X509_NAME* get_dn() {
        AutoX509NAME dn(X509_NAME_new());
        unsigned long chtype = MBSTRING_ASC;
        STACK_OF(CONF_VALUE)* dn_sk = NULL;
        dn_sk = get_section(REQ_SECTION, DISTINGUISHED_NAME);
        if(dn_sk == NULL) {
          logctx->LogFormat(Context::LogError, "Failed to get distinguished_name section");
          return NULL;
        }
        CONF_VALUE* v;
        char *p,*q;
        char* type;
        for (int i = 0; i < sk_CONF_VALUE_num(dn_sk); i++) {
          int mval;
          v = sk_CONF_VALUE_value(dn_sk,i);
          p = q = NULL;
          type=v->name;
          for(p = v->name; *p ; p++)
#ifndef CHARSET_EBCDIC
            if ((*p == ':') || (*p == ',') || (*p == '.')) {
#else
            if ((*p == os_toascii[':']) || (*p == os_toascii[',']) || (*p == os_toascii['.'])) {
#endif
              p++; if(*p) type = p; break;
            }
#ifndef CHARSET_EBCDIC
          if (*p == '+')
#else
          if (*p == os_toascii['+'])
#endif
          {
            p++; mval = -1;
          }
          else mval = 0;
          if (!X509_NAME_add_entry_by_txt(dn,type, chtype, (unsigned char *) v->value,-1,-1,mval)) {
            logctx->Log(Context::LogError,GetOpenSSLError());
            return NULL;
          }
        }
        if(!X509_NAME_entry_count(dn)) {
          logctx->Log(Context::LogError, "No objects specified in config file");
          return NULL;
        }
        X509_NAME* name = X509_NAME_dup(dn);
        return name;
      }

      //Check the syntax of the section
      const char* load_conf_check(const char* section_name, const char* item_name) {
        char* conf_sect = NULL;
        if(section_name && item_name) {
          conf_sect = NCONF_get_string(conf, section_name, item_name);
          if(!conf_sect) { logctx->Log(Context::LogError, GetOpenSSLError()); return NULL; }
        }
        else if(section_name && !item_name) conf_sect = (char*)section_name;
        else return NULL;

        X509V3_CTX ctx;
        X509V3_set_ctx_test(&ctx);
        X509V3_set_nconf(&ctx, conf);
        if(!X509V3_EXT_add_nconf_sk(conf, &ctx, conf_sect, NULL)) {
          logctx->LogFormat(Context::LogError, "Failed to load conf section %s", conf_sect);
          return NULL;
        }
        return conf_sect;
      }

      bool load_conf_tox509(const char* section_name, const char* item_name, X509* ca, X509* cert) {
        char* conf_sect = NULL;
        if(section_name && item_name) {
          conf_sect = NCONF_get_string(conf, section_name, item_name);
          if(!conf_sect) { logctx->Log(Context::LogError, GetOpenSSLError()); return false; }
        }
        else if(section_name && !item_name) conf_sect = (char*)section_name;
        else return false;

        X509V3_CTX ctx;
        X509V3_set_ctx(&ctx, ca, cert, NULL, NULL, 0);
        X509V3_set_nconf(&ctx, conf);
        if(!X509V3_EXT_add_nconf(conf, &ctx, conf_sect, cert)) {
          logctx->LogFormat(Context::LogError, "Failed to load extension section %s to X509 object", conf_sect);
          return false;
        }
        return true;
      }
    
      // a specific method for the load of proxy extension, since the 
      // proxyCertInfo configuarion can not be recognized by openssl
      Status load_proxy_conf_tox509(X509* cert) {
        Status status;

        char* conf_pci = NCONF_get_string(conf, PROXY_EXT_SECTION, PROXY_CERTINFO_EXTENSION_CONF);
        if(conf_pci == NULL) {
          logctx->LogFormat(Context::LogInfo, "Cannot find %s from configuration file", PROXY_CERTINFO_EXTENSION_CONF);
          return Status(0);
        }

        //bool conf_pci_critical = false;
        std::string conf_pci_language;
        std::string conf_pci_pathlen;
        std::string conf_pci_policy;
        std::size_t pos1, pos2;
        std::string conf_pci_str = conf_pci;

        //if(conf_pci_str.find("critical") != std::string::npos) conf_pci_critical = true;
        pos1 = conf_pci_str.find("language");
        if(pos1 != std::string::npos) {
          pos1 = conf_pci_str.find(":", pos1);
          if(pos1 != std::string::npos) {
            pos2 = conf_pci_str.find(",", pos1);
            if(pos2 != std::string::npos)
              conf_pci_language = conf_pci_str.substr(pos1+1, pos2 - pos1 -1);
            else
              conf_pci_language = conf_pci_str.substr(pos1+1);
          }
        }
        pos1 = conf_pci_str.find("pathlen");
        if(pos1 != std::string::npos) {
          pos1 = conf_pci_str.find(":", pos1);
          if(pos1 != std::string::npos) {
            pos2 = conf_pci_str.find(",", pos1);
            if(pos2 != std::string::npos)
              conf_pci_pathlen = conf_pci_str.substr(pos1+1, pos2 - pos1 -1);
            else
              conf_pci_pathlen = conf_pci_str.substr(pos1+1);
          }
        }
        pos1 = conf_pci_str.find("policy");
        if(pos1 != std::string::npos) {
          pos1 = conf_pci_str.find(":", pos1);
          if(pos1 != std::string::npos) {
            pos2 = conf_pci_str.find(",", pos1);
            if(pos2 != std::string::npos)
              conf_pci_policy = conf_pci_str.substr(pos1+1, pos2 - pos1 -1);
            else
              conf_pci_policy = conf_pci_str.substr(pos1+1);
          }
        }

        X509ExtUtil extutil(cert, logctx);
        status = extutil.set_proxy_policy(conf_pci_policy, conf_pci_language, conf_pci_pathlen);

        return status;
      }

      bool load_conf_tox509req(const char* section_name, const char* item_name, X509_REQ* req) {
        char* conf_sect = NULL;
        if(section_name && item_name) {
          conf_sect = NCONF_get_string(conf, section_name, item_name);
          if(!conf_sect) { logctx->Log(Context::LogError, GetOpenSSLError()); return false; }
        }
        else if(section_name && !item_name) conf_sect = (char*)section_name;
        else return false;

        X509V3_CTX ctx;
        X509V3_set_ctx(&ctx, NULL, NULL, req, NULL, 0);
        X509V3_set_nconf(&ctx, conf);
        if(!X509V3_EXT_REQ_add_nconf(conf, &ctx, conf_sect, req)) {
          logctx->LogFormat(Context::LogError, "Failed to load extension section %s to X509 request object", conf_sect);
          return false;
        }
        return true;
      }

      bool load_conf_tox509crl(const char* section_name, const char* item_name, X509_CRL* crl) {
        char* conf_sect = NULL;
        if(section_name && item_name) {
          conf_sect = NCONF_get_string(conf, section_name, item_name);
          if(!conf_sect) { logctx->Log(Context::LogError, GetOpenSSLError()); return false; }
        }
        else if(section_name && !item_name) conf_sect = (char*)section_name;
        else return false;

        X509V3_CTX ctx;
        X509V3_set_ctx(&ctx, NULL, NULL, NULL, crl, 0);
        X509V3_set_nconf(&ctx, conf);
        if(!X509V3_EXT_CRL_add_nconf(conf, &ctx, conf_sect, crl)) {
          logctx->LogFormat(Context::LogError, "Failed to load extension section %s to X509 request object", conf_sect);
          return false;
        }
        return true;
      } 
  };


//----------
// CertContext Implementation
//----------
  typedef struct {
    AuthN::Utils::Cache* cache;
    int cache_count;
  } Cache_Ref;

  static std::map<std::string, Cache_Ref> cache_counter;

  //AuthN::Utils::Cache* CertContext::cache_ = NULL;
  static pthread_mutex_t cache_lock = PTHREAD_MUTEX_INITIALIZER;
  //static int cache_count = 0;

  AuthN::Utils::Cache* CertContext::create_cache() {
    AuthN::Utils::Cache* ret = NULL;
    pthread_mutex_lock(&cache_lock);
    std::map<std::string, Cache_Ref>::iterator it;
    it = cache_counter.find(ocsp_cache_file);
    if(it == cache_counter.end()) {
      AuthN::Utils::Cache* cache = new AuthN::Utils::Cache(ocsp_cache_file, context);
      Cache_Ref cache_ref;
      cache_ref.cache = cache;
      cache_ref.cache_count = 1;
      cache_counter[ocsp_cache_file] = cache_ref;
      ret = cache;
    }
    else {
      Cache_Ref cache_ref = (*it).second;
      cache_ref.cache_count++;
      ret = cache_ref.cache;
    }
    pthread_mutex_unlock(&cache_lock);
    return ret;
  }

  void CertContext::remove_cache() {
    pthread_mutex_lock(&cache_lock);
    std::map<std::string, Cache_Ref>::iterator it;
    it = cache_counter.find(ocsp_cache_file);
    if(it != cache_counter.end()) {
      Cache_Ref cache_ref = (*it).second;
      cache_ref.cache_count--;
      if(cache_ref.cache_count == 0) { 
        delete cache_ref.cache;
        cache_counter.erase(it);
      }
    }
    pthread_mutex_unlock(&cache_lock);
  }

  void CertContext::setOCSPCache(const std::string& cache_file) {
    if(cache_file.empty()) return;
    if(ocsp_cache_file.empty()) {
      ocsp_cache_file = cache_file;
      ocsp_cache_ = create_cache();
    }
    else if(cache_file.compare(ocsp_cache_file) == 0){
      // If the path is the same as the original one, the cache object 
      // should be already created.
      return;
    }
    else {
      // If the path is changed, then we need to remove the cache object
      // attached to the original path
      remove_cache();
      ocsp_cache_file = cache_file;
      ocsp_cache_ = create_cache();
    }
  }

  CertContext::~CertContext() {
    remove_cache();
    if(context) delete context;
  }

  CertContext::CertContext(Context& ctx) : ocsp_cache_(NULL) {
    context = new ContextOpenSSLUI();
    cert_.logctx = context; key_.logctx = context;

    std::string certfile = ctx.GetCertPath();
    std::string keyfile = ctx.GetKeyPath();
    std::string capath = ctx.GetCAPath();
    if(!certfile.empty()) context->SetCredentials(certfile, keyfile);
    if(!capath.empty()) context->SetCAPath(capath);

    if(!certfile.empty()) cert_.fromfile(certfile.c_str(), OSSLItem::TypeCert);
    if(!keyfile.empty()) key_.fromfile(keyfile.c_str(), OSSLItem::TypeKey);
    else if(!certfile.empty()) key_.fromfile(certfile.c_str(), OSSLItem::TypeKey); // This case for proxy certificate
    setProps();
  }

  CertContext::CertContext(Context& ctx, const char* certfilename) : ocsp_cache_(NULL) {
    context = new ContextOpenSSLUI();
    cert_.logctx = context; key_.logctx = context;

    std::string certfile = ctx.GetCertPath();
    std::string keyfile = ctx.GetKeyPath();
    std::string capath = ctx.GetCAPath();
    if(!certfile.empty()) context->SetCredentials(certfile, keyfile);
    if(!capath.empty()) context->SetCAPath(capath);

    if(certfilename != NULL) {
      context->SetCredentials(certfilename, "");
      cert_.fromfile(certfilename, OSSLItem::TypeCert);
      key_.fromfile(certfilename, OSSLItem::TypeKey); // This case for proxy certificate
      setProps();
    }
  }

  CertContext::CertContext(Context& ctx, const char* certfilename, 
      const char* keyfilename) : ocsp_cache_(NULL) {
    context = new ContextOpenSSLUI();
    cert_.logctx = context; key_.logctx = context;

    std::string certfile = ctx.GetCertPath();
    std::string keyfile = ctx.GetKeyPath();
    std::string capath = ctx.GetCAPath();
    if(!certfile.empty()) context->SetCredentials(certfile, keyfile);
    if(!capath.empty()) context->SetCAPath(capath);

    context->SetCredentials(certfilename, keyfilename);

    if(certfilename != NULL)
      cert_.fromfile(certfilename, OSSLItem::TypeCert);
    if(keyfilename != NULL)
      key_.fromfile(keyfilename, OSSLItem::TypeKey);
    else if(certfilename != NULL)
      key_.fromfile(certfilename, OSSLItem::TypeKey); // This case for proxy certificate
  }

  CertContext::CertContext(Context& ctx, const std::string& certstr) : ocsp_cache_(NULL) {
    context = new ContextOpenSSLUI();
    cert_.logctx = context; key_.logctx = context;

    std::string certfile = ctx.GetCertPath();
    std::string keyfile = ctx.GetKeyPath();
    std::string capath = ctx.GetCAPath();
    if(!certfile.empty()) context->SetCredentials(certfile, keyfile);
    if(!capath.empty()) context->SetCAPath(capath);

    if(!certstr.empty()) {
      cert_.fromstr(certstr, OSSLItem::TypeCert);
      key_.fromstr(certstr, OSSLItem::TypeKey);
    }
  }

  CertContext::CertContext(Context& ctx, const std::string& certstr, 
      const std::string& keystr) : ocsp_cache_(NULL) {
    context = new ContextOpenSSLUI();
    cert_.logctx = context; key_.logctx = context;

    std::string certfile = ctx.GetCertPath();
    std::string keyfile = ctx.GetKeyPath();
    std::string capath = ctx.GetCAPath();
    if(!certfile.empty()) context->SetCredentials(certfile, keyfile);
    if(!capath.empty()) context->SetCAPath(capath);

    if(!certstr.empty()) cert_.fromstr(certstr, OSSLItem::TypeCert);
    if(!keystr.empty()) key_.fromstr(keystr, OSSLItem::TypeKey);
    else if(certstr.empty()) key_.fromstr(keystr, OSSLItem::TypeKey);
  }

  Status CertContext::createSelfSigned(const CertificateOptions& opts, KeyContext& key) {

    AutoX509 x509(X509_new()); 
    X509_NAME* name = NULL;
    unsigned long chtype = MBSTRING_ASC;
    std::string sub_name = opts.subject();
    if(!sub_name.empty()) {
      name = parse_name(context, (char*)(sub_name.c_str()), chtype, 0);
      X509_set_subject_name(x509, name);
      X509_set_issuer_name(x509, name);
      X509_NAME_free(name);
    }
    else {
      context->Log(Context::LogError, "Subject name is empty"); 
      return (last_error_ = Status(-1, "Subject name is empty")); 
    }

    //Set version to V3
    if(!X509_set_version(x509, 2)) {                    
      context->Log(Context::LogError, "Failed to set the V3 version for the X509 object");
      return (last_error_ = Status(-1, "Failed to set the V3 version for X509 object"));
    }

    // Try to find the information from CertificateOptions at first,
    // if not found, then try to find the information from openssl.cnf

    // Get information from CertificateOptions
    long serial = -1;
    std::string subject;
    Constraints certConstraints;
    std::vector<std::string> certPolicies;
    std::vector<std::string> crlLocations;
    std::vector<std::string> issuerLocations;
    std::vector<std::string> ocspLocations;
    //bool isCA;
    int pathLimit;

    serial = opts.serialNumber();
    subject = opts.subject();
    certConstraints = opts.constraints();
    certPolicies = opts.policies();
    crlLocations = opts.crlLocations();
    issuerLocations = opts.issuerLocations();
    ocspLocations = opts.ocspLocations();
    //isCA = opts.isCA();
    pathLimit = opts.pathLimit();

    // Get information from openssl.cnf, if exists
    // A self signed certificate must be a CA certificate
    std::string serfile;
    const char* x509_ext = NULL;
    int default_days = -1;
    std::string default_md;
    std::string distinguished_name;    
 
    if(!conf_file.empty()) {
      OSSLConf ossl_conf(conf_file, context);
      if((bool)ossl_conf) {
        char* serf = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_SERIAL);
        if(serf != NULL) serfile = serf;

        // For a CA certificate, the "x509_extensions" subsection of "req" section 
        // will be used for configure certificate extension
        x509_ext = ossl_conf.load_conf_check(REQ_SECTION, X509_EXTENSIONS_CONF);
        // Add extensions
        if(x509_ext != NULL) ossl_conf.load_conf_tox509(REQ_SECTION, X509_EXTENSIONS_CONF, x509, x509);

        char* days = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_DEFAULT_DAYS);        
        if(days!=NULL) default_days = atoi(days);

        char* md = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_DEFAULT_MD);
        if(md!=NULL) default_md = md;
      
        //TODO: other useful information from configuration file
      }
    }

    // Set serial number
    if(serial != -1) {
      ASN1_INTEGER* asn1 = NULL; 
      if((asn1 = long_to_asn1int(serial)) != NULL) { 
        X509_set_serialNumber(x509, asn1); 
        ASN1_INTEGER_free(asn1); 
      }
    } else if(!serfile.empty()) {
      ASN1_INTEGER* asn1 = NULL;
      asn1 = x509_load_serial(*context, "", serfile);
      if(asn1 != NULL) {
        X509_set_serialNumber(x509, asn1);
        ASN1_INTEGER_free(asn1);
      }
      else rand_serial(NULL, X509_get_serialNumber(x509));
    }
    else {
      rand_serial(NULL, X509_get_serialNumber(x509));
    }

    // Set notBefore and notAfter
    AuthN::Utils::Time from = opts.notValidBefore();
    AuthN::Utils::Time till = opts.notValidAfter();
    if((till <= from) && (default_days != -1)) {
      from = AuthN::Utils::Time();
      till = from + default_days * 24 * 3600;
    }
    ASN1_TIME_set(X509_get_notBefore(x509), from.GetTime());
    ASN1_TIME_set(X509_get_notAfter(x509), till.GetTime());

    X509ExtUtil extutil(x509, context);

    //Set subjectKeyIdentifier
    //extutil.set_subject_key_id(std::string("hash"));

    //Set authorityKeyIdentifier
    //extutil.set_authority_key_id(std::string("keyid:always,issuer:always"));

    //Set basic constraints
    extutil.set_basic_constraints(true, pathLimit);

    certConstraints += KeyCertificateSign;
    certConstraints += CRLSign;

    //Set key usage
    extutil.set_key_usage(certConstraints);

    //Set ext key usage
    extutil.set_ext_key_usage(certConstraints);   

    //Set certitificate policy
    extutil.set_cert_policies(certPolicies);

    // Set public key
    EVP_PKEY* privkey;
    privkey = key.getKey(); 
    if(!privkey) {
      context->Log(Context::LogError, "Failed to load EVP_PKEY object from KeyContext");
      return (last_error_ = Status(-1, "Failed to load EVP_PKEY object from KeyContext"));
    } 
    X509_set_pubkey(x509, privkey);

    //Load digest from configuration
    const EVP_MD* digest = NULL;
    if(!default_md.empty()) {
      digest = get_digest(*context, default_md.c_str(), privkey);
    }
    if(digest == NULL) digest = EVP_sha1();
#ifndef OPENSSL_NO_DSA
    if (privkey->type == EVP_PKEY_DSA)
      digest=EVP_dss1();
#endif
#ifdef EVP_PKEY_EC
#ifndef OPENSSL_NO_ECDSA
    if (privkey->type == EVP_PKEY_EC)
      digest = EVP_ecdsa();
#endif
#endif

    // Sign the certificate
    if (!(X509_sign(x509, (EVP_PKEY*)privkey,digest))) {
      context->LogFormat(Context::LogError, "Failed to sign X509 certificate: %s", GetOpenSSLError().c_str());
      return (last_error_ = Status(-1, "Failed to sign X509 certificate"));
    }

    // Set the X509 item attached to this object
    cert_.reset();
    cert_.cert = X509_dup(x509);

    // Set the CertContextProps 
    setProps();
   
    // Set the EVP_PKEY item attached to this object
    key_.reset();
    RSA* priv_rsa = NULL;
    if(privkey->type == EVP_PKEY_RSA && privkey->pkey.rsa != NULL) {
      priv_rsa = privkey->pkey.rsa;
      key_.pkey = EVP_PKEY_new();
      EVP_PKEY_set1_RSA(key_.pkey, priv_rsa);
    }

    return Status(0);
  }

  CertContext* CertContext::signRequest(const CertificateOptions& opts, CSRContext& csr) {
    CertContext* ret = NULL;
    if((cert_.cert == NULL) || (key_.pkey == NULL)) {
      if(cert_.cert == NULL)
        context->Log(Context::LogError,"Issuer certificate does not exist");
      else
        context->Log(Context::LogError,"Issuer private key does not exist");
      return NULL;
    }

    const CertContextProps& csr_props = *csr.getProps();

    if (!X509_check_private_key(cert_.cert, key_.pkey)) {
      context->Log(Context::LogError,"Issuer certificate and issuer private key do not match");
      return NULL;
    }

    // If the issuer is not a CA certificate, then the certificate 
    // to sign is a proxy certificate
    bool isProxy = false;
    if(!(props_.isCA)) isProxy = true;

    AutoX509 x509(X509_new()); 

    // Set issuer
    X509_set_issuer_name(x509, X509_get_subject_name(cert_.cert));

    //Set version to V3
    if(!X509_set_version(x509, 2)) {                    
      context->Log(Context::LogError, "Failed to set the V3 version for the X509 object");
      return NULL;
    }

    // Try to find the information from CertificatOption at first,
    // if not found, then try to find information from the properties of CSR,
    // if again not found, then try to find the information from openssl.cnf

    // Get information from CertificateOptions
    long serial = -1;
    std::string subject;
    Constraints certConstraints;
    std::vector<std::string> certPolicies;
    std::vector<std::string> crlLocations;
    std::vector<std::string> issuerLocations;
    std::vector<std::string> ocspLocations;
    bool isCA;
    int pathLimit;
    std::string proxyPolicy;

    serial = opts.serialNumber();
    subject = opts.subject();
    certConstraints = opts.constraints();
    certPolicies = opts.policies();
    crlLocations = opts.crlLocations();
    issuerLocations = opts.issuerLocations();
    ocspLocations = opts.ocspLocations();
    isCA = opts.isCA();
    pathLimit = opts.pathLimit();
    proxyPolicy = opts.proxyPolicy();
    if(proxyPolicy.empty()) {
      //Try to retrieve proxy policy from X509 request
      proxyPolicy = csr_props.proxyPolicy.value;
    }

    // Get information from openssl.cnf, if exists
    std::string serfile;
    int default_days = -1;
    std::string default_md;
   
    if(!conf_file.empty()) {
      OSSLConf ossl_conf(conf_file, context);
      if((bool)ossl_conf) {
        if(!isProxy) {
          char* serf = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_SERIAL);
          if(serf != NULL) serfile = serf;

          // For an EEC certificate, the "x509_extensions" subsection of "CA_default" section 
          // will be used for configure certificate extension
          char* ca_default_section = ossl_conf.get_item_string(CA_SECTION, DEFAULT_CA);
          if(ca_default_section != NULL) {
            const char* usr_ext = ossl_conf.load_conf_check(ca_default_section, ENV_EXTENSIONS);
            if(usr_ext != NULL) ossl_conf.load_conf_tox509(ca_default_section, 
              ENV_EXTENSIONS, cert_.cert, x509);
          }

          char* days = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_DEFAULT_DAYS);        
          if(days!=NULL) default_days = atoi(days);
        }
        else {
          // For a proxy certificate, the "proxy_cert_ext"
          // section will be used for configure certificate extension
          if(proxyPolicy.empty() && !ossl_conf.load_proxy_conf_tox509(x509)) 
            context->Log(Context::LogError, "Failed to load proxy configration");
        }
      } 
      char* md = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_DEFAULT_MD);
      if(md!=NULL) default_md = md;
    }

    // Set subject
    if(!isProxy) {
      X509_NAME* name = NULL;
      unsigned long chtype = MBSTRING_ASC;
      std::string sub_name = opts.subject();
      if(sub_name.empty()) sub_name = csr_props.subject;
      if(!sub_name.empty()) {
        name = parse_name(context, (char*)(sub_name.c_str()), chtype, 0);
        X509_set_subject_name(x509, name);
        X509_NAME_free(name);
      }
      else return NULL;
    }
    else {
      if(!insert_proxy_cn(x509, cert_.cert, csr.getReq())) {
        context->Log(Context::LogError, "Failed to and CN name to proxy"); return NULL;
      }
    }

    // Set serial number
    if(serial != -1) {
      ASN1_INTEGER* asn1 = NULL; 
      if((asn1 = long_to_asn1int(serial)) != NULL) { 
        X509_set_serialNumber(x509, asn1); 
        ASN1_INTEGER_free(asn1); 
      }
    } else if(!serfile.empty()) {
      ASN1_INTEGER* asn1 = NULL;
      asn1 = x509_load_serial(*context, "", serfile);
      if(asn1 != NULL) {
        X509_set_serialNumber(x509, asn1);
        ASN1_INTEGER_free(asn1);
      }
      else rand_serial(NULL, X509_get_serialNumber(x509));
    }
    else {
      X509_set_serialNumber(x509, X509_get_serialNumber(cert_.cert));
      //rand_serial(NULL, X509_get_serialNumber(x509));
    }

    // Set notBefore and notAfter
    AuthN::Utils::Time from = opts.notValidBefore();
    AuthN::Utils::Time till = opts.notValidAfter();
    if((till <= from) && (default_days != -1)) {
      from = AuthN::Utils::Time();
      till = from + default_days * 24 * 3600;
    }
    ASN1_TIME_set(X509_get_notBefore(x509), from.GetTime());
    ASN1_TIME_set(X509_get_notAfter(x509), till.GetTime());

    // Set X509 extenstions
    X509ExtUtil extutil(x509, context);

    //Set subjectKeyIdentifier
    //extutil.set_subject_key_id(std::string("hash"));

    //Set authorityKeyIdentifier
    //extutil.set_authority_key_id(std::string("keyid:always,issuer:always"));

    //Set basic constraints
    int path_limit = csr_props.pathLimit;
    if((pathLimit > path_limit) && (path_limit != -1))
      pathLimit = path_limit;

    bool ca = isCA && csr_props.isCA;
    extutil.set_basic_constraints(ca, pathLimit); // The path limit set by signer take precedence

    if(certConstraints.Size() == 0)
      certConstraints = csr_props.constraints;
    if(certConstraints.Size() == 0) 
      certConstraints = props_.constraints;

    //Set key usage
    extutil.set_key_usage(certConstraints, false);
    //Set ext key usage
    extutil.set_ext_key_usage(certConstraints);   

    //Set proxy policy
    if(isProxy) {
      if(!extutil.set_proxy_policy(proxyPolicy))
        context->Log(Context::LogError, "Failed to load proxy policy");
    }

    //Set VOMS extension from CSR
    std::string ext_name = "acseq";
    AuthN::Credentials::Extension ext;
    AuthN::Status stat = csr.getCSRExtension(ext_name, ext);
    if(stat == AuthN::Status(0)) {
      extutil.set_extension(ext);
    }

    //Set certitificate policy
    for(unsigned int i = 0; i<csr_props.policies.size(); i++)
      certPolicies.push_back(csr_props.policies[i]);
    extutil.set_cert_policies(certPolicies);

    // Set public key
    EVP_PKEY* pubkey = NULL;
    X509_REQ* req = csr.getReq();
    if(req)
      pubkey = X509_REQ_get_pubkey(req);
    if(!pubkey) {
      context->Log(Context::LogError, "Failed to load public key from X509_REQ object");
      return NULL;
    } 
    X509_set_pubkey(x509, pubkey);

    // Load digest from configuration
    EVP_PKEY* iss_privkey = NULL;
    iss_privkey = key_.pkey;
    const EVP_MD* digest = NULL;
    if(!default_md.empty()) {
      digest = get_digest(*context, default_md.c_str(), iss_privkey);
    }
    if(digest == NULL) digest = EVP_sha1();
#ifndef OPENSSL_NO_DSA
    if (iss_privkey->type == EVP_PKEY_DSA)
      digest=EVP_dss1();
#endif
#ifdef EVP_PKEY_EC
#ifndef OPENSSL_NO_ECDSA
    if (iss_privkey->type == EVP_PKEY_EC)
      digest = EVP_ecdsa();
#endif
#endif

    // Sign the certificate
    if (!(X509_sign(x509, (EVP_PKEY*)iss_privkey,digest))) {
      context->Log(Context::LogError,GetOpenSSLError());
      return NULL;
    }

    AuthN::Context emptyctx(AuthN::Context::EmptyContext);
    ret = new CertContext(emptyctx);
    ret->certFromX509(x509);

    ret->setProps();
    return ret;
  }

  void CertContext::setProps() {
    X509* x509 = NULL;
    CertContextProps props;

    if(!cert_) return;
    x509 = cert_.cert;
    if(x509 == NULL) return;

    props.version = X509_get_version(x509);

    ASN1_INTEGER* serial = X509_get_serialNumber(x509);
    if(serial) {
      char* chs = i2s_ASN1_INTEGER(NULL, serial);
      std::string str = chs;
      OPENSSL_free(chs);
      props.serial = atoi(str.c_str());
    }

    props.start = asn1_to_utctime(X509_get_notBefore(x509));
    props.end = asn1_to_utctime(X509_get_notAfter(x509));

    props.subject = get_subject_name(x509);
    props.issuer = get_issuer_name(x509);
    props.identity = get_identity_name(x509);

    unsigned long hash = X509_NAME_hash(x509->cert_info->subject);
    char hash_buf[16];
    std::string hash_str;
    snprintf(hash_buf, sizeof(hash_buf), "%0lx", hash);
    hash_str = hash_buf;
    props.hash = hash_str;

    props.is_selfsigned = ( X509_V_OK == X509_check_issued(x509, x509) );

    props.isCA = false;
    props.pathLimit = -1;

    X509ExtUtil extutil(x509, context);
    extutil.get_basic_constraints(&props.isCA, &props.pathLimit);

#ifdef HAVE_OPENSSL_PROXY
    int pos = X509_get_ext_by_NID(x509, NID_proxyCertInfo, -1);
    if(pos != -1) props.isProxy = true;
#endif

    if(props.isProxy) extutil.get_proxy_policy(props.proxyPolicy);

    props.constraints = extutil.get_key_usage();
    props.constraints += extutil.get_ext_key_usage();

    extutil.get_cert_policies(props.policies);

    if(x509->signature) {
      int length = x509->signature->length;
      char* buffer = new char[length];
      for (int i=0; i< length; i++)
        buffer[i] = x509->signature->data[i];
      props.sigData.assign(buffer, length);
      delete[] buffer;
    }

    char buf[256];
    int buflen;
    buflen = OBJ_obj2txt(buf, sizeof(buf), x509->cert_info->signature->algorithm, 0);
    props.sigAlg.assign(buf, buflen);

    props.subjectId = extutil.get_subject_key_id();
    props.issuerId = extutil.get_authority_key_id();

    props_ = props;

  }

  const CertContextProps* CertContext::getProps() const { return &props_; }

  void CertContext::certToDER(std::string& out) const {
    cert_.tostrDER(out);
  }
  
  void CertContext::certToPEM(std::string& out) const {
    cert_.tostrPEM(out);
  }

  bool CertContext::certFromDER(const std::string& in) {
    bool res = cert_.fromstrDER(in, OSSLItem::TypeCert);
    if(res) setProps();
    return res;
  }

  bool CertContext::certFromPEM(const std::string& in) {
    bool res = cert_.fromstrPEM(in, OSSLItem::TypeCert);
    if(res) setProps();
    return res;
  }

  void CertContext::certFromX509(X509* x509) {
    if(x509) { 
      cert_.reset(); 
      cert_.cert = X509_dup(x509);
      setProps();
    }
  }

  void CertContext::certFromX509(X509* x509, STACK_OF(X509)* chain) {
    if(x509 || chain) {
      cert_.reset();
      if(x509) cert_.cert = X509_dup(x509);
      if(chain && (sk_X509_num(chain) > 0)) {
        cert_.cert_chain = sk_X509_new_null();
        for(int n = 0; n < sk_X509_num(chain) ; n++) {
          X509* tmp = X509_dup(sk_X509_value(chain, n));
          sk_X509_push(cert_.cert_chain, tmp);
        }
      }
      setProps();
    }
  }

  bool CertContext::certFromStr(const std::string& in) {
    bool res = cert_.fromstr(in, OSSLItem::TypeCert);
    if(res) setProps();
    return res;
  }

  void CertContext::keyToDER(std::string& out) const {
    key_.tostrDER(out);
  }

  void CertContext::keyToPEM(std::string& out, bool enc, const std::string& passphrase) const {
    key_.tostrPEM(out, enc, passphrase);
  }

  bool CertContext::keyFromDER(const std::string& in) {
    bool res = key_.fromstrDER(in, OSSLItem::TypeKey);
    return res;
  }

  bool CertContext::keyFromPEM(const std::string& in) {
    bool res = key_.fromstrPEM(in, OSSLItem::TypeKey);
    return res;
  }

  bool CertContext::keyFromStr(const std::string& in) {
    bool res = key_.fromstr(in, OSSLItem::TypeKey);
    return res;
  }

  CertContext* CertContext::getChainCert(int pos) {
    CertContext* cert = new CertContext(*context);
    if(!cert) return NULL;
    if(pos == 0) {
      cert->certFromX509(cert_.cert);
    }
    else {
      if(cert_.cert_chain && (sk_X509_num(cert_.cert_chain) != 0) && 
         (pos-1 < sk_X509_num(cert_.cert_chain))) {
        cert->certFromX509(sk_X509_value(cert_.cert_chain, pos-1));
      }
      else { delete cert; return NULL; }
    }
    return cert;
  }

  bool CertContext::getCertExtension(int pos, AuthN::Credentials::Extension& ext) const {
    X509ExtUtil extutil(cert_.cert, context);
    return extutil.get_extension(pos, ext);
  }

  bool CertContext::getCertExtension(const std::string& name, 
      AuthN::Credentials::Extension& ext) const {
    X509ExtUtil extutil(cert_.cert, context);
    return extutil.get_extension(name, ext);
  }
  
  bool CertContext::getCertAttributes(const std::string& name, 
      std::list<std::string>& attrs) const {
    X509ExtUtil extutil(cert_.cert, context);
    return extutil.get_attributes(name, attrs);
  }

  X509* CertContext::getCert() {
    return cert_.cert;
  }

  EVP_PKEY* CertContext::getKey() {
    return key_.pkey;
  }

  bool CertContext::operator==(const CertContext& other) {
    //TODO
    return false;
  }

  bool CertContext::isIssuerOf(const CertContext *other) const {
    //TODO
    return false;
  }

  //We need an explicit checking of the valitidy of crl here, since
  //we need to get the validity for the validate mode option ValidationCRLIfValid
  static Status verify_crl(Context& context, const std::string& crl_file, 
      const std::string& CApath) {
    Status ret = Status(-1);
    X509_CRL* crl = NULL;
    X509_STORE* verify_store = NULL;
    X509_STORE_CTX ctx;
    X509_OBJECT obj;
    EVP_PKEY* pkey = NULL;
    int i;

    CRLContext crl_ctx(&context, crl_file.c_str());
    crl = crl_ctx.getCRL();
    if(!crl) goto end;
    verify_store = setup_verify(context, "", CApath);
    if(!verify_store) goto end;

    if(!X509_STORE_CTX_init(&ctx, verify_store, NULL, NULL)) {
      X509_STORE_free(verify_store); verify_store = NULL;
      ret = Status(-1, "Failed to initialize X509 store");
      goto end;
    }
    //loadCRLFile(context, crl_file, crl);
    //if(!crl) goto end;
    i = X509_STORE_get_by_subject(&ctx, X509_LU_X509, X509_CRL_get_issuer(crl), &obj);
    if(i <= 0) {
      ret = Status(-1, "Failed to get CRL issuer certificate");
      goto end;
    }
    pkey = X509_get_pubkey(obj.data.x509);
    if(!pkey) {
      ret = Status(-1, "Failed to get CRL issuer public key");
      goto end;
    }
    i = X509_CRL_verify(crl, pkey);
    if(i <= 0) {
      ret = Status(-1, "CRL verification failed");
      // TODO: extract exact error
      goto end;
    }
    ret = Status(0);

    end:
    X509_OBJECT_free_contents(&obj);
    if(pkey) EVP_PKEY_free(pkey);
    //if(crl) X509_CRL_free(crl);
    if(verify_store) {
      X509_STORE_CTX_cleanup(&ctx);
      X509_STORE_free(verify_store);
    }
    return ret;
  }

  Status CertContext::validate(const AuthN::Validator::ValidationMode& mode, AuthN::Context* ctx) {
    std::vector<CertContext*> trusted;
    std::vector<CertContext*> untrusted;
    std::vector<CRLContext*> crls;
    return validate(mode, trusted, untrusted, crls, ctx);
  }

  Status CertContext::validate(const AuthN::Validator::ValidationMode& mode, 
      const std::vector<CertContext*>& trusted, 
      const std::vector<CertContext*>& untrusted, 
      const std::vector<CRLContext*>&crls, 
      AuthN::Context* ctx) {
    Status status(-1);
    std::string capath;
    X509_STORE* cert_store = NULL;
    X509_STORE_CTX* store_ctx = NULL;
    AuthN::Context* validate_ctx = NULL;

    if(ctx!=NULL) validate_ctx = ctx;
    else validate_ctx = context;

    capath = validate_ctx->GetCAPath();
    if(capath.empty()) cert_store = X509_STORE_new();
    else cert_store = setup_verify(*validate_ctx, "", capath);

    X509_STORE_set_verify_cb_func(cert_store, verify_callback);

    X509* cert = NULL; // the certificate to be validated
    STACK_OF(X509)* untrusted_certchain = NULL;
    STACK_OF(X509)* trusted_certchain = NULL;
    STACK_OF(X509_CRL)* crl_stack = NULL;

    untrusted_certchain = sk_X509_new_null();
    if(trusted.size()) trusted_certchain = sk_X509_new_null();
    if(crls.size()) crl_stack = sk_X509_CRL_new_null();

    for(unsigned int i=0; i < trusted.size(); i++) {
      X509* tmp = NULL; tmp = X509_dup(trusted[i]->getCert());
      sk_X509_push(trusted_certchain, tmp);
    }

    for(unsigned int i=0; i < untrusted.size(); i++) {
      X509* tmp = NULL; tmp = X509_dup(untrusted[i]->getCert());
      sk_X509_push(untrusted_certchain, tmp);
    }
    if(cert_ && cert_.cert_chain) {
      for(int i=0;i<sk_X509_num(cert_.cert_chain);i++) {
        X509* tmp = X509_dup(sk_X509_value(cert_.cert_chain,i));
        sk_X509_push(untrusted_certchain, tmp);
      }
    }
    for(unsigned int i=0; i < crls.size(); i++) {
      X509_CRL* tmp = NULL; tmp = X509_CRL_dup(crls[i]->getCRL());
      if(tmp) sk_X509_CRL_push(crl_stack, tmp);
    }

    if(cert_) { cert = cert_.cert; }
    if(!cert) {
      validate_ctx->Log(Context::LogError, "Certificate to validate is not defined");
      status = Status(-1,"Certificate to validate is not defined");
      goto err;
    }
   
    X509_STORE_set_flags(cert_store, 0);

    if (X509_STORE_load_locations(cert_store, NULL, capath.empty() ? NULL:capath.c_str())) {
      store_ctx = X509_STORE_CTX_new();
      X509_STORE_CTX_init(store_ctx, cert_store, cert, untrusted_certchain);
      if(trusted_certchain != NULL) X509_STORE_CTX_trusted_stack(store_ctx, trusted_certchain);
#if OPENSSL_VERSION_NUMBER > 0x00908000L
      if (crl_stack && sk_X509_CRL_num(crl_stack)!=0) 
        X509_STORE_CTX_set0_crls(store_ctx, crl_stack);
#endif

      // override the check_issued with our version
      store_ctx->check_issued = check_issued;

      //Handle the CRL
      //Check if the corresponding CRL file is available
      Status crl_valid = Status(-1);
      Status crl_present = Status(-1);
      if(crl_stack && sk_X509_CRL_num(crl_stack)!=0) crl_present = Status(0);
      if(!(mode & AuthN::Validator::ValidationCRLNone) && (crl_present != Status(0))) {
        crl_present = Status(0);
        std::string hash_str;
        char hash[32];
        memset(hash, 0, 32);
        snprintf(hash, 32, "%08lx", X509_issuer_name_hash(cert));
        hash_str = hash;
        std::string crl_loc;
        crl_loc = capath + "/" + hash_str + ".r0";
        struct stat st;
        if(stat(crl_loc.c_str(),&st) != 0) {
          crl_loc = capath + "/" + hash_str + ".crl";
          if(stat(crl_loc.c_str(),&st) != 0) {
            crl_present = Status(-1,"Failed to find the crl file under " + capath + "/" + hash_str + ".[r0|crl]");
          }
        }
        if(crl_present) {
          if(!S_ISREG(st.st_mode)) {
            crl_present = Status(-1, "The location: " + crl_loc + " is not a regular file");
          }
        }
        if(crl_present) {
          validate_ctx->LogFormat(Context::LogVerbose, "The location of crl file: %s ",crl_loc.c_str());
          X509_LOOKUP* lookup = NULL;
          lookup = X509_STORE_add_lookup(cert_store,X509_LOOKUP_file());
          if(!lookup || (!X509_load_crl_file(lookup,crl_loc.c_str(), X509_FILETYPE_PEM)) ) {
            crl_present = Status(-1, "Failed to load crl file: " + crl_loc);
          }
        }
        if(crl_present) {
          //Check if the crl is valid
          if(mode & AuthN::Validator::ValidationCRLIfValid) { //Only check validity if the mode is set
            crl_valid = verify_crl(*validate_ctx, crl_loc, capath);
          }
        }
      }
      bool check_crl = false;
      bool allow_proxy = true; // Allow proxy by default
      // TODO: clean mess below. All combinnations need to be handled
      // properly. Errors need to be propagated.
      if((crl_valid && (mode & AuthN::Validator::ValidationCRLIfValid)) ||
         (crl_present && (mode & AuthN::Validator::ValidationCRLIfPresent)) ||
         (mode & AuthN::Validator::ValidationCRLMandatory)) {
        check_crl = true;
      }
      if(mode & AuthN::Validator::ValidationProxyAllow) allow_proxy = true;
      else if(mode & AuthN::Validator::ValidationProxyDisallow) allow_proxy = false;
      if(check_crl && allow_proxy) {
        unsigned long flags = 0;
#ifdef HAVE_OPENSSL_PROXY
        flags |= X509_V_FLAG_ALLOW_PROXY_CERTS;
#endif
//#if OPENSSL_VERSION_NUMBER  > 0x009080ffL
        flags |= X509_V_FLAG_CRL_CHECK;
        flags |= X509_V_FLAG_CRL_CHECK_ALL;
//#endif  
#ifdef X509_V_FLAG_POLICY_CHECK
        flags |= X509_V_FLAG_POLICY_CHECK;
#endif
//#if OPENSSL_VERSION_NUMBER  > 0x10000000L
#ifdef X509_V_FLAG_EXTENDED_CRL_SUPPORT
        flags |= X509_V_FLAG_EXTENDED_CRL_SUPPORT;      
#endif
        X509_STORE_CTX_set_flags(store_ctx, flags);

/*
#ifdef HAVE_OPENSSL_PROXY
// TODO: exact version number to be identified
#if (OPENSSL_VERSION_NUMBER  > 0x009080ffL)
        X509_STORE_CTX_set_flags(store_ctx, X509_V_FLAG_CRL_CHECK | X509_V_FLAG_CRL_CHECK_ALL | X509_V_FLAG_ALLOW_PROXY_CERTS | X509_V_FLAG_POLICY_CHECK | X509_V_FLAG_EXTENDED_CRL_SUPPORT);
#else
        X509_STORE_CTX_set_flags(store_ctx, X509_V_FLAG_CRL_CHECK | X509_V_FLAG_CRL_CHECK_ALL | X509_V_FLAG_ALLOW_PROXY_CERTS | X509_V_FLAG_POLICY_CHECK);
#endif
#else
        validate_ctx->LogFormat(Context::LogError, "The proxy certificate is not allowed by OpenSSL with version: %0x", OPENSSL_VERSION_NUMBER);
        X509_STORE_CTX_set_flags(store_ctx, X509_V_FLAG_CRL_CHECK | X509_V_FLAG_POLICY_CHECK | X509_V_FLAG_EXTENDED_CRL_SUPPORT);
#endif
*/
      }
      else if(!check_crl && allow_proxy){
#ifdef HAVE_OPENSSL_PROXY
        X509_STORE_CTX_set_flags(store_ctx, X509_V_FLAG_ALLOW_PROXY_CERTS);
#else
        validate_ctx->LogFormat(Context::LogError, "The proxy certificate is not allowed by OpenSSL with version: %0x", OPENSSL_VERSION_NUMBER);
#endif
      }
      else if(check_crl && !allow_proxy) {
        X509_STORE_CTX_set_flags(store_ctx, X509_V_FLAG_CRL_CHECK);
      }

      //Handle the OCSP
      //get the ocsp responder url from AIA (Authority Info Access, authorityInfoAccess) of EEC
      //alternatively, the responder url could be set as configuration info
      OCSPContext::CertStatus cert_status = OCSPContext::CertStatusNotRequired;
#ifdef HAVE_OPENSSL_PROXY
      int pos1 = X509_get_ext_by_NID(cert, NID_proxyCertInfo, -1);
      if(pos1 < 0)
#endif
      {
        std::string aia_name = "authorityInfoAccess";
        bool check_ocsp = false;
        std::string ocsp_uri;
        //Credentials::Extension e;
        //bool res = getCertExtension(aia_name, e);
        //std::string str = e.value;
        std::list<std::string> attrs;
        bool res = getCertAttributes(aia_name, attrs);
        std::string str = "";
        std::list<std::string>::iterator it;
        for(it = attrs.begin(); it != attrs.end(); it++) {
          str = *it;
          //authorityInfoAccess = OCSP;URI:http://ocsp.my.host/
          std::string::size_type find = str.find("OCSP");
          if(find != std::string::npos) {
            find = str.find("URI:", find+1);
            if(find != std::string::npos) {
              ocsp_uri = str.substr(find+4);
            }
          }
        }
        validate_ctx->LogFormat(Context::LogVerbose, "The URI of the OCSP responder is: %s", ocsp_uri.c_str());

        // TODO: Fix error propagation - failed CRL should mean failed validation.
        // TODO: Move log into Status. Probably convert check_ocsp into Status or
        //   do something better.

        if(mode & AuthN::Validator::ValidationOCSPMandatory) {
          if(!res) {
            validate_ctx->Log(Context::LogError,"Failed to get AIA information from certificate extension");
          } else if(ocsp_uri.empty()) {
            validate_ctx->Log(Context::LogError, "Failed to get OCSP information from AIA");
          } else {
            check_ocsp = true;
          }
        }
        else if(mode & AuthN::Validator::ValidationOCSPIfPresent) {
          if(!res) {
            validate_ctx->Log(Context::LogVerbose,"Unable to get AIA information from certificate extension");
          } else if(ocsp_uri.empty()) {
            validate_ctx->Log(Context::LogVerbose, "Unable to get OCSP information from AIA");
          } else {
            check_ocsp = true;
          }
        }

        if(check_ocsp) {
          cert_status = OCSPContext::CertStatusUnknown;

          std::string resp_str;
          std::string cert_path = validate_ctx->GetCertPath();
          std::string key_path = validate_ctx->GetKeyPath();

          const std::string cache_file = "/tmp/ocsp_cache";
          setOCSPCache(cache_file);
          OCSPContext ocsp_ctx(validate_ctx, cert, ocsp_cache_);
          ocsp_ctx.SetRequestSigner(cert_path.c_str(), key_path.c_str());

          // Look for the OCSP response in the cache
          status = ocsp_ctx.RetrieveFromCache();
          if(status) {
            status = ocsp_ctx.CheckOCSPResponse();
            if(!status) {
            // Need to contact responder
              context->Log(Context::LogError, status.GetDescription());
            }
          }
          else context->Log(Context::LogError, status.GetDescription());

          // If not found, contact the OCSP responder
          // TODO: give failure if both the cache and ocsp responder fails?
          if(!status) {
            status = ocsp_ctx.MakeOCSPRequest();
            if(status) {
              status = ocsp_ctx.QueryOCSPResponder();
              if(status) {
                cert_status = ocsp_ctx.GetCertStatus();
/*
                switch(cert_status) {
                  case OCSPContext::CertStatusGood: 
                    X509_STORE_CTX_set_error(store_ctx, X509_V_OK); break;
                  case OCSPContext::CertStatusRevoked:  
                    X509_STORE_CTX_set_error(store_ctx, X509_V_ERR_CERT_REVOKED); break;
                  case OCSPContext::CertStatusUnknown:  
                    X509_STORE_CTX_set_error(store_ctx, X509_V_ERR_APPLICATION_VERIFICATION); break;
                }
*/
              }
              else context->Log(Context::LogError, status.GetDescription());
            }
            else context->Log(Context::LogError, status.GetDescription());
          }
          else
            validate_ctx->Log(Context::LogError, status.GetDescription());
 
          //res = query_ocsp_responder(*validate_ctx, ocsp_uri, cert, issuer_loc, cert_path, key_path, capath, "", resp_str);
          //if(!res) validate_ctx->Log(Context::LogInfo, "The certificate is revoked");
        }
      }

      ValidationContext vctx;
      vctx.ca_path = capath;
      vctx.validation_mode = mode;
      vctx.policy_context.inhibit_anypolicy = -1;
      vctx.policy_context.inhibit_policymapping = -1;
      vctx.path_len = -2;
      vctx.context = validate_ctx;
      vctx.cert_status = cert_status;
      if (!X509_STORE_CTX_set_ex_data(store_ctx, get_proxy_auth_ex_data_idx(), &vctx)) {
        status = Status(-1,"Can not set the certificate store for chain verification: "+GetOpenSSLError());
        goto err;
      }
      //X509_STORE_CTX_set_depth(store_ctx, 10);
      if(!X509_verify_cert(store_ctx)) {
        std::cout<<"Error:  "<<X509_verify_cert_error_string(X509_STORE_CTX_get_error(store_ctx))<<std::endl;
        status = Status(-1,"Certificate verification failed: "+GetOpenSSLError());
        goto err;
      }

    }

    status = Status(0);

err:
    if(cert_store) { X509_STORE_free(cert_store); }
    if(store_ctx) { X509_STORE_CTX_free(store_ctx); }
    sk_X509_pop_free(trusted_certchain, X509_free);
    sk_X509_pop_free(untrusted_certchain, X509_free);
    sk_X509_CRL_pop_free(crl_stack, X509_CRL_free);

    //if(!status) last_error_ = status;
    return status;

  }

//----------
// KeyContext Implementation
//----------

  EVP_PKEY* KeyContext::getKey() {
    return key_.pkey;
  }

  bool KeyContext::isPrivate() {
    std::string key_str;
    key_.tostrPEM(key_str, false, "");
    if(key_str.find("PRIVATE KEY") != std::string::npos) return true;
    else return false;
    // TODO: any openssl method for judge if EVP_PKEY include private key?
  }

  Status KeyContext::createKey(int bits) {
    Status res = Status(0);
    key_.reset();

    RSA* rsa_key = NULL;
    int keybits = -1;
     
    if((bits!=0) && ((bits%1024) == 0)) keybits = bits;
    if((keybits==-1) && (!conf_file.empty())) {
      OSSLConf ossl_conf(conf_file, context);
      if((bool)ossl_conf) {
        char* kbits = ossl_conf.get_item_string(REQ_SECTION, BITS);
        if(kbits!=NULL) keybits = atoi(kbits);
      }
    }
    if(keybits == -1) keybits = 1024;

#ifdef HAVE_OPENSSL_OLDRSA
    unsigned long prime = RSA_F4;
    rsa_key = RSA_generate_key(keybits, prime, NULL, NULL);
    if(!rsa_key) {
      if(rsa_key) RSA_free(rsa_key);
      res = Status(-1,"Failed to generate RSA key: "+GetOpenSSLError());
      return res;
    }
#else
    BIGNUM *prime = BN_new();
    if(!prime) {
      res = Status(-1,"Failed to allocate prime for RSA key: "+GetOpenSSLError());
      return res;
    }
    rsa_key = RSA_new();
    if(!rsa_key) {
      BN_free(prime);
      res = Status(-1,"Failed to allocate RSA key: "+GetOpenSSLError());
      return res;
    }
    int ret = BN_set_word(prime,RSA_F4);
    if(ret != 1) {
      BN_free(prime); RSA_free(rsa_key);
      res = Status(-1,"Failed to set prime for RSA key: "+GetOpenSSLError());
      return res;
    }
    ret = RSA_generate_key_ex(rsa_key, keybits, prime, NULL);
    if(ret != 1) {
      BN_free(prime); RSA_free(rsa_key);
      res = Status(-1,"Failed to generate RSA key: "+GetOpenSSLError());
      return res;
    }
#endif
    EVP_PKEY* pkey = EVP_PKEY_new();
    EVP_PKEY_assign_RSA(pkey, rsa_key);
    key_.pkey = pkey;
    return res;
  }

  void KeyContext::publicToDER(std::string& out) const {
    AutoBIO bio(BIO_new(BIO_s_mem()));
    if(!key_.pkey) return;
    i2d_PUBKEY_bio(bio, key_.pkey);
    bio2string(bio, out);
  }

  void KeyContext::publicToPEM(std::string& out) const {
    AutoBIO bio(BIO_new(BIO_s_mem()));
    if(!key_.pkey) return;
    PEM_write_bio_PUBKEY(bio, key_.pkey);
    bio2string(bio, out);
  }

  void KeyContext::privateToDER(std::string& out, const std::string& passphrase) const {
    key_.tostrDER(out);
  }

  void KeyContext::privateToPEM(std::string& out, bool enc, const std::string& passphrase) const {
    key_.tostrPEM(out, enc, passphrase);
  }

  bool KeyContext::privateFromStr(const std::string& in, const std::string& passphrase) {
    key_.reset();
    return key_.fromstr(in, OSSLItem::TypeKey);
  }

//----------
// CSRContext Implementation
//----------

  CSRContext::CSRContext(Context& ctx, const char* csr_filename) : context(&ctx) {
    csr_.logctx = &ctx;
    csr_.fromfile(csr_filename, OSSLItem::TypeReq);
  }

  CSRContext::CSRContext(Context& ctx, const std::string& csrstr) : context(&ctx) {
    csr_.logctx = &ctx;
    csr_.fromstr(csrstr, OSSLItem::TypeReq);
  }

  bool CSRContext::createRequest(const CertificateOptions& opts, KeyContext& key) {
    csr_.reset();

    AutoX509REQ x509req(X509_REQ_new());

    // Get information from openssl.cnf, if exists
    const char* req_ext = NULL;
    std::string distinguished_name;
    X509_NAME* dn_name = NULL; 
    if(!conf_file.empty()) {
      OSSLConf ossl_conf(conf_file, context);
      if((bool)ossl_conf) {
        // For a CSR, the "req_extensions" subsection of "req" section
        // will be used for configure CSR extension
        req_ext = ossl_conf.load_conf_check(REQ_SECTION, REQ_EXTENSIONS_CONF);
        // Add extensions
        if(req_ext != NULL) ossl_conf.load_conf_tox509req(REQ_SECTION, REQ_EXTENSIONS_CONF, x509req);

        char* md = ossl_conf.get_subitem_string(CA_SECTION, DEFAULT_CA, ENV_DEFAULT_MD);
        if(md!=NULL) default_md = md;

        dn_name = ossl_conf.get_dn();
      }
    }

    // Set subject name if provided
    X509_NAME *name = NULL;
    unsigned long chtype = MBSTRING_ASC;
    std::string sub_name = opts.subject();
    if(!sub_name.empty()) {
      name = parse_name(context, (char*)(sub_name.c_str()), chtype, 0);
      X509_REQ_set_subject_name(x509req, name);
      X509_NAME_free(name);
    }
    if(!name && dn_name) X509_REQ_set_subject_name(x509req, dn_name);
    if(dn_name) X509_NAME_free(dn_name);

    // Set challenge if provided
    std::string challenge = opts.challenge();
    if(!challenge.empty())
      X509_REQ_add1_attr_by_NID(x509req, NID_pkcs9_challengePassword, 
        MBSTRING_UTF8, (const unsigned char *)challenge.c_str(), -1);

    X509ExtUtil extutil(x509req, context);

    // Try to find the information from CertificatOption at first,
    // if not found, then try to find the information from openssl.cnf

    Constraints certConstraints = opts.constraints();
    std::vector<std::string> certPolicies;
    std::string proxy_policy;

    //Set basic constraints
    //extutil.set_basic_constraints(isCA, pathLimit);

    //Set key usage
    extutil.set_key_usage(certConstraints);

    //Set ext key usage
    extutil.set_ext_key_usage(certConstraints);

    //Set certitificate policy
    certPolicies = opts.policies();
    extutil.set_cert_policies(certPolicies);

    //Set proxy policy if exists
    //proxyPolicy is the only value that can be
    //retrieved before setProps()
    proxy_policy = props_.proxyPolicy.value;
    if(!proxy_policy.empty()) extutil.set_proxy_policy(proxy_policy);

    //Set the extensions in "exts_" that is set by setCSRExtension
    for(size_t n = 0; n<exts_.size(); n++) {
      AuthN::Credentials::Extension ext = exts_[n];
      extutil.set_extension(ext);
    }

    //Set the attributes in "attrs_" that is set by setCSRAttribute
    std::map<std::string, std::string>::iterator it;
    for(it = attrs_.begin(); it != attrs_.end(); it++) {
      std::string name = (*it).first;
      std::string attr = (*it).second;
      extutil.set_attributes(name, attr);
    }

    // Set public key
    EVP_PKEY* privkey;
    privkey = key.getKey(); 
    if(!privkey) {
      context->Log(Context::LogError, "Failed to load EVP_PKEY object from KeyContext");
      return false;
    } 
    X509_REQ_set_pubkey(x509req, privkey);


    //Load digest from configuration
    const EVP_MD* digest = NULL;
    if(!default_md.empty()) {
      digest = get_digest(*context, default_md.c_str(), privkey);
    }
    if(digest == NULL) digest = EVP_sha1();
#ifndef OPENSSL_NO_DSA
    if (privkey->type == EVP_PKEY_DSA)
      digest=EVP_dss1();
#endif
#ifdef EVP_PKEY_EC
#ifndef OPENSSL_NO_ECDSA
    if (privkey->type == EVP_PKEY_EC)
      digest = EVP_ecdsa();
#endif
#endif
    if (!(X509_REQ_sign(x509req, (EVP_PKEY*)privkey,digest))) {
      context->Log(Context::LogError,GetOpenSSLError());
      return false;
    }

    csr_.reset();
    csr_.req = X509_REQ_dup(x509req);
    setProps();

    return true;

  }

  const CertContextProps* CSRContext::getProps() const { return &props_; }
      
  void CSRContext::setProps() {
    X509_REQ* x509req = csr_.req;
    CertContextProps props;

    props.subject = get_subject_name(x509req);

    X509ExtUtil extutil(x509req, context);
    extutil.get_basic_constraints(&props.isCA, &props.pathLimit);

    extutil.get_proxy_policy(props.proxyPolicy);

    props.constraints = extutil.get_key_usage();
    props.constraints += extutil.get_ext_key_usage();

    extutil.get_cert_policies(props.policies);

    if(x509req->signature) {
      int length = x509req->signature->length;
      char* buffer = new char[length];
      for (int i=0; i< length; i++)
        buffer[i] = x509req->signature->data[i];
      props.sigData.assign(buffer, length);
      delete[] buffer;
    }

    char buf[256];
    int buflen;
    buflen = OBJ_obj2txt(buf, sizeof(buf), x509req->sig_alg->algorithm, 0);
    props.sigAlg.assign(buf, buflen);

    props.subjectId = extutil.get_subject_key_id();
    props.issuerId = extutil.get_authority_key_id();

    props_ = props;
  }

  void CSRContext::csrToDER(std::string& out) const {
    csr_.tostrDER(out);
  }

  void CSRContext::csrToPEM(std::string& out) const {
    csr_.tostrPEM(out);
  }

  bool CSRContext::csrFromDER(const std::string& in) {
    bool res = csr_.fromstrDER(in, OSSLItem::TypeReq);
    setProps();
    return res;
  }

  bool CSRContext::csrFromPEM(const std::string& in) {
    bool res = csr_.fromstrPEM(in, OSSLItem::TypeReq);
    setProps();
    return res;
  }

  Status CSRContext::getCSRExtension(int pos, AuthN::Credentials::Extension& ext) const {
    if(!(csr_.req)) return Status(-1, "X509 request is empty");
    X509ExtUtil extutil(csr_.req, context);
    return extutil.get_extension(pos, ext);
  }

  Status CSRContext::getCSRExtension(const std::string& name, 
      AuthN::Credentials::Extension& ext) const {
    if(!(csr_.req)) return Status(-1, "X509 request is empty");
    X509ExtUtil extutil(csr_.req, context);
    return extutil.get_extension(name, ext);
  }

  Status CSRContext::setCSRExtension(AuthN::Credentials::Extension& ext) {
    if(!(csr_.req)) {
      exts_.push_back(ext);
      return Status(0);
    }
    else {
      X509ExtUtil extutil(csr_.req, context);
      return extutil.set_extension(ext);
    }
  }

  Status CSRContext::setCSRExtension(AuthN::Credentials::Extension& ext,
	  KeyContext& key) {
    if(!(csr_.req)) return Status(-1, "X509 request is empty");

    // We have to create a new X509_REQ, and copy the
    // DN and extensions. The extension adding WON'T exist
    // in the output (csrToPEM(output)) if the previous
    // X509_REQ is used. Not sure the reason.
    X509_REQ* req = X509_REQ_new();

    // Set subject name
    X509_NAME* dn_name = NULL;
    dn_name = X509_REQ_get_subject_name(csr_.req);
    if(dn_name != NULL) X509_REQ_set_subject_name(req, dn_name);

    // Copy the extensions
    STACK_OF(X509_EXTENSION)* from_exts = NULL;
    from_exts = x509_req_get_extensions(csr_.req);
    X509_REQ_add_extensions(req, from_exts);
    if(from_exts) sk_X509_EXTENSION_pop_free(from_exts, X509_EXTENSION_free);

    csr_.reset();
    csr_.req = req;


    X509ExtUtil extutil(csr_.req, context);
    Status stat = extutil.set_extension(ext);
    if(!stat) return stat;

    // Re-sign the X509 request if this method
    // is called on the X509 request generator side

    // Set public key
    // set the public key again, because the key could
    // be different as the former one.
    X509_REQ* x509req = csr_.req;
    EVP_PKEY* privkey = NULL;
    privkey = key.getKey();
    if(!privkey) {
      context->Log(Context::LogError, "Failed to load EVP_PKEY object from KeyContext");
      return Status(-1, "Failed to load EVP_PKEY object from KeyContext");
    }
    X509_REQ_set_pubkey(x509req, privkey);

    //Load digest from configuration
    const EVP_MD* digest = NULL;
    if(!default_md.empty()) {
      digest = get_digest(*context, default_md.c_str(), privkey);
    }
    if(digest == NULL) digest = EVP_sha1();
#ifndef OPENSSL_NO_DSA
    if (privkey->type == EVP_PKEY_DSA)
      digest=EVP_dss1();
#endif
#ifdef EVP_PKEY_EC
#ifndef OPENSSL_NO_ECDSA
    if (privkey->type == EVP_PKEY_EC)
      digest = EVP_ecdsa();
#endif
#endif
    if (!(X509_REQ_sign(x509req, (EVP_PKEY*)privkey,digest))) {
      context->Log(Context::LogError,GetOpenSSLError());
      return Status(-1, "Faile to sign the x509 request");
    }

    /*
    std::string oid, value;
    STACK_OF(X509_EXTENSION)* req_exts = NULL;
    req_exts = x509_req_get_extensions(x509req);
    int num = sk_X509_EXTENSION_num(req_exts);
    for(int i = 0; i < num; i++) {
      X509_EXTENSION* ex = sk_X509_EXTENSION_value(req_exts, i);
      parse_extension(ex, critical, oid, value);
      std::cout<<"oid: "<<oid<<"  value: "<<value<<std::endl;
    }
    */

    setProps();
    return Status(0);
  }

  Status CSRContext::getCSRAttribute(const std::string& name, std::list<std::string>& attrs) const {
    if(!(csr_.req)) return Status(-1, "X509 request is empty");
	X509ExtUtil extutil(csr_.req, context);
	return extutil.get_attributes(name, attrs);
  }

  Status CSRContext::setCSRAttribute(const std::string& name, const std::string& attrs) {
    if(!(csr_.req)) {
      attrs_[name] = attrs;
      return Status(0);
    }
    else {
      X509ExtUtil extutil(csr_.req, context);
      extutil.set_attributes(name, attrs);
    }

    setProps();
    return Status(0);
  }

  Status CSRContext::setCSRAttribute(const std::string& name,
      const std::string& attrs, KeyContext& key) {
    if(!(csr_.req)) return Status(-1, "X509 request is empty");

    // We have to create a new X509_REQ, and copy the
    // DN and extensions. The extension adding WON'T exist
    // in the output (csrToPEM(output)) if the previous
    // X509_REQ is used. Not sure the reason.
    X509_REQ* req = X509_REQ_new();

    // Set subject name
    X509_NAME* dn_name = NULL;
    dn_name = X509_REQ_get_subject_name(csr_.req);
    if(dn_name != NULL) X509_REQ_set_subject_name(req, dn_name);

    // Copy the extensions
    STACK_OF(X509_EXTENSION)* from_exts = NULL;
    from_exts = x509_req_get_extensions(csr_.req);
    X509_REQ_add_extensions(req, from_exts);
    if(from_exts) sk_X509_EXTENSION_pop_free(from_exts, X509_EXTENSION_free);

    csr_.reset();
    csr_.req = req;

    X509ExtUtil extutil(csr_.req, context);

    Status stat = extutil.set_attributes(name, attrs);
    if(!stat) return stat;

    // Re-sign the X509 request if this method
    // is called on the X509 request generator side

    // Set public key
    // set the public key again, because the key could
    // be different as the former one.
    X509_REQ* x509req = csr_.req;
    EVP_PKEY* privkey = NULL;
    privkey = key.getKey();
    if(!privkey) {
      context->Log(Context::LogError, "Failed to load EVP_PKEY object from KeyContext");
      return Status(-1, "Failed to load EVP_PKEY object from KeyContext");
    }
    X509_REQ_set_pubkey(x509req, privkey);

    //Load digest from configuration
    const EVP_MD* digest = NULL;
    if(!default_md.empty()) {
      digest = get_digest(*context, default_md.c_str(), privkey);
    }
    if(digest == NULL) digest = EVP_sha1();
#ifndef OPENSSL_NO_DSA
    if (privkey->type == EVP_PKEY_DSA)
      digest=EVP_dss1();
#endif
#ifdef EVP_PKEY_EC
#ifndef OPENSSL_NO_ECDSA
    if (privkey->type == EVP_PKEY_EC)
      digest = EVP_ecdsa();
#endif
#endif
    if (!(X509_REQ_sign(x509req, (EVP_PKEY*)privkey,digest))) {
      context->Log(Context::LogError,GetOpenSSLError());
      return Status(-1, "Faile to sign the x509 request");
    }

    setProps();
    return Status(0);
  }


  X509_REQ* CSRContext::getReq() {
    return csr_.req;
  }

  bool CSRContext::isSigned() {
    if((csr_.req != NULL) && (csr_.req->signature != NULL)) return true;
    else return false;
  }

  bool CSRContext::operator==(const CSRContext& other) {

    return true;
  }

//----------
// CRLContext Implementation
//----------

  CRLContext::CRLContext(Context* ctx, const char* crl_filename) : context(ctx) {
    crl_.logctx = ctx;
    crl_.fromfile(crl_filename, OSSLItem::TypeCRL);
  }

  CRLContext::CRLContext(Context* ctx, const std::string& crlstr) : context(ctx) {
    crl_.logctx = ctx;
    crl_.fromstr(crlstr, OSSLItem::TypeCRL);
  }

  const CRLContextProps* CRLContext::getProps() const { return &props_; }

  void CRLContext::setProps() { 
    X509_CRL* crl = crl_.crl;
    CRLContextProps props;

    props.issuer = X509_NAME_oneline(X509_CRL_get_issuer(crl),NULL,0);

    props.this_update = asn1_to_utctime(X509_CRL_get_lastUpdate(crl));
    props.next_update = asn1_to_utctime(X509_CRL_get_nextUpdate(crl)); 

    STACK_OF(X509_REVOKED)* revoked  = X509_CRL_get_REVOKED(crl);
    for(int i = 0; i < sk_X509_REVOKED_num(revoked); ++i) {
      X509_REVOKED* item = sk_X509_REVOKED_value(revoked, i);
      long serial = ASN1_INTEGER_get(item->serialNumber);
      AuthN::Utils::Time rev_time = asn1_to_utctime(item->revocationDate);
      CRLContextProps::Reason reason = CRLContextProps::Unspecified;
      int pos = X509_REVOKED_get_ext_by_NID(item, NID_crl_reason, -1);
      if(pos != -1) {
        X509_EXTENSION* ex = X509_REVOKED_get_ext(item, pos);
        if(ex) {
          int* result = (int*) X509V3_EXT_d2i(ex);
          switch(*result) {
            case 0:
              reason = CRLContextProps::Unspecified;
              break;
            case 1:
              reason = CRLContextProps::KeyCompromise;
              break;
            case 2:
              reason = CRLContextProps::CACompromise;
              break;
            case 3:
              reason = CRLContextProps::AffiliationChanged;
              break;
            case 4:
              reason = CRLContextProps::Superseded;
              break;
            case 5:
              reason = CRLContextProps::CessationOfOperation;
              break;
            case 6:
              reason = CRLContextProps::CertificateHold;
              break;
            case 8:
              reason = CRLContextProps::RemoveFromCRL;
              break;
            case 9:
              reason = CRLContextProps::PrivilegeWithdrawn;
              break;
            case 10:
              reason = CRLContextProps::AACompromise;
              break;
            default:
              reason = CRLContextProps::Unspecified;
              break;
          }
          ASN1_INTEGER_free((ASN1_INTEGER*)result);
        }
      }
      CRLContextProps::CRLEntry entry;
      entry.serial = serial;
      entry.rev_time = rev_time;
      entry.reason = reason;
      props.revoked[serial] = entry;
    }

    if(crl->signature) {
      int length = crl->signature->length;
      char* buffer = new char[length];
      for (int i=0; i< length; i++)
        buffer[i] = crl->signature->data[i];
      props.sigData.assign(buffer, length);
      delete[] buffer;
    }

    char buf[256];
    int buflen;
    buflen = OBJ_obj2txt(buf, sizeof(buf), crl->sig_alg->algorithm, 0);
    props.sigAlg.assign(buf, buflen);

    int pos = X509_CRL_get_ext_by_NID(crl, NID_authority_key_identifier, -1);
    if(pos != -1) {
      X509_EXTENSION* ex = X509_CRL_get_ext(crl, pos);
      if(ex) {
        std::string str;
        AUTHORITY_KEYID* akid = (AUTHORITY_KEYID *)X509V3_EXT_d2i(ex);
        if(akid->keyid) str.assign((const char*)ASN1_STRING_data(akid->keyid), ASN1_STRING_length(akid->keyid));
        AUTHORITY_KEYID_free(akid);
        props.issuerId.keyid = str;
      }
    }

    pos = X509_CRL_get_ext_by_NID(crl, NID_crl_number, -1);
    if(pos != -1) {
      X509_EXTENSION* ex = X509_CRL_get_ext(crl, pos);
      if(ex) {
        int* result = (int*) X509V3_EXT_d2i(ex);
        props.number = (*result);
        ASN1_INTEGER_free((ASN1_INTEGER*)result);
      }
    }

    //issuing_dp  delta_indicator   delta_dp

    props_ = props;
  }

  void CRLContext::crlToDER(std::string& out) const {
    crl_.tostrDER(out);
  }

  void CRLContext::crlToPEM(std::string& out) const {
    crl_.tostrPEM(out);
  }

  bool CRLContext::crlFromDER(const std::string& in) {
    bool res = crl_.fromstrDER(in, OSSLItem::TypeCRL);
    setProps();
    return res;
  }

  bool CRLContext::crlFromPEM(const std::string& in) {
    bool res = crl_.fromstrPEM(in, OSSLItem::TypeCRL);
    setProps();
    return res;
  }

  void CRLContext::crlFromCRL(X509_CRL* crl) {
    if(crl) {
      crl_.reset();
      crl_.crl = X509_CRL_dup(crl);
      setProps();
    }
  }

  X509_CRL* CRLContext::getCRL() {
    return crl_.crl;
  }

  bool CRLContext::operator==(const CRLContext& other) {

    return true;
  }


  std::string convert_ossdn_to_rfc2253(const std::string& str) {
    // Convert the openssl dn "/C=US/O=Test Certificates 2011/CN=Good CA"
    // into rfc 2253 format: "CN=Good CA,O=Test Certificates 2011,C=US"

    std::string ret;
    std::string::size_type pos1, pos2;
    pos2 = std::string::npos;
    while(true) {
      std::string name, value;
      pos1 = str.rfind("=", pos2);

      if((pos2 != std::string::npos) && (pos1 != std::string::npos)) ret.append(",");

      if(pos1 == std::string::npos) break;
      if(pos2 == std::string::npos) value = str.substr(pos1+1);
      else value = str.substr(pos1+1, pos2-pos1-1);
        
      pos2 = str.rfind("/", pos1);
      if(pos2 == std::string::npos) break;
      name = str.substr(pos2+1, pos1-pos2-1);
  
      ret.append(name).append("=").append(value);
    }
    return ret;
  }

  std::string convert_rfc2253_to_ossdn(const std::string& str) {
    // Convert the rfc 2253 format: "CN=Good CA,O=Test Certificates 2011, C = US"
    // into openssl dn "/C=US/O=Test Certificates 2011/CN=Good CA"

    std::string ret;
    std::string::size_type pos1, pos2;
    pos2 = std::string::npos;
    while(true) {
      std::string name, value;

      pos1 = str.rfind("=", pos2);

      if(pos1 == std::string::npos) break;
      if(pos2 == std::string::npos) value = str.substr(pos1+1);
      else value = str.substr(pos1+1, pos2-pos1-1);

      pos2 = str.rfind(",", pos1-1);
      if(pos2 == std::string::npos) {
        name = str.substr(0, pos1);
        trim_blank(name); trim_blank(value);
        ret.append("/").append(name).append("=").append(value);
        break;
      }
      else {
        name = str.substr(pos2+1, pos1-pos2-1);
        trim_blank(name); trim_blank(value);
        ret.append("/").append(name).append("=").append(value);
      }
    }

    return ret;
  }



/////////////////////
//The following code is going to be replaced 
/////////////////////




  static X509_NAME* convert_dn(const std::string& str);


  static BIO* OpenFileBIO(const std::string& file) {
    struct stat st;
    if(stat(file.c_str(),&st) != 0) return NULL;
    if(!S_ISREG(st.st_mode)) return NULL;
//    if(!Glib::file_test(file,Glib::FILE_TEST_IS_REGULAR)) return NULL;
    return  BIO_new_file(file.c_str(), "r");
  }

  //Parse the BIO for certificate and get the format of it
  Credformat getFormat(Context& context, BIO* bio) {
    Credformat format = CRED_UNKNOWN;
    if(bio == NULL) return format;
    //bool is_file = (BIO_get_fd(bio,NULL) != -1);
    bool is_file = (BIO_method_type(bio) == BIO_TYPE_FILE);
    if(is_file) {
      char buf[1];
      char firstbyte;
      int position;
      if((position = BIO_tell(bio))<0 || BIO_read(bio, buf, 1)<=0 || BIO_seek(bio, position)<0) {
        context.Log(Context::LogError,
            "Can't get the first byte of input to determine its format: "+GetOpenSSLError());
        return format;
      }
      firstbyte = buf[0];
      // DER-encoded structure (including PKCS12) will start with ASCII 048.
      // Otherwise, it's PEM.
      if(firstbyte==48) {
        //DER-encoded, PKCS12 or DER? firstly parse it as PKCS12 ASN.1,
        //if can not parse it, then it is DER format
        PKCS12* pkcs12 = NULL;
        if((pkcs12 = d2i_PKCS12_bio(bio,NULL)) == NULL){ format=CRED_DER; PKCS12_free(pkcs12); }
        else { format = CRED_PKCS; PKCS12_free(pkcs12); }
        if( BIO_seek(bio, position) < 0 ) {
          context.Log(Context::LogError, "Can't reset the input: "+GetOpenSSLError());
          return format;
        }
      }
      else { format = CRED_PEM; }
    }
    else {
      unsigned char* bio_str = NULL;
      int len;
      len = BIO_get_mem_data(bio, (unsigned char *) &bio_str);
      char firstbyte;
      if(len>0 && bio_str) {
        firstbyte = bio_str[0];
        if(firstbyte==48)  {
          //DER-encoded, PKCS12 or DER? firstly parse it as PKCS12 ASN.1,
          //if can not parse it, then it is DER format
          AutoBIO pkcs12bio(BIO_new_mem_buf(bio_str,len));
          PKCS12* pkcs12 = NULL;
          if((pkcs12 = d2i_PKCS12_bio(pkcs12bio,NULL)) == NULL){
            format=CRED_DER; ERR_clear_error();
          } else {
            format = CRED_PKCS; PKCS12_free(pkcs12);
          }
        } else { format = CRED_PEM; }
      }
      else {
        context.Log(Context::LogError, "Can't get the first byte of input BIO to get its format");
        return format;
      }
    }
    return format;
  }

  void loadCertificateChain(Context& context, BIO* certchainbio, STACK_OF(X509) **certchain) throw (CredentialError) {
    int n;
    if(certchain == NULL) return;
    if(*certchain == NULL) { *certchain = sk_X509_new_null(); n = 0; }
    else { n = sk_X509_num(*certchain); }

    while(!BIO_eof(certchainbio)){
      X509* tmp = NULL;
      if(!(PEM_read_bio_X509(certchainbio, &tmp, NULL, NULL))){
        ERR_clear_error(); break;
      }
      if(!sk_X509_insert(*certchain, tmp, n)) {
        //std::string str(X509_NAME_oneline(X509_get_subject_name(tmp),0,0));
        X509_free(tmp);
        throw CredentialError("Can not insert cert into certificate's issuer chain");
      }
      ++n;
    }
    if(n==0 && *certchain) { sk_X509_pop_free(*certchain, X509_free); *certchain = NULL; }
  }

  void loadCertificate(Context& context, BIO* certbio, X509* &x509, STACK_OF(X509) **certchain) throw (CredentialError) {
    //Parse the certificate
    Credformat format = CRED_UNKNOWN;
    PKCS12* pkcs12 = NULL;
    STACK_OF(X509)* pkcs12_certs = NULL;
    bool get_chain = false;
    if(certchain != NULL) get_chain = true;

    if(certbio == NULL) return;
    format = getFormat(context,certbio);
    int n = 0;
    if(get_chain && *certchain) {
      sk_X509_pop_free(*certchain, X509_free);
      *certchain = NULL;
    }

    switch(format) {
      case CRED_PEM:
        //OpenSSLUtilLogger.msg(DEBUG,"Certificate format is PEM");
        //Get the certificte, By default, certificate is without passphrase
        //Read certificate
        if(!(PEM_read_bio_X509(certbio, &x509, NULL, NULL))) {
          throw CredentialError("Can not read cert information from BIO");
        }
        if(!get_chain) break;
        //Get the issuer chain
        *certchain = sk_X509_new_null();
        n = 0;
        while(!BIO_eof(certbio)){
          X509 * tmp = NULL;
          if(!(PEM_read_bio_X509(certbio, &tmp, NULL, NULL))){
            ERR_clear_error(); break;
          }
          if(!sk_X509_insert(*certchain, tmp, n)) {
            //std::string str(X509_NAME_oneline(X509_get_subject_name(tmp),0,0));
            X509_free(tmp);
            throw CredentialError("Can not insert cert into certificate's issuer chain");
          }
          ++n;
        }
        break;

      case CRED_DER:
        //OpenSSLUtilLogger.msg(DEBUG,"Certificate format is DER");
        x509=d2i_X509_bio(certbio,NULL);
        if(!x509){
          throw CredentialError("Unable to read DER credential from BIO");
        }
        if(!get_chain) break;
        //Get the issuer chain
        *certchain = sk_X509_new_null();
        n = 0;
        while(!BIO_eof(certbio)){
          X509 * tmp = NULL;
          if(!(d2i_X509_bio(certbio, &tmp))){
            ERR_clear_error(); break;
          }
          if(!sk_X509_insert(*certchain, tmp, n)) {
            //std::string str(X509_NAME_oneline(X509_get_subject_name(tmp),0,0));
            X509_free(tmp);
            throw CredentialError("Can not insert cert into certificate's issuer chain");
          }
          ++n;
        }
        break;

      case CRED_PKCS:
        context.Log(Context::LogDebug,"Certificate format is PKCS");
        pkcs12 = d2i_PKCS12_bio(certbio, NULL);
        if(pkcs12){
          char password[100];
          EVP_read_pw_string(password, 100, "Enter Password for PKCS12 certificate:", 0);
          if(!PKCS12_parse(pkcs12, password, NULL, &x509, &pkcs12_certs)) {
            if(pkcs12) PKCS12_free(pkcs12);
            throw CredentialError("Can not parse PKCS12 file: "+GetOpenSSLError());
          }
        }
        else{
          throw CredentialError("Can not read PKCS12 credential from BIO");
        }
        if(!get_chain) break;
        if (pkcs12_certs && sk_X509_num(pkcs12_certs)){
          X509* tmp;
          for (n = 0; n < sk_X509_num(pkcs12_certs); n++) {
            tmp = X509_dup(sk_X509_value(pkcs12_certs, n));
            sk_X509_insert(*certchain, tmp, n);
          }
        }
        if(pkcs12) { PKCS12_free(pkcs12); }
        if(pkcs12_certs) { sk_X509_pop_free(pkcs12_certs, X509_free); }

        break;

      default:
        context.Log(Context::LogDebug,"Certificate format is unknown");
        break;
     } // end switch

    if(get_chain && *certchain && (n==0)) { sk_X509_pop_free(*certchain, X509_free); *certchain = NULL; }
  }

  void loadCertificateFile(Context& context, const std::string& certfile, X509* &x509, STACK_OF(X509) **certchain) throw (CredentialError) {
    BIO* b = OpenFileBIO(certfile);
    if(!b) {
        context.LogFormat(Context::LogError,
                "Can not find certificate file: %s", certfile.c_str());
        throw CredentialError("Can not find certificate file");
    }
    AutoBIO certbio(b);
    if(!certbio){
      throw CredentialError("Can not read certificate file "+certfile+": "+GetOpenSSLError());
    }
    loadCertificate(context,certbio,x509,certchain);
  }

  static std::string tostring(unsigned int val) {
    std::stringstream ss;
    ss << val;
    return ss.str();
  }

  int ContextOpenSSLUI::passwordcb(char *buf, int bufsize, int verify, void *cb_tmp) {
    if(!buf) return 0;
    memset(buf,0,bufsize);
    Context* context = (Context*)cb_tmp;
    if(!context) return 0;
    std::string pass;
    if(verify==1) {
      pass = context->Password("Enter passphrase to use for encrypting private key: ");
      std::string pass2 = context->Password("Repeat passphrase to use for encrypting private key: ");
      if(pass != pass2) {
        context->Message("Passphrases do not match");
        return 0;
      }
    } else {
      pass = context->Password("Enter passphrase to use for decrypting private key: ");
    }
    if(pass.empty()) {
      context->Message("Passphrase is empty.");
      return 0;
    }
    if(pass.length() > (unsigned int)bufsize) {
      context->Message("Passphrase is too long. Can fit max "+tostring(bufsize)+" symbols.");
      return 0;
    }
    memcpy(buf,pass.c_str(),pass.length());
    return pass.length();
  }

  void loadKey(Context& context, BIO* keybio, EVP_PKEY* &pkey) throw (CredentialError) {
    //Read key
    Credformat format;
    PKCS12* pkcs12 = NULL;

    if(keybio == NULL) return;
    format = getFormat(context, keybio);
    std::string prompt;
    switch(format){
      case CRED_PEM:
        if(!(pkey = PEM_read_bio_PrivateKey(keybio, NULL, &ContextOpenSSLUI::passwordcb, &context))) {
          int reason = ERR_GET_REASON(ERR_peek_error());
          if(reason == PEM_R_BAD_BASE64_DECODE) {
            context.Message("Can not read PEM private key: probably bad password");
          } else if(reason == PEM_R_BAD_DECRYPT) {
            context.Message("Can not read PEM private key: failed to decrypt");
          } else if(reason == PEM_R_BAD_PASSWORD_READ) {
            context.Message("Can not read PEM private key: failed to obtain password");
          } else if(reason == PEM_R_PROBLEMS_GETTING_PASSWORD) {
            context.Message("Can not read PEM private key: failed to obtain password");
          } else {
            context.Message("Can not read PEM private key");
          }
        }
        break;

      case CRED_DER:
        pkey=d2i_PrivateKey_bio(keybio, NULL);
        break;

      case CRED_PKCS:
        pkcs12 = d2i_PKCS12_bio(keybio, NULL);
        if(pkcs12) {
          std::string password = context.Password("Enter Password for PKCS12 certificate: ");
          if(!PKCS12_parse(pkcs12, password.c_str(), &pkey, NULL, NULL)) {
            PKCS12_free(pkcs12);
            throw CredentialError("Can not parse PKCS12");
          }
          PKCS12_free(pkcs12);
        }
        break;

      default:
        break;
    }
  }

  void loadKeyFile(Context& context, const std::string& keyfile, EVP_PKEY* &pkey) throw (CredentialError) {
    BIO* b = OpenFileBIO(keyfile);
    if(!b) {
        throw CredentialError("Can not find key file "+keyfile);
    }
    AutoBIO keybio(b);
    if(!keybio){
      throw CredentialError("Can not open key file " + keyfile + ": " + GetOpenSSLError());
    }
    loadKey(context,keybio,pkey);
  }

  void loadCRLFile(Context& context, const std::string& crlfile, X509_CRL* &crl, bool is_pem) throw (CredentialError) {
    BIO* b = OpenFileBIO(crlfile);
    if(!b) {
      context.LogFormat(Context::LogError,"Can not find crl file %s",crlfile.c_str());
      return;
    }
    AutoBIO crlbio(b);
    if(!crlbio){
      context.LogFormat(Context::LogError,
            "Can not open crl file %s: %s", crlfile.c_str(),GetOpenSSLError().c_str());
      return;
    }

    if(!is_pem) crl = d2i_X509_CRL_bio(crlbio, NULL);
    else if (is_pem) crl = PEM_read_bio_X509_CRL(crlbio,NULL,NULL,NULL);
    if(crl == NULL) {
      context.LogFormat(Context::LogError,"Failed to load CRL: %s",GetOpenSSLError().c_str());
    }
    return;
  }

  int get_proxy_auth_ex_data_idx(void) {
    static volatile int idx = -1;
    if (idx < 0) {
      CRYPTO_w_lock(CRYPTO_LOCK_X509_STORE);
      if (idx < 0) {
        idx = X509_STORE_CTX_get_ex_new_index(0,
              //(void*)"for verify callback", NULL,NULL,NULL);
              (void*)"AuthN Validator", NULL,NULL,NULL);
      }
      CRYPTO_w_unlock(CRYPTO_LOCK_X509_STORE);
    }
    return idx;
  }

  static KeyUsages get_cert_key_usage(X509_EXTENSION *ex) {
    KeyUsages keyusages;
    int bit_table[9] = {
            DigitalSignature,
            NonRepudiation,
            KeyEncipherment,
            DataEncipherment,
            KeyAgreement,
            KeyCertificateSign,
            CRLSign,
            EncipherOnly,
            DecipherOnly
    };

    ASN1_BIT_STRING *keyusage = (ASN1_BIT_STRING *)X509V3_EXT_d2i(ex);
    for(int n = 0; n < 9; ++n) {
      if(ASN1_BIT_STRING_get_bit(keyusage, n)) 
      {
        keyusages += KeyUsage((KeyUsageType)bit_table[n]);
      }
    }
    ASN1_BIT_STRING_free(keyusage);
    return keyusages;
  }

  static KeyUsages get_cert_ext_key_usage(X509_EXTENSION *ex) {
    KeyUsages keyusages;
    EXTENDED_KEY_USAGE *extkeyusage = (EXTENDED_KEY_USAGE *)X509V3_EXT_d2i(ex);
    for(int n = 0; n < sk_ASN1_OBJECT_num(extkeyusage); ++n) {
      ASN1_OBJECT *obj = sk_ASN1_OBJECT_value(extkeyusage, n);
      int nid = OBJ_obj2nid(obj);
      if(nid == NID_undef) continue;

      int t = -1;
      switch(nid) {
        case NID_server_auth: t = ServerAuth; break;
        case NID_client_auth: t = ClientAuth; break;
        case NID_code_sign: t = CodeSigning; break;
        case NID_email_protect: t = EmailProtection; break;
        case NID_ipsecEndSystem: t = IPSecEndSystem; break;
        case NID_ipsecTunnel: t = IPSecTunnel; break;
        case NID_ipsecUser: t = IPSecUser; break;
        case NID_time_stamp: t = TimeStamping; break;
        case NID_OCSP_sign: t = OCSPSigning; break;
      };
      if(t == -1) continue;
      keyusages += KeyUsage((KeyUsageType)t);
    }
    sk_ASN1_OBJECT_pop_free(extkeyusage, ASN1_OBJECT_free);
    return keyusages;
  }


  static X509_EXTENSION* new_cert_key_usage(KeyUsages& keyusages) {
    ASN1_BIT_STRING *keyusage = 0;
    for(int n = 0; n < keyusages.Size(); ++n) {
      int bit = -1;
      switch(keyusages[n].type()) {
        case DigitalSignature: bit = Bit_DigitalSignature; break;
        case NonRepudiation: bit = Bit_NonRepudiation; break;
        case KeyEncipherment: bit = Bit_KeyEncipherment; break;
        case DataEncipherment: bit = Bit_DataEncipherment; break;
        case KeyAgreement: bit = Bit_KeyAgreement; break;
        case KeyCertificateSign: bit = Bit_KeyCertificateSign; break;
        case CRLSign: bit = Bit_CRLSign; break;
        case EncipherOnly: bit = Bit_EncipherOnly; break;
        case DecipherOnly: bit = Bit_DecipherOnly; break;
        default: break;
      }
      if(bit != -1) {
        if(!keyusage)
          keyusage = ASN1_BIT_STRING_new();
        ASN1_BIT_STRING_set_bit(keyusage, bit, 1);
      }
    }
    if(!keyusage) return NULL;

    X509_EXTENSION *ex = X509V3_EXT_i2d(NID_key_usage, 1, keyusage);
    ASN1_BIT_STRING_free(keyusage);
    return ex;
  }

  static X509_EXTENSION* new_cert_ext_key_usage(KeyUsages& keyusages) {
    EXTENDED_KEY_USAGE *extkeyusage = NULL;
    for(int n = 0; n < keyusages.Size(); ++n) {
      int nid = -1;
      switch(keyusages[n].type()) {
        case ServerAuth: nid = NID_server_auth; break;
        case ClientAuth: nid = NID_client_auth; break;
        case CodeSigning: nid = NID_code_sign; break;
        case EmailProtection: nid = NID_email_protect; break;
        case IPSecEndSystem: nid = NID_ipsecEndSystem; break;
        case IPSecTunnel: nid = NID_ipsecTunnel; break;
        case IPSecUser: nid = NID_ipsecUser; break;
        case TimeStamping: nid = NID_time_stamp; break;
        case OCSPSigning: nid = NID_OCSP_sign; break;
        default: break;
      }
      if(nid != -1) {
        if(!extkeyusage)
          extkeyusage = sk_ASN1_OBJECT_new_null();
        ASN1_OBJECT *obj = OBJ_nid2obj(nid);
        sk_ASN1_OBJECT_push(extkeyusage, obj);
      }
    }
    if(!extkeyusage) return NULL;

    X509_EXTENSION *ex = X509V3_EXT_i2d(NID_ext_key_usage, 0, extkeyusage);
    sk_ASN1_OBJECT_pop_free(extkeyusage, ASN1_OBJECT_free);
    return ex;
  }

  static KeyUsages get_cert_key_usage(X509* cert) {
    KeyUsages keyusages;
    int index;
    index = X509_get_ext_by_NID(cert, NID_key_usage, -1);
    if(index != -1) {
      X509_EXTENSION *ex = X509_get_ext(cert, index);
      if(ex) {
        keyusages = get_cert_key_usage(ex);
      }
    }
    return keyusages;
  }

  static KeyUsages get_cert_ext_key_usage(X509* cert) {
    KeyUsages keyusages;
    int index;
    index = X509_get_ext_by_NID(cert, NID_ext_key_usage, -1);
    if(index != -1) {
      X509_EXTENSION *ex = X509_get_ext(cert, index);
      if(ex) {
        keyusages = get_cert_ext_key_usage(ex);
      }
    }
    return keyusages;
  }

/*
  static KeyUsages get_key_usage(X509* cert) {
    KeyUsages keyusages;
    int index;
    index = X509_get_ext_by_NID(cert, NID_key_usage, -1);
    if(index != -1) {
      X509_EXTENSION *ex = X509_get_ext(cert, index);
      if(ex) {
        keyusages = get_cert_key_usage(ex);
      }
    }
    index = X509_get_ext_by_NID(cert, NID_ext_key_usage, -1);
    if(index != -1) {
      X509_EXTENSION *ex = X509_get_ext(cert, index);
      if(ex) {
        keyusages += get_cert_ext_key_usage(ex);
      }
    }

    return keyusages;
  }
*/

   /* from section 4.2 of RFC 3820:
   Application making authorization decisions based on the contents of
   the proxy certificate key usage or extended key usage extensions MUST
   examine the list of key usage, extended key usage and proxy policies
   resulting from proxy certificate path validation and determine the
   effective key usage functions of the proxy certificate as follows:

   *  If a certificate is a proxy certificate with a proxy policy of
      id-ppl-independent or an end entity certificate, the effective key
      usage functions of that certificate is as defined by the key usage
      and extended key usage extensions in that certificate.  The key
      usage functionality of the issuer has no bearing on the effective
      key usage functionality.

   *  If a certificate is a proxy certificate with a policy other than
      id-ppl-independent, the effective key usage and extended key usage
      functionality of the proxy certificate is the intersection of the
      functionality of those extensions in the proxy certificate and the
      effective key usage functionality of the proxy issuer.
  */
  int check_issued(X509_STORE_CTX* ctx, X509* x, X509* issuer) {
    int ret;
    ret = X509_check_issued(issuer, x);

#ifdef HAVE_OPENSSL_PROXY
    if (ret == X509_V_OK) {
      if(X509_cmp(x,issuer) != 0) {
        /*If the the 'issuer' is the issuer of 'x', not self-signed.
          And if the NID_proxyCertInfo of 'x' is not id-ppl-independent,
          (i.e. 'x' is a proxy certificate with a policy other 
          than id-ppl-independent). According to RFC3820, "the effective 
          key usage and extended key usage functionality of the proxy 
          certificate is the intersection of the functionality of 
          those extensions in the proxy certificate and the
          effective key usage functionality of the proxy issuer"
         */
        PROXY_CERT_INFO_EXTENSION *pci = (PROXY_CERT_INFO_EXTENSION*)X509_get_ext_d2i(x, NID_proxyCertInfo,
                                         NULL, NULL);
        if(pci != NULL) {
          int i = OBJ_obj2nid(pci->proxyPolicy->policyLanguage);
          if(i != NID_Independent) {
            KeyUsages ku1 = get_cert_key_usage(x);
            KeyUsages ku2 = get_cert_key_usage(issuer);
            ku1 = ku1.Intersection(ku2);
            KeyUsages ku3 = get_cert_ext_key_usage(x);
            KeyUsages ku4 = get_cert_ext_key_usage(issuer);
            ku3 = ku3.Intersection(ku4);
            X509_EXTENSION* ex;
            ex = new_cert_key_usage(ku1);
            if(ex) {
              X509_add_ext(x, ex, -1);
              X509_EXTENSION_free(ex);
            }
            ex = new_cert_ext_key_usage(ku3);
            if(ex) {
              X509_add_ext(x, ex, -1);
              X509_EXTENSION_free(ex);
            }
          }
        }
      }
      return 1;
    }
#endif

#ifdef HAVE_OPENSSL_X509_VERIFY_PARAM
    // If we haven't asked for issuer errors don't set ctx
    if (!(ctx->param->flags & X509_V_FLAG_CB_ISSUER_CHECK))
      return 0;
#endif

    ctx->error = ret;
    ctx->current_cert = x;
    ctx->current_issuer = issuer;
    return ctx->verify_cb(0, ctx);

    return 0;
  }

  static bool find_crl_issuer(const std::string& capath, const std::string& hash_str, std::string& crl_issuer, bool use_last) {
    //We search the ca_dir independently, rather than via the openssl lookup way
    std::string cafile;
    cafile = capath +  "/" + hash_str + ".0";
    if(AuthN::Utils::File::file_exist(capath, cafile) && use_last) {
      //If the corresponding ca file exists, and still the verification 
      //gives error, we have to use the last found ca file ca file 
      //with the same hash value as prefix
      cafile = capath + "/" + hash_str + ".1";
    }
    if(!AuthN::Utils::File::file_exist(capath, cafile)) return false;
  
    crl_issuer = cafile;
    return true;
  }

  static std::string get_cert_extension(const std::string& ext_name, X509* cert) {
    std::string oid;
    bool crit = false;
    std::string ext_val;
    if(cert == NULL) return std::string();
    int nid;
    nid = OBJ_txt2nid(ext_name.c_str());
    if(nid == NID_undef) nid = OBJ_sn2nid(ext_name.c_str());
    if(nid == NID_undef) nid = OBJ_ln2nid(ext_name.c_str());
    if(nid != NID_undef) {
      int pos = X509_get_ext_by_NID(cert, nid, -1);
      if (pos != -1) {
        X509_EXTENSION* x509_ext = NULL;
        x509_ext = X509_get_ext(cert, pos);
        if(!parse_extension(x509_ext, crit, oid, ext_val)) return std::string();
        else return ext_val;
      }
    }
    return std::string();
  }

  static std::string get_crl_extension(const std::string& ext_name, X509_CRL* crl) {
    std::string oid;
    bool crit = false;
    std::string ext_val;
    if(crl == NULL) return std::string();
    int nid;
    nid = OBJ_txt2nid(ext_name.c_str());
    if(nid == NID_undef) nid = OBJ_sn2nid(ext_name.c_str());
    if(nid == NID_undef) nid = OBJ_ln2nid(ext_name.c_str());
    if(nid != NID_undef) {
      int pos = X509_CRL_get_ext_by_NID(crl, nid, -1);
      if (pos != -1) {
        X509_EXTENSION* x509_ext = NULL;
        x509_ext = X509_CRL_get_ext(crl, pos);
        if(!parse_extension(x509_ext, crit, oid, ext_val)) return std::string();
        else return ext_val;
      }
    }
    return std::string();
  }

// TODO: exact version need to be identified
// TODO: workaround or different solution need to be provided
#if (OPENSSL_VERSION_NUMBER  > 0x009080ffL)
  /** Retrieve crl*/
  static int my_get_crl(X509_STORE_CTX* ctx, X509_CRL** pcrl, X509* cert) {
    X509_CRL* crl = NULL;
    STACK_OF(X509_CRL)* crls;
    X509_NAME* name = NULL;

    std::string crl_distribution_point_str = get_cert_extension("X509v3 CRL Distribution Points", cert);
    if(!crl_distribution_point_str.empty()) {
      std::string cdp_crl_issuer_name;
      std::string::size_type find = crl_distribution_point_str.find("CRL Issuer:");
      if(find != std::string::npos) {
        find = crl_distribution_point_str.find("DirName:", find);
        if(find != std::string::npos) {
          std::string::size_type find1 = crl_distribution_point_str.find("\n", find);
          if(find1 != std::string::npos)
            cdp_crl_issuer_name = crl_distribution_point_str.substr(find + 9, find1-find-9);
          else cdp_crl_issuer_name = crl_distribution_point_str.substr(find + 9);
        }
      }
      name = convert_dn(cdp_crl_issuer_name);
    }

    crls = ctx->lookup_crls(ctx, name);
    //if(sk_X509_CRL_num(crls)>1) context.LogFormat(Context::LogError,"There is multile CRL exist");
    // This case should not happen, since we only consider the issuer distribution point here. 

    if(crls == NULL || sk_X509_CRL_num(crls) == 0) {
      //context.LogFormat(Context::LogError,"Failed to find CRL");
      if(crl != NULL) sk_X509_CRL_pop_free(crls, X509_CRL_free); return 0;
    }

    crl = sk_X509_CRL_value(crls, 0);
    *pcrl = crl;
    //sk_X509_CRL_pop_free(crls, X509_CRL_free);
    if(name != NULL) X509_NAME_free(name);

    return 1;
  }

  static int my_check_crl(X509_STORE_CTX *ctx, X509_CRL *crl) {
    int ret = 0;
    if(crl == NULL) { std::cout<<"CRL is NULL ====="<<std::endl; 
      return 1; }

    //if(check_crl_signing(ctx, crl, false, true)) { 
      ret = 1; 
    //}   
    ctx->current_reasons = CRLDP_ALL_REASONS; //crl->idp_reasons;// CRLDP_ALL_REASONS; 
    return ret;
  }
#endif

  static bool check_cert(Context& context, const std::string& cadir, const std::string& cafile, const std::string& crlfile, X509* cert, bool check_crl, bool my_crl_check) {
    // Check the certificate in a new X509_STORE_CTX with 
    // specified cadir, cafile, and crlfile.
    X509_CRL* crl = NULL;
    X509_STORE* verify_store = NULL;
    X509_STORE_CTX localctx;
    int i = 0;
    bool ret = false;
    ValidationContext vctx;

    if(!crlfile.empty() && check_crl) {
      loadCRLFile(context, crlfile, crl, true); //TODO: detect the format
    }

    verify_store = setup_verify(context, cafile, cadir);
    if(!verify_store) goto end;
    X509_STORE_set_flags(verify_store, 0);
    if(!X509_STORE_CTX_init(&localctx, verify_store, cert, NULL)) {
      context.Log(Context::LogError, "Failed to initialize X509 store");
      goto end;
    }
// TODO: exact version need to be identified
// TODO: workaround or different solution need to be provided
#if (OPENSSL_VERSION_NUMBER  > 0x009080ffL)
    if(my_crl_check) {
      localctx.get_crl = my_get_crl;
      localctx.check_crl = my_check_crl;
    }
#endif

    vctx.ca_path = cadir; 
    vctx.validation_mode = 0; 
    vctx.context = &context;
    X509_STORE_CTX_set_ex_data(&localctx, get_proxy_auth_ex_data_idx(), &vctx);

    if(crl && check_crl) {
      i=X509_STORE_add_crl(verify_store,crl);
      if(!i) context.Log(Context::LogError, "Failed to add CRL object into X509_STORE_CTX");
      X509_CRL_free(crl); crl = NULL;
    }

    //Here we force to check CRL
    if(check_crl)
      X509_STORE_CTX_set_flags(&localctx, X509_V_FLAG_CRL_CHECK);// | X509_V_FLAG_EXTENDED_CRL_SUPPORT);
    else 
      X509_STORE_CTX_set_flags(&localctx, 0);// | X509_V_FLAG_EXTENDED_CRL_SUPPORT);

    i = X509_verify_cert(&localctx);
    if (i > 0) {
      context.Log(Context::LogInfo, "Succeeded to verify certificate");
      ret=true;
    }
    else {
      context.LogFormat(Context::LogError, "Failed to verify certificate: %s", X509_verify_cert_error_string(localctx.error));
      //OpenSSLError();
    }
    end:
    if(verify_store) {
      X509_STORE_CTX_cleanup(&localctx);
      X509_STORE_free(verify_store);
    }
    if(crl) X509_CRL_free(crl);
    return ret;
  }

  static bool check_cert(Context& context, const std::string& cadir, const std::string& cafile, const std::string& crlfile, std::string& certfile, bool check_crl, bool my_crl_check) {
    // Check the certificate in a new X509_STORE_CTX with
    // specified cadir, cafile, and crlfile.
    X509* cert = NULL;
    bool ret = false;

    loadCertificateFile(context, certfile, cert, NULL);
    if(!cert) return false;
    ret = check_cert(context, cadir, cafile, crlfile, cert, check_crl, my_crl_check);
    X509_free(cert);
    return ret;
  }

  static bool check_crl_signing(ValidationContext* vctx, X509_STORE_CTX* ctx, X509_CRL* crl, bool use_last, bool my_crl_check) {
    bool ret = false;
    X509_STORE* verify_store = NULL;
    X509_STORE_CTX localctx;
    X509_OBJECT obj;
    EVP_PKEY* pkey = NULL;
    int i;

    std::string cadir;
    //ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
    //                            get_proxy_auth_ex_data_idx());
    if(vctx != NULL) cadir = vctx->ca_path;
    Context& context = *(vctx->context);
    if(cadir.empty()) return false;

    std::string hash_str;
    if(crl) hash_str = AuthN::OpenSSL::get_crl_hash(crl);
    else return false;
    std::string cafile;
    if(!find_crl_issuer(cadir, hash_str, cafile, use_last)) return false;

    //Verify the validity of crl signer
    if(!check_cert(context, cadir, "", "", cafile, false, my_crl_check)) return false;

    //Verify the validity of crl
    //Check the crl independently, using a seperated X509_STORE_CTX
    verify_store = setup_verify(context, cafile, "");
    if(!verify_store) goto end;
    if(!X509_STORE_CTX_init(&localctx, verify_store, NULL, NULL)) {
      context.Log(Context::LogError, "Failed to initialize X509 store");
      goto end;
    }

    i = X509_STORE_get_by_subject(&localctx, X509_LU_X509, X509_CRL_get_issuer(crl), &obj);
    if(i <= 0) {
      context.Log(Context::LogError, "Failed to get CRL issuer certificate");
      goto end;
    }

    pkey = X509_get_pubkey(obj.data.x509);
    if(!pkey) {
      context.Log(Context::LogError, "Failed to get CRL issuer public key");
      goto end;
    }
    i = X509_CRL_verify(crl, pkey);
    if(i < 0) goto end;
    if(i == 0) context.Log(Context::LogError, "Failed to verify crl");
    else context.LogFormat(Context::LogInfo, "Succeeded to verify crl against signing file %s",cafile.c_str());
    ret = true;

    end:
    X509_OBJECT_free_contents(&obj);
    if(pkey) EVP_PKEY_free(pkey);
    if(verify_store) {
      X509_STORE_CTX_cleanup(&localctx);
      X509_STORE_free(verify_store);
    }
    return ret;
  }

  static int memicmp(const void *s1, const void *s2, size_t n) {
    const unsigned char *p1 = (const unsigned char *) s1;
    const unsigned char *p2 = (const unsigned char *) s2;
    for (; n-- > 0; p1++, p2++) {
      int c;
      if (*p1 != *p2 && (c = (toupper(*p1) - toupper(*p2)))) return c;
    }
    return 0;
  }
 
  static std::string get_cafile_from_cert(Context& context, const std::string& cadir, X509* currentcert) {
    std::string cafile;
    char buf[256];
    std::string issuer;
    // Get the issuer name of cert
    X509_NAME_oneline(X509_get_issuer_name(currentcert),buf, sizeof buf);
    issuer = buf;

    if(cadir.empty() || !currentcert) return cafile;

    // Tansverse the CA directory to find a match
    X509* cert = NULL;
    DIR* dir;
    dir = opendir(cadir.c_str());
    if(!dir) { return cafile; }
    for(;;) {
      struct dirent* d = ::readdir(dir);
      if(!d) break;
      if(strcmp(d->d_name,".") == 0) continue;
      if(strcmp(d->d_name,"..") == 0) continue;
      std::string file = cadir + "/" + d->d_name;
  
      if((file.find(".0") == std::string::npos) &&  (file.find(".1") == std::string::npos)) continue;
      loadCertificateFile(context, file, cert, NULL);
      if(!cert) continue;
      char subject[256];
      X509_NAME_oneline(X509_get_subject_name(cert),subject, sizeof subject);
      X509_free(cert); cert = NULL;
      std::string sub_str = subject;
    
      // Case insensitive string comparison
      if(issuer.length() != sub_str.length()) continue;
      int i = memicmp(issuer.c_str(), sub_str.c_str(), issuer.length());
      if(!i) {
        cafile = file;
        break;
      }
    }
    closedir(dir);
    return cafile;
  }

  //static std::string convert_dn(const std::string& str) {
    //Convert the dn "C = US, O = Test Certificates 2011, CN = Good CA"
    //into "/C=US/O=Test Certificates 2011/CN=Good CA"
/*
    std::string ret;
    std::string::size_type pos1, pos2;
    pos1 = 0;
    while(true) {
      ret.append("/");
      if(pos1 == 0) pos1 = str.find_first_not_of(" ", pos1);
      else pos1 = str.find_first_not_of(" ", pos1+1);
      if(pos1 == std::string::npos) break;
      pos2 = str.find_first_of(" ", pos1);
      if(pos2 != std::string::npos) ret.append(str.substr(pos1, pos2-pos1));
      else break;
      pos1 = str.find("=", pos2);
      if(pos1 != std::string::npos) ret.append("=");
      else break;
      pos2 = str.find_first_not_of(" ", pos1+1);
      if(pos2 == std::string::npos) break;
      pos1 = str.find(",", pos2);
      if(pos1 != std::string::npos) ret.append(str.substr(pos2, pos1-pos2));
      else { ret.append(str.substr(pos2)); break; }
    }
    return ret;
  }
*/

  static X509_NAME* convert_dn(const std::string& str) {
    //Convert the dn "C = US, O = Test Certificates 2011, CN = Good CA"
    //into "/C=US/O=Test Certificates 2011/CN=Good CA"
    X509_NAME* name = NULL;
    name = X509_NAME_new();

    std::string::size_type pos1, pos2;
    pos1 = 0;
    std::string str1, str2;
    while(true) {
      if(pos1 == 0) pos1 = str.find_first_not_of(" ", pos1);
      else pos1 = str.find_first_not_of(" ", pos1+1);
      if(pos1 == std::string::npos) break;
      pos2 = str.find_first_of(" ", pos1);
      if(pos2 != std::string::npos)  str1 = str.substr(pos1, pos2-pos1);
      else break;
      pos1 = str.find("=", pos2);
      if(pos1 == std::string::npos) break;
      pos2 = str.find_first_not_of(" ", pos1+1);
      if(pos2 == std::string::npos) break;
      pos1 = str.find(",", pos2);
      if(pos1 != std::string::npos) {
        str2 = str.substr(pos2, pos1-pos2);
#if OPENSSL_VERSION_NUMBER >= 0x0090800fL
        X509_NAME_add_entry_by_txt(name,(const char*)str1.c_str(), MBSTRING_ASC, (const unsigned char*)str2.c_str(), -1, -1, 0);
#else
        X509_NAME_add_entry_by_txt(name,(char*)str1.c_str(), MBSTRING_ASC, (unsigned char*)str2.c_str(), -1, -1, 0);
#endif
      }
      else {
        str2 = str.substr(pos2);
#if OPENSSL_VERSION_NUMBER >= 0x0090800fL
        X509_NAME_add_entry_by_txt(name,(const char*)str1.c_str(), MBSTRING_ASC, (const unsigned char*)str2.c_str(), -1, -1, 0);
#else
        X509_NAME_add_entry_by_txt(name,(char*)str1.c_str(), MBSTRING_ASC, (unsigned char*)str2.c_str(), -1, -1, 0);
#endif
        break;
      }
    }
    return name;
  }

  int verify_callback(int preverify_ok, X509_STORE_CTX *ctx) {
    X509* current_cert = X509_STORE_CTX_get_current_cert(ctx);
    int err = X509_STORE_CTX_get_error(ctx);
    int err_depth = X509_STORE_CTX_get_error_depth(ctx);
    std::string sub_name;
    sub_name = get_subject_name(current_cert);
    ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                get_proxy_auth_ex_data_idx());
    if (!vctx) return preverify_ok;
    Context& context = *(vctx->context);
    if (preverify_ok == 0)
      context.LogFormat(Context::LogError,"Verification error number: %d; error string: %s; error depth: %d; subject: %s", err,X509_verify_cert_error_string(err),err_depth,sub_name.c_str());

    context.LogFormat(Context::LogInfo, "Subject name of current X509 to be verified: %s", sub_name.c_str());

    int pos;
    if (preverify_ok == 0) {
      switch (err) {
// TODO: exact version need to be identified
// TODO: workaround or different solution need to be provided
#if (OPENSSL_VERSION_NUMBER  > 0x009080ffL)
        case X509_V_ERR_DIFFERENT_CRL_SCOPE:
#endif
        case X509_V_ERR_UNABLE_TO_GET_CRL:
#ifdef HAVE_OPENSSL_PROXY
        pos = X509_get_ext_by_NID(current_cert, NID_proxyCertInfo, -1);
        // Since the CRL normally include the EEC certificate that is revoked,
        // if the certificate is a proxy, the CRL will not be checked,
        // and the "unable to get certificate CRL" error is ignored.
        // But the X509_V_FLAG_CRL_CHECK_ALL should be set, so that the 
        // certificates in the whole chain will be check, and the EEC
        // will be checked against the related CRL.
        if(pos >= 0) { preverify_ok = 1; break; }
        else
#endif
        {
          // If the CRL is configured to check, but unable to be found under
          // the CA directory according to the hash value of certificate's issuer,
          // probably the CRL is issued by an entity other than the certificate issuer.
          // This entity should be specified in cRLIssuer field of the 
          // cRLDistributionPoints extension.
          // Note cRLDistributionPoint extension can only be handled by openssl 
          // version after 1.0.0.
           
          std::string crl_distribution_point_str = get_cert_extension("X509v3 CRL Distribution Points", current_cert);
          if(!crl_distribution_point_str.empty()) {
            std::string cdp_crl_issuer_name;
            std::string cdp_fullname;
            std::string::size_type find = crl_distribution_point_str.find("CRL Issuer:");
            if(find != std::string::npos) {
              find = crl_distribution_point_str.find("DirName:", find);
              if(find != std::string::npos) cdp_crl_issuer_name = crl_distribution_point_str.substr(find + 9);
              find = cdp_crl_issuer_name.find("\n");
              if(find != std::string::npos)
                cdp_crl_issuer_name = cdp_crl_issuer_name.substr(0, find);
            }
            find = crl_distribution_point_str.find("Full Name:");
            if(find != std::string::npos) {
              find = crl_distribution_point_str.find("DirName:", find);
              if(find != std::string::npos) {
                cdp_fullname = crl_distribution_point_str.substr(find + 9);
                find = cdp_fullname.find("\n");
                if(find != std::string::npos)
                  cdp_fullname = cdp_fullname.substr(0, find);
              }
            }
            context.LogFormat(Context::LogInfo, "CRL Issuer in CRL Distribution Point: %s", cdp_crl_issuer_name.c_str());
            context.LogFormat(Context::LogInfo, "Full name in CRL Distribution Point: %s", cdp_fullname.c_str());

            //std::string str = convert_dn(cdp_crl_issuer_name); 
            X509_NAME* name = convert_dn(cdp_crl_issuer_name);

            //std::string str(X509_NAME_oneline(name,0,0));

            //Get the hash value of the dn
            std::string hash_str = get_hash_value(name);
            X509_NAME_free(name);

            //Find the crl related to the hash value
            std::string cadir;
            ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
            if(vctx != NULL) cadir = vctx->ca_path;
            if(!hash_str.empty()) {
              std::string crl_file;
              crl_file = cadir + "/" + hash_str + ".r0";
              // The end entity certificate includes a cRLDistributionPoints 
              // extension with a cRLIssuer field indicating that the CRL 
              // is issued by an entity other than the certificate issuer.
              //
              // Now we need to chech whether the CRL issued by the entity 
              // specified in the cRLIssuer field includes an 
              // issuingDistributionPoint extension.
              // See section 4.14.27 (Invalid cRLIssuer Test27) of
              // http://csrc.nist.gov/groups/ST/crypto_apps_infra/documents/PKITS.pdf
              bool issuer_dist_point = false;
              X509_CRL* crl = NULL;
              loadCRLFile(context, crl_file, crl, true); //TODO: detect format
              if(crl != NULL) {
                std::string issuer_distribution_point_str = get_crl_extension("X509v3 Issuing Distrubution Point", crl);
                if(!issuer_distribution_point_str.empty()) { 
                  // Check if there is the cRLIssuer field does not include an
                  // issuingDistributionPoint extension 
                  issuer_dist_point = true; X509_CRL_free(crl); 
                }
                else { 
                  context.Log(Context::LogError, "The CRLDistributionPoints extension of the current cert tells the CRL is issued by an entity other than the certificate issuer, but the CRL does not include an issuingDistributionPoint extension");
                  preverify_ok = 0; X509_CRL_free(crl); break; 
                }
              }
              else {
                context.LogFormat(Context::LogInfo, "Can not find CRL file at %s", crl_file.c_str());
                // There is no CRL available from the entity specified 
                // in cRLIssuer, we check whether the certificate issuer has issued a 
                // CRL with an issuingDistributionPoint extension that 
                // includes a distributionPoint that matches the 
                // distributionPoint in the certificate
                // See section 4.14.35

                X509_NAME* issuer_name = X509_get_issuer_name(current_cert);
                std::string hash_str = get_hash_value(issuer_name);
                if(!hash_str.empty()) {
                  std::string crl_file;
                  crl_file = cadir + "/" + hash_str + ".r0";
                  X509_CRL* crl = NULL;
                  loadCRLFile(context, crl_file, crl, true); //TODO: detect format
                  if(crl != NULL) {
                    std::string issuer_distribution_point_str = get_crl_extension("X509v3 Issuing Distrubution Point", crl);
                    if(!issuer_distribution_point_str.empty()) {
                      // Check if there is the cRLIssuer field does not include an
                      // issuingDistributionPoint extension
                      std::string::size_type find;

std::cout<<"====== "<<issuer_distribution_point_str<<"    "<<cdp_fullname<<std::endl;

                      find = issuer_distribution_point_str.find(cdp_fullname);
                      if(find!=std::string::npos) {
                        context.LogFormat(Context::LogInfo, "The certificate issuer has issued a CRL with an issuingDistributionPoint extension that includes a distributionPoint that matches the distributionPoint in the certificate, the distribution point is: %s", cdp_fullname.c_str()); 
                        preverify_ok = 0; X509_CRL_free(crl); break; 
                      }
                      else { 
                        // The only CRL available from the issuer of the end entity certificate 
                        // includes an issuingDistributionPoint extension with a distributionPoint 
                        // that does not match the distributionPoint specified in the end entity certificate.
                        // The path should not validate successfully since the status of the end
                        // entity certificate can not be determined. See section 4.14.3
                        context.LogFormat(Context::LogInfo, "The certificate issuer has issued a CRL with an issuingDistributionPoint extension that includes a distributionPoint %s that does not matche the distributionPoint %s in the certificate", issuer_distribution_point_str.c_str(), cdp_fullname.c_str());
                        preverify_ok = 0; X509_CRL_free(crl); break;
                      }
                    }
                    X509_CRL_free(crl);
                  }
                  else {
                    // If both the entity specified in the cRLIssuer field does not exist,
                    // and the certificate issuer has not issued a CRL with an 
                    // issuingDistributionPoint extension that
                    // includes a distributionPoint that matches the
                    // distributionPoint in the certificate. See section 4.14.26
                    preverify_ok = 0; break;
                  }
                }
              }

              // If the CRL includes an issuingDistributionPoint extension that matches,
              // we add the CRL file for looking up.
              if(issuer_dist_point) {
                //X509_LOOKUP* lookup = NULL;
                X509_STORE* store = NULL;
                store = ctx->ctx;
// TODO: exact version need to be identified
// TODO: workaround or different solution need to be provided
#if (OPENSSL_VERSION_NUMBER  > 0x009080ffL)
                if(store != NULL) {
                  //lookup=X509_STORE_add_lookup(store,X509_LOOKUP_file());
                  //if(lookup != NULL) {
                  //  if(X509_load_crl_file(lookup,crl_file.c_str(), X509_FILETYPE_PEM)) {
                      if(check_cert(context, cadir, "", crl_file, current_cert, true, true)) { 
                        ctx->current_reasons = CRLDP_ALL_REASONS;
                        // Hack due to the fact that is this is not set,
                        // openssl's check_crl code will goes into loop
                        preverify_ok = 1; break;
                      }
                  //  }
                  //}
                }
#endif
              }
              else {
                if(check_cert(context, cadir, "", "", current_cert, false, false)) { 
                  preverify_ok = 1;
                  std::cout<<"Succeed to verify"<<std::endl; }
              }
            }
          } //crl_distribution_point_str not empty
        }; break;
#if OPENSSL_VERSION_NUMBER >= 0x0090704fL  //(since 0.9.7d)
        case X509_V_ERR_KEYUSAGE_NO_CRL_SIGN: {
          // If the current cert issuer cert does not have CRL signing key usage, then 
          // the CRL could be signed by another issuer, therefore we need to find it.
          // Please find the requirement example from section 4.4.19 of 
          // http://csrc.nist.gov/groups/ST/crypto_apps_infra/documents/PKITS.pdf
          std::string cadir;
          if(vctx != NULL) cadir = vctx->ca_path;
          std::string hash_str;
          X509_CRL* crl = ctx->current_crl;
          if(crl) hash_str = AuthN::OpenSSL::get_crl_hash(crl);
          if(!hash_str.empty()) {
            std::string cafile;
            if(find_crl_issuer(cadir, hash_str, cafile, true)) preverify_ok =1; 
          }
        }; break;
#endif
        case X509_V_ERR_CRL_SIGNATURE_FAILURE: {
          X509_CRL* crl = ctx->current_crl;
          if(check_crl_signing(vctx, ctx, crl, true, false)) preverify_ok =1;
        }; break;
        case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY: {
          // Please find the requirement example from section 4.3.5 of 
          // http://csrc.nist.gov/groups/ST/crypto_apps_infra/documents/PKITS.pdf

          std::string cadir;
          ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
          if(vctx != NULL) cadir = vctx->ca_path;

          //Try to find the preceding intermediate certificate(CA), which may
          //differ in capitalization, but match when a case insensitive match is performed
          std::string cafile; //"./NIST_PKITSTest_data/certs/GoodCACert.crt";
          //X509* current_cert = ctx->current_cert;
          cafile = get_cafile_from_cert(context, cadir, current_cert);       
          std::cout<<"cafile:  "<<cafile<<std::endl;

          //Complete the chain since we have found the CA file
          X509_STORE* store;
          X509_LOOKUP* lookup;
          store = ctx->ctx;
          if(store != NULL) { 
            lookup=X509_STORE_add_lookup(store,X509_LOOKUP_file());
            if (lookup != NULL) {
              if (!cafile.empty()) {
                int i = X509_LOOKUP_load_file(lookup, cafile.c_str(),X509_FILETYPE_PEM);
                if(!i) i = X509_LOOKUP_load_file(lookup, cafile.c_str(),X509_FILETYPE_ASN1);
                if(!i) context.LogFormat(Context::LogError, "Failed to load CA file %s", cafile.c_str());

                X509* cert = NULL;
                cert = ctx->current_cert;

                //Lookup certs from the certificate store
                for (;;) {
                  int ok;
                  X509* xtmp = NULL;
                  // If self signed, break
                  ok = ctx->check_issued(ctx,cert,cert);
                  if(ok) break;
                  ok = ctx->get_issuer(&xtmp, ctx, cert);

                  if (ok < 0) { preverify_ok = 0; break; } //some error happens
                  if (ok == 0) { preverify_ok = 1; break; } //cerificate not found

                  cert = xtmp;
                  sk_X509_push(ctx->chain,cert);
                }
                preverify_ok = 1;
              }
            }
          }
        }; break;
        default: {
          context.LogFormat(Context::LogError,"Verification error number: %d; error string: %s", err, X509_verify_cert_error_string(err));
        }; break;
      }
    }

    /** Handle the case of delta-CRL
        The end entity certificate is listed as revoked on the delta-CRL,
        but not on the complete CRL, see Section 4.15.4 of PKITS file.
    */
    if (preverify_ok == 1 || ((preverify_ok == 0) && (err == X509_V_ERR_CERT_REVOKED))) {
      std::string cadir;
      ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
      if(vctx != NULL) cadir = vctx->ca_path;

      std::string ca_ext = get_cert_extension("X509v3 Basic Constraints", current_cert);
      bool is_ca = false;
      if(ca_ext.find("CA:TRUE") != std::string::npos) is_ca = true;

      X509_NAME* issuer_name = X509_get_issuer_name(current_cert);
      std::string hash_str = get_hash_value(issuer_name);
      if(!hash_str.empty() && !is_ca) {
        std::string crl_file;
        for(int i=0;; i++) {
          char suffix = '0' + i;
          crl_file = cadir + "/" + hash_str + ".r" + suffix;
          X509_CRL* crl = NULL;

          if(AuthN::Utils::File::file_exist(crl_file))
            loadCRLFile(context, crl_file, crl, true);
          else break;

          if(crl != NULL) {
            std::string delta_crl_indicator = get_crl_extension("X509v3 Delta CRL Indicator", crl);
            if(!delta_crl_indicator.empty() && i==0) {
              // the CRL covering the end entity certificate includes 
              // a deltaCRLIndicator extension, but no other CRLs are 
              // available for the intermediate certificate, see Section 4.15.1
              context.Log(Context::LogError, "The CRL covering the end entity certificate includes a deltaCRLIndicator extension, but no other CRLs are available for the intermediate certificate.");
              preverify_ok = 0; X509_CRL_free(crl); break;
            }
            if(!delta_crl_indicator.empty()) {
              // Due to the fact that openssl can not handle the situation
              // when both *.r0 and *.r1 (with the same hash) exist,
              // we need to temporarily move the *.r0 into some temporary location,
              // so that we can use the *.r1 for certificate cheking.
              std::string tmp = cadir + "/tmplocation";
              std::string crl_file_r0 = cadir + "/" + hash_str + ".r0";
              AuthN::Utils::File::movefile(crl_file_r0, tmp);
              AuthN::Utils::File::movefile(crl_file, crl_file_r0);
              if(!check_cert(context, cadir, "", "", current_cert, true, false)) { 
                preverify_ok = 0; 
                AuthN::Utils::File::movefile(crl_file_r0, crl_file);
                AuthN::Utils::File::movefile(tmp, crl_file_r0);
                X509_CRL_free(crl); break; 
              }
              else {
// TODO: exact version need to be identified
// TODO: workaround or different solution need to be provided
#if (OPENSSL_VERSION_NUMBER  > 0x009080ffL)
                if((preverify_ok == 0) && (err == X509_V_ERR_CERT_REVOKED)) {
                  /** Look for serial number of certificate in delta CRL, 
                   if the serial number is found and the reason is 
                   set as removeFromCRL, it means even though the certificate
                   is listed as on hold on the complete CRL, the delta-CRL
                   indicate that it should be removed from the CRL, 
                   see section 4.15.5
                  */
                  X509_CRL* crl = NULL;
                  loadCRLFile(context, crl_file_r0, crl, true);
                  if(crl != NULL) {
                    X509_REVOKED* rev = NULL;
                    if(X509_CRL_get0_by_cert(crl, &rev, current_cert)) {
                      if(rev->reason == CRL_REASON_REMOVE_FROM_CRL)
                        preverify_ok = 1;
                    }
                  }
                }
#endif
                AuthN::Utils::File::movefile(crl_file_r0, crl_file);
                AuthN::Utils::File::movefile(tmp, crl_file_r0);
              }
            }
            X509_CRL_free(crl);
          }
          else break;
        }
      }
    }


    /** Handle the OCSP certificate status
    */
    if(preverify_ok ==1) {
      ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
      if(vctx != NULL) {
        switch(vctx->cert_status) {
          case OCSPContext::CertStatusGood:
            X509_STORE_CTX_set_error(ctx, X509_V_OK); break;
          case OCSPContext::CertStatusRevoked:
            preverify_ok=0;
            X509_STORE_CTX_set_error(ctx, X509_V_ERR_CERT_REVOKED); break;
          case OCSPContext::CertStatusUnknown:
            //preverify_ok=0;
            //X509_STORE_CTX_set_error(ctx, X509_V_ERR_APPLICATION_VERIFICATION); break;
            // TODO: how to deal with Unknown status?
          default: break;
        }
      }
    }

    /** Handle the certificate policy
    */

    // Handle Inhibit Any Policy
    if(preverify_ok == 1) {
      ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
      if(vctx != NULL) {
        std::string cert_policy_ext = get_cert_extension("X509v3 Certificate Policies", current_cert);
        if(!cert_policy_ext.empty()) {
          // Get the policy content from "X509v3 Certificate Policies" extension
          bool any_policy = false;
          std::string::size_type find = cert_policy_ext.find("Policy:");
          if(find != std::string::npos) {
            std::string cert_policy = cert_policy_ext.substr(find + 8);
            if(cert_policy.find("X509v3 Any Policy") != std::string::npos) any_policy = true;
          }
 
          // Check against the inhibit_anypolicy from ValidationContext
          if((vctx->policy_context.inhibit_anypolicy == 0) && any_policy) { 
            context.Log(Context::LogInfo, "This certificate is not allow to assert X509v3 Any Policy");
            preverify_ok = 0;
          }
        }
        if(vctx->policy_context.inhibit_anypolicy > 0) (vctx->policy_context.inhibit_anypolicy)--;
          
        // Get information from "X509v3 Inhibit Any Policy"
        // extension, and store it into ValidationContext
        std::string inhibit_any_policy = get_cert_extension("X509v3 Inhibit Any Policy", current_cert);
        if(!inhibit_any_policy.empty()) {
          int num = atoi(inhibit_any_policy.c_str());
          if((vctx->policy_context.inhibit_anypolicy == -1) || vctx->policy_context.inhibit_anypolicy > num) 
            vctx->policy_context.inhibit_anypolicy = num; 
          // Note here we only set inhibit_anypolicy if it has not been set by any cert in the path,
          // or value in the ValiationContext is bigger than the value of this cert.
          // See section 4.12.5 and 4.12.6 for the situation when two certs in the same path both have
          // "X509v3 Inhibit Any Policy" as extension
        }
      }
    }

    // Handle Inhibit Policy Mapping
    if(preverify_ok == 1) {
      ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
      if(vctx != NULL) {
        std::string policy_mapping_ext = get_cert_extension("X509v3 Policy Mappings", current_cert);
        bool policy_mapping = false;
        if(!policy_mapping_ext.empty()) {
          policy_mapping = true;

          // Check against the inhibit_policymapping from ValidationContext
          if((vctx->policy_context.inhibit_policymapping == 0) && policy_mapping) {
            context.Log(Context::LogInfo, "This certificate is not allow to assert X509v3 Policy Mappings");
            preverify_ok = 0;
          }
        }
        if(vctx->policy_context.inhibit_policymapping > 0) (vctx->policy_context.inhibit_policymapping)--;

        // Get information from "X509v3 Policy Constraints"
        // extension, and store it into ValidationContext
        std::string policy_constraints = get_cert_extension("X509v3 Policy Constraints", current_cert);
        if(!policy_constraints.empty()) {
          std::string inhibit_policy_mapping;
          std::string::size_type find = policy_constraints.find("Inhibit Policy Mapping");
          if(find != std::string::npos) {
            inhibit_policy_mapping = policy_constraints.substr(find + 23);
            int num = atoi(inhibit_policy_mapping.c_str());
            if((vctx->policy_context.inhibit_policymapping == -1) || vctx->policy_context.inhibit_policymapping > num)
              vctx->policy_context.inhibit_policymapping = num;
            // Note here we only set inhibit_policymapping if it has not been set by any cert in the path,
            // or value in the ValidationContext is bigger than the value of this cert.
            // See section 4.11.5 and 4.11.6 for the situation when two certs in the same path both have
            // "Inhibit Policy Mapping" as extension
          }
        }
      }
    }

    // Handle pathLenConstraint
    if(preverify_ok == 1) {
      ValidationContext* vctx  = (ValidationContext*)X509_STORE_CTX_get_ex_data(ctx,
                                      get_proxy_auth_ex_data_idx());
      if(vctx != NULL) {
        // Check the path_len in ValidationContext
        if(vctx->path_len == -1) {
          context.Log(Context::LogInfo, "The length of path violates the pathLenConstraint");
          preverify_ok = 0;
        }

        // Get information from "X509v3 Basic Constraints"
        // extension, and store it into ValidationContext
        std::string basic_constraints = get_cert_extension("X509v3 Basic Constraints", current_cert);
        if(!basic_constraints.empty()) {
          std::string path_len;
          std::string::size_type find = basic_constraints.find("pathlen");
          if(find != std::string::npos) {
            path_len = basic_constraints.substr(find + 8);
            int num = atoi(path_len.c_str());
            if((vctx->path_len == -2) || vctx->path_len > num)
              vctx->path_len = num;
          }
        }
        if(vctx->path_len >= 0) (vctx->path_len)--;
      }
    }

#ifdef HAVE_OPENSSL_PROXY
    /**The proxy related checking goes as the following. 
       Note that the openssl is assumpted to be able to recognize proxy cert info extension;
       We are trying to strictly follow RFC 3820 as much as possible.
    */
    if (preverify_ok == 1) {
      if (current_cert->ex_flags & EXFLAG_PROXY) {
        ProxyPolicyList* proxy_policy_list = &(vctx->ppl);
        if(vctx != NULL) {
          PROXY_CERT_INFO_EXTENSION *pci = (PROXY_CERT_INFO_EXTENSION*)X509_get_ext_d2i(current_cert, NID_proxyCertInfo,
                                           NULL, NULL);
          switch (OBJ_obj2nid(pci->proxyPolicy->policyLanguage)) {
          /**RFC 3820:
          The policyLanguage field has two values of special importance,
          defined in Appendix A, that MUST be understood by all parties
          accepting Proxy Certificates:

         *id-ppl-inheritAll indicates that this is an unrestricted proxy
          that inherits all rights from the issuing PI.  An unrestricted
          proxy is a statement that the Proxy Issuer wishes to delegate all
          of its authority to the bearer (i.e., to anyone who has that proxy
          certificate and can prove possession of the associated private
          key).  For purposes of authorization, this an unrestricted proxy
          effectively impersonates the issuing PI.

         *id-ppl-independent indicates that this is an independent proxy
          that inherits no rights from the issuing PI.  This PC MUST be
          treated as an independent identity by relying parties.  The only
          rights this PC has are those granted explicitly to it.

          For either of the policyLanguage values listed above, the policy
          field MUST NOT be present.
          */
            case NID_Independent:
            /**If there is no explicit rights granted (usually by pulling the 
              rights from some database) to this particular proxy
              certificate, the policy (rights) is set to be empty.
              this particular proxy certificate.
              which means the subsequent proxy is void of any rights, even 
              if it is with pci nid id-ppl-inheritAll
            */
              (*proxy_policy_list).clear();
              break;

            case NID_id_ppl_inheritAll:
            /* This is basically a NOP, we simply let the current
             rights stand as they are. */
              break;

            default:
              ProxyPolicy pp;
              pp.subject_name = sub_name; 
              if((pci->proxyPolicy) && (pci->proxyPolicy->policy) &&
                 (pci->proxyPolicy->policy->data))
                pp.policy = std::string((const char*)(pci->proxyPolicy->policy->data), pci->proxyPolicy->policy->length);
              (*proxy_policy_list).push_back(pp);
              break;
          }	
          PROXY_CERT_INFO_EXTENSION_free(pci);
        }
      }
    } 
#endif

    if(preverify_ok == 1) {
      //check CA policy matching
      char* subject_name = X509_NAME_oneline(X509_get_subject_name(current_cert),NULL,0);
      // eugridpma namespaces policy
      // Do not apply to proxies and self-signed CAs.
#ifdef HAVE_OPENSSL_PROXY
      int pos = X509_get_ext_by_NID(current_cert, NID_proxyCertInfo, -1);
      if(pos < 0)
#endif

     if(vctx != NULL) {
        //Search the namespace policy file
        NamespacesPolicy* in = NULL;
        if(vctx->validation_mode & Validator::ValidationNamespaceCheck) {
          in = new NamespacesPolicy(X509_get_issuer_name(current_cert), vctx->ca_path);
          if(!*in) {
            context.LogFormat(Context::LogError,
                  "Failed to open namespaces policy: %s", in->Error().c_str());
            delete in; in = NULL;
          }
        }
        else if(vctx->validation_mode & Validator::ValidationNamespaceLax) {
          in = new NamespacesPolicy(X509_get_issuer_name(current_cert), vctx->ca_path);
          if(!*in) { delete in; in = NULL; }; // No suitable namespace => no check
        }
        if(in) {
          if(!in->Match(X509_get_subject_name(current_cert))) {
            context.LogFormat(Context::LogError,"Certificate %s failed namespaces policy", subject_name);
            preverify_ok=0;
            X509_STORE_CTX_set_error(ctx,X509_V_ERR_SUBJECT_ISSUER_MISMATCH);
          } else {
            context.LogFormat(Context::LogInfo, "Certificate %s passed namespaces policy", subject_name);
          }
        };
        delete in;
      };

      //Check the left validity time of the peer certificate;
      //Give warning if the certificate is going to be expired
      //in a while of time
      AuthN::Utils::Time exptime = asn1_to_utctime(X509_get_notAfter(current_cert));
      if(exptime <= AuthN::Utils::Time()) {
        context.LogFormat(Context::LogWarning,"Certificate %s already expired", subject_name);
      } else {
        AuthN::Utils::Period timeleft = exptime - AuthN::Utils::Time();
#ifdef HAVE_OPENSSL_PROXY
        int pos = X509_get_ext_by_NID(current_cert,NID_proxyCertInfo,-1);
#else
        int pos = -1;
#endif
        //for proxy certificate, give warning 1 hour in advance
        //for EEC certificate, give warning 5 days in advance
        if(((pos < 0) && (timeleft <= 5*24*3600)) ||
           (timeleft <= 3600)) {
          context.LogFormat(Context::LogWarning,"Certificate %s will expire in %s", subject_name, ((std::string)timeleft).c_str());
        }
      }
      OPENSSL_free(subject_name);
    };

    return(preverify_ok);
  }


/*
  static int BIO_write_filename_User(BIO *b, const char* file) {
    return BIO_write_filename(b, (char*)file);
  }

  static int BIO_read_filename_User(BIO *b, const char* file) {
    return BIO_read_filename(b, (char*)file);
  }
*/

  static const EVP_MD* get_digest(Context& context, const std::string& digest_name, const EVP_PKEY* privkey) {
    const EVP_MD* digest = NULL;
    int digest_nid;
    char* digest_str = NULL;

    //If private key is available, try to get the digest information from it
    //Because EVP_PKEY_get_default_digest_nid is only available on openssl version newer than 1.0.0,
    //for relatively older version of openssl, digest info of private key will not be obtained.
#if OPENSSL_VERSION_NUMBER >= 0x10000000L
    int i = EVP_PKEY_get_default_digest_nid((EVP_PKEY*)privkey, &digest_nid);
    if(privkey != NULL) { 
      if(i <= 0) {
        context.LogFormat(Context::LogInfo,"There is no digest in issuer's private key object");
      }
      else if(i == 2) { //return 2 if the digest is mandatory (other digests can not be used)
        digest_str = (char *)OBJ_nid2sn(digest_nid);
      }
      else { //returns 1 if the message digest is advisory (that is other digests can be used) 
        if(digest_name.empty()) digest_str = (char *)OBJ_nid2sn(digest_nid);
      }
    }
#endif
    if(digest_str == NULL && !digest_name.empty())  digest_str = (char*)(digest_name.c_str());
    if(digest_str != NULL)
      if((digest = EVP_get_digestbyname(digest_str)) == NULL)
        context.LogFormat(Context::LogInfo, "%s is an unsupported digest type", digest_str);

    if(digest == NULL) digest = EVP_sha1();

    // Check whether the digest algorithm is SHA1 or SHA2
    // For openssl version older than 0.9.8f, SHA2 is not supported
    digest_nid = EVP_MD_type(digest);
#if OPENSSL_VERSION_NUMBER >= 0x0090800fL
    if((digest_nid != NID_sha1) && (digest_nid != NID_sha224) && (digest_nid != NID_sha256) && (digest_nid != NID_sha384) && (digest_nid != NID_sha512)) {
      context.LogFormat(Context::LogError, "The signing algorithm %s is not allowed,it should be SHA1 or SHA2 to sign certificate requests",
      OBJ_nid2sn(digest_nid));
      return NULL;
    }
#else
    if(digest_nid != NID_sha1) {
      context.LogFormat(Context::LogError, "The signing algorithm %s is not allowed,it should be SHA1 to sign certificate requests",
      OBJ_nid2sn(digest_nid));
      return NULL;
    }
#endif

    context.LogFormat(Context::LogInfo, "The digest that is used is: %s", OBJ_nid2sn(digest_nid));

    return digest;
  } 



  X509_STORE* setup_verify(Context& context, const std::string& CAfile, const std::string& CApath) {
    X509_STORE* store;
    X509_LOOKUP* lookup;
    if(!(store = X509_STORE_new())) goto end;
    lookup=X509_STORE_add_lookup(store,X509_LOOKUP_file());
    if(lookup == NULL) goto end;
    if(!CAfile.empty()) {
      if(!X509_LOOKUP_load_file(lookup,CAfile.c_str(),X509_FILETYPE_PEM)) {
        context.LogFormat(Context::LogError, "Failed to load CA file %s", CAfile.c_str());
        goto end;
      }
    } else X509_LOOKUP_load_file(lookup,NULL,X509_FILETYPE_DEFAULT);

    lookup=X509_STORE_add_lookup(store,X509_LOOKUP_hash_dir());
    if(lookup == NULL) goto end;
    if(!CApath.empty()) {
      if(!X509_LOOKUP_add_dir(lookup,CApath.c_str(),X509_FILETYPE_PEM)) {
        context.LogFormat(Context::LogError, "Failed to load CA directory %s", CApath.c_str());
        goto end;
      }
    } else X509_LOOKUP_add_dir(lookup,NULL,X509_FILETYPE_DEFAULT);

    ERR_clear_error();
    return store;
    end:
    X509_STORE_free(store);
    return NULL;
  }

  std::string get_cert_hash(X509* cert) {
    std::string hash_str;
    if(!cert) return std::string();
    char hash[32];
    memset(hash, 0, 32);
    snprintf(hash, 32, "%08lx", X509_subject_name_hash(cert));
    hash_str = hash;
    return hash_str;
  }

  std::string get_crl_hash(X509_CRL* crl) {
    std::string hash_str;
    if(!crl) return std::string();
    char hash[32];
    memset(hash, 0, 32);
    snprintf(hash, 32, "%08lx", X509_NAME_hash(X509_CRL_get_issuer(crl)));
    hash_str = hash;
    return hash_str;
  }

  // Using pthreads as sufficiently portable interface.
  pthread_mutex_t * OpenSSLInitializer::ssl_locks = NULL;
  int OpenSSLInitializer::ssl_locks_num = 0;

  OpenSSLInitializer::OpenSSLInitializer(void) {
    static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;
    static bool initialized = false;
    pthread_mutex_lock(&lock);
    if(initialized) return;

    SSL_load_error_strings();
    SSL_library_init();

    // We could RAND_seed() here. But since 0.9.7 OpenSSL
    // knows how to make use of OS specific source of random
    // data. I think it's better to let OpenSSL do a job.

    // Always make sure our own locks are installed
    int num = CRYPTO_num_locks();
    if(num > 0) {
      ssl_locks = new pthread_mutex_t[num];
      for(int n = 0; n<num; ++n) pthread_mutex_init(ssl_locks+n,NULL);
      ssl_locks_num = num;
      CRYPTO_set_locking_callback(&ssl_locking_cb);
      CRYPTO_set_id_callback(&ssl_id_cb);
    }
    OpenSSL_add_all_algorithms();

    initialized=true;

    pthread_mutex_unlock(&lock);
  }

  ContextOpenSSLUI::ContextOpenSSLUI() : Context() {
  }


  std::string ContextOpenSSLUI::Password(const std::string& message) {
    if(!password.empty()) return password;

    const int pass_min_length = 1;
    const int pass_max_length = 256;
    UI *ui = UI_new();
    if (!ui) return "";
    char buf[pass_max_length+1];
    int ok = 0;
    int ui_flags = 0;

    memset(buf,0,pass_max_length+1);
    //ui_flags |= UI_INPUT_FLAG_DEFAULT_PWD;
    if((ok = UI_add_input_string(ui,message.c_str(),ui_flags,buf,pass_min_length,pass_max_length)) < 0) {
      UI_free(ui); return "";
    }
    do{
      ok = UI_process(ui);
      if(ok == -2) break; // Abort request
      if(ok == -1) { // Password error
        unsigned long errcode = ERR_get_error();
        const char* errstr = ERR_reason_error_string(errcode);
        if(errstr == NULL) {
          Message("Error code " + tostring(errcode));
        } else if(strstr(errstr,"result too small")) {
          Message("Password is too short, need at least "+tostring(pass_min_length)+" charcters");
        } else if(strstr(errstr,"result too large")) {
          Message("Password is too long, need at most "+tostring(pass_max_length)+" characters");
        } else {
          Message(std::string(errstr));
        }
      }
    }while (ok < 0 && UI_ctrl(ui, UI_CTRL_IS_REDOABLE, 0, 0, 0));
    UI_free(ui);
    return buf;
  }

  void ContextOpenSSLUI::Message(const std::string& message) {
    UI *ui = UI_new();
    if (!ui) return;
    if(UI_add_info_string(ui,message.c_str())) {
      UI_free(ui); return;
    }
    UI_process(ui);
    UI_free(ui);
  }

  void ContextOpenSSLUI::SetPassword(const std::string& pass) {
    password = pass;
  }

  Context& ContextOpenSSLUI::Copy(void) const {
    return *(new ContextOpenSSLUI(*this));
  }

  unsigned long OpenSSLInitializer::ssl_id_cb(void) {
#ifdef WIN32
    return (uintptr_t)pthread_self();
#else
    return (unsigned long)pthread_self();
#endif
  }

  void OpenSSLInitializer::ssl_locking_cb(int mode, int n, const char * /*s_*/, int /*n_*/){
    if(!ssl_locks) return; // Non-threaded environment? Or maybe _exit(-1) ?
    if((n < 0) || (n >= ssl_locks_num)) _exit(-1); // fatal
    if(mode & CRYPTO_LOCK) {
      pthread_mutex_lock(ssl_locks+n);
    } else {
      pthread_mutex_unlock(ssl_locks+n);
    };
  }

  void OpenSSLInitializer::ThreadCleanup(void) {
    ERR_remove_state(0);
  }

  static OpenSSLInitializer openssl_initializer;

}
}

