/*
** A program to convert the XML rendered by Calligra into LATEX.
**
** SPDX-FileCopyrightText: 2000-2002 Robert JACOLIN
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
**
** To receive a copy of the GNU Library General Public License, write to the
** Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
**
*/

#include "xmlparser.h"

#include "LatexDebug.h"

#include <KoStore.h>

#include <QFile>

/* Init static data */
FileHeader* XmlParser::_fileHeader = nullptr;
Document* XmlParser::_root = nullptr;
KoStore* XmlParser::_in = nullptr;

XmlParser::XmlParser(const QString &filename): _filename(filename)
{
    QFile f(filename);
    if (!f.open(QIODevice::ReadOnly))
        return;
    if (!_document.setContent(&f)) {
        f.close();
        return;
    }
    f.close();
    //_eltCurrent = _document.documentElement();
}

XmlParser::XmlParser(QByteArray in)
{
    _document.setContent(in);
}

XmlParser::XmlParser(const KoStore* in)
{
    _in = const_cast<KoStore*>(in);
    if (!_in->open("root")) {
        errorLatex << "Unable to open input file!" << Qt::endl;
        return;
    }
    /* input file Reading */
    QByteArray array = _in->read(_in->size());
    _document.setContent(array);
}

XmlParser::XmlParser()
= default;

XmlParser::~XmlParser()
{
    if (_in != nullptr)
        _in->close();
}

QDomNode XmlParser::getChild(const QDomNode &node, QString name)
{
    QDomNode childNode = getChild(node, name, 0);
    debugLatex << childNode.nodeName();
    return childNode;
}

bool XmlParser::isChild(const QDomNode &node, QString name)
{
    if (node.isElement())
        return node.toElement().elementsByTagName(name).count();
    return false;
}

QDomNode XmlParser::getChild(const QDomNode &node, QString name, int index)
{
    if (node.isElement()) {
        QDomNodeList children = node.toElement().elementsByTagName(name);
        if (children.count())
            return children.item(index);
    }
    return QDomNode();
}

QDomNode XmlParser::getChild(const QDomNode &node, int index)
{
    QDomNodeList children = node.childNodes();
    if (children.count())
        return children.item(index);
    return QDomNode();
}

QString XmlParser::getData(const QDomNode &node, int index)
{
    return getChild(getChild(node, index), 0).nodeValue();
}

QString XmlParser::getData(const QDomNode &node, QString name)
{
    return getChild(getChild(node, name), 0).nodeValue();
}

int XmlParser::getNbChild(const QDomNode &node)
{
    return node.childNodes().count();
}

int XmlParser::getNbChild(const QDomNode &node, QString name)
{
    if (node.isElement())
        return node.toElement().elementsByTagName(name).count();
    return -1;
}

QString  XmlParser::getChildName(const QDomNode &node, int index)
{
    return node.childNodes().item(index).nodeName();
}

QString  XmlParser::getAttr(const QDomNode &node, QString name) const
{
    if (node.isElement())
        return node.toElement().attributeNode(name).value();
    return QString();
}
