// Copyright 2018 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.skyframe;

/**
 * Hermeticity of a {@link SkyFunction}, meaning whether it accesses external state untracked by
 * Skyframe during its evaluation. A classic example is a {@link SkyFunction} that consumes a file
 * on a filesystem: that state is untracked by Skyframe. Skyframe must be more conservative when
 * using values generated by a non-hermetic function: for instance, a non-hermetic function may need
 * to be re-run even if all its Skyframe dependencies are unchanged: such a node may be explicitly
 * dirtied due to outside changes.
 *
 * <p>Note that Skyframe does <i>not</i> explicitly re-evaluate non-hermetic functions on every
 * build: it just relaxes some of its graph-pruning logic to be more conservative with such nodes.
 */
public enum FunctionHermeticity {

  /**
   * A fully hermetic function. If its Skyframe dependencies are unchanged, it will always produce
   * the same result.
   */
  HERMETIC,

  /**
   * A function that is mostly hermetic, but may act non-hermetic in extenuating circumstances such
   * as when a flag value is changed.
   *
   * <p>Nodes produced by this type of function are treated the same as HERMETIC with respect to
   * versioning and graph pruning. Unlike HERMETIC, however, no error will occur if the function
   * exhibits nondeterminism.
   */
  SEMI_HERMETIC,

  /**
   * A function that is expected to routinely produce different results even if its Skyframe
   * dependencies are unchanged.
   */
  NONHERMETIC
}
