/*
 *   This file is part of Auralquiz
 *   Copyright 2011-2012  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "auralwindow.h"


/*****************************************************************************************
 *
 * Constructor
 */
AuralWindow::AuralWindow(QWidget *parent) : QWidget(parent)
{
    QSettings settings;
    this->setWindowTitle("Auralquiz");
    this->setWindowIcon(QIcon(":/icon/auralquiz.png"));
    this->resize(settings.value("size", QSize(800, 540)).toSize());
    this->firstRun = settings.value("firstRun", true).toBool();


    if (firstRun)
    {
        qDebug() << "This is the first run";
        QMessageBox::about(this,
                           "Auralquiz - " + tr("First usage"),
                           tr("This seems to be the first time you use Auralquiz.\n"
                              "Before playing, your music will be analyzed.\n"
                              "If needed, you should click the Options button "
                              "and select the folder where your "
                              "Ogg, FLAC and MP3 files are stored.\n\n"
                              "This folder, and sub-folders will be scanned "
                              "so Auralquiz can generate questions and answers "
                              "about your music.\n"
                              "\n"
                              "You need files correctly tagged in order for the game to work correctly.\n"
                              "\n"
                              "The scan can take some time, and the program "
                              "will not be responsive until it is complete. "
                              "Please be patient."));
    }

    useOwnColorTheme = settings.value("useOwnColorTheme", false).toBool();
    if (useOwnColorTheme)
    {
        qDebug() << "Using own color theme";
    }
    else
    {
        qDebug() << "Using system colors";
    }


    musicDirectory = settings.value("musicDirectory",
                                    QDesktopServices::storageLocation(QDesktopServices::MusicLocation)).toString();
    if (!musicDirectory.endsWith("/"))
    {
        // Adding final "/" if it's not present in chosen path
        musicDirectory.append("/");
    }
    qDebug() << "Music directory:" << this->musicDirectory;


    // get data directory path
    this->dataDirectory = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
    qDebug() << "Data directory:" << this->dataDirectory;


    // Create data directory if needed, to store music info later...
    QDir dataDir;
    if (!dataDir.exists(this->dataDirectory))
    {
        qDebug() << "Data directory did not exist. Creating...";
        if (dataDir.mkpath(this->dataDirectory))
        {
            qDebug() << this->dataDirectory << "directory created successfully!";
        }
        else
        {
            qDebug() << this->dataDirectory << "directory could NOT be created";
        }
    }

    difficultyLevel = settings.value("difficultyLevel", 2).toInt(); // normal(2) by default
    numQuestions = settings.value("numQuestions", 25).toInt(); // 25 questions by default
    numPlayers = settings.value("numPlayers", 1).toInt(); // 1 player by default


    qDebug() << "Phonon version:" << Phonon::phononVersion();
    qDebug() << "Backend::audioEffects:" << Phonon::BackendCapabilities::availableAudioEffects();
    //qDebug() << "Backend::mimeTypes:" << Phonon::BackendCapabilities::availableMimeTypes();
    //qDebug() << "Backend::audioDevices:" << Phonon::BackendCapabilities::availableAudioOutputDevices();


    playing = false;

    mainLayout = new QVBoxLayout(this);

    initWelcomeScreen();
    initPlayingScreen();

    this->playerNames.clear();
    playerNames = settings.value("playerNames",
                                 QStringList() << "1" << "2"  // overriden
                                               << "3" << "4"  // in OptionsDialog
                                               << "5" << "6"
                                               << "7" << "8").toStringList();

    this->score.clear(); // Clear list of players' scores, etc, and set all to 0
    this->goodAnswers.clear();
    this->badAnswers.clear();
    this->timedOutAnswers.clear();
    for (int counter = 0; counter != MAXPLAYERS; ++counter)
    {
        score << 0;
        goodAnswers << 0;
        badAnswers << 0;
        timedOutAnswers << 0;
    }


    this->setLayout(mainLayout);

    postInitTimer = new QTimer(this);
    postInitTimer->setSingleShot(true);
    postInitTimer->setInterval(500);
    connect(postInitTimer, SIGNAL(timeout()), this, SLOT(loadSongList()));
    postInitTimer->start(); // Call loadSongList() from the timer, to avoid the
                            // first-run analyzing all songs without visible window


    // Timer used to show the Ranking window after a moment
    rankingTimer = new QTimer(this);


    // TEMPORARY ranking tests - START
/*
    this->goodAnswers.clear();
    goodAnswers << 3 << 17 << 3 << 4 << 5 << 6 << 8 << 8;
    this->badAnswers.clear();
    badAnswers << 11 << 22 << 33 << 44 << 55 << 66 << 77 << 88;
    this->timedOutAnswers.clear();
    timedOutAnswers << 111 << 222 << 333 << 444 << 555 << 666 << 777 << 888;
    this->score.clear();
    score << goodAnswers[0]*123 << goodAnswers[1]*123 << goodAnswers[2]*123
          << goodAnswers[3]*123 << goodAnswers[4]*123 << goodAnswers[5]*123
          << goodAnswers[6]*123 << goodAnswers[7]*123;

    qDebug() << "Testing scores:" << score;
    Ranking *rankingTest;
    rankingTest = new Ranking(this->score.length(), this->playerNames, this->score,
                          this->goodAnswers, this->badAnswers, this->timedOutAnswers);
    rankingTest->show();
    qDebug() << "test ranking created and shown";
*/
    // TEMPORARY ranking tests - END
}


/*********************************************************************************
 *
 *  Destructor
 */
AuralWindow::~AuralWindow()
{
    qDebug() << "AuralWindow destroyed";
}


/*********************************************************************************
 *
 * Store app config upon exit
 */
void AuralWindow::closeEvent(QCloseEvent *event)
{
    QSettings settings;
    settings.setValue("firstRun", false);

    settings.setValue("size", this->size());

    settings.setValue("useOwnColorTheme", this->useOwnColorTheme);


    settings.setValue("musicDirectory", this->musicDirectory);
    settings.setValue("numQuestions", this->numQuestions);
    settings.setValue("difficultyLevel", this->difficultyLevel);
    settings.setValue("numPlayers", this->numPlayers);

    settings.setValue("playerNames", this->playerNames);

    qDebug("closeEvent: config saved");
    event->accept();
}



/*******************************************************************************
 *
 * Shuffle song list
 */
void AuralWindow::shuffleMusicFiles()
{
    uint randomSeed;
    randomSeed = (QTime::currentTime().hour())
               + (QTime::currentTime().minute() * 4)
               + (QTime::currentTime().second() * 5)
               + (QTime::currentTime().msec() * 6);
    randomSeed *= 8;
    randomSeed += qrand() % (randomSeed / (QTime::currentTime().second()+1));


    qsrand(randomSeed); // ensure decent randomness based on current time



    int newPosition;
    for (int counter=0; counter != musicFiles[0].length(); ++counter)
    {   newPosition = qrand() % musicFiles[0].length();
        musicFiles[0].swap(0, newPosition); // filename
        musicFiles[1].swap(0, newPosition); // artist
        musicFiles[2].swap(0, newPosition); // title
    }
    qDebug() << "Music Files shuffled. randomSeed:" << randomSeed;
}



/*
 * Update statistics panel
 */
void AuralWindow::updateStatistics()
{
    this->statisticsLabel->setText("\n"
                                   + tr("Good:  %1").arg(this->goodAnswers[currentPlayer])
                                   + "\n\n"
                                   + tr("Bad:  %2").arg(this->badAnswers[currentPlayer])
                                   + "\n\n"
                                   + tr("Timed out:  %3").arg(this->timedOutAnswers[currentPlayer])
                                   + "\n");
}


/*****************************************************************************************
 *
 *  Set up the welcome screen, with the logo and main menu
 */
void AuralWindow::initWelcomeScreen()
{
    qDebug() << "Init welcome screen";
    welcomeWidget = new QWidget();

    welcomeLayout = new QVBoxLayout();
    welcomeLayout->setAlignment(Qt::AlignHCenter);

    logoLabel = new QLabel(this);
    logoLabel->setPixmap(QPixmap(":/images/logo.png"));


    optionsDialog = new OptionsDialog();


    startGameButton = new QPushButton(QIcon::fromTheme("media-playback-start"),
                                      "\n" + tr("&Start game") + "\n");
    connect(startGameButton, SIGNAL(clicked()), optionsDialog, SLOT(showPlayMode()));
    startGameButton->setDisabled(true);


    connect(optionsDialog, SIGNAL(optionsChanged(bool,QString,bool,int,int,int,QStringList,bool)),
                     this, SLOT(updateConfig(bool,QString,bool,int,int,int,QStringList,bool)));

    configureButton = new QPushButton(QIcon::fromTheme("configure"), tr("&Options"));
    connect(configureButton, SIGNAL(clicked()), optionsDialog, SLOT(showConfigMode()));


    aboutButton = new QPushButton(QIcon::fromTheme("help-about"), tr("&About..."));
    connect(aboutButton, SIGNAL(clicked()), this, SLOT(showAbout()));

    quitButton = new QPushButton(QIcon::fromTheme("application-exit"), tr("&Quit"));
    quitButton->setShortcut(QKeySequence::Quit);
    connect(quitButton, SIGNAL(clicked()), this, SLOT(close()));


    welcomeLayout->addWidget(logoLabel);
    welcomeLayout->addSpacing(16);

    welcomeLayout->addWidget(startGameButton);
    welcomeLayout->addSpacing(8);

    welcomeLayout->addWidget(configureButton);
    welcomeLayout->addWidget(aboutButton);
    welcomeLayout->addWidget(quitButton);
    welcomeLayout->addSpacing(16);

    welcomeWidget->setLayout(welcomeLayout);

    mainLayout->addWidget(welcomeWidget);
}



/*****************************************************************************************
 *
 *  Set up the playing screen, with progress bar, time bar, score, etc.
 */
void AuralWindow::initPlayingScreen()
{
    qDebug() << "Init playing screen";
    playingWidget = new QWidget();

    playingLayout = new QVBoxLayout();
    playingLayout->setAlignment(Qt::AlignRight);

    playingTopLayout = new QVBoxLayout();
    playingMiddleLayout = new QHBoxLayout();
    statisticsLayout = new QVBoxLayout();
    answersLayout = new QVBoxLayout();
    playingBottomLayout = new QHBoxLayout();


    QFont playerFont;
    playerFont = QFont();
    playerFont.setPointSize(12);
    playerFont.setBold(true);

    playerNameLabel = new QLabel(":: PLAYER ::");
    playerNameLabel->setAlignment(Qt::AlignRight | Qt::AlignTop);
    playerNameLabel->setFont(playerFont);
    //playerNameLabel->setFrameStyle(QFrame::Raised | QFrame::StyledPanel);

    QFont questionFont;
    questionFont = QFont();
    questionFont.setPointSize(28);
    questionFont.setBold(true);

    questionLabel = new QLabel(":: QUESTION ::");
    questionLabel->setAlignment(Qt::AlignCenter);
    questionLabel->setFont(questionFont);
    questionLabel->setFrameStyle(QFrame::Raised | QFrame::StyledPanel);


    QFont infoFont;
    infoFont = QFont();
    infoFont.setPointSize(11);
    infoFont.setBold(true);
    infoFont.setItalic(true);

    infoLabel = new QLabel(":: INFO ::");
    infoLabel->setAlignment(Qt::AlignCenter);
    infoLabel->setFont(infoFont);


    gameTimer = new QTimer(this);
    gameTimer->setInterval(100); // every 100ms, so it moves fast
    connect(gameTimer, SIGNAL(timeout()), this, SLOT(timerTick()));


    // This will call newQuestion
    preQuestionTimer = new QTimer(this);
    preQuestionTimer->setSingleShot(true);
    //preQuestionTimer->setInterval(3000); // Now changed in updateConfig()
    connect(preQuestionTimer, SIGNAL(timeout()), this, SLOT(newQuestion()));


    // This will call preQuestion
    postQuestionTimer = new QTimer(this);
    postQuestionTimer->setSingleShot(true);
    postQuestionTimer->setInterval(1500); // 1,5 seconds
    connect(postQuestionTimer, SIGNAL(timeout()), this, SLOT(preQuestion()));


    timeBar = new QProgressBar();
    timeBar->setOrientation(Qt::Vertical);
    timeBar->setFormat(tr("Time")); // "%v seconds"
    // timeBar's range() and value() will be set upon game start, on toggleScreen()
    timeBar->setToolTip(tr("Remaining time to answer this question"));

    gameProgressBar = new QProgressBar();
    gameProgressBar->setFormat(tr("%v out of %m questions - %p%"));
    // gameProgressBar's range() and value() will be set upon game start, on toggleScreen()
    gameProgressBar->setToolTip(tr("How many questions you've answered"));

    gameScoreLabel = new QLabel(tr("Score"));
    gameScoreLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    gameScore = new QLCDNumber(5);
    gameScore->setSegmentStyle(QLCDNumber::Flat);
    gameScore->display(0);
    gameScore->setToolTip(tr("Your current score"));

    endGameButton = new QPushButton(QIcon::fromTheme("media-playback-stop"), tr("&End game"));
    connect(endGameButton, SIGNAL(clicked()), this, SLOT(toggleScreen()));


    playingTopLayout->addWidget(playerNameLabel);
    playingTopLayout->addWidget(questionLabel);
    playingTopLayout->addWidget(infoLabel);


    aniNoteLabel = new QLabel();
    QMovie *aniNoteMovie = new QMovie(":/images/aninote.gif");
    aniNoteLabel->setMovie(aniNoteMovie);
    aniNoteLabel->setAlignment(Qt::AlignHCenter);
    statisticsLayout->addWidget(aniNoteLabel);

    statisticsLayout->addSpacing(12);


    statisticsLabel = new QLabel("STATS");
    statisticsLabel->setAlignment(Qt::AlignVCenter | Qt::AlignLeft);
    statisticsLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken); // QFrame::Box

    statisticsBoxLayout = new QVBoxLayout();
    statisticsBoxLayout->addWidget(statisticsLabel);

    statisticsBox = new QGroupBox(tr("Statistics"));
    statisticsBox->setLayout(statisticsBoxLayout);
    statisticsBox->setMinimumWidth(190);  // FIXME: don't hardcode
    statisticsBox->setMaximumWidth(190);

    statisticsLayout->addWidget(statisticsBox);


    /*
     *   Add the 4 answer buttons, and their font
     */
    answerButton[0] = new QPushButton(QIcon::fromTheme("arrow-right"),
                                      "ANSWER 1 ----------------");
    connect(answerButton[0], SIGNAL(clicked()), this, SLOT(answer1()));
    answersLayout->addWidget(answerButton[0]);

    answersLayout->addSpacing(8);

    answerButton[1] = new QPushButton(QIcon::fromTheme("arrow-right"),
                                      "ANSWER 2 ----------------");
    connect(answerButton[1], SIGNAL(clicked()), this, SLOT(answer2()));
    answersLayout->addWidget(answerButton[1]);

    answersLayout->addSpacing(8);

    answerButton[2] = new QPushButton(QIcon::fromTheme("arrow-right"),
                                      "ANSWER 3 ----------------");
    connect(answerButton[2], SIGNAL(clicked()), this, SLOT(answer3()));
    answersLayout->addWidget(answerButton[2]);

    answersLayout->addSpacing(8);

    answerButton[3] = new QPushButton(QIcon::fromTheme("arrow-right"),
                                      "ANSWER 4 ----------------");
    connect(answerButton[3], SIGNAL(clicked()), this, SLOT(answer4()));
    answersLayout->addWidget(answerButton[3]);

    QFont buttonFont;
    buttonFont = QFont();
    buttonFont.setPointSize(13);
    buttonFont.setBold(true);
    // set font and minimum height in all 4 buttons
    for (int counter = 0; counter != 4; ++counter)
    {
        answerButton[counter]->setFont(buttonFont);

        // Set a minimum width, so it's ok for most titles/names
        answerButton[counter]->setMinimumWidth(512);

        // Make buttons use all space available
        answerButton[counter]->setSizePolicy(QSizePolicy::MinimumExpanding,
                                             QSizePolicy::MinimumExpanding);
    }

    // Add the AnswerBox, used in the highest difficulty mode, type-the-answer
    answerBox = new AnswerBox();
    answerBox->setFont(buttonFont);
    answerBox->setMinimumWidth(512);
    connect(answerBox, SIGNAL(answered(bool)),
            this, SLOT(answerFromAnswerBox(bool)));
    answersLayout->addWidget(answerBox);


    playingMiddleLayout->addLayout(statisticsLayout);
    playingMiddleLayout->addSpacing(16);
    playingMiddleLayout->addLayout(answersLayout);
    playingMiddleLayout->addSpacing(16);
    playingMiddleLayout->addWidget(timeBar);

    playingBottomLayout->addWidget(gameProgressBar);
    playingBottomLayout->addSpacing(32);
    playingBottomLayout->addWidget(gameScoreLabel);
    playingBottomLayout->addSpacing(16);
    playingBottomLayout->addWidget(gameScore);

    playingLayout->addLayout(playingTopLayout);
    playingLayout->addSpacing(12);
    playingLayout->addLayout(playingMiddleLayout);
    playingLayout->addSpacing(12);
    playingLayout->addLayout(playingBottomLayout);
    playingLayout->addSpacing(32);
    playingLayout->addWidget(endGameButton);
    playingWidget->setLayout(playingLayout);

    mainLayout->addWidget(playingWidget);
    if (useOwnColorTheme)
    {
        this->setThemedColors();
    }


    musicPlayer = Phonon::createPlayer(Phonon::MusicCategory,
                                       Phonon::MediaSource());
    qDebug() << "Phonon::createPlayer()->isValid()? " << musicPlayer->isValid();
    if (!musicPlayer->isValid())
    {
        QMessageBox::warning(this, tr("Sound system error"),
                             tr("There seems to be a problem with your sound system."));
    }

    connect(musicPlayer, SIGNAL(stateChanged(Phonon::State,Phonon::State)),
            this, SLOT(playerStateChanged(Phonon::State,Phonon::State)));




    playingWidget->hide();
}


/*
 *  Set styles to app-specific, or set them empty for user/system-defined
 */
void AuralWindow::setThemedColors() /* everything here is quite temporary*/
{
    // set transparency
    this->setWindowOpacity(0.98);

    // set light-blue style to the program in general
    this->setStyleSheet("background: lightblue;"
                        "color: qlineargradient(spread:pad, x1:0, y1:0, x2:3, y2:2,"
                        "stop:0.0 rgba(10, 10, 120, 255),"
                        "stop:0.5 rgba(20, 190, 20, 255),"
                        "stop:1.0 rgba(120, 10, 10, 255) );"
                        "");

    // set colors on answer buttons
    answerButton[0]->setStyleSheet("background: #20DD20; color: darkblue");
    answerButton[1]->setStyleSheet("background: #20CC20; color: darkblue");
    answerButton[2]->setStyleSheet("background: #20BB20; color: darkblue");
    answerButton[3]->setStyleSheet("background: #20AA20; color: darkblue");


    // set color of LCD score indicator
    gameScore->setStyleSheet("color: darkBlue");


    statisticsLabel->setStyleSheet("color: darkred");
}



/************************************************************************/


/*
 *  Load music collection info from files created in previous scan
 */
void AuralWindow::loadSongList()
{
    qDebug() << "loadSongList()";

    QFile cachedMetaData0(this->dataDirectory + "/musicFiles0.aq");
    cachedMetaData0.open(QIODevice::ReadOnly);

    QFile cachedMetaData1(this->dataDirectory + "/musicFiles1.aq");
    cachedMetaData1.open(QIODevice::ReadOnly);

    QFile cachedMetaData2(this->dataDirectory + "/musicFiles2.aq");
    cachedMetaData2.open(QIODevice::ReadOnly);

    if (!cachedMetaData0.isReadable()
        || !cachedMetaData1.isReadable()
        || !cachedMetaData2.isReadable())
    {
        cachedMetaData0.close();
        cachedMetaData1.close();
        cachedMetaData2.close();
        qDebug() << "Couldn't load some of the cached metadata files; creating song list";

        this->createSongList();
        return;
    }

    this->musicFiles[0].clear();
    this->musicFiles[1].clear();
    this->musicFiles[2].clear();

    /*
     * FIXME: PLENTY OF ERROR CONTROL MISSING HERE
     */

    QString fileName;
    QString artistName;
    QString songTitle;

    qDebug() << "Loading cached metadata from files";
    while (!cachedMetaData0.atEnd())
    {
        // FIXME: separate this mess (avoids encoding problems)
        fileName = QString::fromUtf8(cachedMetaData0.readLine(1024).trimmed().data());
        musicFiles[0].append(fileName);

        // FIXME: assuming Utf8 in artist and title tags, for now
        artistName = QString::fromUtf8(cachedMetaData1.readLine(1024).trimmed().data());
        musicFiles[1].append(artistName);

        songTitle = QString::fromUtf8(cachedMetaData2.readLine(1024).trimmed().data());
        musicFiles[2].append(songTitle);
    }

    cachedMetaData0.close();
    cachedMetaData1.close();
    cachedMetaData2.close();

    this->startGameButton->setEnabled(true);
    this->startGameButton->setFocus();

    qDebug() << "loadSongList() done";
}


/*
 *  Analyze all songs in given directory, get the metadata and store it
 */
void AuralWindow::createSongList()
{
    startGameButton->setDisabled(true);
    startGameButton->setText(tr("Analyzing music. Please wait"));
    qDebug() << "Creating song list...";


    this->musicFiles[0].clear(); // make file-list empty
    this->musicFiles[1].clear(); // make artists-list empty
    this->musicFiles[2].clear(); // make titles-list empty

    directoryFiles = QStringList();

    QDirIterator dirList(QDir(this->musicDirectory, "*.mp3 *.ogg *.flac",
                              QDir::NoSort, QDir::Files),
                         QDirIterator::Subdirectories | QDirIterator::FollowSymlinks);
    while (dirList.hasNext())
    {
        directoryFiles.append(dirList.next());
        //qDebug() << directoryFiles.last();
    }

    if (directoryFiles.length() == 0)
    {
        qDebug() << "There are no files!";
        QMessageBox::critical(this,
                              tr("There is no music!"),
                              tr("There are no valid music files"
                                 " in your selected music directory.\n"
                                 "Please, select another directory containing "
                                 "Ogg, FLAC or MP3 files, and try again."));
        this->startGameButton->setText(tr("Please choose another music folder"));

        return;
    }


    // Start analyzer loop
    int validFiles = 0;
    int totalFiles = directoryFiles.length();

    QProgressDialog analyzerProgressDialog(tr("Analyzing %1 files\nunder %2", "%1 is a number, %2 is a folder")
                                           .arg(totalFiles).arg(musicDirectory),
                                           tr("Cancel analysis"),
                                           0, totalFiles, this);

    analyzerProgressDialog.setWindowTitle("Auralquiz - "
                                          + tr("Analyzing your music..."));


    for (currentMusicFile = 0; currentMusicFile != totalFiles; ++currentMusicFile)
    {
        qDebug() << "currentMusicFile:" << currentMusicFile+1 << "of" << totalFiles;
        qDebug() << "filename:" << directoryFiles.at(currentMusicFile);

        QString artistTag;
        QString titleTag;

        TagLib::FileRef tagFile(directoryFiles.at(currentMusicFile).toUtf8());
        if (!tagFile.isNull())
        {
            //artistTag = tagFile.tag()->artist().toCString();
            artistTag = QString::fromStdWString(tagFile.tag()->artist().toWString());
            //titleTag = tagFile.tag()->title().toCString();
            titleTag = QString::fromStdWString(tagFile.tag()->title().toWString());

            qDebug() << artistTag << titleTag;
            qDebug() << tagFile.audioProperties()->length();
        }
        else
        {
            qDebug() << "-- tagFile error:" << directoryFiles.at(currentMusicFile).toUtf8();
            qDebug() << "-- tagFile name - " << tagFile.file()->name();
            sleep(1);
        }


        if (!artistTag.isEmpty() && !titleTag.isEmpty())
        {
            qDebug() << "Has artist AND title, OK ->" << artistTag << titleTag;
            this->musicFiles[0].append(directoryFiles.at(currentMusicFile));

            this->musicFiles[1].append(artistTag);
            this->musicFiles[2].append(titleTag);

            ++validFiles;
        }
        else
        {
            qDebug() << "This file doesn't seem to have metadata, or an error related occurred";
        }

        analyzerProgressDialog.setValue(currentMusicFile); // update progressbar
        if (analyzerProgressDialog.wasCanceled())
        {
            qDebug() << "createSongList(): analysis cancelled by button";
            musicFiles[0].clear();
            musicFiles[1].clear();
            musicFiles[2].clear();
            //this->startGameButton->setText(tr("Please reload music info"));

            return; // you sure?
        }

    } // end analyzer loop



    // We're done scanning

    qDebug() << "\n\n---------------- SCAN COMPLETE -------";
    //memory required for the lists?
    qDebug() << "memory used, about: "<< musicFiles[0].length() * 40 * 3 << "bytes";
    qDebug() << "Storing data in:" << this->dataDirectory;
    qDebug() << "Valid files:" << validFiles;


    if (validFiles < 6)  // If not enough files
    {
        qDebug() << "ERROR: Not enough valid files (i.e. no metadata in them)";
        // FIXME: tmp message in disabled button
        this->startGameButton->setText(tr("Please choose another music folder"));
        return;
    }


    // Store info about the scanned music files
    QFile cachedMetadata0(this->dataDirectory + "/musicFiles0.aq");
    cachedMetadata0.open(QIODevice::WriteOnly);
    QString cachedMetadata0Item;

    QFile cachedMetadata1(this->dataDirectory + "/musicFiles1.aq");
    cachedMetadata1.open(QIODevice::WriteOnly);
    QString cachedMetadata1Item;

    QFile cachedMetadata2(this->dataDirectory + "/musicFiles2.aq");
    cachedMetadata2.open(QIODevice::WriteOnly);
    QString cachedMetadata2Item;

    qDebug() << "datafiles created";

    for (int counter = 0; counter < this->musicFiles[0].length(); ++counter)
        {
            // filenames
            cachedMetadata0Item = musicFiles[0].at(counter);
            cachedMetadata0Item.append("\n");
            cachedMetadata0.write(cachedMetadata0Item.toUtf8());

            // FIXME: encoding problems with tags

            // artists
            cachedMetadata1Item = musicFiles[1].at(counter);
            cachedMetadata1Item.append("\n");
            cachedMetadata1.write(cachedMetadata1Item.toUtf8());

            // titles
            cachedMetadata2Item = musicFiles[2].at(counter);
            cachedMetadata2Item.append("\n");
            cachedMetadata2.write(cachedMetadata2Item.toUtf8());
        }
    cachedMetadata0.close();
    cachedMetadata1.close();
    cachedMetadata2.close();

    qDebug() << "Metadata stored in .aq files";


    this->startGameButton->setEnabled(true); // songlist is ready, allow starting game
    this->startGameButton->setText("\n" + tr("&Start game") + "\n");
    this->startGameButton->setFocus();


    qDebug() << "start button re-enabled and with focus";


}





// Used from newQuestion()
void AuralWindow::playerStateChanged(Phonon::State newState, Phonon::State oldState)
{

    qDebug() << "playerStateChanged() (5 means error) -> " << oldState << ">" << newState;

    if (oldState == Phonon::LoadingState
     && newState == Phonon::StoppedState)
    {
        this->nextSongLoaded();
    }

    if (newState == Phonon::PausedState)
    {
        qDebug() << "PausedState";
    }
    if (musicPlayer->errorType() != 0) // if some error
    {
        qDebug() << "playerStateChanged(), some error!! State:" << newState;
        qDebug() << "Error string and type:" << musicPlayer->errorString() << musicPlayer->errorType();
        QMessageBox::warning(this, tr("Error playing sound"),
                             tr("An error occurred while playing sound.\n"
                                "The error message was: %1\n\n"
                                "Maybe your Phonon-backend does not have support for the MP3 file format.").arg(musicPlayer->errorString()));
    }
}



/******************************************************************************/



/*
 *  Decrease time left to guess song. Called by gameTimer
 */
void AuralWindow::timerTick()
{
    if (timeBar->value() == this->maxTime-1) // Just once upon start
    {
        // Seek to a random part of the music file
        qint64 seekTime = 5000; // Start at least at 00:05
        // add random ms avoiding last 100 sec, resulting in range 00:05 -> END -01:35
        seekTime += qrand() % (musicPlayer->totalTime() - 100000);

        qDebug() << "Seeking to:" << seekTime / 1000 << "sec / Seekable:" << musicPlayer->isSeekable() << "/ Total time:" << musicPlayer->totalTime() / 1000;

        musicPlayer->play();
        //musicPlayer->pause();
        /*
         *  play() BEFORE seek()
         *  because gstreamer-backend seeks to 0 again on play() apparently...
         */
        musicPlayer->seek(seekTime); // TMPFIX, add error control
        //musicPlayer->play();
    }


    timeBar->setValue(timeBar->value() - 1);

    // Stop music earlier than the time limit to answer the question
    if (timeBar->value() == this->pieceDuration)
    {
        this->musicPlayer->stop();
        this->aniNoteLabel->movie()->stop(); // TMPFIX
        qDebug() << "totalTime, again:" << musicPlayer->totalTime() / 1000; // TMP tests
    }


    // Clear colored info label (good/bad), soon after new question
    if (timeBar->value() == this->maxTime - 10)   // very very TMPFIX
    {
        this->infoLabel->setStyleSheet("");
        this->infoLabel->setText("");
    }


    // if remaining time is low, change timeBar colors

    if (timeBar->value() == this->warningTime) // warning level
    {
        timeBar->setStyleSheet("background-color: lightYellow"); // orange?
    }

    if (timeBar->value() == this->dangerTime)  // danger level
    {
        timeBar->setStyleSheet("background-color: red");
    }


    // time to answer ended
    if (timeBar->value() == 0)
    {
        qDebug() << "Time's up!";
        musicPlayer->stop();
        this->infoLabel->setStyleSheet("background: qradialgradient(spread:pad, cx:0.5, cy:0.5, radius:0.5, fx:0.5, fy:0.5, stop:0 rgba(64, 0, 0, 255) stop:1 rgba(128, 0, 0, 255)); color: white");
        this->infoLabel->setText(tr("Time's up!!") + "    "
                               + tr("It was:") + "  "
                               + musicFiles[questionType].at(currentMusicFile));



        answerQuestion(0);  // 0 means it's timeout, not a button
    }
}



/*
 *  Update configuration from optionsDialog SIGNAL
 */
void AuralWindow::updateConfig(bool startGame, QString directory, bool forceReload,
                               int difficulty, int questions, int players,
                               QStringList playerNameList, bool ownColors)
{
    bool mustReload = false;
    if (this->musicDirectory != directory)
    {
        qDebug() << "musicDirectory has changed, mustReload = true";
        qDebug() << "musicDirectory:" << musicDirectory << "; directory:" << directory;
        mustReload = true; // if music directory's been changed, reload collection info
    }

    this->musicDirectory = directory;
    this->difficultyLevel = difficulty;
    this->numQuestions = questions;
    this->numPlayers = players;
    this->playerNames.clear();
    playerNames = playerNameList;
    this->useOwnColorTheme = ownColors;

    qDebug() << "Updated config with: " << startGame << directory << forceReload
                                        << difficulty << questions << players
                                        << playerNameList << ownColors;


    // reload music collection information here if needed
    if (mustReload || forceReload)
    {
        qDebug() << "Reloading music collection information";
        this->createSongList();
    }


    // if optionsDialog was called in PlayMode, start game now
    if (startGame)
    {
        if (numPlayers > 1) // in multiplayer, give extra time to prepare
        {
            this->preQuestionTimer->setInterval(2000); // 2 sec
            // FIXME: make configurable?
        }
        else
        {
            this->preQuestionTimer->setInterval(500); // half a sec
        }

        this->toggleScreen();  // Start game!
    }
}



/*
 *  About... box
 */
void AuralWindow::showAbout()
{
    QMessageBox::about(this, tr("About Auralquiz"),
                       "<b>Auralquiz v0.8.1</b>"
                       "<p>(C) 2011-2012  JanKusanagi"
                       "<p><a href=\"http://jancoding.wordpress.com/auralquiz\">http://jancoding.wordpress.com/auralquiz</a>"
                       "<p>"
                       "<p>" +
                       tr("Auralquiz loads all your music from a specified folder and asks "
                          "questions about it, playing a short piece of each music file as clue.")
                       +  "<p>"
                          "<p>"
                       +  tr("Czech translation by Pavel Fric.")
                       +  "<p>"
                       +  tr("Thanks to the betatesters, Jorchube, Coyote, Panko and Eomer for their help.")
                       +  "<p>"
                       +  tr("Also thanks to all the packagers out there who built Auralquiz for different distributions.")
                       +  "<p>"
                       +  tr("Main screen image based on Oxygen icons: http://www.oxygen-icons.org/ (LGPL licensed)")
                       +  "<p>");

}






/*****************************************************************************************
 *
 *  Switch between the welcome screen and the playing screen, in either way
 */
void AuralWindow::toggleScreen()
{
    if (!playing)
    {   // START game!
        qDebug() << "Starting game";
        welcomeWidget->hide();

        shuffleMusicFiles();

        // Reset everything...
        currentMusicFile = 0;
        this->gameProgressBar->setRange(0, this->numQuestions);
        this->gameProgressBar->setValue(0);
        this->gameScore->display(0); // reset score


        maxTime = 500 / (difficultyLevel + 1);  // In millisec/10
        if (difficultyLevel == 5)
        {
            maxTime += 100;  // Extra time in Hardcore/type-the-answer mode!
        }
        warningTime = maxTime / 4;
        dangerTime = maxTime / 8;

        //this->pieceDuration = maxTime - (maxTime / (difficultyLevel+1));
        this->pieceDuration = maxTime - ( 90 / (difficultyLevel+1) );
                                                         // +1 bc diflev can be 0
        if (difficultyLevel == 5)
        {
            pieceDuration -= 5;  // Extra duration in Hardcore level!
        }

        qDebug() << "Max/Warning/Danger times:" << maxTime/6 << warningTime/6 << dangerTime/6 << "secs";
        qDebug() << "Piece duration:" << (maxTime - pieceDuration) / 6 << "secs";

        timeBar->setRange(0, this->maxTime);
        timeBar->setValue(this->maxTime);


        if (difficultyLevel < 5) // Regular mode: show buttons, hide AnswerBox
        {
            answerButton[0]->show();
            answerButton[1]->show();
            answerButton[2]->show();
            answerButton[3]->show();

            answerBox->hide();
        }
        else // Type-the-answer mode: hide buttons and show AnswerBox
        {
            answerBox->show();

            answerButton[0]->hide();
            answerButton[1]->hide();
            answerButton[2]->hide();
            answerButton[3]->hide();
        }


        if (numPlayers > 1)
        {
            this->playerNameLabel->show();
        }
        else
        {
            this->playerNameLabel->hide();
        }


        // Reset stats for all players
        for (int counter = 0; counter != numPlayers; ++counter)
        {
            this->score[counter] = 0;
            this->goodAnswers[counter] = 0;
            this->badAnswers[counter] = 0;
            this->timedOutAnswers[counter] = 0;
        }

        this->currentPlayer = 0;


        QStringList startStrings; // Randomly choose from a list of "get ready" strings
        startStrings << tr("Starting!") << tr("Let's go!") << tr("GO!!") << tr("Good luck!");
        this->infoLabel->setStyleSheet("background: qradialgradient(spread:pad, cx:0.5, cy:0.5, radius:0.5, fx:0.5, fy:0.5, stop:0 rgba(0, 64, 64, 255) stop:1 rgba(0, 255, 255, 255)); color: white");
        this->infoLabel->setText(startStrings.at(qrand() % startStrings.length()));



        playingWidget->show();

        playing = true;
        this->updateStatistics();


        newQuestion();
    }
    else
    {   // STOP game        
        qDebug() << "Stopping game";
        musicPlayer->stop();    // stop music, if any

        gameTimer->stop();

        preQuestionTimer->stop();  // These 2 lines fix problems when clicking
        postQuestionTimer->stop(); // "End Game" right after answering
                                   // (Music in title screen)

        playingWidget->hide();
        welcomeWidget->show();
        playing = false;
    }
}



/*************************************************************************************
 *
 *  Prepare a new question, its answers, and play the new song file
 */
void AuralWindow::newQuestion()
{
    qDebug() << "new question";

    timeBar->setValue(this->maxTime);
    timeBar->setStyleSheet("");

    this->correctAnswer = (qrand() % 4) + 1;  // 1, 2, 3, 4; 0 would mean TIMED UP
    //qDebug() << "correctAnswer:" << this->correctAnswer;

    questionType = (currentMusicFile & 1) + 1; // quite TMPFIX, odd=artist, even=title
    //qDebug() << "questionType: " << questionType;

    if (questionType == 1)
    {
        questionLabel->setText(tr("Who plays this song?"));
    }
    else
    {
        questionLabel->setText(tr("What's the title of this song?"));
    }

    // show current player name in multiplayer mode
    if (numPlayers > 1)
    {
        playerNameLabel->setText(tr("Player %1").arg(currentPlayer + 1)
                                 + "  --  "
                                 + QString("%1\n").arg(playerNames.at(currentPlayer)));
        gameScore->display(score.at(currentPlayer));
    }


    QString buttonTexts[4];
    int answerCount = 0;

    // FIXME: These checks should be moved to the analyzer, in createSongList()
    int tries = 0;
    while (answerCount != 4)
    {
        buttonTexts[answerCount] = musicFiles[questionType]
                                   .at(qrand() % this->musicFiles[0].length());


        //qDebug() << "answerCount" << answerCount;

        // If text in current button is NOT the correct answer (lowercase comparison!)
        if (buttonTexts[answerCount].toLower() != musicFiles[questionType].at(currentMusicFile).toLower())
        {
            bool isGood = true;
            for (int previousAnswer = 0; previousAnswer != answerCount; ++previousAnswer)
            {
                // Compare strings in LOWERCASE (Metallica ~= metallica)
                if (buttonTexts[answerCount].toLower() != buttonTexts[previousAnswer].toLower())
                {
                    isGood = true;
                    //qDebug() << "is GOOD" << previousAnswer;
                    //qDebug() << "is GOOD" << buttonTexts[answerCount] << buttonTexts[previousAnswer];
                }
                else
                {
                    isGood = false;
                    //qDebug() << "is NOT good; duplicated, break" << previousAnswer;
                    //qDebug() << "is NOT good" << buttonTexts[answerCount] << buttonTexts[previousAnswer];
                    break;
                }
            }

            if (isGood)
            {
                // Set random artists/titles as labels on buttons
                answerButton[answerCount]->setText(buttonTexts[answerCount]);

                ++answerCount;
                tries = 0;
            }

            //qDebug() << "buttonTexts[answerCount]:" << buttonTexts[answerCount];
        }
        ++tries;  /* TMPFIX
                     Temporary way of catching infinite loops
                     if there are not enough different artists or titles */

        if (tries > 50)
        {
            // FIXME: these checks should be done after createSongList()
            qDebug() << "Not enough different titles or artists to create question!";
            answerCount = 4; // "break"
        }
    }

    // quite hacky, TMPFIX
    if (tries < 50) // if no problem with duplicates, meaning we have enough valid files
    {
        // Set correct artist/title on one button
        answerButton[correctAnswer-1]->setText(musicFiles[questionType].at(currentMusicFile));

        // replace "&" by "&&" in buttons, so it doesn't turn into an accelerator
        // Also truncate if answer is too long
        for (int counter=0; counter != 4; ++counter)
        {
            QString fixedAnswer = answerButton[counter]->text();
            fixedAnswer.replace(QRegExp("&"), "&&"); // fix non-accelerator
            fixedAnswer.truncate(64); // avoid wiiiiide buttons
            // FIXME 0.9: truncate position should be calculated, not hardcoded
            // For now, 64 seems sane

            answerButton[counter]->setText(fixedAnswer);
        }


        // Set answer for AnswerBox too
        answerBox->setAnswer(musicFiles[questionType].at(currentMusicFile));


        musicPlayer->stop();
        musicPlayer->clear(); // Stop playing file and clear queues


        musicPlayer->setCurrentSource(musicFiles[0].at(currentMusicFile));

        qDebug() << this->musicPlayer->isSeekable(); // checks needed for gstreamer
        qDebug() << "newQuestion(): musicPlayer->setCurrentSource() done";
    }
    else
    {
        this->toggleScreen();  // Recursive!! :P

        QMessageBox::critical(this, tr("Not enough music files"),
                              tr("You don't have enough music files, "
                                 "or from enough different artists.\n"
                                 "You need music from at least 5 different artists "
                                 "to be able to generate questions."));
    }

}



/*
 *  Called from PlayerStateChanged()
 */
void AuralWindow::nextSongLoaded()
{
    qDebug() << "(playerStateChanged) > nextSongLoaded()";


    if (difficultyLevel < 5)
    {
        // Allow using 1~4 keys to answer
        answerButton[0]->setShortcut(Qt::Key_1);
        answerButton[1]->setShortcut(Qt::Key_2);
        answerButton[2]->setShortcut(Qt::Key_3);
        answerButton[3]->setShortcut(Qt::Key_4);

        // re-enable buttons
        this->answerButton[0]->setEnabled(true);
        this->answerButton[1]->setEnabled(true);
        this->answerButton[2]->setEnabled(true);
        this->answerButton[3]->setEnabled(true);
    }
    else
    {
        this->answerBox->setEnabled(true);
        this->answerBox->setFocus();
    }


    //musicPlayer->play();  // put into PlayingState so seeking works later
    //musicPlayer->pause(); // Pause, so the user can't hear anything
    this->aniNoteLabel->movie()->start();
    gameTimer->start(); // will call "timerTick()"
}



void AuralWindow::preQuestion()
{
    if (this->gameProgressBar->value() < this->numQuestions)
    {
        // There are more questions!

        ++currentMusicFile;
        if (currentMusicFile == this->musicFiles[0].length())
        {
            currentMusicFile = 0;
            qDebug() << "Not enough music files; back to 1st";
        }


        this->updateStatistics();


        infoLabel->setStyleSheet("background: qradialgradient(spread:pad, cx:0.5, cy:0.5, radius:0.5, fx:0.5, fy:0.5, stop:0 rgba(0, 0, 128, 255) stop:1 rgba(0, 128, 255, 127)); color: white");
        if (numPlayers > 1)
        {
            infoLabel->setText(tr("Go, %1!").arg(playerNames.at(currentPlayer)));
        }
        else
        {
            infoLabel->setText(tr("Next!"));
        }

        preQuestionTimer->start(); // this will call newQuestion()
    }
    else
    {
        // No more questions!

        qDebug() << "End of questions";

        // Show the ranking window
        ranking = new Ranking(numPlayers, playerNames, score,
                               goodAnswers, badAnswers, timedOutAnswers);
        ranking->setWindowModality(Qt::ApplicationModal);
        connect(ranking, SIGNAL(closed()), this, SLOT(killRanking()));

        rankingTimer->setSingleShot(true);
        connect(rankingTimer, SIGNAL(timeout()), ranking, SLOT(show()));

        rankingTimer->start(2000); // Wait 2 seconds before showing the ranking

        this->infoLabel->setStyleSheet("background: qradialgradient(spread:pad, cx:0.5, cy:0.5, radius:0.5, fx:0.5, fy:0.5, stop:0 rgba(0, 0, 128, 255) stop:1 rgba(0, 128, 255, 255)); color: white");
        this->infoLabel->setText(tr("All done!"));
    }
}


/**************************************************************************************
 *
 *  Check if answer is correct, give score, and get new question
 */
void AuralWindow::answerQuestion(int numAnswer)
{
    qDebug() << "Answered question with button" << numAnswer;

    this->answerButton[0]->setDisabled(true);
    this->answerButton[1]->setDisabled(true);
    this->answerButton[2]->setDisabled(true);
    this->answerButton[3]->setDisabled(true);

    this->answerBox->setDisabled(true);

    this->aniNoteLabel->movie()->stop();

    musicPlayer->stop(); // stop music
                         // This helps Phonon's stateChanged(), avoids silent songs
    gameTimer->stop();



    int questionPoints;
    if (timeBar->value() != 0)
    {
        //questionPoints = (100 * difficultyLevel) + (timeBar->value() / 2); // OLD

        questionPoints = 50 * difficultyLevel+1;
        questionPoints += timeBar->value() / 2;
        questionPoints += 100;
    }
    else
    {
        questionPoints = 0;
        this->timedOutAnswers[currentPlayer]++;
        qDebug() << "Answered by timeout";
    }



    if (questionPoints != 0)
    {
        if (numAnswer == correctAnswer)
        {
            QStringList rightAnswerStrings;
            rightAnswerStrings << tr("Correct!!") << tr("Yeah!") << tr("That's right!") << tr("Good!") << tr("Great!");

            this->infoLabel->setStyleSheet("background: qradialgradient(spread:pad, cx:0.5, cy:0.5, radius:0.5, fx:0.5, fy:0.5, stop:0 rgba(0, 192, 0, 255) stop:1 rgba(0, 255, 0, 0)); color: white");
            this->infoLabel->setText(rightAnswerStrings
                                     .at(qrand() % rightAnswerStrings.length())
                                     + QString("  " + tr("+%1 points!")).arg(questionPoints));
                                    // quite TMPFIX

            this->score[currentPlayer] += questionPoints;
            gameScore->display(score.at(currentPlayer));
            this->goodAnswers[currentPlayer]++;

            qDebug() << "correct!!" << questionPoints;
        }
        else
        {
            QStringList wrongAnswerStrings;
            wrongAnswerStrings << tr("Wrong!") << tr("No!") << tr("That's not it.") << tr("Ooops!") << tr("FAIL!");

            this->infoLabel->setStyleSheet("background: qradialgradient(spread:pad, cx:0.5, cy:0.5, radius:0.5, fx:0.5, fy:0.5, stop:0 rgba(192, 0, 0, 255) stop:1 rgba(255, 0, 0, 0)); color: white");
            this->infoLabel->setText(wrongAnswerStrings
                                     .at(qrand() % wrongAnswerStrings.length())
                                     + "    " + tr("It was:") + "  "
                                     + musicFiles[questionType].at(currentMusicFile));
                                    // quite TMPFIX

            this->badAnswers[currentPlayer]++;

            qDebug() << "wrong!";
        }

    }


    this->updateStatistics();


    if (numPlayers > 1)
    {
        this->currentPlayer++;
        if (currentPlayer == numPlayers)
        {
            this->currentPlayer = 0; // back to first player
            qDebug() << "Next question, back to Player 1";
        }
    }



    if (currentPlayer == 0)
    {   // Always on single-player, and after all players answer, in multi
        gameProgressBar->setValue(gameProgressBar->value() + 1);
    }


    postQuestionTimer->start(); // will call preQuestion()
}


// These below... suck
void AuralWindow::answer1()
{
    answerQuestion(1);
}
void AuralWindow::answer2()
{
    answerQuestion(2);
}
void AuralWindow::answer3()
{
    answerQuestion(3);
}
void AuralWindow::answer4()
{
    answerQuestion(4);
}

// This one doesn't suck, actually!
void AuralWindow::answerFromAnswerBox(bool correct)
{
    if (correct)
    {
        answerQuestion(this->correctAnswer);
    }
    else
    {
        answerQuestion(-1); // Answer incorrectly
    }
}

void AuralWindow::killRanking()
{
    qDebug() << "deleting Ranking object";
    ranking->deleteLater();

    this->endGameButton->click();
}
