; Solidity Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "SOLIDITY")

(include-book "boolean-values")
(include-book "integer-values")

(include-book "ihs/basic-definitions" :dir :system)

(local (include-book "arithmetic/equalities" :dir :system))
(local (include-book "ihs/logops-definitions" :dir :system))

(local (in-theory (disable loghead logext)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ integer-operations
  :parents (integer-values)
  :short (xdoc::topstring "Integer operations " xdoc::*sld-types-integers* ".")
  :long
  (xdoc::topstring
   (xdoc::p
    "If a binary operator is applied to operands of different types,
     there is an (attempted) implicit conversion
     from the type of one operand to the type of the other operand "
    xdoc::*sld-types-conversions-implicit*
    ". Thus, once the operator is actually applied to the two operand values,
     the implicit conversion must have succeeded
     and thus the values must have the same time.
     Therefore, the ACL2 functions that model binary integer operations
     have guards requiring the two operands to have the same type
     (i.e. the same signedness and the same size).")
   (xdoc::p
    "However, the requirement above that the operands of a binary operator
     always are of, or can be converted to, the same type,
     may need to be relaxed for some operations.
     The wording in "
    xdoc::*sld-integer-exponentiation*
    " and "
    xdoc::*sld-integer-shifts*
    " suggests that operands may have different types for these operations,
     in a way that no conversion to a common type may be possible.")
   (xdoc::p
    "For now we strictly follow the same-rule type,
     and therefore must exclude exponentiation and shifts on signed integers.
     However, we may need to adjust our formalization as noted above,
     and add exponentiation and shifts involving signed integers
     (for their first operands only, not their second operands."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection def-uint/int-comparison
  :short "Macro to formalize the Solidity integer comparisons."
  :long
  (xdoc::topstring
   (xdoc::p
    "This macro takes, as required arguments,
     the name of the function to define (which formalizes the comparison),
     an indication of whether the comparison is signed or not,
     and an untranslated term that defines the comparison on the core values.
     This term must have @('x') and @('y') as the only free variables,
     which the boilerplate code generated by the macro binds to
     the core value of the left and right operand.")
   (xdoc::p
    "This macro also takes optional arguments for XDOC documentation:
     parents, short string, and long string.
     The latter may be forms, e.g. involving XDOC constructors.")
   (xdoc::@def "def-uint/int-comparison"))

  (define def-uint/int-comparison-fn ((name symbolp)
                                      (signedp booleanp)
                                      (operation "An untranslated term.")
                                      (parents symbol-listp)
                                      (parents-suppliedp booleanp)
                                      (short "A string or form or @('nil').")
                                      (short-suppliedp booleanp)
                                      (long "A string or form or @('nil').")
                                      (long-suppliedp booleanp))
    :returns (event "A @(tsee acl2::pseudo-event-formp).")
    :parents nil
    (b* ((type (if signedp 'int 'uint))
         (typep (add-suffix type "P"))
         (type->size (add-suffix type "->SIZE"))
         (type->value (add-suffix type "->VALUE")))
      `(define ,name ((left ,typep) (right ,typep))
         :guard (= (,type->size left) (,type->size right))
         :returns (result boolp)
         ,@(and parents-suppliedp (list :parents parents))
         ,@(and short-suppliedp (list :short short))
         ,@(and long-suppliedp (list :long long))
         (b* ((x (,type->value left))
              (y (,type->value right)))
           (bool ,operation))
         :hooks (:fix)
         :no-function t)))

  (defmacro def-uint/int-comparison (name
                                     &key
                                     signedp
                                     operation
                                     (parents 'nil parents-suppliedp)
                                     (short 'nil short-suppliedp)
                                     (long 'nil long-suppliedp))
    `(make-event
      (def-uint/int-comparison-fn
        ',name
        ',signedp
        ',operation
        ',parents ,parents-suppliedp
        ,short ,short-suppliedp
        ,long ,long-suppliedp))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-comparison uint-eq
  :signedp nil
  :operation (equal x y)
  :short "Equality comparison of unsigned integer values.")

(def-uint/int-comparison uint-ne
  :signedp nil
  :operation (not (equal x y))
  :short "Non-equality comparison of unsigned integer values.")

(def-uint/int-comparison uint-lt
  :signedp nil
  :operation (< x y)
  :short "Less-than comparison of unsigned integer values.")

(def-uint/int-comparison uint-le
  :signedp nil
  :operation (<= x y)
  :short "Less-than-or-equal-to comparison of unsigned integer values.")

(def-uint/int-comparison uint-gt
  :signedp nil
  :operation (> x y)
  :short "Greater-than comparison of unsigned integer values.")

(def-uint/int-comparison uint-ge
  :signedp nil
  :operation (>= x y)
  :short "Greater-than-or-equal-to comparison of unsigned integer values.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-comparison int-eq
  :signedp t
  :operation (equal x y)
  :short "Equality comparison of signed integer values.")

(def-uint/int-comparison int-ne
  :signedp t
  :operation (not (equal x y))
  :short "Non-equality comparison of signed integer values.")

(def-uint/int-comparison int-lt
  :signedp t
  :operation (< x y)
  :short "Less-than comparison of signed integer values.")

(def-uint/int-comparison int-le
  :signedp t
  :operation (<= x y)
  :short "Less-than-or-equal-to comparison of signed integer values.")

(def-uint/int-comparison int-gt
  :signedp t
  :operation (> x y)
  :short "Greater-than comparison of signed integer values.")

(def-uint/int-comparison int-ge
  :signedp t
  :operation (>= x y)
  :short "Greater-than-or-equal-to comparison of signed integer values.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection def-uint/int-unary-op
  :short "Macro to formalize a Solidity integer unary operation."
  :long
  (xdoc::topstring
   (xdoc::p
    "This macro takes, as required arguments,
     the name of the function to define (which formalizes the operation),
     an indication of whether the operation is signed or not,
     and an untranslated term that defines the operation on the core value.
     This term must have @('size') and @('x') as the only free variables,
     which the boilerplate code generated by the macro binds to
     the size of the operand and the core value of the operand.")
   (xdoc::p
    "This macro also takes optional arguments for XDOC documentation:
     parents, short string, and long string.
     The latter may be forms, e.g. involving XDOC constructors.")
   (xdoc::@def "def-uint/int-unary-op"))

  (define def-uint/int-unary-op-fn ((name symbolp)
                                    (signedp booleanp)
                                    (operation "An untranslated term.")
                                    (parents symbol-listp)
                                    (parents-suppliedp booleanp)
                                    (short "A string or form or @('nil').")
                                    (short-suppliedp booleanp)
                                    (long "A string or form or @('nil').")
                                    (long-suppliedp booleanp))
    :returns (event "A @(tsee acl2::pseudo-event-formp).")
    :parents nil
    (b* ((type (if signedp 'int 'uint))
         (typep (add-suffix type "P"))
         (type->size (add-suffix type "->SIZE"))
         (type->value (add-suffix type "->VALUE"))
         (make-type (packn-pos (list "MAKE-" type) type)))
      `(define ,name ((operand ,typep))
         :returns (result ,typep)
         ,@(and parents-suppliedp (list :parents parents))
         ,@(and short-suppliedp (list :short short))
         ,@(and long-suppliedp (list :long long))
         (b* ((size (,type->size operand))
              (x (,type->value operand)))
           (,make-type :size (,type->size operand) :value ,operation))
         :hooks (:fix)
         :no-function t)))

  (defmacro def-uint/int-unary-op (name
                                   &key
                                   signedp
                                   operation
                                   (parents 'nil parents-suppliedp)
                                   (short 'nil short-suppliedp)
                                   (long 'nil long-suppliedp))
    `(make-event
      (def-uint/int-unary-op-fn
        ',name
        ',signedp
        ',operation
        ',parents ,parents-suppliedp
        ,short ,short-suppliedp
        ,long ,long-suppliedp))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-unary-op uint-not
  :signedp nil
  :operation (loghead size (lognot x))
  :short "Bitwise negation of unsigned integer values.")

(def-uint/int-unary-op uint-minus
  :signedp nil
  :operation (loghead size (- x))
  :short "Arithmetic negation of unsigned integer values.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-unary-op int-not
  :signedp t
  :operation (logext size (lognot x))
  :short "Bitwise negation of signed integer values.")

(def-uint/int-unary-op int-minus
  :signedp t
  :operation (logext size (- x))
  :short "Arithmetic negation of signed integer values.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection def-uint/int-binary-op
  :short "Macro to formalize a Solidity integer binary operation."
  :long
  (xdoc::topstring
   (xdoc::p
    "This macro takes, as required arguments,
     the name of the function to define (which formalizes the operation),
     an indication of whether the operation is signed or not,
     and an untranslated term that defines the operation on the core value.
     This term must have @('size'), @('x'), and @('y')
     as the only free variables,
     which the boilerplate code generated by the macro binds to
     the size of the operands and the core values of the operands.")
   (xdoc::p
    "This macro also takes an optional argument
     to impose the additional guard restriction
     that the right operand is not zero;
     this is used for the division and modulo operations.")
   (xdoc::p
    "This macro also takes optional arguments for XDOC documentation:
     parents, short string, and long string.
     The latter may be forms, e.g. involving XDOC constructors.")
   (xdoc::@def "def-uint/int-binary-op"))

  (define def-uint/int-binary-op-fn ((name symbolp)
                                     (signedp booleanp)
                                     (operation "An untranslated term.")
                                     (nonzerop booleanp)
                                     (parents symbol-listp)
                                     (parents-suppliedp booleanp)
                                     (short "A string or form or @('nil').")
                                     (short-suppliedp booleanp)
                                     (long "A string or form or @('nil').")
                                     (long-suppliedp booleanp))
    :returns (event "A @(tsee acl2::pseudo-event-formp).")
    :parents nil
    (b* ((type (if signedp 'int 'uint))
         (typep (add-suffix type "P"))
         (type->size (add-suffix type "->SIZE"))
         (type->value (add-suffix type "->VALUE"))
         (make-type (packn-pos (list "MAKE-" type) type))
         (guard? (and nonzerop `(not (equal (,type->value right-operand) 0)))))
      `(define ,name ((left-operand ,typep) (right-operand ,typep))
         ,@(and guard? (list :guard guard?))
         :returns (result ,typep)
         ,@(and parents-suppliedp (list :parents parents))
         ,@(and short-suppliedp (list :short short))
         ,@(and long-suppliedp (list :long long))
         (b* ((size (,type->size left-operand))
              (x (,type->value left-operand))
              (y (,type->value right-operand)))
           (,make-type :size (,type->size left-operand) :value ,operation))
         :hooks (:fix)
         :no-function t)))

  (defmacro def-uint/int-binary-op (name
                                    &key
                                    signedp
                                    operation
                                    nonzerop
                                    (parents 'nil parents-suppliedp)
                                    (short 'nil short-suppliedp)
                                    (long 'nil long-suppliedp))
    `(make-event
      (def-uint/int-binary-op-fn
        ',name
        ',signedp
        ',operation
        ',nonzerop
        ',parents ,parents-suppliedp
        ,short ,short-suppliedp
        ,long ,long-suppliedp))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-binary-op uint-and
  :signedp nil
  :operation (loghead size (logand x y))
  :short "Bitwise conjunction of unsigned integer values.")

(def-uint/int-binary-op uint-ior
  :signedp nil
  :operation (loghead size (logior x y))
  :short "Bitwise inclusive disjunction of unsigned integer values.")

(def-uint/int-binary-op uint-xor
  :signedp nil
  :operation (loghead size (logxor x y))
  :short "Bitwise exclusive disjunction of unsigned integer values.")

(def-uint/int-binary-op uint-add
  :signedp nil
  :operation (loghead size (+ x y))
  :short "Addition of unsigned integer values.")

(def-uint/int-binary-op uint-sub
  :signedp nil
  :operation (loghead size (- x y))
  :short "Subtraction of unsigned integer values.")

(def-uint/int-binary-op uint-mul
  :signedp nil
  :operation (loghead size (* x y))
  :short "Multiplication of unsigned integer values.")

(def-uint/int-binary-op uint-div
  :signedp nil
  :operation (loghead size (truncate x y))
  :nonzerop t
  :short "Division of unsigned integer values."
  :long
  (xdoc::topstring-p
   "Integer division rounds towards 0 in Solidity "
   xdoc::*sld-integer-division* ", so we use @(tsee truncate)."))

(def-uint/int-binary-op uint-mod
  :signedp nil
  :operation (loghead size (rem x y))
  :nonzerop t
  :short "Modulo of unsigned integer values."
  :long
  (xdoc::topstring-p
   "This has the usual relation with division "
   xdoc::*sld-integer-modulo* ", so we use @(tsee rem) here."))

(def-uint/int-binary-op uint-exp
  :signedp nil
  :operation (loghead size (expt x y))
  :short "Exponentiation of unsigned integer values.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-binary-op int-and
  :signedp t
  :operation (logext size (logand x y))
  :short "Bitwise conjunction of signed integer values.")

(def-uint/int-binary-op int-ior
  :signedp t
  :operation (logext size (logior x y))
  :short "Bitwise inclusive disjunction of signed integer values.")

(def-uint/int-binary-op int-xor
  :signedp t
  :operation (logext size (logxor x y))
  :short "Bitwise exclusive disjunction of signed integer values.")

(def-uint/int-binary-op int-add
  :signedp t
  :operation (logext size (+ x y))
  :short "Addition of signed integer values.")

(def-uint/int-binary-op int-sub
  :signedp t
  :operation (logext size (- x y))
  :short "Subtraction of signed integer values.")

(def-uint/int-binary-op int-mul
  :signedp t
  :operation (logext size (* x y))
  :short "Multiplication of signed integer values.")

(def-uint/int-binary-op int-div
  :signedp t
  :operation (logext size (truncate x y))
  :nonzerop t
  :short "Division of signed integer values."
  :long
  (xdoc::topstring-p
   "Integer division rounds towards 0 in Solidity "
   xdoc::*sld-integer-division* ", so we use @(tsee truncate)."))

(def-uint/int-binary-op int-mod
  :signedp t
  :operation (logext size (rem x y))
  :nonzerop t
  :short "Modulo of signed integer values."
  :long
  (xdoc::topstring-p
   "This has the usual relation with division "
   xdoc::*sld-integer-modulo* ", we use @(tsee rem) here."))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def-uint/int-binary-op uint-shl
  :signedp nil
  :operation (loghead size (ash x y))
  :short "Left shift of unsigned integer values."
  :long
  (xdoc::topstring-p
   "We prove that @('x << y') is equivalent to @('x * 2**y'),
    as mentioned in " xdoc::*sld-integer-shifts* "."))

(defrule loghead-of-*-of-loghead-left
  (implies (and (natp x)
                (natp y))
           (equal (loghead size (* (loghead size x)
                                   y))
                  (loghead size (* x y))))
  :enable loghead
  :disable acl2::expt2
  :prep-books ((include-book "arithmetic-5/top" :dir :system)))

(defrule loghead-of-*-of-loghead-right
  (implies (and (natp x)
                (natp y))
           (equal (loghead size (* x
                                   (loghead size y)))
                  (loghead size (* x y))))
  :enable loghead
  :disable acl2::expt2
  :prep-books ((include-book "arithmetic-5/top" :dir :system)))

(defrule equal-expt-base>2-and-base-rewrite-to-exp-one
  (implies (and (natp base)
                (> base 1))
           (equal (equal (expt base exp) base)
                  (equal (ifix exp) 1)))
  :prep-books ((include-book "arithmetic-5/top" :dir :system)))

(defruled uint-shl-alt-def
  (implies (equal (uint->size x)
                  (uint->size y))
           (equal (uint-shl x y)
                  (uint-mul x
                            (uint-exp (make-uint :size (uint->size x) :value 2)
                                      y))))
  :enable (uint-shl uint-mul uint-exp)
  :prep-books ((include-book "ihs/logops-lemmas" :dir :system)))

;;;;;;;;;;;;;;;;;;;;

(def-uint/int-binary-op uint-shr
  :signedp nil
  :operation (loghead size (ash x (- y)))
  :short "Right shift of unsigned integer values."
  :long
  (xdoc::topstring-p
   "We prove that @('x >> y') is equivalent to @('x / 2**y'),
    as mentioned in " xdoc::*sld-integer-shifts* "."))

(defrule expt-of-2-mono
  (implies (and (natp a)
                (natp b)
                (< a b))
           (< (expt 2 a)
              (expt 2 b)))
  :rule-classes :linear
  :prep-books ((include-book "arithmetic-5/top" :dir :system)))

(defrule loghead-of-expt-2
  (implies (and (natp size)
                (natp y)
                (< y size))
           (equal (loghead size (expt 2 y))
                  (expt 2 y)))
  :prep-books ((include-book "ihs/logops-lemmas" :dir :system)))

(defrule uint-shr-alt-def
  (implies (and (equal (uint->size x)
                       (uint->size y))
                (< (uint->value y)
                   (uint->size x)))
           (equal (uint-shr x y)
                  (uint-div x
                            (uint-exp (make-uint :size (uint->size x)
                                                 :value 2)
                                      y))))
  :enable (uint-shr uint-div uint-exp)
  :disable (truncate
            floor
            acl2::|(/ (expt x n))| ; to avoid circularities
            acl2::arith-5-active-flag) ; to avoid theory invariant violation
  :prep-books ((include-book "arithmetic-5/top" :dir :system)))
