/* twpsk  - A gui application for PSK
 * Copyright (C) 1999-2014 Ted Williams WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version: 4.3 - Aug 2017
 *
 * Disp Class - methods to draw the waterfall and spectrum analyser
 */

#include <stdio.h>
#include "GUI.h"
#include "twpskWF.h"
#include "twpskWids.h"
#include "icons/wfCursorShape.xbm"
#include "icons/wfCursorMask.xbm"
#include "server/server.h"


extern Wids pskwids;
extern AppRes appRes;

void Disp::find_visual_info(int *truecolor, XVisualInfo *vinfo)
{
   int scr = DefaultScreen(display);
   if( XMatchVisualInfo(display, scr, 8, PseudoColor, vinfo) )
   {
      fprintf(stderr,"find_visual_info: Using 8 bpp Pseudo Color\n");
      *truecolor = 0; return;
   }
   if( XMatchVisualInfo(display, scr, 15, TrueColor, vinfo) )
   {
      fprintf(stderr,"find_visual_info: Using 15 bpp True Color\n");
      *truecolor = 1; return;
   }
   if( XMatchVisualInfo(display, scr, 16, TrueColor, vinfo) )
   {
      fprintf(stderr,"find_visual_info: Using 16 bpp True Color\n");
      *truecolor = 1; return;
   }
   if( XMatchVisualInfo(display, scr, 24, TrueColor, vinfo) )
   {
      fprintf(stderr,"find_visual_info: Using 24 bpp True Color\n");
      *truecolor = 1; return;
   }
   if( XMatchVisualInfo(display, scr, 32, TrueColor, vinfo) )
   {
      fprintf(stderr,"find_visual_info: Using 32 bpp True Color\n");
      *truecolor = 1; return;
   }
}


/* color: 16 bit values! 0x0000..0xFFFF */
unsigned long Disp::get_mask_color(int color, unsigned long mask)
{
   int shift = 0;
   int notused = 16;
   
   while (!(mask&0x01))
   {
      shift++;
      mask >>=1;
   }
   
   while(mask&0x01)
   {
      notused--;
      mask >>=1;
   }

  unsigned long c = (color>>notused) << shift;
  return c;
}


/*
 * setup for wf/sa drawing area
 */
void Disp::setup (Widget wfDA)
{
   /* constants */
   int i;
   int bad_grays = 0;
   int rtn = 0;

   /* window values */
   display = XtDisplay (wfDA);
   window  = XtWindow (wfDA);
   screen  = DefaultScreen (display);
   gc = XDefaultGC (display, screen);
   cmap = DefaultColormap (display, screen);
   int wfCursorHotX = 7;
   int wfCursorHotY = 15;
   XColor cursorShapePix, cursorMaskPix;
   Pixmap shape, mask;

   /* create the pixmap for WF backstore, clear wfDA and pixmap */
   pmap = XCreatePixmap (display, window, WF_WIDTH, WF_HEIGHT, 
                           DefaultDepth(display, screen));
   XClearWindow (display, window);
   XCopyArea (display, window, pmap, gc, 0, 0, WF_WIDTH, WF_HEIGHT, 0, 0);

   /* Set some line attributes */
   XSetLineAttributes (display, gc, 0, LineSolid, CapRound, JoinRound);

   /* create the waterfall cursor */
   shape = XCreatePixmapFromBitmapData (display, window,
      (char*) wfCursorShape_bits, wfCursorShape_width, wfCursorShape_height, 1, 0, 1);

   mask = XCreatePixmapFromBitmapData (display, window,
      (char *) wfCursorMask_bits, wfCursorMask_width, wfCursorMask_height, 1, 0, 1);

   XParseColor (display, cmap, "red", &cursorShapePix);
   XParseColor (display, cmap, "black", &cursorMaskPix); /*not the best color*/
   
   wfCursor = XCreatePixmapCursor (display, shape, mask, 
         &cursorShapePix, &cursorMaskPix, wfCursorHotX, wfCursorHotY);

   XDefineCursor (display, XtWindow (wfDA), wfCursor);

   cnt = 0;

   /*
    * create 100 Gray pixels
    */
   XVisualInfo vinfo;
   int truecolor;
   find_visual_info(&truecolor, &vinfo);

   if(!truecolor)
   {
      /* 8 bit pseudo color (with color map) */
      for (i=0; i<MAX_GRAYS; i++)                       /* and 99 grays */
      {
         if (XAllocColorCells (display, cmap, False, NULL, 0,
                               &dispPix[i].pixel, 1) == 0) 
         {
            bad_grays++;                                /* count failures */
            continue;
         }
         dispPix[i].flags = DoRed | DoGreen | DoBlue;
         dispPix[i].red = (i + START) * 256;            /* scale to 65K */
         dispPix[i].green = (i + START) * 256;
         dispPix[i].blue = (i + START) * 256;
	 
         XStoreColor (display, cmap, &dispPix[i]);
      }
      if (bad_grays != 0)                               /* tell me about it */
         fprintf (stderr, "Could not allocate %d %s\n",
                  bad_grays, bad_grays == 1 ? "color" : "colors");
   }
   else  /* 15, 16, 24 or 32 bpp true color */
   {
      int r,g,b;
      for(i=0; i<MAX_GRAYS; i++)
      {
         // r=g=b= (i+START)*256;
         // The previous line is equivalent to your pseudo color code,
         // but I personally prefer a bigger range of color intensity,
         // thus using 255,253,251,249,.. instead of 255,254,253,252,...
         r = g = b = (i*2+55)*256;
         dispPix[i].pixel = (get_mask_color(r, vinfo.red_mask) |
                             get_mask_color(g, vinfo.green_mask) |
                             get_mask_color(b, vinfo.blue_mask) );
      }
   }

   /* get pixel for red line */
   rtn = XAllocNamedColor (display, cmap, "red", &redPix, &defPix);
   rtn = rtn;         /* silence the compiler warning */

   /* Setup fft values */
   fft_setup (512 * pskwids.getFFTsample() , pskwids.getFFTspeed());    
}


/*
 * fft_setup method - set fft values for sample sizes
 * speed: 0..100 --> overlap=2048..256
 * 0 -> 2^11   100-> 2^8
 */
void Disp::fft_setup(int samps, int speed)
{
   static int fftspeed=0;
   if(samps > 0)
   {
      samples = samps;               /* Size of the sample - 512, 1K, 2K */
   }
   if (speed >= 0)
   {
      if(speed > 100)
      {
         fprintf(stderr,"fft_setup: invalid speed speed=%d\n",speed);
      }
      else
      {
         float expo = 8+(100-speed)/99.0*3;  // 8..11
         fftspeed = (int)pow(2.0, expo);
         // fprintf(stderr,"setting fft speed to %d\n",fftspeed);
      }
   }

   ffts = samples/2;                          /* Number of filters */
// TJW   deltaf=(float)speed/samples;         /* Hz/fft sample */
   deltaf=(float)SAMPLE_RATE/samples;         /* Hz/fft sample */
   delta = (int)(fc / deltaf - WF_WIDTH/2);   /* Offset in ffts */
   cnt = 0;

   /* setup IMD Calculations constants for the current sample rate */
   upperSig = WF_WIDTH/2 + (samples / 512);
   lowerSig = WF_WIDTH/2 - (samples / 512);

   upperIMD = WF_WIDTH/2 + (samples / 512) * 3;
   lowerIMD = WF_WIDTH/2 - (samples / 512) * 3;

   commControl(COMM_FFTCH, COMM_FFTN, samples);
   commControl(COMM_FFTCH, COMM_FFTOVERLAP, fftspeed);
}


/*
 * getFFT handles fft data
 * when buffer full, this funciton will calculate IMD, scale the
 * data and display it.
 */
void Disp::getFFT(float val[], int len)
{
   IMDcalc(val);			/* calculate the IMD */
   mkPoints(val);			/* scale the fft data for display */
   drawDisplay();			/* draw the wf or sa */
}


/*
 * IMDcalc - calculates the IMD of the signal
 * If DCD is on then calculate the IMD of each main sig, and add to
 * IMDtotal.  After IMD_CNT pairs are collected, compute the average
 * and display.  If there are BLOCK_IMD_SIZE blocks in a row without DCD,
 * put "---" in the widget, so we don't display trash when there is
 * no signal
 */

#define IMD_CNT 20
//TJW orig val #define BLOCK_IMD_SIZE 10
#define BLOCK_IMD_SIZE 5

void Disp::IMDcalc(float *val)
{
   static int cnt;
   static int block;
   static float IMDtotal = 0.0;
   static Widget imdTF = 0;
   char imdVal[8];
   float ratio;
  
   if (imdTF == 0)
      imdTF = pskwids.getImdTF();      /* get the IMD widget id */

   if (pskwids.getDCD() == 1)          /* no IMD if no signal */
   {
      cnt++;
      block = 0;

      ratio = val[delta + lowerIMD] / val[delta + lowerSig];
      IMDtotal = IMDtotal + 20.0 * log10 (ratio);
      
      ratio = val[delta + upperIMD] / val[delta + upperSig];
      IMDtotal = IMDtotal + 20.0 * log10 (ratio);

      if (cnt == IMD_CNT)
      {
         sprintf (imdVal, "%2.0f ", IMDtotal/(IMD_CNT * 2));
         XmTextFieldSetString (imdTF, imdVal);

         cnt = 0;
         IMDtotal = 0.0;
      }
   }
   else
   {
      block++;
      if (block == BLOCK_IMD_SIZE)
      {
         XmTextFieldSetString (imdTF, (char *)"---");
         block = 0;
      }
   } 
}
 
/*
 * mkPoints method - gets WF_WIDTH floats from fft buffer
 * For negative freqs and freqs > MAXFREQ, set values to -1.
 * At one time, the fft data values varied with sample size so, 
 * samples/1024.0 -> 1,2,4 -  kept the fft data values constant 
 */
void Disp::mkPoints(float val[])
{
#define SCALE 2048

   int x;
   float scale = 512.0 / samples;

   for (x=1; x<WF_WIDTH; x++)
   {
                                              /* for -freq or freq < MAXFREQ */
      if (x+delta <= 1 || fc + deltaf * (x-WF_WIDTH/2) > MAXFREQ)
      {
         values[x] = -1.0;                    /* mark it so to be gray */
      }
      else
      {
//TJW now
         values[x] = 5 * val[x+delta] * scale;    /* scale valid freqs */
      }
   }
}


/*
 * drawDisplay method - draws display as spectrum or waterfall
 */
void Disp::drawDisplay()
{
   int x;
   int pixIndex;
   int last=WF_HEIGHT;

   switch (displayFlag)
   {
      /*
       * draw waterfall
       */
      case WF:
           /* move everything up or down one line */
           /* wf_up -> 1=up, 0=down */
           XCopyArea (display, pmap, pmap, gc,
                      0, wf_up,
                      WF_WIDTH, WF_HEIGHT-1,
                      0, !wf_up);

           for (x=0; x < WF_WIDTH; x++)
           {
              /* but don't draw over the red line */
              if ((x < (WF_WIDTH / 2) - 1) || (x > (WF_WIDTH / 2) + 1))
               {
                  if (values[x] == -1)       /* if gray flag */
                  {
                     pixIndex = 50;
                  }
                  else
                  {
                     pixIndex = (int)(log10 (5 * values[x]) * (brightness));
                     //pixIndex = (int)(log10 (10 * values[x]) * brightness);
                     if (pixIndex > 99)
                     {
                        pixIndex = 99;
                     }
                     if (pixIndex < 0)
                        pixIndex = 0;
                  }
                  /* draw to pixmap */
                  XSetForeground (display, gc, dispPix[pixIndex].pixel);
                  if (wf_up == 1)
                     XDrawPoint (display, pmap, gc, x, WF_HEIGHT-1);
                  else
                     XDrawPoint (display, pmap, gc, x, 0);
               }
               else  /* draw the vertical red line to back store pixmap */
               {
                  XSetForeground (display, gc, redPix.pixel);
                  XDrawLine (display, pmap, gc,  x, 0, x, WF_HEIGHT);
               }
            }
         /* new row is done, so copy pixmap to window */
         XCopyArea (display, pmap, window, gc,
                    0, 0,
                    WF_WIDTH, WF_HEIGHT,
                    0, 0);
         break;

      /*
       * draw spectrum analyser
       */
      case SA:
         XClearWindow (display, window);
         XSetLineAttributes (display, gc, 0, LineSolid, CapRound, JoinRound);
         XSetForeground (display, gc, WhitePixel(display, screen));

         for (x=1; x<WF_WIDTH; x++)
         {
            pixIndex = (int) (values[x] * 5.0);
            if (pixIndex > WF_HEIGHT - 10)
            {
               pixIndex = WF_HEIGHT - 10;
            }

            if ((x < (WF_WIDTH / 2) - 1)
               || (x > (WF_WIDTH / 2) + 1))
            {
               if (x+delta <= 1 || x+delta > samples/2)
               /* for 0 > freq > MAXFREQ draw gray line */
               {
                  XSetForeground (display, gc, dispPix[50].pixel);
                  XDrawLine (display, pmap, gc, x, 0, x, WF_HEIGHT-1);
                  XSetForeground (display, gc, WhitePixel(display, screen));
               }
               else
               /* draw signal line */
               {
                  XDrawLine (display, pmap, gc, x, last,
                             x, WF_HEIGHT - pixIndex);
                  last = WF_HEIGHT - pixIndex;
               }
            }
            else   /* Draw the center line */
            {
               XSetForeground (display, gc, redPix.pixel);
               XDrawLine (display, pmap, gc, x, 0, x, WF_HEIGHT);
               XSetForeground (display, gc, WhitePixel(display, screen));
            }
         }

         /* all done so copy pixmap to the window */
         XCopyArea (display, pmap, window, gc,
                    0, 0, WF_WIDTH, WF_HEIGHT, 0, 0);
         /* clear the pixmap */
         XSetForeground (display, gc, BlackPixel(display, screen));
         XFillRectangle (display, pmap, gc, 0, 0, WF_WIDTH, WF_HEIGHT);
         break;

      default:
         fprintf (stderr,"twpskDisplay.drawDisplay - Broken WF/SA value\n");
         break;
   }
}


/*
 * offset method - sets delta to starting index of values array
 */
void Disp::offset(float freq)
{
   fc = freq;
   delta = (int)(fc / deltaf - WF_WIDTH/2);
}


/*
 * new_fc method - given the cursor's x position in the waterfall,
 * return the frequency 
 */
float Disp::new_fc (int x)
{
   int newx;
   float newfreq;

   if ((newx = find_center(x)) != -1)      /* find the center */
      x = newx;
   newfreq = fc + deltaf * (x-WF_WIDTH/2);

   if(newfreq < 0.0 ) return 0.0;
   else if (newfreq > MAXFREQ)
      return MAXFREQ;
   else return newfreq;
}


/*
 * find_center method - trys to find the center of the signal that
 * was clicked.  Returns -1 if it can't figure it out.
 * Time for the squelch again????
 */
#define SENSITIVITY 0.25

int Disp::find_center (int x)
{
   int /*TJW r,*/ i, start, stop;
   int high = 0, low = 0;
   int width = (int)(31.25 / deltaf);
   int center, sigWidth;
   float max = 0;

   //r = row - 1;
   //if (r < 0)                          /* work with the latest row */
      //row = WF_HEIGHT - 1;

   start = (int)((float)x-1.5*width);
   stop  = (int)((float)x+1.5*width);

   /* find maximum */
   for (i=start; i!=stop; i++)
   {
      if (i<0)                         /* it could be neg if near left edge */
         continue;
      if (values[i] > max) 
         max = values[i];           /* find the max value in the range */
   }

   /* find the edges */
   /* first the low edge */
   for (i=start; i<stop; i++)
   {
      if (values[i] < SENSITIVITY * max)
         continue;
      else
      {
         /* found the low edge */
         low = i;
         break;
      }
   }

   /* then the high edge */
   for (i=stop; i>start; i--)
   {
      if (values[i] < SENSITIVITY * max)
         continue;
      else
      {
         /* found the high edge */
         high = i;
         break;
      }
   }         
   sigWidth = high - low;
   center = (high+low) / 2;

//printf ("%d\n ", max);

   if (sigWidth > 0.5 * width && sigWidth < 1.5 * width) // && max >= 3)
   {
      //printf ("calc\n");
      return center;
   }
   else
   {
      //printf ("pointer\n");
      return -1;
   }
}
