//
// System.Web.UI.WebControls.GridView.cs
//
// Authors:
//	Lluis Sanchez Gual (lluis@novell.com)
//
// Copyright (C) 2005-2010 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Web.UI;
using System.Security.Permissions;
using System.Text;
using System.IO;
using System.Reflection;

namespace System.Web.UI.WebControls
{
	[SupportsEventValidation]
	[DesignerAttribute ("System.Web.UI.Design.WebControls.GridViewDesigner, " + Consts.AssemblySystem_Design, "System.ComponentModel.Design.IDesigner")]
	[ControlValuePropertyAttribute ("SelectedValue")]
	[DefaultEventAttribute ("SelectedIndexChanged")]
	[DataKeyProperty ("DataKey")]
	[AspNetHostingPermissionAttribute (SecurityAction.LinkDemand, Level = AspNetHostingPermissionLevel.Minimal)]
	[AspNetHostingPermissionAttribute (SecurityAction.InheritanceDemand, Level = AspNetHostingPermissionLevel.Minimal)]
	public class GridView: CompositeDataBoundControl, ICallbackEventHandler, ICallbackContainer, IPostBackEventHandler, IPostBackContainer, IPersistedSelector
		, IDataKeysControl, IDataBoundListControl, IDataBoundControl, IFieldControl
	{
		Table table;
		GridViewRowCollection rows;
		GridViewRow bottomPagerRow;
		GridViewRow topPagerRow;
		
		IOrderedDictionary currentEditRowKeys;
		IOrderedDictionary currentEditNewValues;
		IOrderedDictionary currentEditOldValues;
		
		ITemplate pagerTemplate;
		ITemplate emptyDataTemplate;
		
		PropertyDescriptor[] cachedKeyProperties;
		PropertyDescriptor[] cachedSuffixKeyProperties;
		// View state
		DataControlFieldCollection columns;
		PagerSettings pagerSettings;
		
		TableItemStyle alternatingRowStyle;
		TableItemStyle editRowStyle;
		TableItemStyle emptyDataRowStyle;
		TableItemStyle footerStyle;
		TableItemStyle headerStyle;
		TableItemStyle pagerStyle;
		TableItemStyle rowStyle;
		TableItemStyle selectedRowStyle;
		TableItemStyle sortedAscendingCellStyle;
		TableItemStyle sortedAscendingHeaderStyle;
		TableItemStyle sortedDescendingCellStyle;
		TableItemStyle sortedDescendingHeaderStyle;

		List <DataKey> _dataKeySuffixList;
		DataKeyArray rowSuffixKeys;
		List <DataKey> _dataKeyList;
		DataKeyArray keys;
		DataKey oldEditValues;
		AutoGeneratedFieldProperties[] autoFieldProperties;
		string [] dataKeyNames = null;
		readonly string[] emptyKeys = new string[0];
		IEnumerator _dataEnumerator;
		
		static readonly object PageIndexChangedEvent = new object();
		static readonly object PageIndexChangingEvent = new object();
		static readonly object RowCancelingEditEvent = new object();
		static readonly object RowCommandEvent = new object();
		static readonly object RowCreatedEvent = new object();
		static readonly object RowDataBoundEvent = new object();
		static readonly object RowDeletedEvent = new object();
		static readonly object RowDeletingEvent = new object();
		static readonly object RowEditingEvent = new object();
		static readonly object RowUpdatedEvent = new object();
		static readonly object RowUpdatingEvent = new object();
		static readonly object SelectedIndexChangedEvent = new object();
		static readonly object SelectedIndexChangingEvent = new object();
		static readonly object SortedEvent = new object();
		static readonly object SortingEvent = new object();
		
		// Control state
		int pageIndex;
		int selectedIndex = -1;
		int editIndex = -1;
		int pageCount = 0;
		SortDirection sortDirection = SortDirection.Ascending;
		string sortExpression;
		
		public GridView ()
		{
			EnableModelValidation = true;
		}
		
		public event EventHandler PageIndexChanged {
			add { Events.AddHandler (PageIndexChangedEvent, value); }
			remove { Events.RemoveHandler (PageIndexChangedEvent, value); }
		}
		
		public event GridViewPageEventHandler PageIndexChanging {
			add { Events.AddHandler (PageIndexChangingEvent, value); }
			remove { Events.RemoveHandler (PageIndexChangingEvent, value); }
		}
		
		public event GridViewCancelEditEventHandler RowCancelingEdit {
			add { Events.AddHandler (RowCancelingEditEvent, value); }
			remove { Events.RemoveHandler (RowCancelingEditEvent, value); }
		}
		
		public event GridViewCommandEventHandler RowCommand {
			add { Events.AddHandler (RowCommandEvent, value); }
			remove { Events.RemoveHandler (RowCommandEvent, value); }
		}
		
		public event GridViewRowEventHandler RowCreated {
			add { Events.AddHandler (RowCreatedEvent, value); }
			remove { Events.RemoveHandler (RowCreatedEvent, value); }
		}
		
		public event GridViewRowEventHandler RowDataBound {
			add { Events.AddHandler (RowDataBoundEvent, value); }
			remove { Events.RemoveHandler (RowDataBoundEvent, value); }
		}
		
		public event GridViewDeletedEventHandler RowDeleted {
			add { Events.AddHandler (RowDeletedEvent, value); }
			remove { Events.RemoveHandler (RowDeletedEvent, value); }
		}
		
		public event GridViewDeleteEventHandler RowDeleting {
			add { Events.AddHandler (RowDeletingEvent, value); }
			remove { Events.RemoveHandler (RowDeletingEvent, value); }
		}
		
		public event GridViewEditEventHandler RowEditing {
			add { Events.AddHandler (RowEditingEvent, value); }
			remove { Events.RemoveHandler (RowEditingEvent, value); }
		}
		
		public event GridViewUpdatedEventHandler RowUpdated {
			add { Events.AddHandler (RowUpdatedEvent, value); }
			remove { Events.RemoveHandler (RowUpdatedEvent, value); }
		}
		
		public event GridViewUpdateEventHandler RowUpdating {
			add { Events.AddHandler (RowUpdatingEvent, value); }
			remove { Events.RemoveHandler (RowUpdatingEvent, value); }
		}
		
		public event EventHandler SelectedIndexChanged {
			add { Events.AddHandler (SelectedIndexChangedEvent, value); }
			remove { Events.RemoveHandler (SelectedIndexChangedEvent, value); }
		}
		
		public event GridViewSelectEventHandler SelectedIndexChanging {
			add { Events.AddHandler (SelectedIndexChangingEvent, value); }
			remove { Events.RemoveHandler (SelectedIndexChangingEvent, value); }
		}
		
		public event EventHandler Sorted {
			add { Events.AddHandler (SortedEvent, value); }
			remove { Events.RemoveHandler (SortedEvent, value); }
		}
		
		public event GridViewSortEventHandler Sorting {
			add { Events.AddHandler (SortingEvent, value); }
			remove { Events.RemoveHandler (SortingEvent, value); }
		}
		
		protected virtual void OnPageIndexChanged (EventArgs e)
		{
			if (Events != null) {
				EventHandler eh = (EventHandler) Events [PageIndexChangedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnPageIndexChanging (GridViewPageEventArgs e)
		{
			if (Events != null) {
				GridViewPageEventHandler eh = (GridViewPageEventHandler) Events [PageIndexChangingEvent];
				if (eh != null) {
					eh (this, e);
					return;
				}
			}
			if (!IsBoundUsingDataSourceID)
				throw new HttpException (String.Format ("The GridView '{0}' fired event PageIndexChanging which wasn't handled.", ID));
		}
		
		protected virtual void OnRowCancelingEdit (GridViewCancelEditEventArgs e)
		{
			if (Events != null) {
				GridViewCancelEditEventHandler eh = (GridViewCancelEditEventHandler) Events [RowCancelingEditEvent];
				if (eh != null) {
					eh (this, e);
					return;
				}
			}
			if (!IsBoundUsingDataSourceID)
				throw new HttpException (String.Format ("The GridView '{0}' fired event RowCancelingEdit which wasn't handled.", ID));
		}
		
		protected virtual void OnRowCommand (GridViewCommandEventArgs e)
		{
			if (Events != null) {
				GridViewCommandEventHandler eh = (GridViewCommandEventHandler) Events [RowCommandEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowCreated (GridViewRowEventArgs e)
		{
			if (Events != null) {
				GridViewRowEventHandler eh = (GridViewRowEventHandler) Events [RowCreatedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowDataBound (GridViewRowEventArgs e)
		{
			if (Events != null) {
				GridViewRowEventHandler eh = (GridViewRowEventHandler) Events [RowDataBoundEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowDeleted (GridViewDeletedEventArgs e)
		{
			if (Events != null) {
				GridViewDeletedEventHandler eh = (GridViewDeletedEventHandler) Events [RowDeletedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowDeleting (GridViewDeleteEventArgs e)
		{
			if (Events != null) {
				GridViewDeleteEventHandler eh = (GridViewDeleteEventHandler) Events [RowDeletingEvent];
				if (eh != null) {
					eh (this, e);
					return;
				}
			}
			if (!IsBoundUsingDataSourceID)
				throw new HttpException (String.Format ("The GridView '{0}' fired event RowDeleting which wasn't handled.", ID));
		}
		
		protected virtual void OnRowEditing (GridViewEditEventArgs e)
		{
			if (Events != null) {
				GridViewEditEventHandler eh = (GridViewEditEventHandler) Events [RowEditingEvent];
				if (eh != null) {
					eh (this, e);
					return;
				}
			}
			if (!IsBoundUsingDataSourceID)
				throw new HttpException (String.Format ("The GridView '{0}' fired event RowEditing which wasn't handled.", ID));
		}
		
		protected virtual void OnRowUpdated (GridViewUpdatedEventArgs e)
		{
			if (Events != null) {
				GridViewUpdatedEventHandler eh = (GridViewUpdatedEventHandler) Events [RowUpdatedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnRowUpdating (GridViewUpdateEventArgs e)
		{
			if (Events != null) {
				GridViewUpdateEventHandler eh = (GridViewUpdateEventHandler) Events [RowUpdatingEvent];
				if (eh != null) {
					eh (this, e);
					return;
				}
			}
			if (!IsBoundUsingDataSourceID)
				throw new HttpException (String.Format ("The GridView '{0}' fired event RowUpdating which wasn't handled.", ID));
		}
		
		protected virtual void OnSelectedIndexChanged (EventArgs e)
		{
			if (Events != null) {
				EventHandler eh = (EventHandler) Events [SelectedIndexChangedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSelectedIndexChanging (GridViewSelectEventArgs e)
		{
			if (Events != null) {
				GridViewSelectEventHandler eh = (GridViewSelectEventHandler) Events [SelectedIndexChangingEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSorted (EventArgs e)
		{
			if (Events != null) {
				EventHandler eh = (EventHandler) Events [SortedEvent];
				if (eh != null) eh (this, e);
			}
		}
		
		protected virtual void OnSorting (GridViewSortEventArgs e)
		{
			if (Events != null) {
				GridViewSortEventHandler eh = (GridViewSortEventHandler) Events [SortingEvent];
				if (eh != null) {
					eh (this, e);
					return;
				}
			}
			if (!IsBoundUsingDataSourceID)
				throw new HttpException (String.Format ("The GridView '{0}' fired event Sorting which wasn't handled.", ID));
		}
		
		
		[WebCategoryAttribute ("Paging")]
		[DefaultValueAttribute (false)]
		public virtual bool AllowPaging {
			get {
				object ob = ViewState ["AllowPaging"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == AllowPaging)
					return;
				ViewState ["AllowPaging"] = value;
				RequireBinding ();
			}
		}
		
		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AllowSorting {
			get {
				object ob = ViewState ["AllowSorting"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == AllowSorting)
					return;
				ViewState ["AllowSorting"] = value;
				RequireBinding ();
			}
		}
		
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle AlternatingRowStyle {
			get {
				if (alternatingRowStyle == null) {
					alternatingRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						alternatingRowStyle.TrackViewState();
				}
				return alternatingRowStyle;
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AutoGenerateEditButton {
			get {
				object ob = ViewState ["AutoGenerateEditButton"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == AutoGenerateEditButton)
					return;
				ViewState ["AutoGenerateEditButton"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AutoGenerateDeleteButton {
			get {
				object ob = ViewState ["AutoGenerateDeleteButton"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == AutoGenerateDeleteButton)
					return;
				ViewState ["AutoGenerateDeleteButton"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool AutoGenerateSelectButton {
			get {
				object ob = ViewState ["AutoGenerateSelectButton"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == AutoGenerateSelectButton)
					return;
				ViewState ["AutoGenerateSelectButton"] = value;
				RequireBinding ();
			}
		}

		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (true)]
		public virtual bool AutoGenerateColumns {
			get {
				object ob = ViewState ["AutoGenerateColumns"];
				if (ob != null)
					return (bool) ob;
				return true;
			}
			set {
				if (value == AutoGenerateColumns)
					return;
				ViewState ["AutoGenerateColumns"] = value;
				RequireBinding ();
			}
		}
		
		[UrlPropertyAttribute]
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute ("")]
		[EditorAttribute ("System.Web.UI.Design.ImageUrlEditor, " + Consts.AssemblySystem_Design, "System.Drawing.Design.UITypeEditor, " + Consts.AssemblySystem_Drawing)]
		public virtual string BackImageUrl {
			get {
				if (ControlStyleCreated)
					return ((TableStyle) ControlStyle).BackImageUrl;
				return String.Empty;
			}
			set { ((TableStyle) ControlStyle).BackImageUrl = value; }
		}

		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow BottomPagerRow {
			get {
				EnsureDataBound ();
				return bottomPagerRow;
			}
		}
	
		[WebCategoryAttribute ("Accessibility")]
		[DefaultValueAttribute ("")]
		[LocalizableAttribute (true)]
		public virtual string Caption {
			get {
				object ob = ViewState ["Caption"];
				if (ob != null)
					return (string) ob;
				return String.Empty;
			}
			set { ViewState ["Caption"] = value; }
		}
		
		[WebCategoryAttribute ("Accessibility")]
		[DefaultValueAttribute (TableCaptionAlign.NotSet)]
		public virtual TableCaptionAlign CaptionAlign
		{
			get {
				object o = ViewState ["CaptionAlign"];
				if(o != null)
					return (TableCaptionAlign) o;
				return TableCaptionAlign.NotSet;
			}
			set { ViewState ["CaptionAlign"] = value; }
		}

		[WebCategoryAttribute ("Layout")]
		[DefaultValueAttribute (-1)]
		public virtual int CellPadding
		{
			get {
				if (ControlStyleCreated)
					return ((TableStyle) ControlStyle).CellPadding;
				return -1;
			}
			set { ((TableStyle) ControlStyle).CellPadding = value; }
		}

		[WebCategoryAttribute ("Layout")]
		[DefaultValueAttribute (0)]
		public virtual int CellSpacing
		{
			get {
				if (ControlStyleCreated)
					return ((TableStyle) ControlStyle).CellSpacing;
				return 0;
			}
			set { ((TableStyle) ControlStyle).CellSpacing = value; }
		}
		
		[EditorAttribute ("System.Web.UI.Design.WebControls.DataControlFieldTypeEditor, " + Consts.AssemblySystem_Design, "System.Drawing.Design.UITypeEditor, " + Consts.AssemblySystem_Drawing)]
		[MergablePropertyAttribute (false)]
		[PersistenceModeAttribute (PersistenceMode.InnerProperty)]
		[DefaultValueAttribute (null)]
		[WebCategoryAttribute ("Misc")]
		public virtual DataControlFieldCollection Columns {
			get {
				if (columns == null) {
					columns = new DataControlFieldCollection ();
					columns.FieldsChanged += new EventHandler (OnFieldsChanged);
					if (IsTrackingViewState)
						((IStateManager)columns).TrackViewState ();
				}
				return columns;
			}
		}

		[BrowsableAttribute(false)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Hidden)]
		public IAutoFieldGenerator ColumnsGenerator {
			get;
			set;
		}

		[DefaultValueAttribute (null)]
		[WebCategoryAttribute ("Data")]
		[TypeConverter (typeof(StringArrayConverter))]
		[EditorAttribute ("System.Web.UI.Design.WebControls.DataFieldEditor, " + Consts.AssemblySystem_Design, "System.Drawing.Design.UITypeEditor, " + Consts.AssemblySystem_Drawing)]
		public virtual string[] DataKeyNames {
			get {
				if (dataKeyNames != null)
					return dataKeyNames;
				return emptyKeys;
			}
			set {
				dataKeyNames = value;
				RequireBinding ();
			}
		}

		List <DataKey> DataKeyList {
			get {
				if (_dataKeyList == null)
					_dataKeyList = new List <DataKey> ();
				
				return _dataKeyList;
			}
		}
		List <DataKey> DataKeySuffixList {
			get {
				if (_dataKeySuffixList == null)
					_dataKeySuffixList = new List <DataKey> ();
				
				return _dataKeySuffixList;
			}
		}
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual DataKeyArray DataKeys {
			get {
				if (keys == null) {
					keys = new DataKeyArray (DataKeyList);
					if (IsTrackingViewState)
						((IStateManager) keys).TrackViewState ();
				}
				return keys;
			}
		}

		DataKey OldEditValues {
			get {
				if (oldEditValues == null) {
					oldEditValues = new DataKey (new OrderedDictionary ());
				}
				return oldEditValues;
			}
		}

		[WebCategoryAttribute ("Misc")]
		[DefaultValueAttribute (-1)]
		public virtual int EditIndex {
			get { return editIndex; }
			set {
				if (value == editIndex)
					return;
				editIndex = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle EditRowStyle {
			get {
				if (editRowStyle == null) {
					editRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						editRowStyle.TrackViewState();
				}
				return editRowStyle;
			}
		}
		
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle EmptyDataRowStyle {
			get {
				if (emptyDataRowStyle == null) {
					emptyDataRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						emptyDataRowStyle.TrackViewState();
				}
				return emptyDataRowStyle;
			}
		}
		
		[DefaultValue (null)]
		[TemplateContainer (typeof(GridViewRow), BindingDirection.OneWay)]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[Browsable (false)]
		public virtual ITemplate EmptyDataTemplate {
			get { return emptyDataTemplate; }
			set { emptyDataTemplate = value; }
		}
		
		[LocalizableAttribute (true)]
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute ("")]
		public virtual string EmptyDataText {
			get {
				object ob = ViewState ["EmptyDataText"];
				if (ob != null)
					return (string) ob;
				return String.Empty;
			}
			set {
				if (value == EmptyDataText)
					return;
				ViewState ["EmptyDataText"] = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Behavior")]
		[DefaultValueAttribute (false)]
		public virtual bool EnableSortingAndPagingCallbacks {
			get {
				object ob = ViewState ["EnableSortingAndPagingCallbacks"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == EnableSortingAndPagingCallbacks)
					return;
				ViewState ["EnableSortingAndPagingCallbacks"] = value;
				RequireBinding ();
			}
		}

		[MonoTODO ("Make use of it in the code")]
		[DefaultValue (true)]
		public virtual bool EnableModelValidation {
			get;
			set;
		}
		
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow FooterRow {
			get {
				if (table != null) {
					for (int index = table.Rows.Count - 1; index >= 0; index--) {
						GridViewRow row = (GridViewRow) table.Rows [index];
						switch (row.RowType) {
							case DataControlRowType.Separator:
							case DataControlRowType.Pager:
								continue;
							case DataControlRowType.Footer:
								return row;
							default:
								break;
						}
					}
				}
				return null;
			}
		}
	
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DefaultValue (null)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle FooterStyle {
			get {
				if (footerStyle == null) {
					footerStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						footerStyle.TrackViewState();
				}
				return footerStyle;
			}
		}
		
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute (GridLines.Both)]
		public virtual GridLines GridLines {
			get {
				if (ControlStyleCreated)
					return ((TableStyle) ControlStyle).GridLines;
				return GridLines.Both;
			}
			set { ((TableStyle) ControlStyle).GridLines = value; }
		}

		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow HeaderRow {
			get {
				if (table != null) {
					for (int index = 0, total = table.Rows.Count; index < total; index++) {
						GridViewRow row = (GridViewRow) table.Rows [index];
						switch (row.RowType) {
							case DataControlRowType.Separator:
							case DataControlRowType.Pager:
								continue;
							case DataControlRowType.Header:
								return row;
							default:
								break;
						}
					}
				}
				return null;
			}
		}
	
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DefaultValue (null)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle HeaderStyle {
			get {
				if (headerStyle == null) {
					headerStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						headerStyle.TrackViewState();
				}
				return headerStyle;
			}
		}
		
		[Category ("Layout")]
		[DefaultValueAttribute (HorizontalAlign.NotSet)]
		public virtual HorizontalAlign HorizontalAlign {
			get {
				if (ControlStyleCreated)
					return ((TableStyle) ControlStyle).HorizontalAlign;
				return HorizontalAlign.NotSet;
			}
			set { ((TableStyle) ControlStyle).HorizontalAlign = value; }
		}

		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual int PageCount {
			get { return pageCount; }
			private set { pageCount = value; }
		}

		[WebCategoryAttribute ("Paging")]
		[BrowsableAttribute (true)]
		[DefaultValueAttribute (0)]
		public virtual int PageIndex {
			get { return pageIndex; }
			set {
				if (value == pageIndex)
					return;
				pageIndex = value;
				RequireBinding ();
			}
		}
	
		[DefaultValueAttribute (10)]
		[WebCategoryAttribute ("Paging")]
		public virtual int PageSize {
			get {
				object ob = ViewState ["PageSize"];
				if (ob != null)
					return (int) ob;
				return 10;
			}
			set {
				if (value == PageSize)
					return;
				ViewState ["PageSize"] = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Paging")]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Content)]
		[NotifyParentPropertyAttribute (true)]
		[PersistenceModeAttribute (PersistenceMode.InnerProperty)]
		public virtual PagerSettings PagerSettings {
			get {
				if (pagerSettings == null) {
					pagerSettings = new PagerSettings (this);
					if (IsTrackingViewState)
						((IStateManager)pagerSettings).TrackViewState ();
				}
				return pagerSettings;
			}
		}
	
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle PagerStyle {
			get {
				if (pagerStyle == null) {
					pagerStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						pagerStyle.TrackViewState();
				}
				return pagerStyle;
			}
		}
		
		
		[DefaultValue (null)]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[Browsable (false)]
		[TemplateContainerAttribute(typeof(GridViewRow))]
		public virtual ITemplate PagerTemplate {
			get { return pagerTemplate; }
			set { pagerTemplate = value; }
		}
		
		[DefaultValueAttribute ("")]
		[WebCategoryAttribute ("Accessibility")]
		//		[TypeConverterAttribute (typeof(System.Web.UI.Design.DataColumnSelectionConverter)]
		public virtual string RowHeaderColumn {
			get {
				object ob = ViewState ["RowHeaderColumn"];
				if (ob != null)
					return (string) ob;
				return String.Empty;
			}
			set {
				if (value == RowHeaderColumn)
					return;
				ViewState ["RowHeaderColumn"] = value;
				RequireBinding ();
			}
		}
		
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRowCollection Rows {
			get {
				EnsureChildControls ();
				if (rows == null)
					rows = new GridViewRowCollection (new ArrayList ());
				return rows;
			}
		}
		
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle RowStyle {
			get {
				if (rowStyle == null) {
					rowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						rowStyle.TrackViewState();
				}
				return rowStyle;
			}
		}
		
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Hidden)]
		public virtual DataKey SelectedDataKey {
			get {
				if (DataKeyNames.Length == 0)
					throw new InvalidOperationException (String.Format ("Data keys must be specified on GridView '{0}' before the selected data keys can be retrieved.  Use the DataKeyNames property to specify data keys.", ID));

				if (selectedIndex >= 0 && selectedIndex < DataKeys.Count)
					return DataKeys [selectedIndex];
				else
					return null;
			}
		}

		[MonoTODO]
		[Browsable(false)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Hidden)]
		public virtual DataKey SelectedPersistedDataKey {
			get; set;
		}

		[MonoTODO]
		DataKey IPersistedSelector.DataKey {
			get { return SelectedPersistedDataKey; }
			set { SelectedPersistedDataKey = value; }
		}
		
		[BindableAttribute (true)]
		[DefaultValueAttribute (-1)]
		public virtual int SelectedIndex {
			get { return selectedIndex; }
			set {
				if (rows != null && selectedIndex >= 0 && selectedIndex < Rows.Count) {
					int oldIndex = selectedIndex;
					selectedIndex = -1;
					Rows [oldIndex].RowState = GetRowState (oldIndex);
				}
				selectedIndex = value;
				if (rows != null && selectedIndex >= 0 && selectedIndex < Rows.Count)
					Rows [selectedIndex].RowState = GetRowState (selectedIndex);
			}
		}
	
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual GridViewRow SelectedRow {
			get {
				if (selectedIndex >= 0 && selectedIndex < Rows.Count)
					return Rows [selectedIndex];
				else
					return null;
			}
		}
		
		[WebCategoryAttribute ("Styles")]
		[PersistenceMode (PersistenceMode.InnerProperty)]
		[NotifyParentProperty (true)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		public TableItemStyle SelectedRowStyle {
			get {
				if (selectedRowStyle == null) {
					selectedRowStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						selectedRowStyle.TrackViewState();
				}
				return selectedRowStyle;
			}
		}
		
		[BrowsableAttribute (false)]
		public object SelectedValue {
			get {
				if (SelectedDataKey != null)
					return SelectedDataKey.Value;
				else
					return null;
			}
		}
		
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute (false)]
		public virtual bool ShowFooter {
			get {
				object ob = ViewState ["ShowFooter"];
				if (ob != null)
					return (bool) ob;
				return false;
			}
			set {
				if (value == ShowFooter)
					return;
				ViewState ["ShowFooter"] = value;
				RequireBinding ();
			}
		}
	
		[WebCategoryAttribute ("Appearance")]
		[DefaultValueAttribute (true)]
		public virtual bool ShowHeader {
			get {
				object ob = ViewState ["ShowHeader"];
				if (ob != null)
					return (bool) ob;
				return true;
			}
			set {
				if (value == ShowHeader)
					return;
				ViewState ["ShowHeader"] = value;
				RequireBinding ();
			}
		}
		
		[PersistenceModeAttribute (PersistenceMode.InnerProperty)]
		[BrowsableAttribute (false)]
		[DefaultValueAttribute (SortDirection.Ascending)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual SortDirection SortDirection {
			get { return sortDirection; }
			private set {
				if (sortDirection == value)
					return;
				sortDirection = value;
				RequireBinding ();
			}
		}
		
		[BrowsableAttribute (false)]
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		public virtual string SortExpression {
			get {
				if (sortExpression == null)
					return String.Empty;
				return sortExpression;
			}
			private set {
				if (sortExpression == value)
					return;
				sortExpression = value;
				RequireBinding ();
			}
		}
		
		[DesignerSerializationVisibilityAttribute (DesignerSerializationVisibility.Hidden)]
		[BrowsableAttribute (false)]
		public virtual GridViewRow TopPagerRow {
			get {
				EnsureDataBound ();
				return topPagerRow;
			}
		}
	
		[WebCategoryAttribute ("Accessibility")]
		[DefaultValueAttribute (true)]
		public virtual bool UseAccessibleHeader {
			get {
				object ob = ViewState ["UseAccessibleHeader"];
				if (ob != null)
					return (bool) ob;
				return true;
			}
			set {
				if (value == UseAccessibleHeader)
					return;
				ViewState ["UseAccessibleHeader"] = value;
				RequireBinding ();
			}
		}
		[TypeConverter (typeof(StringArrayConverter))]
		[DefaultValue (null)]
		public virtual string[] ClientIDRowSuffix {
			get;
			set;
		}

		[BrowsableAttribute(false)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Hidden)]
		public DataKeyArray ClientIDRowSuffixDataKeys {
			get {
				if (rowSuffixKeys == null) {
					rowSuffixKeys = new DataKeyArray (DataKeySuffixList);
					if (IsTrackingViewState)
						((IStateManager) rowSuffixKeys).TrackViewState ();
				}

				return rowSuffixKeys;
			}
		}

		[DefaultValue (false)]
		public virtual bool EnablePersistedSelection {
			get {
				throw new NotImplementedException ();
			}
			
			set {
				throw new NotImplementedException ();
			}
		}

		IAutoFieldGenerator IFieldControl.FieldsGenerator {
			get {
				throw new NotImplementedException ();
			}
			
			set {
				throw new NotImplementedException ();
			}
		}

		[DefaultValue (false)]
		public virtual bool ShowHeaderWhenEmpty {
			get { return ViewState.GetBool ("ShowHeaderWhenEmpty", false); }
			set {
				if (value == ShowHeaderWhenEmpty)
					return;
				
				ViewState ["ShowHeaderWhenEmpty"] = value;
				RequireBinding ();
			}
		}

		[PersistenceMode (PersistenceMode.InnerProperty)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		[NotifyParentProperty (true)]
		public TableItemStyle SortedAscendingCellStyle {
			get {
				if (sortedAscendingCellStyle == null) {
					sortedAscendingCellStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						((IStateManager)sortedAscendingCellStyle).TrackViewState ();
				}

				return sortedAscendingCellStyle;
			}
		}

		[PersistenceMode (PersistenceMode.InnerProperty)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		[NotifyParentProperty (true)]
		public TableItemStyle SortedAscendingHeaderStyle {
			get {
				if (sortedAscendingHeaderStyle == null) {
					sortedAscendingHeaderStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						((IStateManager)sortedAscendingHeaderStyle).TrackViewState ();
				}

				return sortedAscendingHeaderStyle;
			}
		}

		[PersistenceMode (PersistenceMode.InnerProperty)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		[NotifyParentProperty (true)]
		public TableItemStyle SortedDescendingCellStyle {
			get {
				if (sortedDescendingCellStyle == null) {
					sortedDescendingCellStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						((IStateManager)sortedDescendingCellStyle).TrackViewState ();
				}

				return sortedDescendingCellStyle;
			}
		}

		[PersistenceMode (PersistenceMode.InnerProperty)]
		[DesignerSerializationVisibility (DesignerSerializationVisibility.Content)]
		[NotifyParentProperty (true)]
		public TableItemStyle SortedDescendingHeaderStyle {
			get {
				if (sortedDescendingHeaderStyle == null) {
					sortedDescendingHeaderStyle = new TableItemStyle ();
					if (IsTrackingViewState)
						((IStateManager)sortedDescendingHeaderStyle).TrackViewState ();
				}

				return sortedDescendingHeaderStyle;
			}
		}		
		public virtual bool IsBindableType (Type type)
		{
			return type.IsPrimitive || type == typeof (string) || type == typeof (decimal) || type == typeof (DateTime) || type == typeof (Guid);
		}
		
		// MSDN: The CreateDataSourceSelectArguments method is a helper method called by 
		// the GridView control to create the DataSourceSelectArguments object that 
		// contains the arguments passed to the data source. In this implementation, 
		// the DataSourceSelectArguments object contains the arguments for paging operations.
		protected override DataSourceSelectArguments CreateDataSourceSelectArguments ()
		{
			DataSourceSelectArguments arg = DataSourceSelectArguments.Empty;
			DataSourceView view= GetData();
			if (AllowPaging && view.CanPage) {
				arg.StartRowIndex = PageIndex * PageSize;
				if (view.CanRetrieveTotalRowCount) {
					arg.RetrieveTotalRowCount = true;
					arg.MaximumRows = PageSize;
				} else
					arg.MaximumRows = -1;
			}

			if (IsBoundUsingDataSourceID && !String.IsNullOrEmpty (sortExpression)) {
				if (sortDirection == SortDirection.Ascending)
					arg.SortExpression = sortExpression;
				else
					arg.SortExpression = sortExpression + " DESC";
			}
			
			return arg;
		}
		
		protected virtual ICollection CreateColumns (PagedDataSource dataSource, bool useDataSource)
		{
			bool autoGenerate = AutoGenerateColumns;

			if (autoGenerate) {
				IAutoFieldGenerator fieldGenerator = ColumnsGenerator;
				if (fieldGenerator != null)
					return fieldGenerator.GenerateFields (this);
			}
			
			ArrayList fields = new ArrayList ();
			
			if (AutoGenerateEditButton || AutoGenerateDeleteButton || AutoGenerateSelectButton) {
				CommandField field = new CommandField ();
				field.ShowEditButton = AutoGenerateEditButton;
				field.ShowDeleteButton = AutoGenerateDeleteButton;
				field.ShowSelectButton = AutoGenerateSelectButton;
				fields.Add (field);
			}

			fields.AddRange (Columns);
			
			if (autoGenerate) {
				if (useDataSource)
					autoFieldProperties = CreateAutoFieldProperties (dataSource);
	
				if (autoFieldProperties != null) {
					foreach (AutoGeneratedFieldProperties props in autoFieldProperties)
						fields.Add (CreateAutoGeneratedColumn (props));
				}
			}
			
			return fields;
		}
		
		protected virtual AutoGeneratedField CreateAutoGeneratedColumn (AutoGeneratedFieldProperties fieldProperties)
		{
			return new AutoGeneratedField (fieldProperties);
		}
		
		AutoGeneratedFieldProperties[] CreateAutoFieldProperties (PagedDataSource source)
		{
			if(source == null) return null;
			
			PropertyDescriptorCollection props = source.GetItemProperties (new PropertyDescriptor[0]);
			Type prop_type = null;
			
			var retVal = new List <AutoGeneratedFieldProperties> ();
			
			if (props == null) {
				object fitem = null;
				PropertyInfo prop_item =  source.DataSource.GetType().GetProperty("Item",
												  BindingFlags.Instance | BindingFlags.Static |
												  BindingFlags.Public, null, null,
												  new Type[] { typeof(int) }, null);
				
				if (prop_item != null)
					prop_type = prop_item.PropertyType;
				
				if (prop_type == null || prop_type == typeof(object)) {
					IEnumerator en = source.GetEnumerator();
					if (en != null && en.MoveNext ()) {
						fitem = en.Current;
						_dataEnumerator = en;
					}
					if (fitem != null)
						prop_type = fitem.GetType();
				}
				
				if (fitem != null && fitem is ICustomTypeDescriptor)
					props = TypeDescriptor.GetProperties(fitem);
				else if (prop_type != null) {
					if (IsBindableType (prop_type)) {
						AutoGeneratedFieldProperties field = new AutoGeneratedFieldProperties ();
						((IStateManager)field).TrackViewState();
						field.Name = "Item";
						field.DataField = BoundField.ThisExpression;
						field.Type = prop_type;
						retVal.Add (field);
					} else
						props = TypeDescriptor.GetProperties (prop_type);
				}
			}
			
			if (props != null && props.Count > 0) {
				foreach (PropertyDescriptor current in props) {
					if (IsBindableType (current.PropertyType) && (prop_type == null || current.ComponentType == prop_type)) {
						AutoGeneratedFieldProperties field = new AutoGeneratedFieldProperties ();
						((IStateManager)field).TrackViewState();
						field.Name = current.Name;
						field.DataField = current.Name;
						for (int i = 0; i < DataKeyNames.Length; i++) {
							if (string.Compare (DataKeyNames [i], current.Name, StringComparison.InvariantCultureIgnoreCase) == 0) {
								field.IsReadOnly = true;
								break;
							}
						}
						field.Type = current.PropertyType;
						retVal.Add (field);
					}
				}
			}

			if (retVal.Count > 0)
				return retVal.ToArray ();
			else
				return new AutoGeneratedFieldProperties [0];
		}
		
		protected virtual GridViewRow CreateRow (int rowIndex, int dataSourceIndex, DataControlRowType rowType, DataControlRowState rowState)
		{
			GridViewRow row = new GridViewRow (rowIndex, dataSourceIndex, rowType, rowState);
			return row;
		}
		
		void RequireBinding ()
		{
			if (Initialized)
				RequiresDataBinding = true;
		}
		
		protected virtual Table CreateChildTable ()
		{
			return new ContainedTable (this);
		}

		void CreateHeaderRow (Table mainTable, DataControlField[] fields, bool dataBinding)
		{
			GridViewRow headerRow = CreateRow (-1, -1, DataControlRowType.Header, DataControlRowState.Normal);
			InitializeRow (headerRow, fields);
			OnRowCreated (new GridViewRowEventArgs (headerRow));
			mainTable.Rows.Add (headerRow);
			if (dataBinding) {
				headerRow.DataBind ();
				OnRowDataBound (new GridViewRowEventArgs (headerRow));
			}
		}
		
		protected override int CreateChildControls (IEnumerable dataSource, bool dataBinding)
		{
			// clear GridView
			Controls.Clear ();
			table = null;
			rows = null;

			if (dataSource == null)
				return 0;

			PagedDataSource pagedDataSource;

			if (dataBinding) {
				DataSourceView view = GetData ();
				pagedDataSource = new PagedDataSource ();
				pagedDataSource.DataSource = dataSource;
				
				if (AllowPaging) {
					pagedDataSource.AllowPaging = true;
					pagedDataSource.PageSize = PageSize;
					if (view.CanPage) {
						pagedDataSource.AllowServerPaging = true;
						if (SelectArguments.RetrieveTotalRowCount)
							pagedDataSource.VirtualCount = SelectArguments.TotalRowCount;
					}
					if (PageIndex >= pagedDataSource.PageCount)
						pageIndex = pagedDataSource.PageCount - 1;
					pagedDataSource.CurrentPageIndex = PageIndex;
				}
				
				PageCount = pagedDataSource.PageCount;
			} else {
				pagedDataSource = new PagedDataSource ();
				pagedDataSource.DataSource = dataSource;
				if (AllowPaging) {
					pagedDataSource.AllowPaging = true;
					pagedDataSource.PageSize = PageSize;
					pagedDataSource.CurrentPageIndex = PageIndex;
				}
			}

			bool createPager = AllowPaging && (PageCount >= 1) && PagerSettings.Visible;

			ArrayList list = new ArrayList ();
			
			// Creates the set of fields to show

			_dataEnumerator = null;
			ICollection fieldCollection = CreateColumns (pagedDataSource, dataBinding);
			int fieldCount = fieldCollection.Count;
			DataControlField dcf;
			DataControlField[] fields = new DataControlField [fieldCount];
			fieldCollection.CopyTo (fields, 0);
			
			for (int i = 0; i < fieldCount; i++) {
				dcf = fields [i];
				dcf.Initialize (AllowSorting, this);
				if (EnableSortingAndPagingCallbacks)
					dcf.ValidateSupportsCallback ();
			}

			bool skip_first = false;
			IEnumerator enumerator;
			if (_dataEnumerator != null) {
				// replaced when creating bound columns
				enumerator = _dataEnumerator;
				skip_first = true;
			} else
				enumerator = pagedDataSource.GetEnumerator ();

			// Main table creation
			Table mainTable = ContainedTable;
			List <DataKey> dataKeyList;
			string[] dataKeyNames;
			List <DataKey> dataKeySuffixList;
			string[] clientIDRowSuffix;
			if (dataBinding) {
				dataKeyList = DataKeyList;
				dataKeyNames = DataKeyNames;
				dataKeySuffixList = DataKeySuffixList;
				clientIDRowSuffix = ClientIDRowSuffix;
			} else {
				dataKeyList = null;
				dataKeyNames = null;
				dataKeySuffixList = null;
				clientIDRowSuffix = null;
			}

			while (skip_first || enumerator.MoveNext ()) {
				skip_first = false;
				object obj = enumerator.Current;
				
				if (list.Count == 0) {
					if (createPager && (PagerSettings.Position == PagerPosition.Top || PagerSettings.Position == PagerPosition.TopAndBottom)) {
						topPagerRow = CreatePagerRow (fieldCount, pagedDataSource);
						OnRowCreated (new GridViewRowEventArgs (topPagerRow));
						mainTable.Rows.Add (topPagerRow);
						if (dataBinding) {
							topPagerRow.DataBind ();
							OnRowDataBound (new GridViewRowEventArgs (topPagerRow));
						}
						if (PageCount == 1)
							topPagerRow.Visible = false;
					}

					if (ShowHeader)
						CreateHeaderRow (mainTable, fields, dataBinding);
				}
				
				DataControlRowState rstate = GetRowState (list.Count);
				GridViewRow row = CreateRow (list.Count, list.Count, DataControlRowType.DataRow, rstate);
				row.DataItem = obj;
				list.Add (row);
				InitializeRow (row, fields);
				OnRowCreated (new GridViewRowEventArgs (row));
				mainTable.Rows.Add (row);
				if (dataBinding) {
					row.DataBind ();					
					if (EditIndex == row.RowIndex)
						oldEditValues = new DataKey (GetRowValues (row, true, true));
					dataKeyList.Add (new DataKey (CreateRowDataKey (row), dataKeyNames));
					dataKeySuffixList.Add (new DataKey (CreateRowSuffixDataKey (row), clientIDRowSuffix));
					OnRowDataBound (new GridViewRowEventArgs (row));
				} 
			}

			if (list.Count == 0) {
 				if (ShowHeader && ShowHeaderWhenEmpty)
 					CreateHeaderRow (mainTable, fields, dataBinding);
				GridViewRow emptyRow = CreateEmptyrRow (fieldCount);
				if (emptyRow != null) {
					OnRowCreated (new GridViewRowEventArgs (emptyRow));
					mainTable.Rows.Add (emptyRow);
					if (dataBinding) {
						emptyRow.DataBind ();
						OnRowDataBound (new GridViewRowEventArgs (emptyRow));
					}
				}
				if (mainTable.Rows.Count == 0)
					table = null;
				return 0;
			} else {
				GridViewRow footerRow = CreateRow (-1, -1, DataControlRowType.Footer, DataControlRowState.Normal);
				InitializeRow (footerRow, fields);
				OnRowCreated (new GridViewRowEventArgs (footerRow));
				mainTable.Rows.Add (footerRow);
				if (dataBinding) {
					footerRow.DataBind ();
					OnRowDataBound (new GridViewRowEventArgs (footerRow));
				}

				if (createPager && (PagerSettings.Position == PagerPosition.Bottom || PagerSettings.Position == PagerPosition.TopAndBottom)) {
					bottomPagerRow = CreatePagerRow (fieldCount, pagedDataSource);
					OnRowCreated (new GridViewRowEventArgs (bottomPagerRow));
					mainTable.Rows.Add (bottomPagerRow);
					if (dataBinding) {
						bottomPagerRow.DataBind ();
						OnRowDataBound (new GridViewRowEventArgs (bottomPagerRow));
					}
					if (PageCount == 1)
						bottomPagerRow.Visible = false;
				}
			}

			rows = new GridViewRowCollection (list);

			if (!dataBinding)
				return -1;

			if (AllowPaging)
				return pagedDataSource.DataSourceCount;
			else
				return list.Count;
		}

		Table ContainedTable  {
			get {
				if (table == null) {
					table = CreateChildTable ();
					Controls.Add (table);
				}
				
				return table;
			}
		}

		protected override Style CreateControlStyle ()
		{
			TableStyle style = new TableStyle (ViewState);
			style.GridLines = GridLines.Both;
			style.CellSpacing = 0;
			return style;
		}
		
		DataControlRowState GetRowState (int index)
		{
			DataControlRowState rstate = (index % 2) == 0 ? DataControlRowState.Normal : DataControlRowState.Alternate;
			if (index == SelectedIndex)
				rstate |= DataControlRowState.Selected;
			if (index == EditIndex)
				rstate |= DataControlRowState.Edit;
			return rstate;
		}
		
		GridViewRow CreatePagerRow (int fieldCount, PagedDataSource dataSource)
		{
			GridViewRow row = CreateRow (-1, -1, DataControlRowType.Pager, DataControlRowState.Normal);
			InitializePager (row, fieldCount, dataSource);
			return row;
		}
		
		protected virtual void InitializePager (GridViewRow row, int columnSpan, PagedDataSource pagedDataSource)
		{
			TableCell cell = new TableCell ();
			if (columnSpan > 1)
				cell.ColumnSpan = columnSpan;
			
			if (pagerTemplate != null)
				pagerTemplate.InstantiateIn (cell);
			else
				cell.Controls.Add (PagerSettings.CreatePagerControl (pagedDataSource.CurrentPageIndex, pagedDataSource.PageCount));
			
			row.Cells.Add (cell);
		}
		
		GridViewRow CreateEmptyrRow (int fieldCount)
		{
			if (emptyDataTemplate == null && String.IsNullOrEmpty (EmptyDataText))
				return null;

			GridViewRow row = CreateRow (-1, -1, DataControlRowType.EmptyDataRow, DataControlRowState.Normal);
			TableCell cell = new TableCell ();
			cell.ColumnSpan = fieldCount;
			
			if (emptyDataTemplate != null)
				emptyDataTemplate.InstantiateIn (cell);
			else
				cell.Text = EmptyDataText;
			
			row.Cells.Add (cell);
			return row;
		}
		
		protected virtual void InitializeRow (GridViewRow row, DataControlField[] fields)
		{
			DataControlCellType ctype;
			bool accessibleHeader = false;

			switch (row.RowType) {
				case DataControlRowType.Header:
					ctype = DataControlCellType.Header; 
					accessibleHeader = UseAccessibleHeader;
					break;
				case DataControlRowType.Footer:
					ctype = DataControlCellType.Footer;
					break;
				default:
					ctype = DataControlCellType.DataCell;
					break;
			}
			
			for (int n=0; n<fields.Length; n++) {
				DataControlField field = fields [n];				
				DataControlFieldCell cell;

				if (((field is BoundField) && ((BoundField)field).DataField == RowHeaderColumn) || accessibleHeader)
					cell = new DataControlFieldHeaderCell (field, accessibleHeader ? TableHeaderScope.Column : TableHeaderScope.Row);
				else
					cell = new DataControlFieldCell (field);
				row.Cells.Add (cell);
				field.InitializeCell (cell, ctype, row.RowState, row.RowIndex);
			}
		}
		
		void LoadAndCacheProperties (string[] names, object dataItem, ref PropertyDescriptor[] cache)
		{
			if (cache != null)
				return;

			PropertyDescriptorCollection props = TypeDescriptor.GetProperties (dataItem);
			int len = names != null ? names.Length : 0;
			cache = new PropertyDescriptor [len];
			for (int n = 0; n < len; n++) {
				string propName = names [n];
				PropertyDescriptor p = props.Find (propName, true);
				if (p == null)
					throw new InvalidOperationException ("Property '" + propName + "' not found in object of type " + dataItem.GetType ());
				cache [n] = p;
			}
		}

		IOrderedDictionary CreateDictionaryFromProperties (PropertyDescriptor[] cache, object dataItem)
		{
			OrderedDictionary dic = new OrderedDictionary ();
			foreach (PropertyDescriptor p in cache)
				dic [p.Name] = p.GetValue (dataItem);
			return dic;
		}
		
		IOrderedDictionary CreateRowDataKey (GridViewRow row)
		{
			object dataItem = row.DataItem;
			LoadAndCacheProperties (DataKeyNames, dataItem, ref cachedKeyProperties);
			return CreateDictionaryFromProperties (cachedKeyProperties, dataItem);
		}
		IOrderedDictionary CreateRowSuffixDataKey (GridViewRow row)
		{
			object dataItem = row.DataItem;
			LoadAndCacheProperties (ClientIDRowSuffix, dataItem, ref cachedSuffixKeyProperties);
			return CreateDictionaryFromProperties (cachedSuffixKeyProperties, dataItem);
		}
		IOrderedDictionary GetRowValues (GridViewRow row, bool includeReadOnlyFields, bool includePrimaryKey)
		{
			OrderedDictionary dic = new OrderedDictionary ();
			ExtractRowValues (dic, row, includeReadOnlyFields, includePrimaryKey);
			return dic;
		}
		
		protected virtual void ExtractRowValues (IOrderedDictionary fieldValues, GridViewRow row, bool includeReadOnlyFields, bool includePrimaryKey)
		{
			DataControlField field;
			foreach (TableCell cell in row.Cells) {
				DataControlFieldCell c = cell as DataControlFieldCell;
				if (c == null)
					continue;
				
				field = c.ContainingField;
				if (field != null && !field.Visible)
					continue;
				
				c.ContainingField.ExtractValuesFromCell (fieldValues, c, row.RowState, includeReadOnlyFields);
			}
			if (!includePrimaryKey && DataKeyNames != null)
				foreach (string key in DataKeyNames)
					fieldValues.Remove (key);
		}
		
		protected override HtmlTextWriterTag TagKey {
			get {
				if (EnableSortingAndPagingCallbacks)
					return HtmlTextWriterTag.Div;
				else
					return HtmlTextWriterTag.Table;
			}
		}
		
		public sealed override void DataBind ()
		{
			DataKeyList.Clear ();
			cachedKeyProperties = null;
			base.DataBind ();

			keys = new DataKeyArray (DataKeyList);
			
			GridViewRow row = HeaderRow;
			if (row != null)
				row.Visible = ShowHeader;

			row = FooterRow;
			if (row != null)
				row.Visible = ShowFooter;
		}
		
		protected internal override void PerformDataBinding (IEnumerable data)
		{
			base.PerformDataBinding (data);
		}

		protected internal virtual void PrepareControlHierarchy ()
		{
			if (table == null)
				return;

			table.Caption = Caption;
			table.CaptionAlign = CaptionAlign;
			table.CopyBaseAttributes (this);
			
			foreach (GridViewRow row in table.Rows) {
				switch (row.RowType) {
					case DataControlRowType.Header:
						if (headerStyle != null && !headerStyle.IsEmpty)
							row.ControlStyle.MergeWith(headerStyle);
						row.Visible = ShowHeader;
						break;
					case DataControlRowType.Footer:
						if (footerStyle != null && !footerStyle.IsEmpty)
							row.ControlStyle.MergeWith (footerStyle);
						row.Visible = ShowFooter;
						break;
					case DataControlRowType.Pager:
						if (pagerStyle != null && !pagerStyle.IsEmpty)
							row.ControlStyle.MergeWith (pagerStyle);
						break;
					case DataControlRowType.EmptyDataRow:
						if (emptyDataRowStyle != null && !emptyDataRowStyle.IsEmpty)
							row.ControlStyle.MergeWith (emptyDataRowStyle);
						break;
					case DataControlRowType.DataRow:
						if ((row.RowState & DataControlRowState.Edit) != 0 && editRowStyle != null && !editRowStyle.IsEmpty)
							row.ControlStyle.MergeWith (editRowStyle);
						if ((row.RowState & DataControlRowState.Selected) != 0 && selectedRowStyle != null && !selectedRowStyle.IsEmpty)
							row.ControlStyle.MergeWith (selectedRowStyle);
						if ((row.RowState & DataControlRowState.Alternate) != 0 && alternatingRowStyle != null && !alternatingRowStyle.IsEmpty)
							row.ControlStyle.MergeWith (alternatingRowStyle);
						if (rowStyle != null && !rowStyle.IsEmpty)
							row.ControlStyle.MergeWith (rowStyle);
						break;
					default:
						break;
				}
				string sortExpression = SortExpression;
				bool haveSorting = !String.IsNullOrEmpty (sortExpression);
				foreach (TableCell cell in row.Cells) {
					DataControlFieldCell fcell = cell as DataControlFieldCell;
					if (fcell != null) {
						DataControlField field = fcell.ContainingField;
						if (field == null)
							continue;
						if (!field.Visible) {
							cell.Visible = false;
							continue;
						}
						
						switch (row.RowType) {
							case DataControlRowType.Header:
								if (field.HeaderStyleCreated && !field.HeaderStyle.IsEmpty)
									cell.ControlStyle.MergeWith (field.HeaderStyle);
								if (haveSorting)
									MergeWithSortingStyle (sortExpression, sortedAscendingHeaderStyle, sortedDescendingHeaderStyle, field, cell);
								break;
							case DataControlRowType.Footer:
								if (field.FooterStyleCreated && !field.FooterStyle.IsEmpty)
									cell.ControlStyle.MergeWith (field.FooterStyle);
								break;
							default:
								if (field.ControlStyleCreated && !field.ControlStyle.IsEmpty) {
									foreach (Control c in cell.Controls) {
										WebControl wc = c as WebControl;
										if (wc != null)
											wc.ControlStyle.MergeWith (field.ControlStyle);
									}
								}
								if (field.ItemStyleCreated && !field.ItemStyle.IsEmpty)
									cell.ControlStyle.MergeWith (field.ItemStyle);
								if (haveSorting)
									MergeWithSortingStyle (sortExpression, sortedAscendingCellStyle, sortedDescendingCellStyle, field, cell);
								break;
						}
					}
				}
			}
		}
		void MergeWithSortingStyle (string sortExpression, TableItemStyle ascending, TableItemStyle descending, DataControlField field, TableCell cell)
		{
			if (String.Compare (field.SortExpression, sortExpression, StringComparison.OrdinalIgnoreCase) != 0)
				return;

			cell.ControlStyle.MergeWith (SortDirection == SortDirection.Ascending ? ascending : descending);
		}
		protected internal override void OnInit (EventArgs e)
		{
			Page page = Page;
			if (page != null)
				page.RegisterRequiresControlState (this);
			base.OnInit (e);
		}
		
		void OnFieldsChanged (object sender, EventArgs args)
		{
			RequireBinding ();
		}
		
		protected override void OnDataPropertyChanged ()
		{
			base.OnDataPropertyChanged ();
			RequireBinding ();
		}
		
		protected override void OnDataSourceViewChanged (object sender, EventArgs e)
		{
			base.OnDataSourceViewChanged (sender, e);
			RequireBinding ();
		}
		
		protected override bool OnBubbleEvent (object source, EventArgs e)
		{
			GridViewCommandEventArgs args = e as GridViewCommandEventArgs;
			if (args != null) {
				bool causesValidation = false;
				IButtonControl button = args.CommandSource as IButtonControl;
				if (button != null && button.CausesValidation) {
					Page.Validate (button.ValidationGroup);
					causesValidation = true;
				}
				OnRowCommand (args);
				string param = args.CommandArgument as string;
				if (param == null || param.Length == 0) {
					GridViewRow row = args.Row;
					if (row != null)
						param = row.RowIndex.ToString();
				}
				ProcessEvent (args.CommandName, param, causesValidation);
				return true;
			}
			return base.OnBubbleEvent (source, e);
		}
		
		void IPostBackEventHandler.RaisePostBackEvent (string eventArgument)
		{
			ValidateEvent (UniqueID, eventArgument);
			RaisePostBackEvent (eventArgument);
		}

		// This is prolly obsolete
		protected virtual void RaisePostBackEvent (string eventArgument)
		{
			GridViewCommandEventArgs args;
			int i = eventArgument.IndexOf ('$');
			if (i != -1)
				args = new GridViewCommandEventArgs (this, new CommandEventArgs (eventArgument.Substring (0, i), eventArgument.Substring (i + 1)));
			else
				args = new GridViewCommandEventArgs (this, new CommandEventArgs (eventArgument, null));

			OnRowCommand (args);
			ProcessEvent (args.CommandName, (string) args.CommandArgument, false);
		}

		void ProcessEvent (string eventName, string param, bool causesValidation)
		{
			switch (eventName) {
				case DataControlCommands.PageCommandName:
					int newIndex = -1;
					switch (param) {
						case DataControlCommands.FirstPageCommandArgument:
							newIndex = 0;
							break;
						case DataControlCommands.LastPageCommandArgument:
							newIndex = PageCount - 1;
							break;
						case DataControlCommands.NextPageCommandArgument:
							newIndex = PageIndex + 1;
							break;
						case DataControlCommands.PreviousPageCommandArgument:
							newIndex = PageIndex - 1;
							break;
						default:
							int paramIndex = 0;
							int.TryParse (param, out paramIndex);
							newIndex = paramIndex - 1;
							break;
					}
					SetPageIndex (newIndex);
					break;
					
				case DataControlCommands.FirstPageCommandArgument:
					SetPageIndex (0);
					break;

				case DataControlCommands.LastPageCommandArgument:
					SetPageIndex (PageCount - 1);
					break;
					
				case DataControlCommands.NextPageCommandArgument:
					if (PageIndex < PageCount - 1)
						SetPageIndex (PageIndex + 1);
					break;

				case DataControlCommands.PreviousPageCommandArgument:
					if (PageIndex > 0)
						SetPageIndex (PageIndex - 1);
					break;
					
				case DataControlCommands.SelectCommandName:
					SelectRow (int.Parse (param));
					break;
					
				case DataControlCommands.EditCommandName:
					SetEditRow (int.Parse (param));
					break;
					
				case DataControlCommands.UpdateCommandName:
					int editIndex = int.Parse (param);
					UpdateRow (Rows [editIndex], editIndex, causesValidation);
					break;
					
				case DataControlCommands.CancelCommandName:
					CancelEdit ();
					break;
					
				case DataControlCommands.DeleteCommandName:
					DeleteRow (int.Parse (param));
					break;
					
				case DataControlCommands.SortCommandName:
					Sort (param);
					break;
			}
		}
		
		void Sort (string newSortExpression)
		{
			SortDirection newDirection = SortDirection.Ascending;
			if (sortExpression == newSortExpression && sortDirection == SortDirection.Ascending)
				newDirection = SortDirection.Descending;

			Sort (newSortExpression, newDirection);
		}
		
		public virtual void Sort (string sortExpression, SortDirection sortDirection)
		{
			GridViewSortEventArgs args = new GridViewSortEventArgs (sortExpression, sortDirection);
			OnSorting (args);
			if (args.Cancel)
				return;
			
			if (IsBoundUsingDataSourceID) {
				EditIndex = -1;
				PageIndex = 0;
				SortExpression = args.SortExpression;
				SortDirection = args.SortDirection;
			}
			
			OnSorted (EventArgs.Empty);
		}
		public
		void SelectRow (int rowIndex)
		{
			GridViewSelectEventArgs args = new GridViewSelectEventArgs (rowIndex);
			OnSelectedIndexChanging (args);
			if (!args.Cancel) {
				RequireBinding ();
				SelectedIndex = args.NewSelectedIndex;
				OnSelectedIndexChanged (EventArgs.Empty);
			}
		}
		public
		void SetPageIndex (int rowIndex)
		{
			GridViewPageEventArgs args = new GridViewPageEventArgs (rowIndex);
			OnPageIndexChanging (args);
			
			if (args.Cancel || !IsBoundUsingDataSourceID)
				return;
			
			EndRowEdit ();
			PageIndex = args.NewPageIndex;
			OnPageIndexChanged (EventArgs.Empty);
		}
		public
		void SetEditRow (int rowIndex)
		{
			GridViewEditEventArgs args = new GridViewEditEventArgs (rowIndex);
			OnRowEditing (args);
			
			if (args.Cancel || !IsBoundUsingDataSourceID)
				return;
			
			EditIndex = args.NewEditIndex;
		}
		
		void CancelEdit ()
		{
			GridViewCancelEditEventArgs args = new GridViewCancelEditEventArgs (EditIndex);
			OnRowCancelingEdit (args);
			
			if (args.Cancel || !IsBoundUsingDataSourceID)
				return;

			EndRowEdit ();
		}

		[MonoTODO ("Support two-way binding expressions")]
		public virtual void UpdateRow (int rowIndex, bool causesValidation)
		{
			if (rowIndex != EditIndex)
				throw new NotSupportedException ();
			
			GridViewRow row = Rows [rowIndex];
			UpdateRow (row, rowIndex, causesValidation);
		}

		void UpdateRow (GridViewRow row, int rowIndex, bool causesValidation)
		{
			if (causesValidation && Page != null && !Page.IsValid)
				return;

			currentEditOldValues = CopyOrderedDictionary (OldEditValues.Values);
			currentEditRowKeys = CopyOrderedDictionary (DataKeys [rowIndex].Values);
			currentEditNewValues = GetRowValues (row, false, false);
			
			GridViewUpdateEventArgs args = new GridViewUpdateEventArgs (rowIndex, currentEditRowKeys, currentEditOldValues, currentEditNewValues);
			OnRowUpdating (args);
			
			if (args.Cancel || !IsBoundUsingDataSourceID)
				return;
			
			DataSourceView view = GetData ();
			if (view == null)
				throw new HttpException ("The DataSourceView associated to data bound control was null");
			view.Update (currentEditRowKeys, currentEditNewValues, currentEditOldValues, new DataSourceViewOperationCallback (UpdateCallback));
		}

		static IOrderedDictionary CopyOrderedDictionary (IOrderedDictionary sourceDic)
		{
			OrderedDictionary copyDic = new OrderedDictionary ();
			foreach (object key in sourceDic.Keys)
				copyDic.Add (key, sourceDic [key]);
			return copyDic;
		}

		bool UpdateCallback (int recordsAffected, Exception exception)
		{
			GridViewUpdatedEventArgs dargs = new GridViewUpdatedEventArgs (recordsAffected, exception, currentEditRowKeys, currentEditOldValues, currentEditNewValues);
			OnRowUpdated (dargs);

			if (!dargs.KeepInEditMode)		
				EndRowEdit ();

			return dargs.ExceptionHandled;
		}
		
		public virtual void DeleteRow (int rowIndex)
		{
			GridViewRow row = Rows [rowIndex];
			currentEditRowKeys = CopyOrderedDictionary (DataKeys [rowIndex].Values);
			currentEditNewValues = GetRowValues (row, true, true);
			
			GridViewDeleteEventArgs args = new GridViewDeleteEventArgs (rowIndex, currentEditRowKeys, currentEditNewValues);
			OnRowDeleting (args);

			if (args.Cancel || !IsBoundUsingDataSourceID)
				return;
			
			RequireBinding ();
			DataSourceView view = GetData ();
			if (view != null)
				view.Delete (currentEditRowKeys, currentEditNewValues, new DataSourceViewOperationCallback (DeleteCallback));
			else {
				GridViewDeletedEventArgs dargs = new GridViewDeletedEventArgs (0, null, currentEditRowKeys, currentEditNewValues);
				OnRowDeleted (dargs);
			}
		}

		bool DeleteCallback (int recordsAffected, Exception exception)
		{
			GridViewDeletedEventArgs dargs = new GridViewDeletedEventArgs (recordsAffected, exception, currentEditRowKeys, currentEditNewValues);
			OnRowDeleted (dargs);
			return dargs.ExceptionHandled;
		}
		
		void EndRowEdit ()
		{
			EditIndex = -1;
			oldEditValues = new DataKey (new OrderedDictionary ());
			currentEditRowKeys = null;
			currentEditOldValues = null;
			currentEditNewValues = null;
		}

		protected internal override void LoadControlState (object savedState)
		{
			if (savedState == null)
				return;
			object[] state = (object[]) savedState;
			base.LoadControlState (state[0]);
			pageIndex = (int) state[1];
			selectedIndex = (int) state[2];
			editIndex = (int) state[3];
			sortExpression = (string) state[4];
			sortDirection = (SortDirection) state[5];
			DataKeyNames = (string []) state [6];
			if (state [7] != null)
				LoadDataKeyArrayState ((object []) state [7], out keys);
			if (state [8] != null)
				((IStateManager) OldEditValues).LoadViewState (state [8]);
			pageCount = (int)state [9];
			if (state [10] != null)
				ClientIDRowSuffix = (string[]) state [10];
			if (state [11] != null)
				LoadDataKeyArrayState ((object []) state [11], out rowSuffixKeys);
		}
		
		protected internal override object SaveControlState ()
		{
			if (EnableSortingAndPagingCallbacks) {
				Page page = Page;
				ClientScriptManager scriptManager = page != null ? page.ClientScript : null;

				if (scriptManager != null) {
					scriptManager.RegisterHiddenField (ClientID + "_Page", PageIndex.ToString ());
					scriptManager.RegisterHiddenField (ClientID + "_SortExpression", SortExpression);
					scriptManager.RegisterHiddenField (ClientID + "_SortDirection", ((int)SortDirection).ToString());
				}
			}

			object bstate = base.SaveControlState ();
			return new object [] {
				bstate, 
				pageIndex, 
				selectedIndex, 
				editIndex, 
				sortExpression, 
				sortDirection, 
				DataKeyNames,
				SaveDataKeyArrayState (keys),
				(oldEditValues == null ? null : ((IStateManager)oldEditValues).SaveViewState ()),
				pageCount,
				ClientIDRowSuffix,
				SaveDataKeyArrayState (rowSuffixKeys)
			};
		}

		object [] SaveDataKeyArrayState (DataKeyArray keys)
		{
			if (keys == null)
				return null;

			object [] state = new object [keys.Count];
			for (int i = 0; i < keys.Count; i++)
				state [i] = ((IStateManager) keys [i]).SaveViewState ();
			return state;
		}

		void LoadDataKeyArrayState (object [] state, out DataKeyArray keys)
		{
			List <DataKey> dataKeyList = DataKeyList;
			string[] dataKeyNames = DataKeyNames;
			int dataKeyNamesLen = dataKeyNames.Length;
			
			for (int i = 0; i < state.Length; i++) {
				DataKey dataKey = new DataKey (new OrderedDictionary (dataKeyNamesLen), dataKeyNames);
				((IStateManager) dataKey).LoadViewState (state [i]);
				dataKeyList.Add (dataKey);
			}
			keys = new DataKeyArray (dataKeyList);
		}

		protected override void TrackViewState ()
		{
			base.TrackViewState();
			if (columns != null)
				((IStateManager)columns).TrackViewState();
			if (pagerSettings != null)
				((IStateManager)pagerSettings).TrackViewState();
			if (alternatingRowStyle != null)
				((IStateManager)alternatingRowStyle).TrackViewState();
			if (footerStyle != null)
				((IStateManager)footerStyle).TrackViewState();
			if (headerStyle != null)
				((IStateManager)headerStyle).TrackViewState();
			if (pagerStyle != null)
				((IStateManager)pagerStyle).TrackViewState();
			if (rowStyle != null)
				((IStateManager)rowStyle).TrackViewState();
			if (selectedRowStyle != null)
				((IStateManager)selectedRowStyle).TrackViewState();
			if (editRowStyle != null)
				((IStateManager)editRowStyle).TrackViewState();
			if (emptyDataRowStyle != null)
				((IStateManager)emptyDataRowStyle).TrackViewState();
			if (sortedAscendingCellStyle != null)
				((IStateManager)sortedAscendingCellStyle).TrackViewState ();
			
			if (sortedAscendingHeaderStyle != null)
				((IStateManager)sortedAscendingHeaderStyle).TrackViewState ();
			
			if (sortedDescendingCellStyle != null)
				((IStateManager)sortedDescendingCellStyle).TrackViewState ();
			
			if (sortedDescendingHeaderStyle != null)
				((IStateManager)sortedDescendingHeaderStyle).TrackViewState ();
			if (rowSuffixKeys != null)
				((IStateManager) rowSuffixKeys).TrackViewState ();
			if (keys != null)
				((IStateManager)keys).TrackViewState();
			if (autoFieldProperties != null) {
				foreach (IStateManager sm in autoFieldProperties)
					sm.TrackViewState ();
			}
		}

		protected override object SaveViewState ()
		{
			object[] autoFieldsData = null;
			
			if (autoFieldProperties != null) {
				object[] data = new object [autoFieldProperties.Length];
				bool allNull = true;
				for (int n = 0; n < data.Length; n++) {
					data [n] = ((IStateManager)autoFieldProperties [n]).SaveViewState ();
					if (data [n] != null)
						allNull = false;
				}
				if (!allNull)
					autoFieldsData = data;
			}
			
			object[] states = {
				base.SaveViewState(), // 0
				(columns == null ? null : ((IStateManager)columns).SaveViewState()), // 1
				(pagerSettings == null ? null : ((IStateManager)pagerSettings).SaveViewState()), // 2
				(alternatingRowStyle == null ? null : ((IStateManager)alternatingRowStyle).SaveViewState()), // 3
				(footerStyle == null ? null : ((IStateManager)footerStyle).SaveViewState()), // 4
				(headerStyle == null ? null : ((IStateManager)headerStyle).SaveViewState()), // 5
				(pagerStyle == null ? null : ((IStateManager)pagerStyle).SaveViewState()), // 6
				(rowStyle == null ? null : ((IStateManager)rowStyle).SaveViewState()), // 7
				(selectedRowStyle == null ? null : ((IStateManager)selectedRowStyle).SaveViewState()), // 8
				(editRowStyle == null ? null : ((IStateManager)editRowStyle).SaveViewState()), // 9
				(emptyDataRowStyle == null ? null : ((IStateManager)emptyDataRowStyle).SaveViewState()), // 10
				autoFieldsData, // 11
				sortedAscendingCellStyle == null ? null : ((IStateManager)sortedAscendingCellStyle).SaveViewState (), // 12
				sortedAscendingHeaderStyle == null ? null : ((IStateManager)sortedAscendingHeaderStyle).SaveViewState (), // 13
				sortedDescendingCellStyle == null ? null : ((IStateManager)sortedDescendingCellStyle).SaveViewState (), // 14
				sortedDescendingHeaderStyle == null ? null : ((IStateManager)sortedDescendingHeaderStyle).SaveViewState () // 15
			};

			for (int i = states.Length - 1; i >= 0; i--) {
				if (states [i] != null)
					return states;
			}

			return null;
		}

		protected override void LoadViewState (object savedState)
		{
			if (savedState == null) {
				base.LoadViewState (null);
				return;
			}

			object [] states = (object []) savedState;
			
			if (states[11] != null) {
				object[] data = (object[]) states [11];
				autoFieldProperties = new AutoGeneratedFieldProperties [data.Length];
				for (int n=0; n<data.Length; n++) {
					IStateManager p = new AutoGeneratedFieldProperties ();
					p.TrackViewState ();
					p.LoadViewState (data [n]);
					autoFieldProperties [n] = (AutoGeneratedFieldProperties) p;
				}
			}

			base.LoadViewState (states[0]);
			
			if (states[1] != null)
				((IStateManager)Columns).LoadViewState (states[1]);
			if (states[2] != null)
				((IStateManager)PagerSettings).LoadViewState (states[2]);
			if (states[3] != null)
				((IStateManager)AlternatingRowStyle).LoadViewState (states[3]);
			if (states[4] != null)
				((IStateManager)FooterStyle).LoadViewState (states[4]);
			if (states[5] != null)
				((IStateManager)HeaderStyle).LoadViewState (states[5]);
			if (states[6] != null)
				((IStateManager)PagerStyle).LoadViewState (states[6]);
			if (states[7] != null)
				((IStateManager)RowStyle).LoadViewState (states[7]);
			if (states[8] != null)
				((IStateManager)SelectedRowStyle).LoadViewState (states[8]);
			if (states[9] != null)
				((IStateManager)EditRowStyle).LoadViewState (states[9]);
			if (states[10] != null)
				((IStateManager)EmptyDataRowStyle).LoadViewState (states[10]);
			if (states [12] != null)
				((IStateManager)sortedAscendingCellStyle).LoadViewState (states [12]);
			if (states [13] != null)
				((IStateManager)sortedAscendingHeaderStyle).LoadViewState (states [13]);
			if (states [14] != null)
				((IStateManager)sortedDescendingCellStyle).LoadViewState (states [14]);
			if (states [15] != null)
				((IStateManager)sortedDescendingHeaderStyle).LoadViewState (states [15]);
		}
		
		void ICallbackEventHandler.RaiseCallbackEvent (string eventArgument)
		{
			RaiseCallbackEvent (eventArgument);
		}
		
		protected virtual void RaiseCallbackEvent (string eventArgument)
		{
			string[] clientData = eventArgument.Split ('|');
			PageIndex = int.Parse (clientData[0]);
			SortExpression = HttpUtility.UrlDecode (clientData [1]);
			SortDirection = (SortDirection) int.Parse (clientData [2]);
			
			RaisePostBackEvent (clientData[3]);
			DataBind ();
		}
		
		string ICallbackEventHandler.GetCallbackResult ()
		{
			return GetCallbackResult ();
		}

		protected virtual string GetCallbackResult ()
		{
			PrepareControlHierarchy ();
			
			StringWriter sw = new StringWriter ();
			sw.Write (PageIndex.ToString () + '|' + SortExpression + '|' + (int) SortDirection + '|');

			HtmlTextWriter writer = new HtmlTextWriter (sw);
			RenderGrid (writer);
			return sw.ToString ();
		}

		string ICallbackContainer.GetCallbackScript (IButtonControl buttonControl, string argument)
		{
			return GetCallbackScript (buttonControl, argument);
		}
		
		protected virtual string GetCallbackScript (IButtonControl buttonControl, string argument)
		{
			if (EnableSortingAndPagingCallbacks) {
				Page page = Page;
				if (page != null)
					page.ClientScript.RegisterForEventValidation (UniqueID, argument);
				return "javascript:GridView_ClientEvent (\"" + ClientID + "\",\"" + buttonControl.CommandName + "$" + buttonControl.CommandArgument + "\"); return false;";
			} else
				return null;
		}
		
		protected override void OnPagePreLoad (object sender, EventArgs e)
		{
			base.OnPagePreLoad (sender, e);
			Page page = Page;
			
			if (page != null && page.IsPostBack && EnableSortingAndPagingCallbacks) {
				HttpRequest req = page.Request;
				int value;
				if (req != null) {
					if (int.TryParse (req.Form [ClientID + "_Page"], out value))
						PageIndex = value;

					if (int.TryParse (req.Form [ClientID + "_SortDirection"], out value))
						SortDirection = (SortDirection) value;
					SortExpression = req.Form [ClientID + "_SortExpression"];
				}
			}
		}

		const string onPreRenderScript = @"var {0} = new Object ();
{0}.pageIndex = {1};
{0}.sortExp = {2};
{0}.sortDir = {3};
{0}.uid = {4};
{0}.form = {5};
";
		protected internal override void OnPreRender (EventArgs e)
		{
			base.OnPreRender (e);

			if (EnableSortingAndPagingCallbacks) {
				Page page = Page;
				ClientScriptManager scriptManager = page != null ? page.ClientScript : null;
				if (scriptManager != null) {
					if (!scriptManager.IsClientScriptIncludeRegistered (typeof(GridView), "GridView.js")) {
						string url = scriptManager.GetWebResourceUrl (typeof(GridView), "GridView.js");
						scriptManager.RegisterClientScriptInclude (typeof(GridView), "GridView.js", url);
					}
				
					string cgrid = ClientID + "_data";
					string script = String.Format (onPreRenderScript,
								       cgrid,
								       ClientScriptManager.GetScriptLiteral (PageIndex),
								       ClientScriptManager.GetScriptLiteral (SortExpression == null ? String.Empty : SortExpression),
								       ClientScriptManager.GetScriptLiteral ((int) SortDirection),
								       ClientScriptManager.GetScriptLiteral (UniqueID),
								       page.theForm);
				
					scriptManager.RegisterStartupScript (typeof(TreeView), this.UniqueID, script, true);
				
					// Make sure the basic script infrastructure is rendered
					scriptManager.GetCallbackEventReference (this, "null", String.Empty, "null");
					scriptManager.GetPostBackClientHyperlink (this, String.Empty);
				}
			}
		}
		
		protected internal override void Render (HtmlTextWriter writer)
		{
			PrepareControlHierarchy ();
			if (EnableSortingAndPagingCallbacks)
				writer.AddAttribute (HtmlTextWriterAttribute.Id, ClientID + "_div");
			writer.RenderBeginTag (HtmlTextWriterTag.Div);

			RenderGrid (writer);

			writer.RenderEndTag ();
		}
		
		void RenderGrid (HtmlTextWriter writer)
		{
			if (table == null)
				return;

			table.Render (writer);
		}

		PostBackOptions IPostBackContainer.GetPostBackOptions (IButtonControl control)
		{
			if (control == null)
				throw new ArgumentNullException ("control");
			
			if (control.CausesValidation)
				throw new InvalidOperationException ("A button that causes validation in GridView '" + ID + "' is attempting to use the container GridView as the post back target.  The button should either turn off validation or use itself as the post back container.");
			
			PostBackOptions options = new PostBackOptions (this);
			options.Argument = control.CommandName + "$" + control.CommandArgument;
			options.RequiresJavaScriptProtocol = true;

			return options;
		}
	}
}
