#include "adns_rr.h"
using namespace System::Net;


namespace ADNS {

	/*
	All RRs have the same top level format shown below:

										1  1  1  1  1  1
		  0  1  2  3  4  5  6  7  8  9  0  1  2  3  4  5
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                                               |
		/                                               /
		/                      NAME                     /
		|                                               |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                      TYPE                     |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                     CLASS                     |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                      TTL                      |
		|                                               |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                   RDLENGTH                    |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--|
		/                     RDATA                     /
		/                                               /
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+

*/		

	String^ ResourceRecord::PrintHeader()
	{
		String^ output;

		output = owner->Print();
		output += " " + Convert::ToString(ttl);
		output += " " + Enum::GetName(RR_CLASS::typeid,rr_class);
		output += " " + Enum::GetName(RR_TYPE::typeid,rr_type);
		output += " ";

		return output;

	}

	array<Byte>^ ResourceRecord::HeaderToWire()
	{
		array<Byte>^ output;
		array<Byte>^ owner_name;

		owner_name = owner->GetName();
		output = gcnew array<Byte>(owner_name->Length + 10); // 2 each for type,class,rdlength 4 for ttl
		owner_name->CopyTo(output,0);

		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) rr_type))->CopyTo(output,owner_name->Length);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) rr_class))->CopyTo(output,owner_name->Length + 2);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) ttl))->CopyTo(output,owner_name->Length + 4);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) rdata->Length))->CopyTo(output,owner_name->Length + 8);

		return output;

	}

	array<Byte>^ ResourceRecord::ToWire()
	{
		array<Byte>^ output;
		int outputlen = 0;

		outputlen += HeaderToWire()->Length;

		if (rdata)
			outputlen += rdata->Length;

		output = gcnew array<Byte>(outputlen);
		HeaderToWire()->CopyTo(output,0);

		if (rdata)
			rdata->CopyTo(output,HeaderToWire()->Length);

		return output;
	}

	//Note:  This assumes that each resource record in the set is already in canonical form.
	//Also Note:  According to RFC 4034, all Owners, Types, and Classes must be IDENTICAL in all
	//members of the set for it to be valid.  Therefore all we're really doing at this point is
	//sorting by RDATA.

	array<int>^ ResourceRecord::SortRRSetCanonicalByIndicies(List<int>^ indicies, List<ResourceRecord^>^ l)
	{
	

		int i;
		bool swapped = true;
		int n;
		int v;
		array<int>^ index;


		index = indicies->ToArray();
		n = index->Length;
		while(swapped)
		{
			swapped = false;
			for (i = 0; i < n - 1; ++i)
			{		
				if (l[i]->RdataIsLess(l[index[i+1]],l[index[i]]))
				{ 
					v = index[i];
					index[i] = index[i+1];
					index[i+1] = v;
					swapped = true;
				}
			}
			n--;
		}

		return index;
	}

	//NOTE:  By RFC 2181, no RRSet can contain duplicate records, so we don't worry about that here, and if it occurs, return false. (equal isn't less, after all)

	//Returns true if r1 < r2 by RFC 4034 section 6.3

	bool ResourceRecord::RdataIsLess(ResourceRecord^ r1, ResourceRecord^ r2)
	{
		int minlen;
		int i = 0;

		if ((r1 == nullptr) && (r2 != nullptr))
			return true;
		if ((r1 == nullptr) || (r2 == nullptr))
			return false;

		if (r1->rdata->Length < r2->rdata->Length)
			minlen = r1->rdata->Length;
		else
			minlen = r2->rdata->Length;

		for (i = 0; i < minlen; ++i)
		{
			if (r1->rdata[i] < r2->rdata[i])
				return true;
			if (r1->rdata[i] > r2->rdata[i])
				return false;
		}

		//if we're here, check lengths.

		if (r1->rdata->Length < r2->rdata->Length)
			return true;

		//at this point, either r2 is shorter, or they're identical.  Either way, we return false.
		return false;
	}


}